#
# Copyright (C) 2003,2004,2006 Dmitry Fedorov <dm.fedorov@gmail.com>
#
# This file is part of Offmirror.
#
# Offmirror is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Offmirror is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Offmirror; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA

package OffMirror::FileList;

require 5.004;
use strict;
local $^W=1; # use warnings only since 5.006

use integer;
use Cwd 'abs_path';
use File::Find;

use OffMirror::FileAttrRecord;


use constant FORMAT_VERSION => scalar 1;

# printf(fmt, hostname, basedir[, digest]);
use constant HEADER_FMT     => scalar 'header: version='.FORMAT_VERSION.' hostdir=%s:%s';
use constant FOOTER_FMT     => scalar 'footer: version='.FORMAT_VERSION.' hostdir=%s:%s digest=%s';

# $1: version, $2: hostname, $3: basedir, $4: digest
use constant HDRFTR_RE      => scalar ':\s+version=(\d+)\s+hostdir=(.*):(.+)';
use constant HEADER_RE      => scalar '^header'.HDRFTR_RE.'$';
use constant FOOTER_RE      => scalar '^footer'.HDRFTR_RE.'\s+digest=(\w+)$';


BEGIN
{
    use vars qw($VERSION);
    $VERSION = do { my @r = (q$Revision: 0.36 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r };

    #+ use MD5 || Digest::MD5
    use vars qw( $md5_module );
    my @md5_modules = qw(Digest::MD5 MD5);
    foreach my $nm ( @md5_modules )
    {
	eval "use $nm";
	if (! $@)
	{
            $md5_module = $nm;
            last;
        }
    }
    die "Cannot find any MD5 module from: @md5_modules\n" if $@;
    #warn "md5_module: $md5_module\n";
    #- use MD5 || Digest::MD5
}


my $_make_header_footer = sub
{
    my ($fmt, $hostname, $basedir, $digest) = @_;

    return sprintf $fmt, $hostname, $basedir, $digest;
};

my $make_header = sub
{
    return &$_make_header_footer(HEADER_FMT, @_);
};

my $make_footer = sub
{
    return &$_make_header_footer(FOOTER_FMT, @_);
};



# ($version, $hostname, $basedir, $digest) = _parse_header_footer($re, $line)
# or (undef,undef,undef,undef)
my $_parse_header_footer = sub
{
    my ($re, $line) = @_;
    my ($version, $hostname, $basedir, $digest);

    $line =~ m/$re/		&&
	return ($1, $2, OffMirror::FileAttrRecord::decode_file_name($3), $4);

    return undef;
};

# ($version, $hostname, $basedir) = parse_header($line)
# or (undef,undef,undef)
my $parse_header = sub
{
    return $_parse_header_footer->(HEADER_RE, $_[0]);
};

# ($version, $hostname, $basedir, $digest) = parse_footer($line)
# or (undef,undef,undef,undef)
my $parse_footer = sub
{
    return $_parse_header_footer->(FOOTER_RE, $_[0]);
};


my $read_line = sub
{
    local *FH = shift;

    while (my $line = <FH>)
    {
	chomp $line;
	next if length($line) == 0;

	return $line;
    }
};


# generate_list( $basedir, \@exclude_list, $md5sums, $pretty )
# generate and print list to stdout
sub generate_list($$$$)
{
    my ($basedir, $exclude_list, $md5sums, $pretty) = @_;

    $basedir =~ s:(.*)/$:$1:;	# strip '/' from end
    $basedir = abs_path($basedir) or
	die "can't get abs path of $basedir: $!";

    my $host = `hostname` or warn "can't run hostname: $!";
    chomp $host;

    print ( &$make_header($host, $basedir) . "\n" );

    my $md5 = $md5_module->new;

    my $onfile = sub ()
    {
	my $shortname = $_;

	my $fname = "$File::Find::name";
	return if $fname =~ m:^$basedir$:o;

	$fname =~ s:^$basedir/::so; # strip basedir

	map { return if $fname =~ m/$_/ } (@$exclude_list);

	my $r = OffMirror::FileAttrRecord::stat_file($shortname, $md5sums);

	if ( $r->field('type') eq '?' )
	{
	    print STDERR "unknown type, skip file $fname\n";
	}

	my $line = $r->make_record_line($fname, 'l', $pretty);
	$md5->add($line);
	print ($line."\n");
    };

    find($onfile, $basedir);

    print ( &$make_footer($host, $basedir, $md5->hexdigest()) . "\n" );
}


# ctor
# $objref = OffMirror::FileList::new( *FH )
sub new($)
{
    my $self =
    {
	'fh'	=> shift,
	'footer'=> 0,
	'host'	=> undef,
	'dir'	=> undef,
	'md5'	=> $md5_module->new,
	'list'	=> {},
    };

    return bless($self, __PACKAGE__);
}


# ctor
# $objref = OffMirror::FileList::new_from_file( $filename )
my $new_from_file = sub ($)
{
    my $file = shift;

    $file =~ s/^(.*\.(gz|z|Z|zip))$/gzip -dc < $1|/
	or
    $file =~ s/^(.*\.(bz2|bz))$/bzip2 -dc < $1|/
	or
    $file =~ s/^(.*)$/< $1/;

    local *LIST_IN;
    open   LIST_IN, "$file"
	or die "Can't open file list $file: $!";

    return new( *LIST_IN );
};


sub parse_first($)
{
    my $this = shift;
    my $line = &$read_line( $this->{'fh'} )
	or die "can't read header at line $., stopped";

    my ($hdr_version, $hdr_hostname, $hdr_basedir) = &$parse_header($line);
    die "invalid header format at line $., stopped"
	unless defined $hdr_version;
    die "unsupported format version $hdr_version in header"
	." at line $., stopped"
	if $hdr_version != FORMAT_VERSION;

    $this->{'footer'} = 0;
    $this->{'host'} = $hdr_hostname;
    $this->{'dir' } = $hdr_basedir;
    $this->{'list'} = {};
}


my $record_re = OffMirror::FileAttrRecord::RECORD_RE;
my $footer_re = FOOTER_RE;

# returns ($r, $fname)
sub parse_line($)
{
    my $this  = shift;

    my $line = &$read_line( $this->{'fh'} );
    unless (defined $line)
    {
	die "no footer record found, stopped"
	    unless $this->{'footer'};

	return (undef,undef);
    }

    if ( $line =~ m/$record_re/o )
    {
	$this->{'md5'}->add($line);

	my ($r, $fname) = OffMirror::FileAttrRecord::parse_record_line($line);

	return ($r, $fname);
    }
    elsif ( $line =~ m/$footer_re/o )
    {
	my ($ftr_version, $ftr_hostname, $ftr_basedir, $ftr_digest) =
	    &$parse_footer($line);

	die "invalid footer format at line $., stopped"
	    unless defined $ftr_version;
	die "unsupported format version $ftr_version in footer"
	    ." at line $., stopped" if $ftr_version != FORMAT_VERSION;
	die "header/footer host:dir mismatch at line $., stopped"
	    if	$ftr_hostname ne $this->{'host'} or
		$ftr_basedir  ne $this->{'dir'};
	die "MD5 digest mismatch at line $., stopped"
	    if $ftr_digest ne $this->{'md5'}->hexdigest();


	return (undef,undef);
    }
    else
    {
	die "unrecognized record format at line $., stopped";
    }
}


sub parse_list($)
{
    my $this  = shift;

    $this->parse_first();

    my ($r, $fname);
    while ( (($r, $fname) = $this->parse_line()), defined $r )
    {
	$this->{'list'}->{$fname} = $r;
    }
}


# ctor
# $objref = OffMirror::FileList::list_from_file( $filename )
sub list_from_file($)
{
    my $file = shift;

    my $this = &$new_from_file($file);
    $this->parse_list();
    return $this;
}


# $basedir = basedir_from_file($file)
sub basedir_from_file($)
{
    my $list = &$new_from_file(shift);
    $list->parse_first();
    my $basedir = $list->{'dir'};
    undef $list;

    return $basedir;
}


# ($src_list_ref, $dst_list_ref, $pretty)
sub diff_list($$$)
{
    my $srcobj = shift; my $srclist = $srcobj->{'list'};
    my $dstobj = shift; my $dstlist = $dstobj->{'list'};
    my $pretty = shift;

    print ( &$make_header( $dstobj->{'host'}, $dstobj->{'dir'}) . "\n" );
    my $md5 = $md5_module->new;

    my $print_line = sub
    {
	my ($rec, $fname, $action ) = @_;

	my $line = $rec->make_record_line($fname, $action, $pretty);
	$md5->add($line);
	print ( $line . "\n");
    };

    # what to mirror
    while ( (my $fname, my $srcrec) = each(%$srclist) )
    {
	if ( $srcrec->field('type') eq '?' )
	{
	    print STDERR "unknown type, skip file $fname\n";
	    next;
	}

	if ( ! exists $dstlist->{$fname} )
	{   # copy
	    &$print_line($srcrec, $fname, 'c');
	}
	else
	{
	    my $cmp = OffMirror::FileAttrRecord::compare_records(
					$srcrec, $dstlist->{$fname} );

	    next if $cmp eq '=';

	    &$print_line($srcrec, $fname, $cmp);
	}
    }


    # what to delete
    while ( (my $fname, my $dstrec) = each(%$dstlist) )
    {
	if ( $dstrec->field('type') eq '?' or ! exists $srclist->{$fname} )
	{
	    &$print_line($dstrec, $fname, 'd');
	}
    }


    print( &$make_footer($dstobj->{'host'}, $dstobj->{'dir'}, $md5->hexdigest())
	  . "\n" );
}


1;
