#!/usr/bin/perl -w
#
# Copyright (C) 2003 Dmitry Fedorov <fedorov@inp.nsk.su>
#
# This file is part of Offmirror.
#
# Offmirror is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Offmirror is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Offmirror; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


=head1 NAME

offmirror-chattr - change attributes of files on destination side.

=head1 SYNOPSIS

offmirror-chattr [options]

=head1 DESCRIPTION

This program is part of files tree offline mirroring utility package,
which is used to synchronize file trees without any direct connection
between them.

This program reads diff list from stdin and delete files/dirs, creates dirs,
changes attributes of files on destination side.

=cut

require 5.003;
use strict;
use Getopt::Long;

use Errno qw(EEXIST ENOENT);

use File::Stat::ModeString;

use FindBin;
use lib "$FindBin::Bin";
use OffMirror::FileList;
use OffMirror::FileAttrRecord;
use OffMirror::ChangeAttr;


my $progname = 'offmirror-chattr';

sub usage
{
	warn "\n".join(" ", @_)."\n" if @_;
	warn <<EOF;

Usage: $progname	[--verbose] [--dry-run]		\
			[--force-{chown,chmod,mtime}]	\
			[--help]

For details, see man page.
EOF
	exit(1);
}


=head1 OPTIONS

=over 4

=item --verbose

Show what to do.

=item --dry-run

Simulate.

=item --force-chown

Always do chown and chgrp.

=item --force-chmod

Always do chmod.

=item --force-mtime

Always set mtime.

=item --help

Display a usage summary.

=back

=head1 SEE ALSO

L<offmirror(1)>

L<offmirror-untar(1)>

=cut


my $verbose=0;
my $dry_run=0;
my $force_chown=0;
my $force_chmod=0;
my $force_mtime=0;
my $help=0;

GetOptions(
	'verbose'	=> \$verbose,
	'dry-run'	=> \$dry_run,
	'force-chown'	=> \$force_chown,
	'force-chmod'	=> \$force_chmod,
	'force-mtime'	=> \$force_mtime,
	'help'		=> \$help
) or usage;

usage if $help;

shift @ARGV and usage("extra parameter specified");

my $list = OffMirror::FileList->new(*STDIN);
$list->parse_list();

my $basedir = $list->{'dir'};


#+ select
my (@delete_files, @delete_dirs, @create_dirs);

while ( my ($fname,$r) = each %{$list->{'list'}} )
{
    if ( $r->field('action') eq 'd' )
    {
	if ( $r->field('type') eq 'd' )
	{
	    push @delete_dirs, $fname;
	}
	else
	{
	    push @delete_files, $fname;
	}
    }
    elsif ( $r->field('action') eq 'c' and $r->field('type') eq 'd' )
    {
	push @create_dirs, $fname;
    }
}


# delete nested first
@delete_dirs = sort { length($main::b) <=> length($main::a) } @delete_dirs;

# create nested last
@create_dirs = sort { length($main::a) <=> length($main::b) } @create_dirs;

#- select


#+ delete files
foreach my $fname (@delete_files)
{
    my $fullname = $basedir.'/'.$fname;

    print STDERR "delete file $fullname\n" if $verbose;

    unless ( $dry_run )
    {
	warn "can't unlink $fullname: $!"
	    if unlink($fullname) != 1 and $! != ENOENT;
    }

    delete $list->{'list'}->{$fname};
}

print "\n" if $verbose and scalar(@delete_files)!=0;

undef @delete_files;
#- delete files


#+ delete directories
foreach my $dname (@delete_dirs)
{
    my $fullname = $basedir.'/'.$dname;

    print STDERR "delete dir  $fullname\n" if $verbose;

    unless ( $dry_run )
    {
	warn "can't rmdir $fullname: $!"
	    if !rmdir($fullname) and $! != ENOENT;
    }

    delete $list->{'list'}->{$dname};
}

print "\n" if $verbose and scalar(@delete_dirs)!=0;

undef @delete_dirs;
#- delete directories


umask 0; # to allow to create files/dirs with any permissions


#+ create directories
foreach my $dname (@create_dirs)
{
    my $fullname = $basedir.'/'.$dname;

    print STDERR "create dir  $fullname\n" if $verbose;

    unless ( $dry_run )
    {
	my $mode = string_to_mode( $list->{'list'}->{$dname}->field('mode') );

	my $rc = mkdir $fullname, $mode;
	if ( !$rc and $! != EEXIST)
	{
	    warn "can't mkdir $fullname: $!";
	}
    }
}

print "\n" if $verbose and scalar(@create_dirs)!=0;

undef @create_dirs;
#- create directories


sub chattr($)
{
    my $hashref = shift;

    while ( my ($fname,$r) = each %{$hashref} )
    {
	my $fullname = $basedir.'/'.$fname;

	my $r_action = $r->field('action');

	if ( $r_action eq 'a' or ($r_action eq 'c' and $r->field('type') eq 'd') )
	{
	    my $locr = OffMirror::FileAttrRecord->stat_file($fullname);

	    # chown

	    my $user = $r->field('user');
	    if ( $force_chown or
			($> == 0 and $user ne $locr->field('user')) )
	    {
		print STDERR "chown $user $fname\n" if $verbose;
		my_chown($fullname, $user) unless ($dry_run);
	    }

	    my $group = $r->field('group');
	    if ( $force_chown or
			($> == 0 and $group ne $locr->field('group')) )
	    {
		print STDERR "chgrp $group $fname\n" if $verbose;
		my_chgrp($fullname, $group) unless ($dry_run);
	    }

	    # chmod

	    my $mode = $r->field('mode');
	    if ( $force_chmod or $mode ne $locr->field('mode') )
	    {
		print STDERR "chmod $mode $fname\n" if $verbose;
		my_chmod($fullname, $mode) unless ($dry_run);
	    }

	    # utime

	    my $mtime = $r->field('mtime');
	    if ( $force_mtime or $mtime ne $locr->field('mtime') )
	    {
		print STDERR "set mtime $mtime $fname\n" if $verbose;
		my_utime($fullname, $mtime) unless ($dry_run);
	    }
	}
    }
}


chattr($list->{'list'});


exit 0;


=head1 AUTHOR

Dmitry Fedorov <fedorov@inp.nsk.su>

=head1 COPYRIGHT

Copyright (C) 2003 Dmitry Fedorov <fedorov@inp.nsk.su>

=head1 LICENSE

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License,
or (at your option) any later version.

=head1 DISCLAIMER

The author disclaims any responsibility for any mangling of your system
etc, that this script may cause.

=cut

