/*  GNU Ocrad - Optical Character Recognition program
    Copyright (C) 2003, 2004 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <cctype>
#include <cstdio>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "vrhomboid.h"
#include "track.h"
#include "ucs.h"
#include "block.h"
#include "character.h"
#include "textline.h"


int Textline::find_space( int begin ) const throw()
  {
  int end = begin;
  for( ; end < characters(); ++end )
    {
    const Character & c = character( end );
    if( c.guesses() && UCS::isspace( c.guess(0).code ) ) break;
    }
  return end;
  }


Textline::Textline( const Textline & l ) throw()
  : data( l.data ), track( l.track )
  {
  if( l._big_initial ) _big_initial = new Character( *(l._big_initial) );
  else _big_initial = 0;
  }


Textline & Textline::operator=( const Textline & l ) throw()
  {
  if( this != &l )
    {
    delete_big_initial();
    if( l._big_initial ) _big_initial = new Character( *(l._big_initial) );
    data = l.data; track = l.track;
    }
  return *this;
  }


void Textline::initialize_track() throw()
  {
  std::vector< Rectangle > rv;
  for( unsigned int i = 0; i < data.size(); ++i ) rv.push_back( data[i] );
  track.initialize( rv );
  }


void Textline::big_initial( const Character & c ) throw()
  {
  if( !_big_initial ) _big_initial = new Character( c );
  else *_big_initial = c;
  }


void Textline::delete_big_initial() throw()
  {
  if( _big_initial ) { delete _big_initial; _big_initial = 0; }
  }


void Textline::delete_character( int i ) throw()
  {
  if( i < 0 || i >= characters() )
    Ocrad::internal_error( "delete_character, index out of bounds" );
  data.erase( data.begin() + i );
  }


int Textline::shift_character( const Character & c ) throw()
  {
  int i = characters();

  while( i > 0 && c.h_precedes( data[i-1] ) ) --i;
  data.insert( data.begin() + i, c );
  return i;
  }


bool Textline::insert_space( int i ) throw()
  {
  if( i <= 0 || i >= characters() )
    Ocrad::internal_error( "insert_space, index out of bounds" );
  Character & c1 = data[i-1];
  Character & c2 = data[i];
  if( c2.left() - c1.right() <= 1 ) return false;
  int left = c1.right() + 1;
  int top = ( c1.top() + c2.top() ) / 2;
  int right = c2.left() - 1;
  int bottom = ( c1.bottom() + c2.bottom() ) / 2;
  Character c( Rectangle( left, top, right, bottom ), *c1.blockmap() );
  c.add_guess( ' ', 0 );
  data.insert( data.begin() + i, c );
  return true;
  }


Character & Textline::character( int i ) const throw()
  {
  if( i < 0 || i >= characters() )
    Ocrad::internal_error( "character, index out of bounds" );
  return data[i];
  }


int Textline::mean_height() const throw()
  {
  int sum = 0;

  for( int i = 0; i < characters(); ++i ) sum += data[i].height();
  if( characters() ) sum /= characters();
  return sum;
  }


int Textline::mean_width() const throw()
  {
  int sum = 0;

  for( int i = 0; i < characters(); ++i ) sum += data[i].width();
  if( characters() ) sum /= characters();
  return sum;
  }


int Textline::mean_gap_width() const throw()
  {
  int sum = 0;

  if( characters() < 2 ) return sum;
  for( int i = 1; i < characters(); ++i )
    if( data[i].left() > data[i-1].right() )
      sum += data[i].left() - data[i-1].right() - 1;
  return sum / ( characters() - 1 );
  }


int Textline::mean_hcenter() const throw()
  {
  int sum = 0;

  for( int i = 0; i < characters(); ++i ) sum += data[i].hcenter();
  if( characters() ) sum /= characters();
  return sum;
  }


int Textline::mean_vcenter() const throw()
  {
  int sum = 0;

  for( int i = 0; i < characters(); ++i ) sum += data[i].vcenter();
  if( characters() ) sum /= characters();
  return sum;
  }


void Textline::join( Textline & l ) throw()
  {
  if( l._big_initial &&
      ( !_big_initial || l._big_initial->h_precedes( *_big_initial ) ) )
    big_initial( *(l._big_initial) );

  for( int i = 0; i < l.characters(); ++i ) shift_character( l.data[i] );

  l.delete_big_initial(); l.data.clear();
  }


void Textline::print( const Control & control ) const throw()
  {
  if( _big_initial ) _big_initial->print( control );
  for( int i = 0; i < characters(); ++i ) character( i ).print( control );
  std::fputs( "\n", control.outfile );
  }


void Textline::dprint( const Control & control, bool graph, bool recursive )
								const throw()
  {
  if( graph || recursive )
    std::fprintf( control.outfile, "mean_height = %d, track.segments = %d\n",
                  mean_height(), track.segments() );

  if( _big_initial ) _big_initial->dprint( control, graph, recursive );
  for( int i = 0; i < characters(); ++i )
    {
    if( graph || recursive )
      std::fprintf( control.outfile, "track.vcenter = %d\n", track.vcenter( character(i).hcenter() ) );
    character( i ).dprint( control, graph, recursive );
    }
  std::fputs( "\n", control.outfile );
  }


void Textline::xprint( const Control & control ) const throw()
  {
  if( _big_initial ) _big_initial->xprint( control );
  for( int i = 0; i < characters(); ++i )
    character( i ).xprint( control );
  }


void Textline::recognize1( const Charset & charset ) const throw()
  {
  if( _big_initial )
    {
    _big_initial->recognize1( charset, _big_initial->vcenter() );
    if( _big_initial->guesses() == 1 )
      {
      int code = _big_initial->guess( 0 ).code;
      if( UCS::islower( code ) )
        _big_initial->only_guess( UCS::toupper( code ), 0 );
      }
    else if( _big_initial->guesses() > 1 ) _big_initial->clear_guesses();
    }

  for( int i = 0; i < characters(); ++i )
    {
    Character & c = character( i );
    c.recognize1( charset, track.vcenter( c.hcenter() ) );
    }
  }
