/*  GNU Ocrad - Optical Character Recognition program
    Copyright (C) 2003, 2004 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <cstdio>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "bitmap.h"
#include "block.h"
#include "blockmap.h"


namespace {

void ignore_anormal_blocks( std::vector< Block > & block_vector ) throw()
  {
  for( int i = block_vector.size() - 1; i >= 0; --i )
    {
    const Block & b = block_vector[i];
    if( b.height() > 35 * b.width() ||
        b.width() > 20 * b.height() ||
        b.id() < 0 )
      block_vector.erase( block_vector.begin() + i );
    }
  }


void ignore_small_blocks( std::vector< Block > & block_vector ) throw()
  {
  int to = 0, blocks = block_vector.size();
  for( int from = 0; from < blocks; ++from )
    {
    const Block & b = block_vector[from];
    if( b.id() > 0 )	// black block
      {
      if( b.height() > 4 || b.width() > 4 ||
          ( ( b.height() > 2 || b.width() > 2 ) && b.area() > 5 ) )
        block_vector[to++] = b;
      }
    else		// white block (hole)
      {
      if( b.height() > 4 || b.width() > 4 ||
          ( ( b.height() > 2 || b.width() > 2 ) && b.area() > 3 ) )
        block_vector[to++] = b;
      }
    }
  if( to < blocks )
    block_vector.erase( block_vector.begin() + to, block_vector.end() );
  }


void remove_top_bottom_noise( std::vector< Block > & block_vector ) throw()
  {
  int blocks = block_vector.size();
  for( int i = 0; i < blocks; ++i )
    {
    Block & b = block_vector[i];
    if( b.height() < 11 ) continue;
    const Blockmap & bm = *(b.blockmap());

    int c = 0;
    for( int col = b.left(); col <= b.right(); ++col )
      if( bm.id( b.top(), col ) == b.id() && ++c > 1 ) break;
    if( c <= 1 ) b.top( b.top() + 1 );

    c = 0;
    for( int col = b.left(); col <= b.right(); ++col )
      if( bm.id( b.bottom(), col ) == b.id() && ++c > 1 ) break;
    if( c <= 1 ) b.bottom( b.bottom() - 1 );
    }
  }


void remove_left_right_noise( std::vector< Block > & block_vector ) throw()
  {
  int blocks = block_vector.size();
  for( int i = 0; i < blocks; ++i )
    {
    Block & b = block_vector[i];
    if( b.width() < 6 ) continue;
    const Blockmap & bm = *(b.blockmap());

    int c = 0;
    for( int row = b.top(); row <= b.bottom(); ++row )
      if( bm.id( row, b.left() ) == b.id() && ++c > 1 ) break;
    if( c <= 1 ) b.left( b.left() + 1 );

    c = 0;
    for( int row = b.top(); row <= b.bottom(); ++row )
      if( bm.id( row, b.right() ) == b.id() && ++c > 1 ) break;
    if( c <= 1 ) b.right( b.right() - 1 );
    }
  }

} // end namespace


int Blockmap::generate_black_id() { static int id = 0; return ++id; }

int Blockmap::generate_white_id() { static int id = 0; return --id; }


void Blockmap::add_point_to_block( int row, int col, int id ) throw()
  {
  static int ib = -1, iw = -1;
  std::vector< Block > & block_vector = _block_matrix.back();
  int vblocks = block_vector.size();
  if( id > 0 && ib >= 0 && ib < vblocks && block_vector[ib].id() == id )
    { block_vector[ib].add_point( row, col ); return; }
  if( id < 0 && iw >= 0 && iw < vblocks && block_vector[iw].id() == id )
    { block_vector[iw].add_point( row, col ); return; }
  for( int c = 0, i = vblocks - 1; i >= 0 ; --i, ++c )
    if( block_vector[i].id() == id )
      {
      block_vector[i].add_point( row, col );
      if( id > 0 ) ib = i; else iw = i;
      if( c < _width ) return;
      Block temp = block_vector[i];
      block_vector.erase( block_vector.begin() + i );
      block_vector.push_back( temp );
      return;
      }
  Ocrad::internal_error( "add_point_to_block, lost block" );
  }


void Blockmap::create_block( const Rectangle & r, int id ) throw()
  {
  _block_matrix.back().push_back( Block( r, *this, id ) );
  }


void Blockmap::delete_block( int id ) throw()
  {
  std::vector< Block > & block_vector = _block_matrix.back();
  for( int i = block_vector.size() - 1; i >= 0 ; --i )
    {
    Block & b = block_vector[i];
    if( b.id() == id )
      {
      for( int row = b.top(); row <= b.bottom(); ++row )
        for( int col = b.left(); col <= b.right(); ++col )
          if( data[row][col] == id ) data[row][col] = 0;
      block_vector.erase( block_vector.begin() + i ); return;
      }
    }
  Ocrad::internal_error( "delete_block, lost block" );
  }


void Blockmap::join_blocks( int id1, int id2 ) throw()
  {
  if( std::abs( id1 ) > std::abs( id2 ) )
    { int temp = id1; id1 = id2; id2 = temp; }

  std::vector< Block > & block_vector = _block_matrix.back();
  int i1 = block_vector.size();
  int i2 = i1;
  while( --i1 >= 0 && block_vector[i1].id() != id1 );
  while( --i2 >= 0 && block_vector[i2].id() != id2 );
  if( i1 < 0 || i2 < 0 ) Ocrad::internal_error( "join_blocks, lost block" );

  Block & b1 = block_vector[i1];
  Block & b2 = block_vector[i2];
  for( int row = b2.top(); row <= b2.bottom(); ++row )
    for( int col = b2.left(); col <= b2.right(); ++col )
      if( data[row][col] == id2 ) data[row][col] = id1;
  b1.add_rectangle( b2 );
  block_vector.erase( block_vector.begin() + i2 );
  }


void Blockmap::ignore_wide_blocks() throw()
  {
  for( unsigned int cut = 0; cut < _block_matrix.size(); ++cut )
    {
    std::vector< Block > & block_vector = _block_matrix[cut];
    bool frame_found = false;
    for( unsigned int i = 0; i < block_vector.size(); )
      {
      if( 2 * block_vector[i].width() < _width ) { ++i; continue; }
      Block b( block_vector[i] );
      block_vector.erase( block_vector.begin() + i );
      if( b.id() > 0 && 4 * b.area() > b.size() )	// image, not frame
        {
        if( 10 * b.width() > 8 * _width && 10 * b.height() > 8 * _height )
          { block_vector.clear(); break; }
        for( int j = block_vector.size() - 1; j >= 0; --j )
          {
          if( b.includes( block_vector[j] ) )
            block_vector.erase( block_vector.begin() + j );
          else if( block_vector[j].top() < b.top() ) break;
          }
        }
      else frame_found = true;
      }
    if( frame_found )		// Make cuts from blocks inside deleted frame
      {
      int bottom = 0;
      for( unsigned int i = 0; i < _block_matrix[cut].size(); ++i )
        {
        const Block & b( _block_matrix[cut][i] );
        if( b.bottom() > bottom )
          {
          int old_bottom = bottom; bottom = b.bottom();
          if( b.top() > old_bottom && i > 0 )
            {
            std::vector< Block > new_block_vector( _block_matrix[cut].begin() + i, _block_matrix[cut].end() );
            _block_matrix[cut].erase( _block_matrix[cut].begin() + i, _block_matrix[cut].end() );
            _block_matrix.insert( _block_matrix.begin() + cut + 1, new_block_vector );
            ++cut; i = 0;
            }
          }
        }
      }
    }
  }


Blockmap::Blockmap( const Bitmap & page_image, int rindex, int debug_level )
                                                                     throw()
  {
  if( rindex < 0 || rindex >= page_image.rectangles() )
    Ocrad::internal_error( "Blockmap, text rectangle index out of range" );

  const Rectangle & r = page_image.rectangle_vector()[rindex];
  int row0 = r.top();
  int col0 = r.left();
  _height =  r.height();
  _width =   r.width();
  data.resize( _height );

  for( int row = 0; row < _height; ++row )
    { data[row].resize( _width ); data[row][0] = 0; }
  for( int col = 0; col < _width; ++col ) data[0][col] = 0;
  _block_matrix.resize( 1 );

  for( int row = 1; row < _height; ++row )
    {
    std::vector< int > & datarow = data[row];
    std::vector< int > & datarow1 = data[row-1];
    bool blank_row = true;
    for( int col = 1; col < _width; ++col )
      {
      if( page_image.get_bit( row0 + row, col0 + col ) )	// black point
        {
        blank_row = false;
        if( datarow1[col] > 0 )
          {
          int id = datarow[col] = datarow1[col];
          add_point_to_block( row, col, id );
          if( datarow[col-1] > 0 && datarow[col-1] != id )
            join_blocks( datarow[col-1], id );
          }
        else if( datarow[col-1] > 0 )
          {
          int id = datarow[col] = datarow[col-1];
          add_point_to_block( row, col, id );
          }
        else if( datarow1[col-1] > 0 )
          {
          int id = datarow[col] = datarow1[col-1];
          add_point_to_block( row, col, id );
          }
        else
          {
          int id = datarow[col] = generate_black_id();
          create_block( Rectangle( col, row, col, row ), id );
          }
        }
      else						// white point
        {
        if( datarow[col-1] == 0 )
          {
          datarow[col] = 0;
          if( datarow1[col] < 0 ) delete_block( datarow1[col] );
          }
        else if( datarow1[col] == 0 )
          {
          datarow[col] = 0;
          if( datarow[col-1] < 0 ) delete_block( datarow[col-1] );
          }
        else if( datarow1[col] < 0 )
          {
          int id = datarow[col] = datarow1[col];
          add_point_to_block( row, col, id );
          if( datarow[col-1] < 0 && datarow[col-1] != id )
            join_blocks( datarow[col-1], id );
          }
        else if( datarow[col-1] < 0 )
          {
          int id = datarow[col] = datarow[col-1];
          add_point_to_block( row, col, id );
          }
        else
          {
          int id = datarow[col] = generate_white_id();
          create_block( Rectangle( col, row, col, row ), id );
          id = datarow1[col];
          if( datarow[col-1] != id ) join_blocks( datarow[col-1], id );
          }
        }
      }
    if( blank_row && _block_matrix.back().size() )
      _block_matrix.push_back( std::vector< Block >() );
    }

  while( _block_matrix.size() && !_block_matrix.back().size() )
    _block_matrix.pop_back();

  if( debug_level <= 99 )
    {
    ignore_wide_blocks();
    for( unsigned int i = 0; i < _block_matrix.size(); ++i )
      {
      ignore_small_blocks( _block_matrix[i] );
      Block::hierarchize_blocks( _block_matrix[i] );
      ignore_anormal_blocks( _block_matrix[i] );
      remove_top_bottom_noise( _block_matrix[i] );
      remove_left_right_noise( _block_matrix[i] );
      }
    }

  for( int i = _block_matrix.size() - 1; i >= 0; --i )
    if( !_block_matrix[i].size() )
      _block_matrix.erase( _block_matrix.begin() + i );
  }


int Blockmap::blocks() const throw()
  {
  int sum = 0;
  for( unsigned int i = 0; i < _block_matrix.size(); ++i )
    sum += _block_matrix[i].size();
  return sum;
  }


void Blockmap::print( FILE * outfile, int debug_level ) const throw()
  {
  std::fprintf( outfile, "total cuts %d\n", _block_matrix.size() );
  std::fprintf( outfile, "total blocks %d\n", blocks() );
  std::fprintf( outfile, "blockmap size %dw x %dh\n\n", _width, _height );

  int sp = (debug_level & 1) ? 0 : -1;
  for( unsigned int i = 0; i < _block_matrix.size(); ++i )
    {
    std::fprintf( outfile, "cut %d blocks %d\n", i + 1, _block_matrix[i].size() );
    for( unsigned int j = 0; j < _block_matrix[i].size(); ++j )
    _block_matrix[i][j].print( outfile, sp );
    }
  }
