/*  GNU Ocrad - Optical Character Recognition program
    Copyright (C) 2003, 2004 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <algorithm>
#include <cstdio>
#include <cstdlib>
#include "common.h"


void Ocrad::internal_error( const char * msg ) throw()
  {
  std::fprintf( stderr, "ocrad: internal error: %s.\n", msg );
  exit( 3 );
  }


bool Ocrad::similar( int a, int b, int percent_dif, int abs_dif ) throw()
  {
  int difference = std::abs( a - b );
  if( percent_dif > 0 && difference <= abs_dif ) return true;
  int max_abs = std::max( std::abs(a), std::abs(b) );
  return ( difference * 100 <= max_abs * percent_dif );
  }


namespace {

const int charsets = 3;

const Charset::Value charset_value[charsets] =
  { Charset::ascii, Charset::iso_8859_9, Charset::iso_8859_15 };

const char * const charset_name[charsets] =
  { "ascii", "iso-8859-9", "iso-8859-15" };

} // end namespace


bool Charset::enable( const char * name ) throw()
  {
  for( int i = 0; i < charsets; ++i )
    if( std::strcmp( name, charset_name[i] ) == 0 )
      { _charset |= charset_value[i]; return true; }
  return false;
  }


bool Charset::enabled( Value cset ) const throw()
  {
  if( !_charset ) return cset == iso_8859_15;		// default charset
  return _charset & cset;
  }


bool Charset::only( Value cset ) const throw()
  {
  if( !_charset ) return cset == iso_8859_15;		// default charset
  return _charset == cset;
  }


void Charset::show_error( const char * arg ) const throw()
  {
  if( arg && std::strcmp( arg, "help" ) )
    std::fprintf( stderr,"ocrad: bad charset '%s'\n", arg );
  std::fputs( "Valid charset names are:  ", stderr );
  for( int i = 0; i < charsets; ++i )
    std::fprintf( stderr, "%s  ", charset_name[i] );
  std::fputs( "\n", stderr );
  }


bool Control::set_format( const char * name ) throw()
  {
  if( std::strcmp( name, "byte" ) == 0 ) { format = byte; return true; }
  if( std::strcmp( name, "utf8" ) == 0 ) { format = utf8; return true; }
  return false;
  }
