/*  GNU Ocrad - Optical Character Recognition program
    Copyright (C) 2003, 2004 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <cstdio>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "bitmap.h"
#include "block.h"
#include "blockmap.h"


namespace {

void ignore_small_blocks( std::vector< Block > & block_vector ) throw()
  {
  int to = 0, blocks = block_vector.size();
  for( int from = 0; from < blocks; ++from )
    {
    const Block & b = block_vector[from];
    if( b.id() > 0 )	// black block
      {
      if( b.height() > 4 || b.width() > 4 ||
          ( ( b.height() > 2 || b.width() > 2 ) && b.area() > 5 ) )
        block_vector[to++] = b;
      }
    else		// white block (hole)
      {
      if( b.height() > 4 || b.width() > 4 ||
          ( ( b.height() > 2 || b.width() > 2 ) && b.area() > 3 ) )
        block_vector[to++] = b;
      }
    }
  if( to < blocks )
    block_vector.erase( block_vector.begin() + to, block_vector.end() );
  }


void ignore_white_blocks( std::vector< Block > & block_vector ) throw()
  {
  int i = 0, blocks = block_vector.size();
  while( i < blocks )
    {
    if( block_vector[i].id() < 0 )
      { block_vector.erase( block_vector.begin() + i ); --blocks; }
    else ++i;
    }
  }


void order_blocks( std::vector< Block > & block_vector ) throw()
  {
  int blocks = block_vector.size();
  for( int i = 0; i < blocks; )
    {
    int j = i, best = i;
    while( ++j < blocks && block_vector[j].top() <= block_vector[best].bottom() )
      if( block_vector[j].left() < block_vector[best].left() ) best = j;
    if( best == i ) ++i;
    else
      {
      Block temp = block_vector[best];
      while( best > i ) { block_vector[best] = block_vector[best-1]; --best; }
      block_vector[i] = temp;
      }
    }
  }


void remove_top_bottom_noise( std::vector< Block > & block_vector ) throw()
  {
  int blocks = block_vector.size();
  for( int i = 0; i < blocks; ++i )
    {
    Block & b = block_vector[i];
    if( b.height() < 11 ) continue;
    const Blockmap & bm = *(b.blockmap());

    int c = 0;
    for( int col = b.left(); col <= b.right(); ++col )
      if( bm.id( b.top(), col ) == b.id() && ++c > 1 ) break;
    if( c <= 1 ) b.top( b.top() + 1 );

    c = 0;
    for( int col = b.left(); col <= b.right(); ++col )
      if( bm.id( b.bottom(), col ) == b.id() && ++c > 1 ) break;
    if( c <= 1 ) b.bottom( b.bottom() - 1 );
    }
  }


void remove_left_right_noise( std::vector< Block > & block_vector ) throw()
  {
  int blocks = block_vector.size();
  for( int i = 0; i < blocks; ++i )
    {
    Block & b = block_vector[i];
    if( b.width() < 6 ) continue;
    const Blockmap & bm = *(b.blockmap());

    int c = 0;
    for( int row = b.top(); row <= b.bottom(); ++row )
      if( bm.id( row, b.left() ) == b.id() && ++c > 1 ) break;
    if( c <= 1 ) b.left( b.left() + 1 );

    c = 0;
    for( int row = b.top(); row <= b.bottom(); ++row )
      if( bm.id( row, b.right() ) == b.id() && ++c > 1 ) break;
    if( c <= 1 ) b.right( b.right() - 1 );
    }
  }

} // end namespace


int Blockmap::generate_black_id() { static int id = 0; return ++id; }

int Blockmap::generate_white_id() { static int id = 0; return --id; }


void Blockmap::add_point_to_block( int row, int col, int id ) throw()
  {
  static int ib = -1, iw = -1;
  if( id > 0 && ib >= 0 && ib < blocks() && _block_vector[ib].id() == id )
    { _block_vector[ib].add_point( row, col ); return; }
  if( id < 0 && iw >= 0 && iw < blocks() && _block_vector[iw].id() == id )
    { _block_vector[iw].add_point( row, col ); return; }
  for( int c = 0, i = blocks() - 1; i >= 0 ; --i, ++c )
    if( _block_vector[i].id() == id )
      {
      _block_vector[i].add_point( row, col );
      if( id > 0 ) ib = i; else iw = i;
      if( c < _width ) return;
      Block temp = _block_vector[i];
      _block_vector.erase( _block_vector.begin() + i );
      _block_vector.push_back( temp );
      return;
      }
  Ocrad::internal_error( "add_point_to_block, lost block" );
  }


void Blockmap::create_block( const Rectangle & r, int id ) throw()
  {
  _block_vector.push_back( Block( r, *this, id ) );
  }


void Blockmap::delete_block( int id ) throw()
  {
  for( int i = blocks() - 1; i >= 0 ; --i )
    {
    Block & b = _block_vector[i];
    if( b.id() == id )
      {
      for( int row = b.top(); row <= b.bottom(); ++row )
        for( int col = b.left(); col <= b.right(); ++col )
          if( data[row][col] == id ) data[row][col] = 0;
      _block_vector.erase( _block_vector.begin() + i ); return;
      }
    }
  Ocrad::internal_error( "delete_block, lost block" );
  }


void Blockmap::join_blocks( int id1, int id2 ) throw()
  {
  if( std::abs( id1 ) > std::abs( id2 ) )
    { int temp = id1; id1 = id2; id2 = temp; }

  int i1 = blocks();
  int i2 = i1;
  while( --i1 >= 0 && _block_vector[i1].id() != id1 );
  while( --i2 >= 0 && _block_vector[i2].id() != id2 );
  if( i1 < 0 || i2 < 0 ) Ocrad::internal_error( "join_blocks, lost block" );

  Block & b1 = _block_vector[i1];
  Block & b2 = _block_vector[i2];
  for( int row = b2.top(); row <= b2.bottom(); ++row )
    for( int col = b2.left(); col <= b2.right(); ++col )
      if( data[row][col] == id2 ) data[row][col] = id1;
  b1.add_rectangle( b2 );
  _block_vector.erase( _block_vector.begin() + i2 );
  }


void Blockmap::ignore_wide_blocks() throw()
  {
  int i = 0, blocks = _block_vector.size();
  while( i < blocks )
    {
    const Block & b = _block_vector[i];
    if( 2 * b.width() < _width ) { ++i; continue; }
    if( b.id() > 0 && 4 * b.area() > b.size() )
      {
      // image, not frame
      if( 10 * b.width() > 8 * _width && 10 * b.height() > 8 * _height )
        { _block_vector.clear(); return; }
      for( int j = blocks - 1; j > i; --j )
        if( b.includes( _block_vector[j] ) )
          { _block_vector.erase( _block_vector.begin() + j ); --blocks; }
      }
    _block_vector.erase( _block_vector.begin() + i ); --blocks;
    }
  }


Blockmap::Blockmap( const Bitmap & page_image, int rindex, int debug_level )
                                                                     throw()
  {
  if( rindex < 0 || rindex >= page_image.rectangles() )
    Ocrad::internal_error( "Blockmap, text rectangle index out of range" );

  const Rectangle & r = page_image.rectangle_vector()[rindex];
  int row0 = r.top();
  int col0 = r.left();
  _height =  r.height();
  _width =   r.width();
  data.resize( _height );

  for( int row = 0; row < _height; ++row )
    { data[row].resize( _width ); data[row][0] = 0; }
  for( int col = 0; col < _width; ++col ) data[0][col] = 0;

  for( int row = 1; row < _height; ++row )
    {
    for( int col = 1; col < _width; ++col )
      {
      if( page_image.get_bit( row0 + row, col0 + col ) )	// black point
        {
        if( data[row-1][col] > 0 )
          {
          int id = data[row][col] = data[row-1][col];
          add_point_to_block( row, col, id );
          if( data[row][col-1] > 0 && data[row][col-1] != id )
            join_blocks( data[row][col-1], id );
          }
        else if( data[row][col-1] > 0 )
          {
          int id = data[row][col] = data[row][col-1];
          add_point_to_block( row, col, id );
          }
        else
          {
          int id = data[row][col] = generate_black_id();
          create_block( Rectangle( col, row, col, row ), id );
          }
        }
      else						// white point
        {
        if( data[row][col-1] == 0 )
          {
          data[row][col] = 0;
          if( data[row-1][col] < 0 ) delete_block( data[row-1][col] );
          }
        else if( data[row-1][col] == 0 )
          {
          data[row][col] = 0;
          if( data[row][col-1] < 0 ) delete_block( data[row][col-1] );
          }
        else if( data[row-1][col] < 0 )
          {
          int id = data[row][col] = data[row-1][col];
          add_point_to_block( row, col, id );
          if( data[row][col-1] < 0 && data[row][col-1] != id )
            join_blocks( data[row][col-1], id );
          }
        else if( data[row][col-1] < 0 )
          {
          int id = data[row][col] = data[row][col-1];
          add_point_to_block( row, col, id );
          }
        else
          {
          int id = data[row][col] = generate_white_id();
          create_block( Rectangle( col, row, col, row ), id );
          }
        }
      }
    }
  if( debug_level <= 99 )
    {
    ignore_small_blocks( _block_vector );
    ignore_wide_blocks();
    Block::hierarchize_blocks( _block_vector );
    ignore_white_blocks( _block_vector );
    remove_top_bottom_noise( _block_vector );
    remove_left_right_noise( _block_vector );
    if( debug_level <= 97 ) order_blocks( _block_vector );
    }
  }


void Blockmap::print( FILE * outfile, int debug_level ) const throw()
  {
  std::fprintf( outfile, "%d blocks\n", blocks() );
  std::fprintf( outfile, "%d %d\n\n", _width, _height );

  int sp = (debug_level & 1) ? 0 : -1;
  std::vector< Block >::const_iterator p = _block_vector.begin();
  for( ; p != _block_vector.end(); ++p )
    p->print( outfile, sp );
  }
