/*  GNU OCRAD - Optical Character Recognition program
    Copyright (C) 2003 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
/*
    Return values: 0 for a normal exit, 1 for environmental problems
    (file not found, invalid flags, I/O errors, etc), 2 to indicate a
    corrupt or invalid input file, 3 for an internal consistency error
    (eg, bug) which caused OCRAD to panic.
*/

#include <cctype>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <getopt.h>
#include <list>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "bitmap.h"
#include "block.h"
#include "blockmap.h"
#include "character.h"
#include "textline.h"
#include "textblock.h"


void show_version()
  {
  printf( "GNU %s version %s\n", PROGRAM_NAME, PROGRAM_VERSION );
  printf( "Copyright (C) %s Antonio Diaz.\n", PROGRAM_YEAR );
  printf( "This program is free software; you may redistribute it under the terms of\n" );
  printf( "the GNU General Public License.  This program has absolutely no warranty.\n" );
  }

void show_error( char * msg )
  {
  if( msg && msg[0] != 0 ) fprintf( stderr,"%s: %s\n", PROGRAM_NAME, msg );
  fprintf( stderr,"Try `%s --help' for more information.\n", PROGRAM_NAME );
  }

void show_help()
  {
  printf( "GNU %s, Optical Character Recognition program\n", PROGRAM_NAME );
  printf( "Reads pbm file(s), or standard input, and sends text to standard output\n" );
  printf( "\nUsage: %s [options] [files]\n", PROGRAM_NAME );
  printf( "Options:\n");
  printf( "  -h, --help               display this help and exit\n");
  printf( "  -V, --version            output version information and exit\n");
  printf( "  -1, -4                   pbm output file type (debug)\n");
  printf( "  -a, --append             append text to output file\n");
  printf( "  -b, --block=<n>          process only the specified text block\n");
  printf( "  -C, --copy               'copy' input to output (debug)\n");
  printf( "  -D, --debug=<level>      (0-100) output intermediate data (debug)\n");
  printf( "  -f, --force              force overwrite of output file\n");
//  printf( "  -L, --level              level input image before doing OCR\n");
  printf( "  -l, --layout=<n>         layout analysis, 0=none, 1=column, 2=full\n");
  printf( "  -o <file>                place the output into <file>\n");
//  printf( "  -R <scale>               reduce input image by <scale> (debug)\n");
//  printf( "  -s <type>                make a 'special file' (debug)\n");
//  printf( "  -t <threshold>           set threshold for -R option (debug)\n");
  printf( "  -v, --verbose            be verbose\n");
  printf( "  -x <file>                export OCR Results File to <file>\n");

  printf( "\nReport bugs to bug-ocrad@gnu.org\n");
  }


void ocr( const Blockmap & blockmap, FILE * outfile, FILE * exportfile,
          int debug_level ) throw()
  {
  if( debug_level >= 90 )
    { blockmap.print( outfile, debug_level ); return; }

  Textblock textblock( blockmap.block_list() );

  // First pass. Recognize the easy characters.
  textblock.recognize1();

  // Second pass. Use context to resolve ambiguities.
  textblock.recognize2();

  if( debug_level >= 86 )
    {
    bool graph = debug_level >= 88 ? true : false;
    bool recursive = (debug_level & 1) ? true : false;
    textblock.print( outfile, graph, recursive );
    return;
    }

  textblock.print( outfile, false, false );
  if( exportfile ) textblock.xprint( exportfile );
  }


int main(int argc, char *argv[]) 
  {
  FILE *infile = stdin, *outfile = stdout, *exportfile = 0;
  char *outfile_name = 0, *exportfile_name = 0;
  int rindex = -1, debug_level = 0, layout_level = 0;
  int scale = 0, threshold = -1;
  char filetype = 0, specialtype = 0;
  bool append = false, copy = false, force = false, level = false;
  bool verbose = false;

  // scan arguments

  while( true )			// process options
    {
    static struct option long_options[] =
      {
      {"append", no_argument, 0, 'a'},
      {"block", required_argument, 0, 'b'},
      {"copy", no_argument, 0, 'C'},
      {"debug", required_argument, 0, 'D'},
      {"force", no_argument, 0, 'f'},
      {"help", no_argument, 0, 'h'},
      {"layout", required_argument, 0, 'l'},
      {"level", no_argument, 0, 'L'},
      {"verbose", no_argument, 0, 'v'},
      {"version", no_argument, 0, 'V'},
      {0, 0, 0, 0}
      };

    int c = getopt_long( argc, argv, "14ab:CD:fhLl:o:R:s:t:Vvx:",
                         long_options, 0 );
    if( c == -1 ) break;		// all options processed

    switch( c )
      {
      case 0: break;
      case '?': return 1;  // `getopt_long' already printed an error message.
      case '1':
      case '4': filetype = c; break;
      case 'a': append = true; break;
      case 'b': rindex = strtol( optarg, 0, 0 ) - 1; break;
      case 'C': copy = true; break;
      case 'D': debug_level = strtol( optarg, 0, 0 ); break;
      case 'f': force = true; break;
      case 'h': show_help(); return 0;
      case 'L': level = true; break;
      case 'l': layout_level = strtol( optarg, 0, 0 ); break;
      case 'o':	outfile_name = optarg; break;
      case 'R': scale = strtol( optarg, 0, 0 ); break;
      case 's': specialtype = optarg[0]; break;
      case 't': threshold = strtol( optarg, 0, 0 ); break;
      case 'V':	show_version(); return 0;
      case 'v': verbose = true; break;
      case 'x':	exportfile_name = optarg; break;
      default: return 1;
      }
    } // end process options

  if( outfile_name )
    {
    if( append ) outfile = fopen( outfile_name, "a" );
    else if( force ) outfile = fopen( outfile_name, "w" );
    else if( ( outfile = fopen( outfile_name, "wx" ) ) == 0 )
      {
      fprintf( stderr, "Output file %s already exists.\n", outfile_name );
      return 1;
      }
    if( !outfile )
      { fprintf( stderr, "Cannot open %s\n", outfile_name ); return 1; }
    }

  if( exportfile_name )
    {
    exportfile = fopen( exportfile_name, "w" );
    if( !exportfile )
      { fprintf( stderr, "Cannot open %s\n", exportfile_name ); return 1; }
    fprintf( exportfile, "# Ocr Results File. Created by GNU %s version %s\n",
             PROGRAM_NAME, PROGRAM_VERSION );
    }

// process any remaining command line arguments (input files)
  do {
    if( optind < argc )
      {
      if( infile != stdin ) fclose( infile );
      if( strcmp( argv[optind], "-" ) == 0 ) infile = stdin;
      else if( ( infile = fopen( argv[optind], "r" ) ) == 0 )
	{ fprintf(stderr,"Cannot open %s\n",argv[optind]); return 1; }
      ++optind;
      }

// 'infile' contains the scanned text image to be converted to ascii
// (in pbm format).
// 'outfile' is the destination for the ascii version of the scanned
// text image. (Or for a pbm file if debugging).

    try
      {
      Bitmap page_image( infile );

      // make a list of text blocks
      page_image.analyse_layout( layout_level );
      if( verbose )
        fprintf( stderr, "number of text blocks %d\n", page_image.rectangles() );

      if( rindex >= page_image.rectangles() )
        {
        fprintf( stderr,"This page has only %d text blocks(s)\n",
                 page_image.rectangles() );
        return 1;
        }

      if( level )
        fprintf( stderr, "\nslope_best = %d\n",
                 page_image.horizontalify( verbose ) );

      if( scale != 0 )
        {
        Bitmap reduced( page_image, threshold, scale );
        reduced.save( outfile, filetype ); return 0;
        }
      else if( specialtype != 0 )
        {
        Bitmap::type t;
        if( specialtype == 'v' ) t = Bitmap::vertical_histogram ;
        else if( specialtype == 'h' ) t = Bitmap::horizontal_histogram ;
        else if( specialtype == 'g' ) t = Bitmap::connected_ground ;
        else { show_error( "bad special type" ); return 1; }
        Bitmap sb( page_image, t );
        sb.save( outfile, filetype ); return 0;
        }
      else if( copy )
        {
        if( rindex < 0 ) page_image.save( outfile, filetype );
        else
          {
          Bitmap bitmap( page_image, page_image.rectangle_vector()[rindex] );
          bitmap.save( outfile, filetype );
          }
        return 0;
        }

      if( exportfile )
        fprintf( exportfile, "total blocks %d\n",
                 ( rindex < 0 ) ? page_image.rectangles() : 1 );

      // call the character recognizer for every rectangle of text
      for( int c = 0; c < page_image.rectangles(); ++c )
        if( rindex < 0 || rindex == c )
          {
          if( exportfile )
            {
            const Rectangle & r = page_image.rectangle_vector()[c];
            fprintf( exportfile, "block %d %d %d %d %d\n", c + 1,
                     r.left(), r.top(), r.width(), r.height() );
            }
          Blockmap blockmap( page_image, c, debug_level );
          ocr( blockmap, outfile, exportfile, debug_level );
          }
      }

    catch( Bitmap::Error e ) { fputs( e.s, stderr ); return 2; }
    }
  while( optind < argc );
  fclose( outfile );
  return 0;
  }
