/*
--          This file is part of the New World OS and Objectify projects
           Copyright (C) 2006, 2007, 2008, 2009, 2010, 2011  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2011-04-30 19:44:48 -0600 (Sat, 30 Apr 2011) $
--   $Revision: 4893 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.  Also this file was created in the
--   alpha_05_branch so check the logs for this file in it too.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/


#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "md5.h"
#include "sha1.h"

#include "chunk_info.h"
#include "config.h"
#include "disk_io.h"
#include "header.h"
#include "log.h"
#include "progress_bar.h"
#include "user_config.h"


static void print_usage(char *program)
{
    fprintf(stderr, "usage: %s [output-file]\n", program);
    fprintf(stderr, " if no output file is specified it just outputs the checksums.\n");
}


int main(int argc, char* argv[])
{
    int obj_file_desc;
    const char* obj_file_path;
    const char* error_msg;
    off_t chunk;
    uint8 block_map[BIT_MAP_BYTES];
    uint8 block[FILE_BLOCK_SIZE];
    int i;
    int j;
    size_t bytes_read;
    FILE* ofp = NULL;
    int num_blocks;
    char msg[128];
    uint32 ref;
    struct md5_ctx md5_context;    /* MD5 checksum context */
    struct sha1_ctx sha1_context;
    uint8 md5_digest[16];
    uint8 sha1_digest[20];
    int chunk_num;
    Chunk_Info* chunk_info;
    int blocks_with_bad_ids = 0;
    Disk_Header* disk_header;

    if (argc > 2)
    {
	print_usage(argv[0]);
	exit(1);
    }

    if (argc == 2 && *argv[1] == '-')
    {
	fprintf(stderr, "error: this program doesn't have any options\n");
	print_usage(argv[0]);
	exit(1);
    }


    /* Open the storage drive and verify the header info */

    nwos_log_arguments(argc, argv);

    obj_file_path = nwos_get_private_objects_path();

    obj_file_desc = open(obj_file_path, O_RDONLY);

    if (obj_file_desc < 0)
    {
	perror(obj_file_path);
	exit(1);
    }

    bytes_read = read(obj_file_desc, block, sizeof(block));

    if (bytes_read != sizeof(block))
    {
	perror("reading first block");
	exit(1);
    }

    error_msg = nwos_load_private_data(block, sizeof(block), false);   /* don't allow compressed files */

    if (error_msg != NULL)
    {
	fprintf(stderr, "%s: %s\n", error_msg, obj_file_path);
	exit(1);
    }

    assert(nwos_used_private_chunks > 0);

    chunk_info = malloc(nwos_used_private_chunks * sizeof(Chunk_Info));
    assert(chunk_info != NULL);

    bytes_read = read(obj_file_desc, chunk_info, nwos_used_private_chunks * sizeof(Chunk_Info));

    if (bytes_read != nwos_used_private_chunks * sizeof(Chunk_Info))
    {
	perror("reading chunk info");
	exit(1);
    }

    /* fix the byte order on little endian machines */
#ifndef WORDS_BIGENDIAN
	{
	  int i;
	  for (i = 0; i < nwos_used_private_chunks; i++)
	  {
	      chunk_info[i].ref = byteswap_uint64(chunk_info[i].ref);
	      chunk_info[i].flags_used = byteswap_uint32(chunk_info[i].flags_used);
	      chunk_info[i].index = byteswap_uint32(chunk_info[i].index);
	  }
	}
#endif

    if (argc == 2)
    {
	ofp = fopen(argv[1], "w");

	if (ofp == NULL)
	{
	    perror(argv[1]);
	    exit(1);
	}
    }

    md5_init_ctx(&md5_context);   /* initialize the MD5 checksum context */
    sha1_init_ctx(&sha1_context);

    printf("header: %c%c%c%c %c%c%c%c\n",
	   nwos_private_disk_header.magic_number[0], nwos_private_disk_header.magic_number[1], nwos_private_disk_header.magic_number[2], nwos_private_disk_header.magic_number[3], 
	   nwos_private_disk_header.version_string[0], nwos_private_disk_header.version_string[1], nwos_private_disk_header.version_string[2], nwos_private_disk_header.version_string[3]);

    printf("total blocks on disk: %llu  chunks_used: %d\n", (uint64)nwos_total_private_chunks * USABLE_BLOCKS_PER_CHUNK, nwos_used_private_chunks);
    fflush(stdout);

    disk_header = (Disk_Header*)block;
    memcpy(&disk_header->type_code, TYPE_CODE_COMPRESSED, sizeof(disk_header->type_code));

    /* write the first 256 bytes always */
    if (ofp != NULL && fwrite(block, 1, sizeof(block), ofp) != sizeof(block))
    {
	perror(argv[1]);
	close(obj_file_desc);
	exit(1);
    }

    md5_process_bytes(block, sizeof(block), &md5_context);    /* include this data in the md5 checksum */
    sha1_process_bytes(block, sizeof(block), &sha1_context);    /* include this data in the sha1 checksum */

    num_blocks = 0;

    nwos_start_progress_bar();

    // for now we can skip over public blocks because they should always stay the same
    for (chunk_num = 0; chunk_num < nwos_used_private_chunks; chunk_num++)
    {
	nwos_update_progress_bar((float)num_blocks / (float)nwos_used_private_blocks);

	chunk = nwos_block_offset_to_chunks + chunk_info[chunk_num].index * BLOCKS_IN_CHUNK;

	if (lseek(obj_file_desc, chunk << 8, SEEK_SET) < 0)
	{
	    snprintf(msg, sizeof(msg), "lseek chunk:%08x", (uint32)chunk);
	    perror(msg);
	    exit(1);
	}

	bytes_read = read(obj_file_desc, block_map, sizeof(block_map));

	if (bytes_read != sizeof(block_map))
	{
	    snprintf(msg, sizeof(msg), "reading block map: %u", (uint32)chunk);
	    perror(msg);
	    exit(1);
	}

	for (i = 0; i < USABLE_BLOCKS_PER_CHUNK; i++)
	{
	    if ((block_map[i/8] & (0x80 >> (i%8))) != 0)
	    {
		if (lseek(obj_file_desc, (chunk + BIT_MAP_BLOCKS + i) << 8, SEEK_SET) < 0)
		{
		    snprintf(msg, sizeof(msg), "lseek block:%08x", (uint32)(chunk + i));
		    perror(msg);
		    exit(1);
		}

		bytes_read = read(obj_file_desc, block, sizeof(block));

		if (bytes_read != sizeof(block))
		{
		    snprintf(msg, sizeof(msg), "reading block: %u", (uint32)(chunk + i));
		    perror(msg);
		    exit(1);
		}
		
		ref = (uint32)block[4] << 24 | (uint32)block[5] << 16 | (uint32)block[6] << 8 | (uint32)block[7];

		if (ref == 0)
		{
		    nwos_erase_progress_bar();

		    blocks_with_bad_ids++;

		    printf("WARNING: reference in chunk %08llx block %04x is zero, not written:\n", chunk_info[chunk_num].ref, i);

		    for (j = 0; j < FILE_BLOCK_SIZE; j += 16)
		    {
			printf("%02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x\n",
			       block[j+0],  block[j+1],  block[j+2],  block[j+3],
			       block[j+4],  block[j+5],  block[j+6],  block[j+7],
			       block[j+8],  block[j+9],  block[j+10], block[j+11],
			       block[j+12], block[j+13], block[j+14], block[j+15]);
		    }

		    nwos_start_progress_bar();
		}
		else
		{
#if 0
		    /* save this for verbose mode? */
		    printf("id: %08x  block: %08x\n", ref, (uint32)chunk+i);
		    /* printf("id: %08x\n", ref); */
		    fflush(stdout);
#endif

		    if (ofp != NULL && fwrite(block, 1, sizeof(block), ofp) != sizeof(block))
		    {
			perror(argv[1]);
			close(obj_file_desc);
			exit(1);
		    }

		    md5_process_bytes(block, sizeof(block), &md5_context);    /* include this data in the md5 checksum */
		    sha1_process_bytes(block, sizeof(block), &sha1_context);    /* include this data in the sha1 checksum */
		    num_blocks++;
		}
	    }
	}
    }

    nwos_finish_progress_bar();

    printf("Number of blocks: %d\n", num_blocks);

    if (blocks_with_bad_ids > 0)
    {
	printf("WARNING: %d blocks had a reference ID of zero and weren't written!\n", blocks_with_bad_ids);
    }

    md5_finish_ctx(&md5_context, md5_digest);   /* finish computing the md5 sum */
    sha1_finish_ctx(&sha1_context, sha1_digest);

    printf("MD5: ");
    for (i = 0; i < sizeof(md5_digest); i++) printf("%02x", md5_digest[i]);
    printf("\n");

    printf("SHA1: ");
    for (i = 0; i < sizeof(sha1_digest); i++) printf("%02x", sha1_digest[i]);
    printf("\n");


    if (ofp != NULL && fclose(ofp) != 0)
    {
	perror(argv[1]);
	exit(1);
    }

    close(obj_file_desc);

    return 0;
}

