/*
--          This file is part of the New World OS and Objectify projects
--                     Copyright (C) 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2010-01-23 10:02:35 -0700 (Sat, 23 Jan 2010) $
--   $Revision: 4492 $
--
*/


#include <assert.h>
#include <limits.h>    /* define HOST_NAME_MAX */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "progress_bar.h"
#include "types.h"


struct
{
  uint8 map[32];
  uint8 block[256];
  int next;
} sequence[4096];

bool all_slots_filled()
{
    int i;
    int j;

    for (i = 0; i < 4096; i++)
    {
	for (j = 1; j < 32; j++)
	{
	    if (sequence[i].map[j] != 0xff) return false;
	}
    }

    return true;
}

int16 blocks_without[4096];

int find_blocks_without(uint8 byte)
{
    int i;
    int byte_num;
    int mask;
    int result = 0;

    byte_num = byte / 8;
    mask = 0x80 >> (byte & 7);

    for (i = 0; i < 4096; i++)
    {
	if ((sequence[i].map[byte_num] & mask) == 0)
	{
	    blocks_without[result] = i;
	    result++;
	}
    }
#if 0
    for (i = 0; i < result; i++)
    {
	assert ((sequence[blocks_without[i]].map[byte_num] & mask) == 0);
    }
#endif
    return result;
}


int main(int argc, char* argv[])
{
    struct timespec ts;
    int i;
    int j;
    int k;
    uint8 byte;
    int byte_num;
    int mask;
    int block;
    int num_blocks_without;
    int start_time;
    int elapsed_time;
    int loop_count = 0;
    uint32 half_time;
    char hostname[HOST_NAME_MAX];
    char filename[HOST_NAME_MAX+16];
    FILE* fp;
    char *random_device = NULL;
    uint random_word = 0;


    /* REMEMBER: in the real program to generate a key, using the random device should be the default! */

    if (argc > 1)
    {
	if (argc == 2 && strcmp(argv[1], "--use-dev-random") == 0)
	{
	    random_device = "/dev/random";
	}
	else if (argc == 2 && strcmp(argv[1], "--use-dev-srandom") == 0)   /* for OpenBSD */
	{
	    random_device = "/dev/srandom";
	}
	else if (argc == 2 && strcmp(argv[1], "--use-dev-urandom") == 0)
	{
	    random_device = "/dev/urandom";
	}
	else
	{
	    fprintf(stderr, "usage: %s [--use-dev-random |--use-dev-srandom |--use-dev-urandom]\n", argv[0]);
	    exit(1);
	}
    }

    clock_gettime(CLOCK_REALTIME, &ts);
    start_time = (int)ts.tv_sec;

    if (random_device != NULL)
    {
	fp = fopen(random_device, "r");

	if (fp == NULL)
	{
	    perror(random_device);
	    exit(1);
	}

	printf("reading from %s: ", random_device);
	fflush(stdout);

	if (fread(&random_word, 1, sizeof(random_word), fp) != sizeof(random_word))
	{
	    printf("\n");
	    perror(random_device);
	    exit(1);
	}

	fclose(fp);
	printf("%08x\n", random_word);
    }

    nwos_start_progress_bar();

    while (!all_slots_filled())
    {
	clock_gettime(CLOCK_REALTIME, &ts);
	srandom((uint)ts.tv_sec ^ (uint)ts.tv_nsec ^ random_word);

	usleep(random() % 659);

	clock_gettime(CLOCK_REALTIME, &ts);
	half_time = (uint32)ts.tv_nsec;
	usleep(half_time % 347);

	clock_gettime(CLOCK_REALTIME, &ts);

	byte = (((uint32)ts.tv_nsec >> (random() % 7)) ^ (uint32)random() >> (half_time % 23)) % 248 + 8;

	num_blocks_without = find_blocks_without(byte);

	if (num_blocks_without > 0)
	{
	    block = blocks_without[random() % num_blocks_without];

	    byte_num = byte / 8;
	    mask = 0x80 >> (byte & 7);

	    //	    assert((sequence[block].map[byte_num] & mask) == 0);

	    if ((sequence[block].map[byte_num] & mask) != 0)
	    {
		printf("block: %d  byte: %02x  byte_num: %d  mask: %02x  map: %02x\n", block, byte, byte_num, mask, 
		       sequence[block].map[byte_num]);
		while(1);
	    }

	    if (sequence[block].next == 0) sequence[block].next = 8;

	    assert(8 <= sequence[block].next && sequence[block].next <= 255);

	    sequence[block].block[sequence[block].next] = byte;

	    sequence[block].next = sequence[block].next + 1;

	    sequence[block].map[byte_num] |= mask;

	    if (sequence[block].next == 255)
	    {
		byte_num = 0;

		for (i = 1; i < 32; i++)
		{
		    if (sequence[block].map[i] != 0xff)
		    {
			if (byte_num != 0)
			{
			    printf("Multiple holes in map when next is 255\n");
			    for (i = 0; i < 32; i++)
			    {
				printf("%02x", sequence[block].map[i]);
			    }
			    printf("\n");
			    exit(1);
			}

			byte_num = i;
		    }
		}

		assert(byte_num != 0);

		for (i = 0; i < 8; i++)
		{
		    if (sequence[block].map[byte_num] == (uint8) ~(0x80 >> i)) break;
		}

		if (i < 0 || i > 7)
		{
		    printf("single hole not found in map byte_num: %d\n", byte_num);
		    for (i = 0; i < 32; i++)
		    {
			printf("%02x", sequence[block].map[i]);
		    }
		    printf("\n");

		    for (i = 0; i < 8; i++)
		      {
			printf("mask: %02x\n", (uint8) ~(0x80 >> i));
		      }

		    exit(1);
		}

		sequence[block].block[255] = byte_num * 8 + i;

		sequence[block].map[byte_num] |= 0x80 >> i;
	    }

	    loop_count++;

	    nwos_update_progress_bar((float)loop_count / (256.0 * 4096.0));
	}
    }

    nwos_finish_progress_bar();


    printf("Checking... ");
    fflush(stdout);

    nwos_start_progress_bar();

    for (i = 0; i < 4096; i++)
    {
	nwos_update_progress_bar((float)i / 4096.0);

	for (j = 0; j < 256; j++)
	{
	    k = 0;

	    if (j < 8)
	    {
		if (sequence[i].block[j] != 0)
		{
		    k = 256;
		}
	    }
	    else
	    {
		for (k = 8; k < 256; k++)
		{
		    if (sequence[i].block[k] == j) break;
		}
	    }

	    if (k >= 256)
	    {
		printf("block: %d  doesn't have: %02x\n", i, j);
		for (k = 0; k < 256; k++)
		{
		    printf("%02x%c", sequence[i].block[k], k % 16 == 15 ? '\n' : ' ');
		}
		exit(1);
	    }
	}
    }

    nwos_finish_progress_bar();

    /* write file out */

    if (gethostname(hostname, sizeof(hostname)) < 0)
    {
	fprintf(stderr, "gethostname failed, using 'default' for file name\n");
	strncpy(hostname, "default", sizeof(hostname));
    }

    printf("\n");

    clock_gettime(CLOCK_REALTIME, &ts);

    elapsed_time = (int)ts.tv_sec - start_time;

    printf("Elapsed time: %02d:%02d:%02d\n", elapsed_time / 3600, (elapsed_time % 3600) / 60, elapsed_time % 60);

    snprintf(filename, sizeof(filename), "%s-%u.tbl", hostname, (uint)ts.tv_sec);

    fp = fopen(filename, "w");

    if (fp == NULL)
    {
	perror("test.out");
	exit(1);
    }

    for (i = 0; i < 4096; i++)
    {
	if (fwrite(sequence[i].block, 1, 256, fp) != 256)
	{
	    perror("test.out");
	    exit(1);
	}
    }

    if (fclose(fp) != 0)
    {
	perror("test.out");
	exit(1);
    }

    exit(0);
}


