/*
--         This file is part of the New World OS and Objectify projects
--                    Copyright (C) 2004-2009  QRW Software
--              J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                         http://www.qrwsoftware.com
--                         http://nwos.sourceforge.com
--                      http://objectify.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: header.h,v $
-- Revision 1.6  2009/07/11 21:50:07  jsedwards
-- Rearranged header again and changed used public and private blocks to 64 bits.
--
-- Revision 1.5  2009/07/11 13:27:01  jsedwards
-- Added type_code parameter to nwos_check_disk_header function.
--
-- Revision 1.4  2009/07/07 03:09:39  jsedwards
-- Added type_code definitions and element to Disk_Header.
--
-- Revision 1.3  2009/07/02 13:18:37  jsedwards
-- Added external declaration for new nwos_check_disk_header function.
--
-- Revision 1.2  2009/07/02 12:50:13  jsedwards
-- Moved the public and private disk headers from the disk_io.c file and
-- changed the load and store function parameters to pass a buffer instead of
-- a disk header structure.
--
-- Revision 1.1  2009/06/30 10:10:23  jsedwards
-- Moved definitions and declarations related to the disk header from the
-- disk_io.h file.
--
*/

#ifndef HEADER_H
#define HEADER_H

#include "time_stamp.h"
#include "types.h"


#define MAGIC_NUMBER "NWOS"

#define TYPE_CODE_COMPRESSED  "cmpr"
#define TYPE_CODE_DIFF        "diff"
#define TYPE_CODE_DISK        "disk"
#define TYPE_CODE_FILE        "file"
#define TYPE_CODE_PUBLIC      "pblc"


typedef struct {
    char magic_number[4];
    char version_string[4];
    char type_code[4];
    uint8 block_offset_to_chunks[4];

    TimeStamp last_prep_disk;
    uint8 total_chunks[4];               /* this does NOT include the chunk used by the system */
    uint8 used_chunks[4];               /* this does NOT include the chunk used by the system */

    TimeStamp last_change;
    uint8 used_blocks[8];
} Disk_Header;

typedef struct {
    char magic_number[4];
    char version_string[4];
    TimeStamp last_prep_disk;

    uint8 total_blocks[4];
    uint8 used_blocks[4];
    TimeStamp last_change;

    uint8 block_offset_to_chunks[4];
    uint8 used_chunks[4];
    uint8 unused7[4];
    uint8 unused8[4];
} Disk_Header_0022_to_0029;

typedef struct {
    char magic_number[4];
    char version_string[4];
    TimeStamp last_prep_disk;

    uint8 total_public_blocks[4];
    uint8 used_public_blocks[4];
    TimeStamp public_last_change;

    uint8 total_private_blocks[4];
    uint8 used_private_blocks[4];
    TimeStamp private_last_change;

    uint8 unused5[4];
    uint8 next_private_chunk[4];
    uint8 unused6[4];
    uint8 unused7[4];
} Disk_Header_0021;

typedef struct {
    char magic_number[4];
    char version_string[4];
    TimeStamp time_last_change;

    uint8 total_public_blocks[4];
    uint8 total_private_blocks[4];
    uint8 unused1[4];
    uint8 unused2[4];

    uint8 used_public_blocks[4];
    uint8 used_private_blocks[4];
    uint8 unused3[4];
    uint8 unused4[4];

    uint8 unused5[4];
    uint8 next_private_chunk[4];
    uint8 unused6[4];
    uint8 unused7[4];
} Disk_Header_0016_to_0020;


extern Disk_Header nwos_public_disk_header;    /* these should only be accessed by storage.c to pass to backup */
extern Disk_Header nwos_private_disk_header;

extern uint64 nwos_used_public_blocks;
extern uint64 nwos_used_private_blocks;

extern uint32 nwos_total_private_chunks;
extern uint32 nwos_used_private_chunks;
extern uint32 nwos_block_offset_to_chunks;

extern char nwos_version_string[5];

extern const char* nwos_check_disk_header(Disk_Header* disk_header, bool compressed_file, const char* type_code);

extern const char* nwos_load_public_data(void *buffer, size_t buffer_size);
extern void nwos_store_public_data(void *buffer, size_t buffer_size);
extern const char* nwos_load_private_data(void *buffer, size_t buffer_size, bool compressed_file);
extern void nwos_store_private_data(void *buffer, size_t buffer_size, bool archive_is_file);


#endif


