/*
--             This file is part of the New World OS project
--                 Copyright (C) 2005-2009  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: check_objects.c,v $
-- Revision 1.23  2009/03/10 12:04:30  jsedwards
-- Eliminated the use of FILE_BLOCK_SIZE for kludge buffers.
--
-- Revision 1.22  2009/03/08 01:28:37  jsedwards
-- Changed include objectify.h to class_definition.h.
--
-- Revision 1.21  2009/03/07 20:17:19  jsedwards
-- Removed #include of objectify_private.h, not needed.
--
-- Revision 1.20  2008/09/01 03:13:40  jsedwards
-- Change for new nwos_initialize_objectify calling convention (doesn't pass
-- back root_object_reference anymore) and removed call to nwos_set_root_object
-- because initialize calls it now.
--
-- Revision 1.19  2008/08/31 21:53:51  jsedwards
-- Added an assert around calls to nwos_read_variable_sized_object_from_disk
-- and nwos_read_object_from_disk because now they return false when they fail
-- instead of asserting themselves.
--
-- Revision 1.18  2008/08/30 12:46:06  jsedwards
-- Removed code and variables to read pass phrase and pass it to initialize,
-- and change parameters passed to initialize.
--
-- Revision 1.17  2008/02/03 01:05:23  jsedwards
-- Changed DEFAULT_TYPE_RO to READ_ONLY.
--
-- Revision 1.16  2007/07/01 19:44:11  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.15  2007/06/22 14:08:09  jsedwards
-- Update all of the things that were out of date so it runs again.  Only
-- tests "File Path" objects right now.
--
-- Revision 1.14  2007/02/11 16:58:26  jsedwards
-- Changed so DEFAULT_TYPE has to specify RO (Read-Only) or RW (Read-Write).
--
-- Revision 1.13  2006/12/20 12:33:06  jsedwards
-- Changed to have two class definition refs, public and private.
--
-- Revision 1.12  2006/12/19 14:52:15  jsedwards
-- Kludged version to compile with new 'file path' structure without 'file'
-- feature.  DOES NOT function correctly!
--
-- Revision 1.11  2006/12/11 12:32:27  jsedwards
-- Fix check_month function to use new month class format.
--
-- Revision 1.10  2006/12/01 14:37:22  jsedwards
-- Fix the year in the copyright.
--
-- Revision 1.9  2006/12/01 14:31:30  jsedwards
-- Changed to use new malloc_reference_list and free_reference_list functions
-- instead of inlining the code.
--
-- Revision 1.8  2006/11/18 15:09:09  jsedwards
-- Added "max_size" parameter to read_variable_sized_object_from_disk because
-- objects are no longer limited to one file block.
--
-- Revision 1.7  2006/11/11 12:01:01  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.6  2006/11/08 02:17:44  jsedwards
-- Change to check a few objects (spelling, name, file_path, month).
--
-- Revision 1.5  2006/10/26 01:51:23  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.4.2.2  2006/10/25 12:22:26  jsedwards
-- Changed C_struct_class_definition to C_struct_Class_Definition so the case
-- is consistent with all the other C_struct objects.
--
-- Revision 1.4.2.1  2006/09/01 13:27:20  jsedwards
-- Changed "nwos_object_size" to "nwos_reference_list_size" and added the
-- object reference to "nwos_fill_in_common_header" so it can put the "id"
-- in the header now.
--
-- Revision 1.4  2006/01/08 15:28:14  jsedwards
-- Changed to scan all class objects.  Still not very useful.
--
-- Revision 1.3  2005/12/24 16:18:26  jsedwards
-- Removed "host" id from object references (ObjRef).  Host redirection will
-- be done using a "redirection" object in the future.
--
-- Revision 1.2  2005/12/10 15:03:35  jsedwards
-- Fixed header to say the GPL is in the LICENSE file instead of COPYING.
--
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "class_definition.h"


bool reference_list_has_reference(ObjRef* ref_list_ref, ObjRef* ref)
{
    ReferenceList* ref_list;
    int num_refs;
    int i;

    ref_list = nwos_malloc_reference_list(ref_list_ref);

    num_refs = ref_list->common_header.num_refs;

    for (i = 0; i < num_refs; i++)
    {
	if (is_same_object(&ref_list->references[i], ref)) break;
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return i < num_refs;    /* return true if found it */
}

void dump_reference_list(ObjRef* ref_list_ref)
{
    ReferenceList* ref_list;
    int num_refs;
    int i;

    ref_list = nwos_malloc_reference_list(ref_list_ref);

    num_refs = ref_list->common_header.num_refs;

    for (i = 0; i < num_refs; i++)
    {
	printf("   %08x\n", nwos_ref_to_word(&ref_list->references[i]));
    }
}


static size_t get_spelling_object_size(void* spelling_obj)
{
    assert(((C_struct_Spelling*)spelling_obj)->count > 0);

    return sizeof(C_struct_Spelling) + ((C_struct_Spelling*)spelling_obj)->count;
}

void check_spelling(ObjRef* spelling_ref)
{
    int i;
    char buffer[64];
    uint8 kludge[MAX_SPELLING_OBJ_SIZE];
    C_struct_Spelling* spelling_obj_ptr = (C_struct_Spelling*)kludge;
#if 0
    size_t size;
    ObjRef ref;
    ReferenceList* ref_list;
    int num_refs;
#endif

    printf("Spelling: %08x ", nwos_ref_to_word(spelling_ref));
    fflush(stdout);

    assert(nwos_read_variable_sized_object_from_disk(spelling_ref, kludge, sizeof(kludge), &get_spelling_object_size));

    for (i = 0; i < spelling_obj_ptr->count; i++) buffer[i] = spelling_obj_ptr->storage[i];
    buffer[i] = '\0';

    printf(": %s\n", buffer);

    if (!is_void_reference(&spelling_obj_ptr->character_set))
    {
	printf("WARNING: character_set in Spelling object is non-void: %08x\n", nwos_ref_to_word(&spelling_obj_ptr->character_set));
    }
}


static size_t get_name_object_size(void* name_obj)
{
    assert(((C_struct_Name*)name_obj)->count > 0);

    return sizeof(C_struct_Name) + (((C_struct_Name*)name_obj)->count * sizeof(ObjRef));
}

void check_name(ObjRef* name_ref)
{
    int i;
    uint8 kludge[MAX_NAME_OBJ_SIZE];
    C_struct_Name* name_obj_ptr = (C_struct_Name*)kludge;
    uint8 cludge[MAX_SPELLING_OBJ_SIZE];
    C_struct_Spelling* spelling_obj_ptr = (C_struct_Spelling*)cludge;
#if 0
    size_t size;
    ObjRef ref;
#endif

    printf("Name: %08x\n", nwos_ref_to_word(name_ref));
    fflush(stdout);

    assert(nwos_read_variable_sized_object_from_disk(name_ref, kludge, sizeof(kludge), &get_name_object_size));

    for (i = 0; i < name_obj_ptr->count; i++)
    {
	printf("  %08x\n", nwos_ref_to_word(&name_obj_ptr->spelling[i]));

	assert(nwos_read_variable_sized_object_from_disk(&name_obj_ptr->spelling[i], cludge, sizeof(cludge), &get_spelling_object_size));

	if (!reference_list_has_reference(&spelling_obj_ptr->header.object.references, name_ref))
	{
	    printf("ERROR: name object not in spelling reference list: %08x\n",
		   nwos_ref_to_word(&spelling_obj_ptr->header.object.references));
	    dump_reference_list(&spelling_obj_ptr->header.object.references);
	}
    }
}



void check_month(ObjRef* month_ref)
{
    C_struct_Month month_obj;
    char number[4];
#if 0
    int i;
    size_t size;
    ObjRef ref;
    ReferenceList* ref_list;
    int num_refs;
#endif

    printf("Month: %08x\n", nwos_ref_to_word(month_ref));

    assert(nwos_read_object_from_disk(month_ref, &month_obj, sizeof(month_obj)));

    nwos_ordinal_number_to_string(&month_obj.number, number, sizeof(number));

    printf("Number: %s\n", number);

    if (!is_void_reference(&month_obj.definition))
    {
	printf("WARNING: definition in Month object is non-void: %08x\n", nwos_ref_to_word(&month_obj.definition));
    }
}


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}

void check_file_path(ObjRef* file_path_ref)
{
    int i;
    char buffer[64];
    uint8 kludge[MAX_PATH_OBJ_SIZE];
    C_struct_File_Path* file_path_obj_ptr = (C_struct_File_Path*)kludge;
    C_struct_Path_And_File_Association assoc_obj;
    ObjRef assoc_class_ref;
    ObjRef object_class;
#if 0
    size_t size;
    ObjRef ref;
#endif
    ReferenceList* ref_list;
    int num_refs;

    printf("File Path: %08x ", nwos_ref_to_word(file_path_ref));
    fflush(stdout);

    assert(nwos_read_variable_sized_object_from_disk(file_path_ref, kludge, sizeof(kludge), &get_path_object_size));

    for (i = 0; i < file_path_obj_ptr->count; i++) buffer[i] = file_path_obj_ptr->storage[i];
    buffer[i] = '\0';

    printf(": %s\n", buffer);

    ref_list = nwos_malloc_reference_list(&file_path_obj_ptr->header.object.references);

    num_refs = ref_list->common_header.num_refs;

    printf("Number of refrences: %d\n", num_refs);

    assert(nwos_find_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref));

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &assoc_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &assoc_obj, sizeof(assoc_obj)));

	    if (!is_same_object(file_path_ref, &assoc_obj.path))
	    {
		printf("ERROR: file path object not in path and file assoc. object: %08x\n",
		       nwos_ref_to_word(&assoc_obj.path));
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;
}


	// Need this for something?
#if 0
    if (!reference_list_has_reference(&file_obj.header.object.references, file_path_ref))
    {
	printf("ERROR: file path object not in file reference list: %08x\n",
	       nwos_ref_to_word(&file_obj.header.object.references));
	dump_reference_list(&file_obj.header.object.references);
    }
#endif


void check_class(ObjRef* class_ref)
{
    int i;
#if 0
    size_t size;
    ObjRef ref;
#endif
    char buffer[64];
    uint8 kludge[MAX_CLASS_DEFINITION_OBJ_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*) kludge;
    ReferenceList* ref_list;
    int num_refs;

    printf("Reading class definition: %08x\n", nwos_ref_to_word(class_ref));

    assert(nwos_read_variable_sized_object_from_disk(class_ref, kludge, sizeof(kludge), &nwos_get_object_size));

    nwos_name_to_string(&class_def_obj_ptr->name, buffer, sizeof(buffer));

    printf("Class name: %s\n", buffer);

    if (!is_same_object(&class_def_obj_ptr->header.common.class_definition, &nwos_private_class_definition_class_ref))
    {
	printf("Class is wrong: %08x  should be: %08x\n",
	       nwos_ref_to_word(&class_def_obj_ptr->header.common.class_definition),
	       nwos_ref_to_word(&nwos_private_class_definition_class_ref));
    }
    else if (strcasecmp(buffer, "REFERENCE LIST") != 0)    /* reference lists don't have reference lists */
    {
	printf("Reading reference list: %08x\n", nwos_ref_to_word(&class_def_obj_ptr->header.object.references));
	fflush(stdout);

	ref_list = nwos_malloc_reference_list(&class_def_obj_ptr->header.object.references);

	num_refs = ref_list->common_header.num_refs;

	printf("Number of refrences: %d\n", num_refs);

	for (i = 0; i < num_refs; i++)
	{
	    if (strcasecmp(buffer, "SPELLING") == 0)
	    {
		check_spelling(&ref_list->references[i]);
	    }
	    else if (strcasecmp(buffer, "NAME") == 0)
	    {
		check_name(&ref_list->references[i]);
	    }
	    else if (strcasecmp(buffer, "MONTH") == 0)
	    {
		check_month(&ref_list->references[i]);
	    }
	    else if (strcasecmp(buffer, "FILE PATH") == 0)
	    {
		check_file_path(&ref_list->references[i]);
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }
}


int main(int argc, char* argv[])
{
    int i;
#if 0
    size_t size;
    ObjRef ref;
#endif
    char buffer[64];
    uint8 kludge[MAX_CLASS_DEFINITION_OBJ_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*) kludge;
    ReferenceList* ref_list;
    int num_refs;


    nwos_initialize_objectify(READ_ONLY, DEFAULT_FILE);

    printf("Reading class definition class: %08x\n", nwos_ref_to_word(&nwos_private_class_definition_class_ref));

    assert(nwos_read_variable_sized_object_from_disk(&nwos_private_class_definition_class_ref, kludge, sizeof(kludge), &nwos_get_object_size));

    nwos_name_to_string(&class_def_obj_ptr->name, buffer, sizeof(buffer));


    printf("Class name: %s\n", buffer);

    printf("Reading class definition class reference list: %08x\n", nwos_ref_to_word(&class_def_obj_ptr->header.object.references));

    ref_list = nwos_malloc_reference_list(&class_def_obj_ptr->header.object.references);

    num_refs = ref_list->common_header.num_refs;

    printf("Number of refrences: %d\n", num_refs);

    for (i = 0; i < num_refs; i++)
    {
	check_class(&ref_list->references[i]);
    }

    nwos_free_reference_list(ref_list);

    return 0;
}

