/*
--             This file is part of the New World OS project
--                 Copyright (C) 2004-2009  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: bit_map.c,v $
-- Revision 1.18  2009/07/11 22:09:01  jsedwards
-- Changed to use total private chunks instead of total private blocks.
--
-- Revision 1.17  2009/07/07 03:12:57  jsedwards
-- Commented out print statement when testing MD5 sum of chunks.
--
-- Revision 1.16  2009/06/30 10:45:43  jsedwards
-- Added include of new header.h file.
--
-- Revision 1.15  2009/06/24 13:16:19  jsedwards
-- Changed to only verify chunk MD5 sums at random instead of every one that
-- is read.  Otherwise it is painfully slow.
--
-- Revision 1.14  2009/05/09 16:06:09  jsedwards
-- Changed to use new nwos_blocks_used_in_chunk function instead of accessing
-- used block count in chunk_info table directly.
--
-- Revision 1.13  2009/04/19 15:27:31  jsedwards
-- Added nwos_calculate_chunk_md5sum so terminate_chunk_info can update any
-- chunks where blocks were modified but the bit map was not changed.
--
-- Revision 1.12  2009/04/19 14:44:30  jsedwards
-- Created new calculate_chunk_md5sum with code taken from write_bit_map and
-- read_bit_map_into_cache and changed them to call it.
--
-- Revision 1.11  2009/04/13 14:44:28  jsedwards
-- Changed to pass new Private_Archive value instead of Private_Reference to
-- nwos_read_block_from_offset function.
--
-- Revision 1.10  2009/04/13 12:47:00  jsedwards
-- Added include of assert.h file.
--
-- Revision 1.9  2009/04/09 11:49:59  jsedwards
-- Added nwos_block_offset_to_chunks to chunk_start_block and changed to call
-- nwos_read_block_from_offset with private reference instead of public.
--
-- Revision 1.8  2009/04/08 14:54:57  jsedwards
-- Changed to call new nwos_read_block_from_offset function instead of reading
-- the blocks directly.
--
-- Revision 1.7  2009/04/07 14:28:45  jsedwards
-- Merged in changes from disk_io.c file in CVS
-- branch_0030_new_chunk_info branch, see revision 1.89.2.6 in disk_io.c file.
--
-- Revision 1.6  2009/03/14 23:14:04  jsedwards
-- Added include of the new chunk_info.h file.
--
-- Revision 1.5  2009/03/14 11:46:46  jsedwards
-- Added include of gen_id.h file.
--
-- Revision 1.4  2009/03/14 03:31:24  jsedwards
-- Added include of bit_map.h file.
--
-- Revision 1.3  2009/03/08 00:06:18  jsedwards
-- Changed include objectify_private.h to disk_io.h.
--
-- Revision 1.2  2009/03/07 13:24:30  jsedwards
-- Fixed --minimal-security assert bugs and simplified find next block.
--
-- Revision 1.1  2009/03/06 04:01:11  jsedwards
-- Initial version created from functions taken from the disk_io.c file.
--
*/

#include <assert.h>
#include <stdlib.h>    /* define NULL */
#include <string.h>    /* define memcmp */

#include "gnu/md5.h"

#include "bit_map.h"
#include "chunk_info.h"
#include "disk_io.h"
#include "gen_id.h"
#include "header.h"
#include "log.h"


#define BIT_MAP_CACHE_SIZE 256

typedef struct {
  bool8 in_use;
  bool8 dirty;
  int chunk_index;    /* index into chunk (NOT chunk_info index) */
  int age;
  uint8* map;
} Bit_Map_Cache_Entry;

static Bit_Map_Cache_Entry bit_map_cache[BIT_MAP_CACHE_SIZE];

static int bit_map_tick = 0;




/*********************************************/
/* Functions to manipulate the bit map cache */
/*********************************************/

static int8 bits_in_byte[256] = 
{
/*        0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f */
/* 0 */   0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4,
/* 1 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 2 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 3 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 4 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 5 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 6 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 7 */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* 8 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 9 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* a */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* b */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* c */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* d */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* e */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* f */   4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
};


/****************************************************************************/
/* Calculates the MD5 sum of a chunk and returns the number of blocks used. */
/****************************************************************************/

static uint32 calculate_chunk_md5sum(Bit_Map_Cache_Entry* entry, uint8 md5_digest[MD5_DIGEST_SIZE])
{
    int i;
    int j;
    uint32 chunk_start_block;
    uint32 blocks_used = 0;
    struct md5_ctx md5_context;    /* MD5 checksum context */
    uint8 block[FILE_BLOCK_SIZE];
    char log_msg[80];

    chunk_start_block = (int64)entry->chunk_index * BLOCKS_IN_CHUNK + nwos_block_offset_to_chunks;

    md5_init_ctx(&md5_context);   /* initialize the MD5 checksum context */

    for (i = 0; i < BIT_MAP_BYTES; i++)
    {
	blocks_used += bits_in_byte[entry->map[i]];

	for (j = 0; j < 8; j++)
	{
	    if ((entry->map[i] & (0x80 >> j)) != 0)
	    {
		if (!nwos_read_block_from_offset(Private_Archive, (int64)(chunk_start_block + BIT_MAP_BLOCKS + (i * 8) + j) * FILE_BLOCK_SIZE, block))
		{
		    snprintf(log_msg, sizeof(log_msg), "reading block for chunk MD5 sum: %08x", chunk_start_block + BIT_MAP_BLOCKS + (i * 8) + j);
		    perror(log_msg);
		    exit(1);
		}

		md5_process_bytes(block, sizeof(block), &md5_context);    /* include this data in the md5 checksum */
	    }
	}
    }

    md5_finish_ctx(&md5_context, md5_digest);   /* finish computing the md5 sum */

    return blocks_used;
}


static void write_bit_map(Bit_Map_Cache_Entry* entry)
{
    int info_index;
    uint32 blocks_used;


    assert(0 <= entry->chunk_index && entry->chunk_index < nwos_used_private_chunks);

    info_index = nwos_chunk_index_to_info_index(entry->chunk_index);

    assert(0 <= info_index && info_index < nwos_used_private_chunks);
    assert(nwos_chunk_info[info_index].index == entry->chunk_index);

#ifdef LOG_BIT_MAP
    snprintf(log_msg, sizeof(log_msg), "write bit_map - info_index: %d  chunk: %d - %08x", info_index, entry->chunk_index, chunk);
    nwos_log(log_msg);
#endif

    blocks_used = calculate_chunk_md5sum(entry, nwos_chunk_info[info_index].md5_digest);

    nwos_clear_chunk_md5_verified(info_index);

    assert(blocks_used <= BLOCKS_IN_CHUNK - BIT_MAP_BLOCKS);

    if (blocks_used != nwos_blocks_used_in_chunk(info_index))
    {
	nwos_reset_chunk_info_blocks_used(info_index, blocks_used);
    }

    nwos_write_bit_map(entry->chunk_index, entry->map);

    entry->dirty = false;
}


static char* md5sum_to_string(uint8 md5_digest[MD5_DIGEST_SIZE])
{
    int i;
    static char result[MD5_DIGEST_SIZE * 2 + 1];
    static char hex[] = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };

    for (i = 0; i < MD5_DIGEST_SIZE; i++)
    {
	result[i*2] = hex[md5_digest[i] >> 4];
	result[i*2 + 1] = hex[md5_digest[i] & 0xf];
    }

    result[i*2] = '\0';

    return result;
}


static void read_bit_map_into_cache(Bit_Map_Cache_Entry* entry)
{
    int i;
    int info_index;
    uint32 blocks_used = 0;
    uint8 md5_digest[MD5_DIGEST_SIZE];
    char log_msg[80];
    static int md5_skip_count = -1;     // number of chunks to skip until next test

    if (md5_skip_count < 0) md5_skip_count = random() % 1001;

#ifdef LOG_BIT_MAP
    snprintf(log_msg, sizeof(log_msg), "read bit_map: %08x\n", entry->chunk);
    nwos_log(log_msg);
#endif

    assert(0 <= entry->chunk_index && entry->chunk_index < nwos_used_private_chunks);

    info_index = nwos_chunk_index_to_info_index(entry->chunk_index);

    assert(0 <= info_index && info_index < nwos_used_private_chunks);
    assert(nwos_chunk_info[info_index].index == entry->chunk_index);

    if (entry->map == NULL)
    {
	entry->map = malloc(BIT_MAP_BYTES);
	assert(entry->map != NULL);
    }

    nwos_read_bit_map(entry->chunk_index, entry->map);

    if (md5_skip_count == 0 && !nwos_chunk_md5_has_been_verified(info_index))    /* it is time to do another test */
    {
	for (i = 0; i < MD5_DIGEST_SIZE; i++)
	{
	    if (nwos_chunk_info[info_index].md5_digest[i] != 0) break;
	}
    }
    else
    {
	i = MD5_DIGEST_SIZE;    /* don't test */
    }

    if (i < MD5_DIGEST_SIZE)   /* non-zero value was found in checksum */
    {
	snprintf(log_msg, sizeof(log_msg), "Verifying MD5 checksum for chunk: %08llx", nwos_chunk_info[info_index].ref);
	nwos_log(log_msg);
	/* fprintf(stderr, "%s\n", log_msg); */

	blocks_used = calculate_chunk_md5sum(entry, md5_digest);

	if (memcmp(nwos_chunk_info[info_index].md5_digest, md5_digest, sizeof(md5_digest)) != 0)   /* make sure they match */
	{
	    snprintf(log_msg, sizeof(log_msg), "Error: chunk MD5 checksum mismatch for chunk %08llx:", nwos_chunk_info[info_index].ref);
	    nwos_log(log_msg);
	    fprintf(stderr, "%s\n", log_msg);

	    snprintf(log_msg, sizeof(log_msg), "      stored: %s", md5sum_to_string(nwos_chunk_info[info_index].md5_digest));
	    nwos_log(log_msg);
	    fprintf(stderr, "%s\n", log_msg);

	    snprintf(log_msg, sizeof(log_msg), "  calculated: %s", md5sum_to_string(md5_digest));
	    nwos_log(log_msg);
	    fprintf(stderr, "%s\n", log_msg);

	    exit(1);
	}

	nwos_set_chunk_md5_verified(info_index);

	md5_skip_count = random() % 1001;
    }
    else
    {
	assert(md5_skip_count >= 0);

	for (i = 0; i < BIT_MAP_BYTES; i++)
	{
	    blocks_used += bits_in_byte[entry->map[i]];
	}

        if (md5_skip_count > 0) md5_skip_count--;
    }

    assert(blocks_used <= BLOCKS_IN_CHUNK - BIT_MAP_BLOCKS);

    if (blocks_used != nwos_blocks_used_in_chunk(info_index))
    {
	nwos_reset_chunk_info_blocks_used(info_index, blocks_used);
    }

    entry->in_use = true;
    entry->dirty = false;
}


static Bit_Map_Cache_Entry* find_bit_map_in_cache(uint32 block)
{
    int i;
    uint16 chunk_index;
    Bit_Map_Cache_Entry* result = NULL;

    assert(nwos_block_offset_to_chunks <= block && block < nwos_block_offset_to_chunks + nwos_total_private_chunks * BLOCKS_IN_CHUNK);

    /* assert((block % CHUNK_SIZE) != 0);  what was this for? */

    chunk_index = (block - nwos_block_offset_to_chunks) / BLOCKS_IN_CHUNK;


    /* see if it's already in the cache */
    for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
    {
	if (bit_map_cache[i].in_use && bit_map_cache[i].chunk_index == chunk_index) break;
    }

    if (i < BIT_MAP_CACHE_SIZE)   /* found it */
    {
	result = &bit_map_cache[i];
    }
    else                           /* didn't find it */
    {
	/* find an empty one or find the oldest */

	result = bit_map_cache;   /* so we have an age to compare to */

	for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
	{
	    if (!bit_map_cache[i].in_use)
	    {
		result = &bit_map_cache[i];
		break;
	    }

	    if (bit_map_cache[i].age < result->age)
	    {
		result = &bit_map_cache[i];
	    }
	}

	if (i == BIT_MAP_CACHE_SIZE && result->dirty)    /* didn't find an empty one, write the oldest one out */
	{
	    write_bit_map(result);
	}

	result->chunk_index = chunk_index;

	read_bit_map_into_cache(result);
    }

    bit_map_tick++;
    result->age = bit_map_tick;

    assert(result != NULL);

    return result;
}


void nwos_set_bit_in_map(uint32 block)
{
    int byte_num;
    int bit_num;
    int info_index;
    Bit_Map_Cache_Entry* entry;

    entry = find_bit_map_in_cache(block);

    byte_num = (block % BLOCKS_IN_CHUNK - BIT_MAP_BLOCKS) / 8;
    assert(byte_num < BIT_MAP_BYTES);
    bit_num = block % 8;

    if ((entry->map[byte_num] & (0x80 >> bit_num)) == 0)    /* don't count a block that was already used */
    {
	entry->map[byte_num] |= (0x80 >> bit_num);

/* printf("set_bit_in_map block: %08x  chunk: %08x  byte: %d  bit: %d\n", block, entry->chunk, byte_num, bit_num); */

	entry->dirty = true;

	info_index = nwos_chunk_index_to_info_index(entry->chunk_index);

	assert(0 <= info_index && info_index < nwos_used_private_chunks);
	assert(nwos_chunk_info[info_index].index == entry->chunk_index);

	nwos_increment_chunk_info_block_used(info_index);

	nwos_used_private_blocks++;
    }
}


void nwos_clear_bit_in_map(uint32 block)
{
    int byte_num;
    int bit_num;
    int info_index;
    Bit_Map_Cache_Entry* entry;

    entry = find_bit_map_in_cache(block);

    byte_num = (block % BLOCKS_IN_CHUNK - BIT_MAP_BLOCKS) / 8;
    assert(byte_num < BIT_MAP_BYTES);
    bit_num = block % 8;

    if ((entry->map[byte_num] & (0x80 >> bit_num)) != 0)    /* don't count a block that was already clear */
    {
	entry->map[byte_num] &= ~(0x80 >> bit_num);

/* printf("clear_bit_in_map block: %08x  chunk: %08x  byte: %d  bit: %d\n", block, entry->chunk, byte_num, bit_num); */

	entry->dirty = true;

	info_index = nwos_chunk_index_to_info_index(entry->chunk_index);

	assert(0 <= info_index && info_index < nwos_used_private_chunks);
	assert(nwos_chunk_info[info_index].index == entry->chunk_index);

	nwos_decrement_chunk_info_block_used(info_index);

	nwos_used_private_blocks--;
    }
}


bool nwos_test_bit_in_map(uint32 block)
{
    int byte_num;
    int bit_num;
    Bit_Map_Cache_Entry* entry;

    entry = find_bit_map_in_cache(block);

    byte_num = (block % BLOCKS_IN_CHUNK - BIT_MAP_BLOCKS) / 8;
    assert(byte_num < BIT_MAP_BYTES);
    bit_num = block % 8;

    return (entry->map[byte_num] & (0x80 >> bit_num)) != 0;
}


bool nwos_block_used(ObjRef* ref)
{
#ifdef PUBLIC_MODE
    uint8 block[FILE_BLOCK_SIZE];

    return  nwos_read_block(ref, block) && !is_void_reference((ObjRef*)&block[4]);
#else
    uint32 block;
    int byte_num;
    int bit_num;
    Bit_Map_Cache_Entry* entry;

    block = nwos_hash_ref(ref);

    entry = find_bit_map_in_cache(block);

    byte_num = (block % BLOCKS_IN_CHUNK - BIT_MAP_BLOCKS) / 8;
    assert(byte_num < BIT_MAP_BYTES);
    bit_num = block % 8;

    return ((entry->map[byte_num] & (0x80 >> bit_num)) != 0);
#endif
}


uint32 nwos_find_random_block_in_chunk(int info_index)
{
    int i;
    int bit;
    int byte;
    Bit_Map_Cache_Entry* entry;

    entry = find_bit_map_in_cache(nwos_block_offset_to_chunks + ((uint32)nwos_chunk_info[info_index].index * BLOCKS_IN_CHUNK) + BIT_MAP_BLOCKS);

    /* don't pick one of the bytes (currently 0-3) used by the bit map */
    byte = (random() % (BIT_MAP_BYTES - (BIT_MAP_BLOCKS / 8))) + (BIT_MAP_BLOCKS / 8);

    if (entry->map[byte] == 0xff)   /* didn't get lucky */
    {
	for (i = 1; i < BIT_MAP_BYTES; i++)   /* find the nearest empty */
	{
	    if (byte - i >= 0 && entry->map[byte - i] != 0xff)
	    {
		byte -= i;
		break;
	    }

	    if (byte + i < BIT_MAP_BYTES && entry->map[byte + i] != 0xff)
	    {
		byte += i;
		break;
	    }

	    assert(0 <= byte - i || byte + i < BIT_MAP_BYTES);
	}
    }

    assert(BIT_MAP_BLOCKS / 8 <= byte && byte < BIT_MAP_BYTES);
    assert(entry->map[byte] != 0xff);

    bit = random() % 8;

    if ((entry->map[byte] & (0x80 >> bit)) != 0)
    {
	for (i = 1; i < 8; i++)   /* find the nearest empty */
	{
	    if (bit - i >= 0 && (entry->map[byte] & (0x80 >> (bit - i))) == 0)
	    {
		bit -= i;
		break;
	    }

	    if (bit + i < 8 && (entry->map[byte] & (0x80 >> (bit + i))) == 0)
	    {
		bit += i;
		break;
	    }

	    assert(0 <= bit - i || bit + i < 8);
	}
    }

    assert(0 <= bit && bit < 8);

    return nwos_chunk_info[info_index].ref + byte * 8 - BIT_MAP_BLOCKS + bit;
}


int nwos_find_low_density_in_chunk(int info_index, int density)
{
    Bit_Map_Cache_Entry* entry = NULL;
    int byte_num;
    int bits_used = 0;

    entry = find_bit_map_in_cache(nwos_block_offset_to_chunks + nwos_chunk_info[info_index].index * BLOCKS_IN_CHUNK + BIT_MAP_BLOCKS);

    for (byte_num = BIT_MAP_BYTES - 1; byte_num > 0; byte_num--)
    {
	bits_used += bits_in_byte[entry->map[byte_num]];

	if (bits_used / (BIT_MAP_BYTES - byte_num) > density) break;
    }

    return byte_num;
}


uint32 nwos_find_next_empty_block(uint32 start_block, int density)
{
    Bit_Map_Cache_Entry* entry;
    uint32 offset;
    uint32 chunk;
    int    byte_num;
    int    bit_num;
    uint8  byte;
    int    info_index;
    int    skipped = 0;

    offset = (start_block - nwos_block_offset_to_chunks) % BLOCKS_IN_CHUNK;
    chunk = start_block - offset;

    assert(BIT_MAP_BLOCKS <= offset);

    entry = find_bit_map_in_cache(chunk);
    info_index = nwos_chunk_index_to_info_index(entry->chunk_index);

    assert(entry->chunk_index == (chunk - nwos_block_offset_to_chunks) / BLOCKS_IN_CHUNK);

    byte_num = offset / 8;
    bit_num = offset % 8;

    while (byte_num < BIT_MAP_BYTES)
    {
	assert(bit_num < 8);
	assert(0 <= info_index && info_index < nwos_used_private_chunks);
	assert(nwos_chunk_info[info_index].index == entry->chunk_index);
	assert(nwos_chunk_info[info_index].index == (chunk - nwos_block_offset_to_chunks) / BLOCKS_IN_CHUNK);

	byte = entry->map[byte_num];

	if ((byte & (0x80 >> bit_num)) == 0)  /* this block is empty */
	{
	    if ((random() % 8) < density + skipped)   /* randomly skip some blocks */
	    {
		break;                                   /* use this empty */
	    }

	    skipped++;
	}

	bit_num++;                                /* check the next bit */

	if (bit_num == 8)
	{
	    bit_num = 0;
	    byte_num++;
	}
    }

    offset = (byte_num * 8) + bit_num;

    assert(BIT_MAP_BLOCKS <= offset && offset <= BLOCKS_IN_CHUNK);

    return chunk + offset;
}


void nwos_flush_bit_maps()
{
    int i;

    for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
    {
	if (bit_map_cache[i].in_use && bit_map_cache[i].dirty)
	{
	    write_bit_map(&bit_map_cache[i]);
	}
    }
}


void nwos_terminate_bit_map(void)
{
    int i;

    for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
    {
	if (bit_map_cache[i].in_use && bit_map_cache[i].dirty)
	{
	    write_bit_map(&bit_map_cache[i]);

	    bit_map_cache[i].in_use = false;

	    free(bit_map_cache[i].map);
	    bit_map_cache[i].map = NULL;
	}
    }
}


void nwos_calculate_chunk_md5sum(uint32 chunk_index, uint8 md5_digest[MD5_DIGEST_SIZE])
{
    Bit_Map_Cache_Entry* entry;


    entry = find_bit_map_in_cache(nwos_block_offset_to_chunks + (chunk_index * BLOCKS_IN_CHUNK) + BIT_MAP_BLOCKS);

    calculate_chunk_md5sum(entry, md5_digest);    /* ignore the blocks_used that it returns */
}


