/*
--          This file is part of the New World OS and Objectify projects
--               Copyright (C) 2007, 2008, 2009, 2010  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2010-02-28 08:14:08 -0700 (Sun, 28 Feb 2010) $
--   $Revision: 4600 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "disk_io.h"
#include "header.h"
#include "strlcxx.h"


#define MAX_FILES 100
#define MAX_BUFFER_SIZE 1048576
#define MAX_BLOCKS (MAX_BUFFER_SIZE / FILE_BLOCK_SIZE)



typedef struct {
    uint8 block[4];
    ObjRef ref;
    uint8 body[FILE_BLOCK_SIZE - sizeof(ObjRef) - 4];
} Dummy_Block;


typedef struct
{
    char* path;

    Disk_Header header1;
    uint8 unused1[FILE_BLOCK_SIZE - sizeof(Disk_Header)];

    Disk_Header header2;
    uint8 unused2[FILE_BLOCK_SIZE - sizeof(Disk_Header)];

    uint32 offset;

    int num_blocks;
    int index;

    Dummy_Block blocks[0];
} Backup_File;


void print_usage(char* program)
{
    fprintf(stderr, "usage: %s input-file-1 [...] input-file-n output-file\n", program);
    fprintf(stderr, "maximum number of input files is: %d\n", MAX_FILES);
}



/* Returns -1 if error, 0 if end of file, and > 0 if new chunk read in */

int read_next_file_chunk(Backup_File* file_ptr)
{
    FILE* fp;
    int block;
    int errsv;

    file_ptr->index = 0;

    if (file_ptr->num_blocks < MAX_BLOCKS)
    {
	file_ptr->num_blocks = 0;    /* end of file */
	return 0;
    }

    fp = fopen(file_ptr->path, "r");

    if (fp == NULL)
    {
	return -1;
    }

    file_ptr->offset += MAX_BLOCKS;

    if (fseek(fp, (off_t)file_ptr->offset * FILE_BLOCK_SIZE, SEEK_SET) < 0)
    {
	return -1;
    }

    block = 0;

    while (block < MAX_BLOCKS)
    {
	if (fread(&file_ptr->blocks[block].block, FILE_BLOCK_SIZE, 1, fp) != 1)
	{
	    if (feof(fp)) break;

	    errsv = errno;
	    fclose(fp);
	    errno = errsv;

	    return -1;
	}

	block++;
    }

    file_ptr->num_blocks = block;

    if (fclose(fp) != 0)
    {
	return -1;
    }

    return block;
}


int main(int argc, char* argv[])
{
    int i;
    int block;
    int num_input_files;
    int lowest;
    FILE* fp;
    uint32 ref1;
    uint32 ref2;
    bool fatal_error = false;
    uint32 temp32_1;
    uint32 temp32_2;
    uint64 temp64_1;
    uint64 temp64_2;
    char temp_string_1[24];
    char temp_string_2[24];


    Backup_File* files[MAX_FILES];

    for (i = 1; i < argc; i++)
    {
	if (*argv[i] == '-')
	{
	    fprintf(stderr, "This program has no options.\n");
	    print_usage(argv[0]);
	    exit(1);
	}
    }

    if (argc < 4)
    {
	fprintf(stderr, "At least two input files and one output file must be specified.\n");
	print_usage(argv[0]);
	exit(1);
    }


    for (i = 0; i < MAX_FILES; i++) files[i] = NULL;

    if (argc > MAX_FILES + 2)
    {
	fprintf(stderr, "This version can only combine a maximum of %d files.\n", MAX_FILES);
	exit(1);
    }

    fp = fopen(argv[argc-1], "r");

    if (fp != NULL)
    {
	printf("ERROR: output file: %s already exists, will not overwrite.\n", argv[argc-1]);
	fclose(fp);
	exit(1);
    }

    num_input_files = argc - 2;  /* ignore program name and output file name */

    for (i = 0; i < num_input_files; i++)    /* the last name is the output file */
    {
	printf("%s: ", argv[i+1]);
	fflush(stdout);

	files[i] = malloc(sizeof(Backup_File) + MAX_BUFFER_SIZE);   /* 1 MB for blocks */

	if (files[i] == NULL)
	{
	    perror("allocating buffer");
	    fclose(fp);
	    exit(1);
	}

	files[i]->path = argv[i+1];
	files[i]->index = 0;

	fp = fopen(files[i]->path, "r");

	if (fp == NULL)
	{
	    perror(NULL);
	    exit(1);
	}

	if (fread(&files[i]->header1, FILE_BLOCK_SIZE, 1, fp) != 1)
	{
	    perror(NULL);
	    fclose(fp);
	    exit(1);
	}

	if (i > 0 && memcmp(&files[i-1]->header2, &files[i]->header1, sizeof(Disk_Header)) != 0)  /* if the header doesn't match exactly, see if it a significant difference */
	{
	    fprintf(stderr, "\n");

	    if (memcmp(&files[i]->header1.magic_number, &files[i-1]->header2.magic_number, sizeof(files[i]->header1.used_blocks)) != 0)
	    {
		fprintf(stderr, "ERROR: header 1 magic number: %c%c%c%c doesn't match file '%s' header 2 magic number: %c%c%c%c\n",
			files[i]->header1.magic_number[0], files[i]->header1.magic_number[1], files[i]->header1.magic_number[2], files[i]->header1.magic_number[3],
			files[i-1]->path,
			files[i-1]->header2.magic_number[0], files[i-1]->header2.magic_number[1], files[i-1]->header2.magic_number[2], files[i-1]->header2.magic_number[3]);
		fatal_error = true;
	    }

	    if (memcmp(&files[i]->header1.version_string, &files[i-1]->header2.version_string, sizeof(files[i]->header1.used_blocks)) != 0)
	    {
		fprintf(stderr, "ERROR: header 1 version: %c%c%c%c doesn't match file '%s' header 2 version: %c%c%c%c\n",
			files[i]->header1.version_string[0], files[i]->header1.version_string[1], files[i]->header1.version_string[2], files[i]->header1.version_string[3],
			files[i-1]->path,
			files[i-1]->header2.version_string[0], files[i-1]->header2.version_string[1], files[i-1]->header2.version_string[2], files[i-1]->header2.version_string[3]);
		fatal_error = true;
	    }

	    if (memcmp(&files[i]->header1.type_code, &files[i-1]->header2.type_code, sizeof(files[i]->header1.used_blocks)) != 0)
	    {
		fprintf(stderr, "ERROR: header 1 type code: %c%c%c%c doesn't match file '%s' header 2 type code: %c%c%c%c\n",
			files[i]->header1.type_code[0], files[i]->header1.type_code[1], files[i]->header1.type_code[2], files[i]->header1.type_code[3],
			files[i-1]->path,
			files[i-1]->header2.type_code[0], files[i-1]->header2.type_code[1], files[i-1]->header2.type_code[2], files[i-1]->header2.type_code[3]);
		fatal_error = true;
	    }

	    if (memcmp(&files[i]->header1.block_offset_to_chunks, &files[i-1]->header2.block_offset_to_chunks, sizeof(files[i]->header1.block_offset_to_chunks)) != 0)
	    {
		nwos_4_uint8_to_uint32(files[i]->header2.block_offset_to_chunks, &temp32_1);
		nwos_4_uint8_to_uint32(files[i-1]->header2.block_offset_to_chunks, &temp32_2);
		fprintf(stderr, "WARNING: header 1 block offset: %u doesn't match file '%s' header 2 block offset: %u\n", temp32_1, files[i-1]->path, temp32_2);
	    }

	    if (memcmp(&files[i]->header1.last_prep_disk, &files[i-1]->header2.last_prep_disk, sizeof(files[i]->header1.last_prep_disk)) != 0)
	    {
		strlcpy(temp_string_1, nwos_time_stamp_to_string(files[i]->header1.last_prep_disk), sizeof(temp_string_1));
		strlcpy(temp_string_2, nwos_time_stamp_to_string(files[i-1]->header2.last_prep_disk), sizeof(temp_string_2));
		fprintf(stderr, "WARNING: header 1 prep disk time: %s doesn't match file '%s' header 2 prep disk time: %s\n", temp_string_1, files[i-1]->path, temp_string_2);
	    }

	    if (memcmp(&files[i]->header1.total_chunks, &files[i-1]->header2.total_chunks, sizeof(files[i]->header1.total_chunks)) != 0)
	    {
		nwos_4_uint8_to_uint32(files[i]->header2.total_chunks, &temp32_1);
		nwos_4_uint8_to_uint32(files[i-1]->header2.total_chunks, &temp32_2);
		fprintf(stderr, "WARNING: header 1 total chunks: %u doesn't match file '%s' header 2 total chunks: %u\n", temp32_1, files[i-1]->path, temp32_2);
	    }

	    if (memcmp(&files[i]->header1.used_chunks, &files[i-1]->header2.used_chunks, sizeof(files[i]->header1.used_chunks)) != 0)
	    {
		nwos_4_uint8_to_uint32(files[i]->header2.used_chunks, &temp32_1);
		nwos_4_uint8_to_uint32(files[i-1]->header2.used_chunks, &temp32_2);
		fprintf(stderr, "WARNING: header 1 used chunks: %u doesn't match file '%s' header 2 used chunks: %u\n", temp32_1, files[i-1]->path, temp32_2);
	    }

	    if (memcmp(&files[i]->header1.last_change, &files[i-1]->header2.last_change, sizeof(files[i]->header1.last_change)) != 0)
	    {
		strlcpy(temp_string_1, nwos_time_stamp_to_string(files[i]->header1.last_change), sizeof(temp_string_1));
		strlcpy(temp_string_2, nwos_time_stamp_to_string(files[i-1]->header2.last_change), sizeof(temp_string_2));
		fprintf(stderr, "WARNING: header 1 last change time: %s doesn't match file '%s' header 2 last change time: %s\n", temp_string_1, files[i-1]->path, temp_string_2);
	    }

	    if (memcmp(&files[i]->header1.used_blocks, &files[i-1]->header2.used_blocks, sizeof(files[i]->header1.used_blocks)) != 0)
	    {
		nwos_8_uint8_to_uint64(files[i]->header2.used_blocks, &temp64_1);
		nwos_8_uint8_to_uint64(files[i-1]->header2.used_blocks, &temp64_2);
		fprintf(stderr, "ERROR: header 1 used blocks: %llu doesn't match file '%s' header 2 used blocks: %llu\n", temp64_1, files[i-1]->path, temp64_2);
		fatal_error = true;
	    }

	    if (fatal_error)
	    {
		exit(1);
	    }
	}

	if (fread(&files[i]->header2, FILE_BLOCK_SIZE, 1, fp) != 1)
	{
	    perror(NULL);
	    fclose(fp);
	    exit(1);
	}

        block = 0;

	files[i]->offset = 2;

	while (block < MAX_BLOCKS)
	{
	    if (fread(&files[i]->blocks[block].block, FILE_BLOCK_SIZE, 1, fp) != 1)
	    {
		if (feof(fp)) break;

		perror(NULL);
		fclose(fp);
		exit(1);
	    }

	    block++;
	}

	files[i]->num_blocks = block;

	if (fclose(fp) != 0)
	{
	    perror(NULL);
	    exit(1);
	}

	printf("%d blocks\n", files[i]->num_blocks);

	if (files[i]->num_blocks < MAX_BLOCKS)  /* reduce the memory used to what is actually used */
	{
	    files[i] = realloc(files[i], sizeof(Backup_File) + files[i]->num_blocks * FILE_BLOCK_SIZE);
	}

	assert(files[i] != NULL);  /* when reducing memory could it ever fail? */
    }


    /* now we've read all the input files, write the output file */

    fp = fopen(argv[argc-1], "w");

    if (fp == NULL)
    {
	perror(argv[argc-1]);
	exit(1);
    }

    if (fwrite(&files[0]->header1, FILE_BLOCK_SIZE, 1, fp) != 1)
    {
	perror(NULL);
	fclose(fp);
	exit(1);
    }

    if (fwrite(&files[num_input_files-1]->header2, FILE_BLOCK_SIZE, 1, fp) != 1)
    {
	perror(NULL);
	fclose(fp);
	exit(1);
    }

    i = 0;
    while (i < num_input_files)
    {
	for (lowest = 0; lowest < num_input_files; lowest++) if (files[lowest] != NULL) break;

	for (i = lowest + 1; i < num_input_files; i++)
	{
	    if (files[i] != NULL)
	    {
		ref1 = nwos_ref_to_word(&files[lowest]->blocks[files[lowest]->index].ref);
		ref2 = nwos_ref_to_word(&files[i]->blocks[files[i]->index].ref);

		if (ref1 == ref2)  /* second supercedes first */
		{
		    files[lowest]->index++;   /* skip over this block, won't be used */

		    if (files[lowest]->index == files[lowest]->num_blocks)   /* finished with this file? */
		    {
			if (read_next_file_chunk(files[lowest]) < 0)
			{
			    perror(files[lowest]->path);
			    fclose(fp);
			    exit(1);
			}
			if (files[lowest]->num_blocks == 0)             /* yes */
			{
			    free(files[lowest]);
			    files[lowest] = NULL;
			}
		    }

		    lowest = i;
		}
		else if (ref1 > ref2)
		{
		    lowest = i;
		}
	    }
	}

	if (fwrite(&files[lowest]->blocks[files[lowest]->index].block, FILE_BLOCK_SIZE, 1, fp) != 1)
	{
	    perror(argv[argc-1]);
	    fclose(fp);
	    exit(1);
	}

	files[lowest]->index++;   /* this block is done */

	if (files[lowest]->index == files[lowest]->num_blocks)   /* finished with this file */
	{
	    if (read_next_file_chunk(files[lowest]) < 0)
	    {
		perror(files[lowest]->path);
		fclose(fp);
		exit(1);
	    }
	    if (files[lowest]->num_blocks == 0)             /* yes */
	    {
		free(files[lowest]);
		files[lowest] = NULL;
	    }
	}

	/* set i to the first file still in play, so we know when we are done */
	for (i = 0; i < num_input_files; i++) if (files[i] != NULL) break;
    }

    if (fclose(fp) != 0)
    {
	perror(NULL);
	exit(1);
    }

    return 0;
}

