/*
--          This file is part of the New World OS and Objectify projects
--               Copyright (C) 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2010-01-21 05:54:56 -0700 (Thu, 21 Jan 2010) $
--   $Revision: 4470 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revisions of this file and the log_disc.c file which this file
--   was created from.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <assert.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>

#include "bit_map.h"
#include "objectify.h"
#include "crc32.h"


static char* file_names[MAX_FILES_PER_DISC_LIST];
static ObjRef files[MAX_FILES_PER_DISC_LIST];


static void get_disc_id(char* buffer, size_t bufsize)
{
    if (fgets(buffer, bufsize, stdin) == NULL)
    {
	fputc('\n', stderr);

	if (feof(stdin))
	{
	    fprintf(stderr, "ERROR: received end of input from standard input when reading disc ID\n");
	    exit(1);
	}
	else
	{
	    perror("reading disc ID");
	    exit(1);
	}
    }
}


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}


void check_for_previous_version_and_link(ObjRef* new_assoc_ref)
{
#ifdef VERIFY_WRITE
    C_struct_Path_And_File_Association assoc_obj;
#endif
    C_struct_Path_And_File_Association old_assoc_obj;
    C_struct_Path_And_File_Association new_assoc_obj;
    uint8 kludge[MAX_PATH_OBJ_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    ObjRef object_class;

    assert(nwos_read_object_from_disk(new_assoc_ref, &new_assoc_obj, sizeof(new_assoc_obj)));

    assert(is_void_reference(&new_assoc_obj.header.object.prev_version));

    assert(nwos_read_variable_sized_object_from_disk(&new_assoc_obj.path, kludge, sizeof(kludge), &get_path_object_size));

    ref_list = nwos_malloc_reference_list(&ptr_path_obj->header.object.references);

    num_refs = ref_list->common_header.num_refs;

    assert(is_same_object(&ref_list->references[num_refs-1], new_assoc_ref));

    if (num_refs > 1)
    {
	for (i = 0; i < num_refs - 1; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);
	    if (is_same_object(&object_class, &new_assoc_obj.header.common.class_definition))
	    {
		assert(nwos_read_object_from_disk(&ref_list->references[i], &old_assoc_obj, sizeof(old_assoc_obj)));

		printf("Old association: %08x  file: %08x  next_version: %08x\n",
		       nwos_ref_to_word(&ref_list->references[i]),
		       nwos_ref_to_word(&old_assoc_obj.file),
		       nwos_ref_to_word(&old_assoc_obj.header.object.next_version));

		if (is_void_reference(&old_assoc_obj.header.object.next_version)) break;
	    }
	}

	if (i < num_refs - 1)
	{
	    assert(is_same_object(&new_assoc_obj.header.common.id, new_assoc_ref));
	    assert(is_void_reference(&old_assoc_obj.header.object.next_version));
	    assert(is_void_reference(&new_assoc_obj.header.object.prev_version));
	    assert(is_same_object(&old_assoc_obj.path, &new_assoc_obj.path));

	    copy_reference(&old_assoc_obj.header.object.next_version, &new_assoc_obj.header.common.id);
	    copy_reference(&new_assoc_obj.header.object.prev_version, &old_assoc_obj.header.common.id);

	    nwos_crc32_calculate((uint8*) &old_assoc_obj.header.object, sizeof(ObjectHeader), old_assoc_obj.header.common.header_chksum);
	    nwos_crc32_calculate((uint8*) &new_assoc_obj.header.object, sizeof(ObjectHeader), new_assoc_obj.header.common.header_chksum);

	    nwos_overwrite_object_to_disk(&old_assoc_obj.header.common.id, &old_assoc_obj, sizeof(old_assoc_obj));
	    nwos_overwrite_object_to_disk(&new_assoc_obj.header.common.id, &new_assoc_obj, sizeof(new_assoc_obj));

	    printf("linked - old: %08x and new: %08x\n",
		   nwos_ref_to_word(&old_assoc_obj.header.common.id),
		   nwos_ref_to_word(&new_assoc_obj.header.common.id));

#ifdef VERIFY_WRITE
	    assert(nwos_read_object_from_disk(&old_assoc_obj.header.common.id, &assoc_obj, sizeof(assoc_obj)));
	    assert(memcmp(&assoc_obj, &old_assoc_obj, sizeof(assoc_obj)) == 0);

	    assert(nwos_read_object_from_disk(&new_assoc_obj.header.common.id, &assoc_obj, sizeof(assoc_obj)));
	    assert(memcmp(&assoc_obj, &new_assoc_obj, sizeof(assoc_obj)) == 0);
#endif
	}

    }
}


int main(int argc, char* argv[])
{
    ObjRef public_ref;
    ObjRef ref;
    ObjCreateResult result;
    int i;
    time_t start_time;
    char id[14];
    char *p;
    bool ok = false;
    uint32 num_files;
    ObjRef disc_list_ref;
    bool add_revision = false;
    bool ignore_public = false;
    bool public_file_mismatch = false;
    uint32 error_mask = DISC_LIST_ERROR_MASK;
    int argi = 1;

    while (argi < argc - 1)
    {
	if (strcmp(argv[argi], "--add-revision") == 0)
	{
	    add_revision = true;
	    argi++;
	}
	else if (strcmp(argv[argi], "--ignore-public") == 0)
	{
	    ignore_public = true;
	    argi++;
	}
	else if (strcmp(argv[argi], "--ignore-empty-directories") == 0)
	{
	    error_mask &= ~DISC_LIST_EMPTY_DIR_FLAG;
	    argi++;
	}
	else if (strcmp(argv[argi], "--ignore-empty-files") == 0)
	{
	    error_mask &= ~DISC_LIST_EMPTY_FILE_FLAG;
	    argi++;
	}
	else if (*argv[argi] == '-')
	{
	    fprintf(stderr, "\n");
	    fprintf(stderr, "unknown option: %s\n", argv[argi]);
	    break;
	}
    }

    if (argi != argc - 1)
    {
	fprintf(stderr, "\n");
	fprintf(stderr, "usage: %s [options] directory\n", argv[0]);
	fprintf(stderr, "  options:\n");
	fprintf(stderr, "    --add-revision              if file already exists with same name, make it a new revision\n");
	fprintf(stderr, "    --ignore-public             don't check public files for matches\n");
	fprintf(stderr, "    --ignore-empty-directories  ignore any empty directories\n");
	fprintf(stderr, "    --ignore-empty-files        ignore any empty files\n");
	fprintf(stderr, "\n");
	exit(1);
    }

    printf("\n");

    /***********************************************************/
    /* First open the directory and read all of the file names */
    /***********************************************************/

    num_files = nwos_read_files_disc_list(argv[argi], NULL, file_names, 0);    /* NULL for subdirectory indicates this is the root */

    if ((num_files & DISC_LIST_NUM_FILES_MASK) == 0)
    {
	fprintf(stderr, "Error: directory '%s' is empty!\n\n", argv[argi]);

	exit(16);  /* 16 means directory is empty */
    }

    if ((num_files & error_mask) != 0)
    {
	fprintf(stderr, "\n");

	exit(num_files >> DISC_LIST_ERROR_SHIFT);  /* exit code is error code returned by nwos_read_files_disc_list 1-15 */
    }

    num_files &= DISC_LIST_NUM_FILES_MASK;


    /*******************/
    /* Get the disc id */
    /*******************/

    while (!ok)
    {
	printf("Disk id: ");
	fflush(stdout);

	get_disc_id(id, sizeof(id));

	p = strchr(id, '\n');   /* find the newline char */

	if (p == NULL)    /* line was tool long */
	{
	    while (p == NULL) 
	    {
		get_disc_id(id, sizeof(id));
		p = strchr(id, '\n');   /* find the newline char */
	    }
	    printf("id was too long, must be less than %zd characters\n", sizeof(id) - 2);
	}
	else     /* line was ok */
	{
	    *p = '\0';   /* eliminate the newline character */

	    if (strlen(id) < 12)
	    {
		printf("id was too short, must be %zd characters\n", sizeof(id) - 2);
	    }
	    else
	    {
		ok = true;   /* we should be good to go */
	    }
	}
    }


    /*******************************************************************************************/
    /* Before logging any files, open Objectify in public, read only mode and scan all of the  */
    /* files to see if any of their names match any public files.  If any of them do, checksum */
    /* them and if the checksum of any of them doesn't match report an error and exit.         */
    /*******************************************************************************************/

    if (!ignore_public)
    {
	nwos_initialize_objectify(PUBLIC, NULL);

	for (i = 0; i < num_files; i++)
	{
	    p = strrchr(file_names[i], '/');

	    if (p == NULL)    /* no path in file name */
	    {
		p = file_names[i];   /* use it as is */
	    }
	    else
	    {
		p++;   /* point to first character following the slash */
	    }

	    if (nwos_find_file_path(p, &public_ref))
	    {
		printf("checking public file %s: ", p);
		fflush(stdout);

		/* NOTE: when the time comes that we are doing recursive directory searches this will have to be
		         fixed so that the path includes the directory and the part of the path in the file name */
		if (nwos_find_matching_path_and_file_association(argv[argi], file_names[i], &public_ref, IgnoreTime))
		{
		    printf(" OK\n");
		}
		else
		{
		    printf(" does not match public file information!\n");
		    public_file_mismatch = true;
		}
	    }
	}

	nwos_terminate_objectify();
    }

    if (public_file_mismatch)
    {
	fprintf(stderr, "\n");
	fprintf(stderr, "ERROR: disc_list not created!  One or more files with the same name as a public\n");
	fprintf(stderr, "       file were found but the checksums don't match.  You may have a bad\n");
        fprintf(stderr, "       download.  To ignore any public files use the --ignore-public option.\n");
	fprintf(stderr, "\n");

	exit(1);
    }

    printf("\n");


    /***********************************************************************/
    /* Now open Objectify in private, read-write mode to do the real thing */
    /***********************************************************************/


    nwos_log_arguments(argc, argv);

    nwos_initialize_objectify(READ_WRITE, DEFAULT_FILE);

    if (!nwos_check_blocks_available(2 + num_files * 12)) /* 1 file = 14, 2 files = 26, 3 files = 38... */
    {
	fprintf(stderr, "Cannot add disc_list!\n");
	nwos_terminate_objectify();
	exit(1);
    }

    if (nwos_find_disc_list(id, &ref))
    {
	printf("That disc already exists!\n");
	nwos_terminate_objectify();
	exit(1);
    }

    memset(files, 0, sizeof(files));

    for (i = 0; i < num_files; i++)
    {
	start_time = time(NULL);

	result = nwos_create_file_without_storing_data(argv[argi], file_names[i], &files[i]);

	assert(!is_void_reference(&files[i]));

	if (result == CREATED_NEW)
	{
	    printf("created new file path: %02x%02x%02x%02x   time: %d seconds\n", 
		   files[i].id[0], files[i].id[1], files[i].id[2], files[i].id[3], 
		   (int) (time(NULL) - start_time));

	    if (add_revision)   /* check to see if there was a previous revision of this file */
	    {
		check_for_previous_version_and_link(&files[i]);
	    }
	}
	else
	{
	    printf("result: %d\n", result);
	}

	nwos_flush_bit_maps();
    }

    assert(nwos_create_disc_list(id, files, &disc_list_ref) == CREATED_NEW);

    printf("disc list: %02x%02x%02x%02x\n",
	   disc_list_ref.id[0], disc_list_ref.id[1], disc_list_ref.id[2], disc_list_ref.id[3]);

    for (i = 0; i < num_files; i++)
    {
	free(file_names[i]);
    }

    nwos_terminate_objectify();

    return 0;
}


