/*
--          This file is part of the New World OS and Objectify projects
--                       Copyright (C) 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-10-06 08:06:23 -0600 (Tue, 06 Oct 2009) $
--   $Revision: 4384 $
--
--
--  This program is a hack to reorder some photos after I renamed them.
--  I saved it only as an example.  Soon something like this won't be
--  needed, as I plan to order the reference lists.
--
*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "objectify.h"
#include "crc32.h"
#include "security.h"


#define MAX_REFS_IN_REF_LIST (((FILE_BLOCK_SIZE - sizeof(CommonHeader)) / sizeof(ObjRef)) - 1)
#define MAX_REFS_IN_SIDECAR  (((FILE_BLOCK_SIZE - 12) / sizeof(ObjRef)) - 1)

typedef struct ref_list_extra_block {
    struct ref_list_extra_block* next_block_ptr;
    int dirty;
    ObjRef id;
    uint8 checksum[4];
    ObjRef refs[MAX_REFS_IN_SIDECAR];
    ObjRef next_block_ref;
    uint8 ivec[IVEC_SIZE];
} Ref_List_Extra_Block;

typedef struct {
    struct ref_list_extra_block* next_block_ptr;
    ReferenceList list;
    ObjRef refs[MAX_REFS_IN_REF_LIST];
    ObjRef next_block_ref;
    uint8 ivec[IVEC_SIZE];
} Ref_List_First_Block;


#define REF_LIST_CACHE_SIZE 64

typedef struct {
  ObjRef ref;
  int num_refs;
  Ref_List_First_Block* first_block;
  Ref_List_Extra_Block* last_block;        /* null if only one block in list */
  int age;
} Block_Cache;


extern Block_Cache ref_list_cache[REF_LIST_CACHE_SIZE];

extern Block_Cache* find_ref_list_in_cache(ObjRef* ref);


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}


int main(int argc, char* argv[])
{
    C_struct_Class_Definition class_def_obj;
    ObjRef assoc_class_ref;
    ReferenceList* old_assoc_list;
    ReferenceList* new_assoc_list;
    int i;
    int j;
    Block_Cache* assoc_block_cache;
    Ref_List_Extra_Block* extra_block;
    Ref_List_Extra_Block* empty_block;
    Ref_List_Extra_Block* write_block;
    int empty_index;
    int write_seq;
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    char path_name[65];
    enum { Searching, Copying, Moving } state = Searching;
    char* img_string = "20090531-001/img_";
    char* szq_string = "20090531-001/szq_";
    ObjRef szq_refs[512];
    int num_szq = 0;
    int seq;
    uint8 ivec[IVEC_SIZE];

    char* path = DEFAULT_FILE;

    if (argc != 1)
    {
	fprintf(stderr, "usage: %s\n", argv[0]);
	fprintf(stderr, "  This program reorders the 20090531-001 files.\n");
	exit(1);
    }

//    nwos_log_arguments(argc, argv);
    nwos_log_arguments(0, NULL);

    printf("\n");

    nwos_initialize_objectify(READ_WRITE, path);
//    nwos_initialize_objectify(READ_ONLY, path);

    assert(nwos_find_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref));

    nwos_read_class_definition(&assoc_class_ref, &class_def_obj);

    old_assoc_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    assoc_block_cache = find_ref_list_in_cache(&class_def_obj.header.object.references);

    assert(is_same_object(&assoc_block_cache->ref, &assoc_block_cache->first_block->list.common_header.id));

    printf("first block: %08x\n", nwos_ref_to_word(&assoc_block_cache->ref));

    for (i = 0; i < MAX_REFS_IN_REF_LIST; i++)
    {
	if (!nwos_read_object_from_disk(&assoc_block_cache->first_block->refs[i], &assoc_obj, sizeof(assoc_obj)))
	{
	    fprintf(stderr, "Reading association object failed\n");
	    exit(1);
	}

	if (!nwos_read_variable_sized_object_from_disk(&assoc_obj.path, kludge, sizeof(kludge), &get_path_object_size))
	{
	    fprintf(stderr, "Reading path object failed\n");
	    exit(1);
	}

	if (ptr_path_obj->count > 64)
	{
	    fprintf(stderr, "path too long\n");
	    exit(1);
	}

	memcpy(path_name, ptr_path_obj->storage, ptr_path_obj->count);
	path_name[ptr_path_obj->count] = '\0';

	printf("%02d:%08x: %s\n", i, nwos_ref_to_word(&assoc_block_cache->first_block->refs[i]), path_name);
    }

    extra_block = assoc_block_cache->first_block->next_block_ptr;

    assert(is_same_object(&extra_block->id, &assoc_block_cache->first_block->next_block_ref));

    seq = 0;

    while (extra_block != NULL)
    {
	seq++;

	printf("extra block: %03d:%08x\n", seq, nwos_ref_to_word(&extra_block->id));

	for (i = 0; i < MAX_REFS_IN_SIDECAR && !is_void_reference(&extra_block->refs[i]); i++)
	{
	    if (!nwos_read_object_from_disk(&extra_block->refs[i], &assoc_obj, sizeof(assoc_obj)))
	    {
		fprintf(stderr, "Reading asssociation object failed\n");
		exit(1);
	    }

	    if (!nwos_read_variable_sized_object_from_disk(&assoc_obj.path, kludge, sizeof(kludge), &get_path_object_size))
	    {
		fprintf(stderr, "Reading path object failed\n");
		exit(1);
	    }

	    if (ptr_path_obj->count > 64)
	    {
		fprintf(stderr, "path too long\n");
		exit(1);
	    }

	    memcpy(path_name, ptr_path_obj->storage, ptr_path_obj->count);
	    path_name[ptr_path_obj->count] = '\0';

	    switch (state)
	    {
	      case Searching:
		if (memcmp(path_name, szq_string, strlen(szq_string)) == 0)
		{
		    state = Copying;
		    empty_block = extra_block;
		    empty_index = i;
		    write_block = extra_block;
		    write_seq = seq;
		}
		break;

	      case Copying:
		if (memcmp(path_name, img_string, strlen(img_string)) == 0)
		{
		    state = Moving;
		}
		break;

	      case Moving:
		if (memcmp(path_name, szq_string, strlen(szq_string)) == 0)
		{
		    state = Copying;
		}
		break;
	    }

	    switch (state)
	    {
	      case Searching:
		printf("%02d:%08x: %s\n", i, nwos_ref_to_word(&extra_block->refs[i]), path_name);
		break;

	      case Copying:
		copy_reference(&szq_refs[num_szq], &extra_block->refs[i]);
		printf("%02d:%08x: %s - copied to: %d\n", i, nwos_ref_to_word(&extra_block->refs[i]), path_name, num_szq);
		num_szq++;
		break;

	      case Moving:
		copy_reference(&empty_block->refs[empty_index], &extra_block->refs[i]);
		printf("%02d:%08x: %s - moved to: %08x:%d\n", i, nwos_ref_to_word(&extra_block->refs[i]), path_name, nwos_ref_to_word(&empty_block->id), empty_index);
		empty_index++;
		if (empty_index == MAX_REFS_IN_SIDECAR)
		{
		    empty_block = empty_block->next_block_ptr;
		    empty_index = 0;
		}
	    }

	    fflush(stdout);
	}

	if (extra_block->next_block_ptr != NULL)
	{
	    assert(is_same_object(&extra_block->next_block_ptr->id, &extra_block->next_block_ref));
	}
	else
	{
	    assert(is_same_object(&assoc_block_cache->last_block->id, &extra_block->id));
	}

	extra_block = extra_block->next_block_ptr;
    }

    printf("Num sqz refs: %d\n", num_szq);

    for (i = 0; i < num_szq; i++)
    {
	copy_reference(&empty_block->refs[empty_index], &szq_refs[i]);
	printf("%03d:%08x: moved to: %08x:%d\n", i, nwos_ref_to_word(&szq_refs[i]), nwos_ref_to_word(&empty_block->id), empty_index);
	fflush(stdout);
	empty_index++;

	if (empty_index == MAX_REFS_IN_SIDECAR)
	{
	    empty_block = empty_block->next_block_ptr;
	    empty_index = 0;
	}
    }

    /* make sure all references in the old reference list are there in the new reference list */

    new_assoc_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    printf("Old refs: %d  New refs: %d\n", old_assoc_list->common_header.num_refs, new_assoc_list->common_header.num_refs);
    fflush(stdout);

    if (new_assoc_list->common_header.num_refs != old_assoc_list->common_header.num_refs)
    {
	fprintf(stderr, "Num refs mismatch!\n");
	write_block = NULL;
    }
    else
    {
	for (i = 0; i < old_assoc_list->common_header.num_refs; i++)
	{
	    for (j = 0; j < new_assoc_list->common_header.num_refs; j++) if (is_same_object(&new_assoc_list->references[j], &old_assoc_list->references[i])) break;

	    if (j == new_assoc_list->common_header.num_refs)
	    {
		fprintf(stderr, "Could not find ref: %08x in new list\n", nwos_ref_to_word(&old_assoc_list->references[i]));
		write_block = NULL;
	    }
	}

	printf("Check complete!\n");
	fflush(stdout);
    }

    nwos_free_reference_list(old_assoc_list);
    nwos_free_reference_list(new_assoc_list);

    printf("write_block: %p  ref: %08x\n", write_block, nwos_ref_to_word(&write_block->id));
    fflush(stdout);

    if (write_block != NULL)
    {
	while (write_block != assoc_block_cache->last_block)
	{
	    nwos_crc32_calculate((uint8*) &write_block->refs, (MAX_REFS_IN_SIDECAR + 1) * sizeof(ObjRef), write_block->checksum);

	    memcpy(ivec, write_block->ivec, sizeof(ivec));

	    printf("writing: %03d:%08x %02x%02x%02x%02x\n", write_seq, nwos_ref_to_word(&write_block->id),
		   write_block->checksum[0], write_block->checksum[1], write_block->checksum[2], write_block->checksum[3]);
	    fflush(stdout);

	    nwos_write_block_to_disk_and_encrypt(&write_block->id, &write_block->dirty, FILE_BLOCK_SIZE, ivec, write_seq);

	    memcpy(write_block->next_block_ptr->ivec, ivec, sizeof(write_block->next_block_ptr->ivec));

	    write_seq++;

	    write_block = write_block->next_block_ptr;
	}

	nwos_crc32_calculate((uint8*) &write_block->refs, (empty_index + 1) * sizeof(ObjRef), write_block->checksum);

	memcpy(ivec, write_block->ivec, sizeof(ivec));

	printf("writing: %03d:%08x %02x%02x%02x%02x - num_refs: %d\n", write_seq, nwos_ref_to_word(&write_block->id),
	       write_block->checksum[0], write_block->checksum[1], write_block->checksum[2], write_block->checksum[3], (empty_index + 1));
	fflush(stdout);

	nwos_write_block_to_disk_and_encrypt(&write_block->id, &write_block->dirty, FILE_BLOCK_SIZE, ivec, write_seq);

	assert(write_block->next_block_ptr == NULL);
    }

    printf("seq: %d  write_seq %d\n", seq, write_seq);

    nwos_terminate_objectify();

    return 0;
}


