/*
--          This file is part of the New World OS and Objectify projects
--            Copyright (C) 2005, 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-08-04 05:36:52 -0600 (Tue, 04 Aug 2009) $
--   $Revision: 4253 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file and the word.c file which this
--   file was derived from.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/


#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>   /* define memset */

#include "crc32.h"
#include "objectify.h"



static ObjRef* get_public_spelling_class_ref()
{
    static ObjRef public_spelling_class_ref;

    if (is_void_reference(&public_spelling_class_ref))
    {
	nwos_find_public_class_definition("SPELLING", &public_spelling_class_ref);
    }

    assert(!is_void_reference(&public_spelling_class_ref));

    return &public_spelling_class_ref;
}


static ObjRef* get_private_spelling_class_ref()
{
    static ObjRef private_spelling_class_ref;

    if (is_void_reference(&private_spelling_class_ref))
    {
	if (!nwos_find_private_class_definition("SPELLING", &private_spelling_class_ref))
	{
	    return NULL;
	}
    }

    return &private_spelling_class_ref;
}


static size_t get_spelling_object_size(void* spelling_obj)
{
    assert(((C_struct_Spelling*)spelling_obj)->count > 0);

    return sizeof(C_struct_Spelling) + ((C_struct_Spelling*)spelling_obj)->count;
}


bool nwos_find_public_spelling(char* spelling, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    uint8 kludge[MAX_SPELLING_OBJ_SIZE];
    C_struct_Spelling* ptr_spelling_obj = (C_struct_Spelling*)kludge;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    size_t length;
    int i;


    length = strlen(spelling);

    assert(length <= 255);
    
    nwos_read_class_definition(get_public_spelling_class_ref(), &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, get_public_spelling_class_ref()))
	{
	    assert(nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &get_spelling_object_size));

	    /* remember ptr_spelling_obj points to the kludge buffer */

	    if (ptr_spelling_obj->count == length && strncasecmp((char*)ptr_spelling_obj->storage, spelling, length) == 0)   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}


bool nwos_find_private_spelling(char* spelling, ObjRef* ref)
{
    ObjRef* spelling_class_ref_ptr;
    C_struct_Class_Definition class_def_obj;
    uint8 kludge[MAX_SPELLING_OBJ_SIZE];
    C_struct_Spelling* ptr_spelling_obj = (C_struct_Spelling*)kludge;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    size_t length;
    int i;


    length = strlen(spelling);

    assert(length <= 255);

    spelling_class_ref_ptr = get_private_spelling_class_ref();

    if (spelling_class_ref_ptr == NULL) return false;

    nwos_read_class_definition(spelling_class_ref_ptr, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, spelling_class_ref_ptr))
	{
	    assert(nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &get_spelling_object_size));

	    /* remember ptr_spelling_obj points to the kludge buffer */

	    if (ptr_spelling_obj->count == length && strncasecmp((char*)ptr_spelling_obj->storage, spelling, length) == 0)   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}



/* Find existing spelling or create new */

ObjCreateResult nwos_create_public_spelling(char* spelling, ObjRef* ref)
{
    uint8 kludge[MAX_SPELLING_OBJ_SIZE];
    C_struct_Spelling* ptr_spelling_obj = (C_struct_Spelling*)kludge;
    size_t length;
    ObjCreateResult result = FOUND_EXISTING;
    int i;
    C_struct_Class_Definition class_def_obj;

    length = strlen(spelling);

    assert(length <= 255);

    /* first find out if we already have this spelling */

    if (!nwos_find_public_spelling(spelling, ref))   /* didn't find it */
    {
	memset(kludge, 0, sizeof(kludge));  /* zero it out */

	/* remember ptr_spelling_obj points to the kludge buffer */

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&ptr_spelling_obj->header.common, ref, get_public_spelling_class_ref());

	nwos_read_class_definition(get_public_spelling_class_ref(), &class_def_obj);

	ptr_spelling_obj->count = length;
	for (i = 0; i < length; i++) ptr_spelling_obj->storage[i] = tolower(spelling[i]);

	nwos_create_reference_list(ref, &ptr_spelling_obj->header.object.references);

	nwos_crc32_calculate((uint8*) &ptr_spelling_obj->header.object, sizeof(ObjectHeader), ptr_spelling_obj->header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &ptr_spelling_obj->character_set, sizeof(C_struct_Spelling) + length - sizeof(EveryObject), ptr_spelling_obj->header.common.data_chksum);

	nwos_write_object_to_disk(ref, kludge, sizeof(C_struct_Spelling) + length);

	nwos_add_to_references(ref, get_public_spelling_class_ref());

#ifdef VERIFY_WRITE
	ptr_spelling_obj = malloc(sizeof(C_struct_Spelling) + length);
	assert(nwos_read_object_from_disk(ref, ptr_spelling_obj, sizeof(C_struct_Spelling) + length));
	assert(memcmp(kludge, ptr_spelling_obj, sizeof(C_struct_Spelling) + length) == 0);

	memset(kludge, 0, sizeof(kludge));  /* clear it */
	assert(nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_spelling_object_size));  /* read the other way */
	assert(memcmp(ptr_spelling_obj, kludge, sizeof(C_struct_Spelling) + length) == 0);

	free(ptr_spelling_obj);
	ptr_spelling_obj = NULL;
#endif

	result = CREATED_NEW;
    }

    return result;
}


/* Find existing private spelling or clone the public object */

ObjCreateResult nwos_create_private_spelling(char* spelling, ObjRef* ref)
{
    ObjRef spelling_class_ref;
    ObjRef public_ref;
    uint8 kludge[MAX_SPELLING_OBJ_SIZE];
    C_struct_Spelling* ptr_spelling_obj = (C_struct_Spelling*)kludge;
    size_t length;
    ObjCreateResult result = FOUND_EXISTING;

    if (!nwos_find_private_spelling(spelling, ref))
    {
	nwos_find_or_create_private_class_definition("SPELLING", &spelling_class_ref);

	assert(nwos_find_public_spelling(spelling, &public_ref));

	assert(nwos_read_variable_sized_object_from_disk(&public_ref, kludge, sizeof(kludge), &get_spelling_object_size));

	nwos_generate_new_id(ref);

	/* copy the public id into the clone_of field */
	copy_reference(&ptr_spelling_obj->header.object.clone_of, &public_ref);

	copy_reference(&ptr_spelling_obj->header.common.id, ref);

	copy_reference(&ptr_spelling_obj->header.common.class_definition, &spelling_class_ref);

	nwos_create_reference_list(ref, &ptr_spelling_obj->header.object.references);

	nwos_crc32_calculate((uint8*) &ptr_spelling_obj->header.object, sizeof(ObjectHeader), ptr_spelling_obj->header.common.header_chksum);

	length = ptr_spelling_obj->count;

	nwos_write_object_to_disk(ref, kludge, sizeof(C_struct_Spelling) + length);

	nwos_add_to_references(ref, &spelling_class_ref);

#ifdef VERIFY_WRITE
	ptr_spelling_obj = malloc(sizeof(C_struct_Spelling) + length);
	assert(nwos_read_object_from_disk(ref, ptr_spelling_obj, sizeof(C_struct_Spelling) + length));
	assert(memcmp(kludge, ptr_spelling_obj, sizeof(C_struct_Spelling) + length) == 0);

	memset(kludge, 0, sizeof(kludge));  /* clear it */
	assert(nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_spelling_object_size));  /* read the other way */
	assert(memcmp(ptr_spelling_obj, kludge, sizeof(C_struct_Spelling) + length) == 0);

	free(ptr_spelling_obj);
	ptr_spelling_obj = NULL;
#endif

	result = CREATED_NEW;
    }

    return result;
}

