/*
--          This file is part of the New World OS and Objectify projects
--         Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-08-04 05:36:52 -0600 (Tue, 04 Aug 2009) $
--   $Revision: 4253 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file and the log of this file in the
--   alpha_05_branch which had many changes to this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/


#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "bit_map.h"
#include "chunk_info.h"                /* define nwos_hash_ref */
#include "class_definition.h"
#include "crc32.h"
#include "disk_io.h"                   /* define nwos_archive_is_file and nwos_write_empty_chunk */
#include "header.h"
#include "objectify.h"
#include "security.h"
#include "storage.h"
#include "time_stamp.h"


#define MAX_ROOT_TRANSFORMATIONS 128      // the number of different possible root objects for a given id


/* Global variables... BAD!... need to fix */

static bool nwos_ready = false;



bool nwos_is_ready(void)
{
    return nwos_ready;
}

bool nwos_is_writable(void)
{
    return nwos_storage_is_read_write();
}


/***********************************/
/* Variable Length Count functions */
/***********************************/

uint32 nwos_decode_variable_sized_count(uint8 count[4])
{
    uint32 result;

    result = count[0];   /* get the first byte */

    if (result > 127)   /* two bytes */
    {
	assert((result & 0xc0) == 0x80);  /* for now upper two bits must be 10 */

	result = (((result & 0x3f) << 8) | count[1]) + 128;
    }
    else
    {
	/* this byte should always be zero if first bytes is < 128.  the original */
	/* plan was to not store the extra bytes, but the compiler did it anyway */
	assert(count[1] == 0);
    }

    return result;
}

void nwos_encode_variable_sized_count(uint32 count, uint8 result[4])
{
    int adjusted;

    assert(count < (16384 + 128));   /* for now we can only handle two bytes */

    if (count > 127)   /* two bytes */
    {
	adjusted  = count - 128;
	result[0] = (0x80 | (adjusted >> 8));
	result[1] = adjusted & 0xff;
    }
    else
    {
	result[0] = count;
	result[1] = 0;
    }

    result[2] = 0;
    result[3] = 0;

    assert(nwos_decode_variable_sized_count(result) == count);
}



/*********************************/
/* Root object ID transformation */
/*********************************/

static bool is_binary_number(uint n)
{
    uint u;

    for (u = 1; u != 0; u <<= 1)
    {
	if (u == n) return true;
    }

    return false;
}


static void nwos_transform_root(ObjRef* dst_ref, ObjRef* src_ref, int num)
{
    int shift_count;

    assert(16 <= MAX_ROOT_TRANSFORMATIONS && MAX_ROOT_TRANSFORMATIONS < 256 && is_binary_number(MAX_ROOT_TRANSFORMATIONS));
    assert(0 <= num && num < MAX_ROOT_TRANSFORMATIONS);

    shift_count = 1;
    while ((MAX_ROOT_TRANSFORMATIONS << shift_count) != 256) shift_count++;

    copy_reference(dst_ref, src_ref);

    dst_ref->id[0] = src_ref->id[0] + (num << shift_count);
}


/**********************************/
/* Initialization and Termination */
/**********************************/

void nwos_initialize_objectify(AccessType type, const char* path)
{
    ObjRef root_object_ref;

#ifdef PUBLIC_MODE
    assert(type == PUBLIC);
#else
    int i;
    int attempt;  /* number of pass phrase attempts */
    uint8 big_key[16 + 8 + 4];
    ObjRef test_object_ref;
    char* incorrect_messages[] = { "Incorrect, try again.", "Wrong again, one more try.", "Sorry, no go." };

#define NUM_ATTEMPTS (sizeof(incorrect_messages) / sizeof (char*))

#endif

    nwos_log("nwos_initialize_objectify():");

    nwos_crc32_initialize();
    nwos_initialize_storage(type, path);

    if (type == PUBLIC)
    {
	root_object_ref.id[0] = 0;
	root_object_ref.id[1] = 0;
	root_object_ref.id[2] = 0;
	root_object_ref.id[3] = 1;

	assert(nwos_set_root_object(&root_object_ref));
    }
#ifndef PUBLIC_MODE
    else
    {
	for (attempt = 0; attempt < NUM_ATTEMPTS; attempt++)
	{
	    nwos_get_key_from_password(big_key, sizeof(big_key));

	    nwos_setup_from_big_key(big_key, &root_object_ref);

	    for (i = 0; i < MAX_ROOT_TRANSFORMATIONS; i++)
	    {
		nwos_transform_root(&test_object_ref, &root_object_ref, i);

		if (nwos_is_private_reference(&test_object_ref) && nwos_set_root_object(&test_object_ref)) break;  /* we have a winner! */
	    }

	    if (i < MAX_ROOT_TRANSFORMATIONS) break;
	    
	    fprintf(stderr, "%s\n", incorrect_messages[attempt]);
	}

	if (attempt == NUM_ATTEMPTS)
	{
	    nwos_terminate_storage();
	    fprintf(stderr, "\n");
	    exit(1);
	}
    }
#endif
}


/* this is the same as the normal routine in objectify.c except it doesn't translate the reference for the disk */
static void set_public_root_object(ObjRef* ref)
{
    C_struct_Root root_obj;
    ObjRef verify_ref;

    assert(nwos_read_object_from_disk(ref, &root_obj, sizeof(root_obj)));
    copy_reference(&nwos_public_class_definition_class_ref, &root_obj.class_definition_class);
    copy_reference(&nwos_reference_list_class_ref, &root_obj.reference_list_class);

    assert(nwos_find_public_class_definition("REFERENCE LIST", &verify_ref));
    assert(is_same_object(&verify_ref, &nwos_reference_list_class_ref));
}

    
#ifndef PUBLIC_MODE
void nwos_create_root(void)
{
    uint8 big_key_1[16 + 8 + 4];
    uint8 big_key_2[16 + 8 + 4];
    uint32 hash;
    ObjRef public_root_obj_ref;
    ObjRef private_root_obj_ref;
    ObjRef public_root_class_ref;
    ObjRef test_object_ref;
    C_struct_Root root_obj;
    ObjRef root_class_ref;
    int i;
    int random_transform;


    nwos_log("nwos_create_root():");

    nwos_crc32_initialize();
    nwos_initialize_storage(READ_WRITE, DEFAULT_FILE);

    assert(nwos_archive_is_file() || nwos_total_private_chunks > nwos_used_private_chunks);

    public_root_obj_ref.id[0] = 0;
    public_root_obj_ref.id[1] = 0;
    public_root_obj_ref.id[2] = 0;
    public_root_obj_ref.id[3] = 1;

    set_public_root_object(&public_root_obj_ref);

    assert(nwos_find_public_class_definition("ROOT", &public_root_class_ref));


    while (true)
    {
	assert(sizeof(big_key_1) == sizeof(big_key_2));

	nwos_get_key_from_password(big_key_1, sizeof(big_key_1));

	nwos_setup_from_big_key(big_key_1, &test_object_ref);

	for (i = 0; i < MAX_ROOT_TRANSFORMATIONS; i++)
	{
	    nwos_transform_root(&private_root_obj_ref, &test_object_ref, i);

	    if (nwos_is_private_reference(&private_root_obj_ref) && nwos_read_object_from_disk(&private_root_obj_ref, &root_obj, sizeof(root_obj))) break;  /* found it! */
	}

	if (i < MAX_ROOT_TRANSFORMATIONS)   /* found an existing password */
	{
	    i = MAX_ROOT_TRANSFORMATIONS;            /* don't allow it */
	}
	else
	{
	    i = 0;
	}

	random_transform = random() % MAX_ROOT_TRANSFORMATIONS;

	while (i < MAX_ROOT_TRANSFORMATIONS)
	{
	    nwos_transform_root(&private_root_obj_ref, &test_object_ref, random_transform);

	    if (nwos_is_private_reference(&private_root_obj_ref))
	    {
		hash = nwos_hash_ref(&private_root_obj_ref);

		if (hash == 0 || !nwos_test_bit_in_map(hash)) break;   /* we have a winner! */
	    }

	    random_transform = (random_transform + 1) % MAX_ROOT_TRANSFORMATIONS;

	    i++;
	}

	if (i < MAX_ROOT_TRANSFORMATIONS)   /* found a good location */
	{
	    nwos_get_key_from_password(big_key_2, sizeof(big_key_2));

	    if (memcmp(big_key_1, big_key_2, sizeof(big_key_1)) == 0) break;

	    fprintf(stderr, "\nKeys don't match, try again\n\n");
	}
	else
	{
	    fprintf(stderr, "\nThat pass phrase cannot be used, try a different one.\n\n");
	}
    }

    /* create the private root object */

    if (nwos_archive_is_file())
    {
	nwos_write_empty_chunk(nwos_total_private_chunks);
    }

    nwos_allocate_new_chunk(nwos_ref_to_word(&private_root_obj_ref));

    nwos_check_blocks_available(3);

    hash = nwos_hash_ref(&private_root_obj_ref);

    assert(hash != 0 && !nwos_test_bit_in_map(hash));    /* make sure chunk is allocated, but block is not already used */

    nwos_set_bit_in_map(hash);

#ifdef CHANGE_SECURITY_TO_DENSITY
    nwos_set_security_level(Security_Extreme);
#endif

    nwos_clone_class_definition(&nwos_public_class_definition_class_ref, &nwos_private_class_definition_class_ref);


    /* currently I can't think of any reason to have a private root class, since there is no list of root objects */
    assert(nwos_find_public_class_definition("ROOT", &root_class_ref));

    memset(&root_obj, 0, sizeof(root_obj));  /* zero it out */

    nwos_fill_in_common_header(&root_obj.header.common, &private_root_obj_ref, &root_class_ref);

    memcpy(&root_obj.class_definition_class, &nwos_private_class_definition_class_ref, sizeof(root_obj.class_definition_class));

    memcpy(&root_obj.reference_list_class, &nwos_reference_list_class_ref, sizeof(root_obj.reference_list_class));

    nwos_crc32_calculate((uint8*) &root_obj.header.object, sizeof(ObjectHeader), root_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &root_obj.class_definition_class, sizeof(root_obj) - sizeof(EveryObject), root_obj.header.common.data_chksum);

    nwos_write_object_to_disk(&private_root_obj_ref, &root_obj, sizeof(root_obj));

    nwos_terminate_objectify();
}
#endif


bool nwos_set_root_object(ObjRef* ref)
{
    C_struct_Root root_obj;
    ObjRef verify_ref;
    C_struct_Class_Definition class_def;

    assert(!nwos_ready);

    if (nwos_read_object_from_disk(ref, &root_obj, sizeof(root_obj)))
    {
	if (nwos_reference_type(ref) == Public_Reference)
	{
	    assert(nwos_storage_is_public_only());   /* only allow if no private objects are in play */

	    copy_reference(&nwos_public_class_definition_class_ref, &root_obj.class_definition_class);
	}
	else
	{
	    assert(!nwos_storage_is_public_only());   /* only allow if private objects are in play */

	    copy_reference(&nwos_private_class_definition_class_ref, &root_obj.class_definition_class);

	    nwos_read_class_definition(&nwos_private_class_definition_class_ref, &class_def);

	    copy_reference(&nwos_public_class_definition_class_ref, &class_def.header.object.clone_of);
	}

	copy_reference(&nwos_reference_list_class_ref, &root_obj.reference_list_class);

	assert(nwos_find_public_class_definition("REFERENCE LIST", &verify_ref));
	assert(is_same_object(&verify_ref, &nwos_reference_list_class_ref));

	nwos_ready = true;
    }

    return nwos_ready;
}


void nwos_flush_to_disk()
{
    nwos_flush_dirty_ref_lists();  /* flush any reference lists that are dirty out to disk */
    nwos_flush_storage();
}


void nwos_terminate_objectify()
{
    nwos_log("nwos_terminate_objectify():");

    nwos_ready = false;

    nwos_flush_dirty_ref_lists();  /* flush any reference lists that are dirty out to disk */

    nwos_terminate_storage();

#ifdef CHECK_MEMORY_LEAKS
    for (i = 0; i < MAX_MEM_ADDR_SAVED; i++)
    {
	if (save_malloc_addr[i] != NULL) printf("WARNING - memory block not freed: %p\n", save_malloc_addr[i]);
    }
#endif
}


bool nwos_in_public_mode()
{
    return is_void_reference(&nwos_private_class_definition_class_ref);
}


const char* nwos_archive_version()
{
    return nwos_version_string;
}


bool nwos_is_quit_command(char* command)
{
    if (strcasecmp(command, "quit") == 0) return true;
    if (strcasecmp(command, "quit now") == 0) return true;
    if (strcasecmp(command, "exit") == 0) return true;
    if (strcasecmp(command, "stop") == 0) return true;
    if (strcasecmp(command, "stop now") == 0) return true;
    if (strcasecmp(command, "done") == 0) return true;
    if (strcasecmp(command, "done now") == 0) return true;
    if (strcasecmp(command, "finish") == 0) return true;
    if (strcasecmp(command, "finish now") == 0) return true;
    if (strcasecmp(command, "finished") == 0) return true;
    if (strcasecmp(command, "terminate") == 0) return true;

    return false;
}


