/*
--          This file is part of the New World OS and Objectify projects
--         Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-08-01 08:34:47 -0600 (Sat, 01 Aug 2009) $
--   $Revision: 4227 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <stdio.h>
#include <time.h>
#include <stdlib.h>
#include "time_stamp.h"

  /*
  -- A time stamp is stored as 8 8-bit unsigned values,
  -- that represent broken down time.  See the header
  -- file for the break down of bits.
  */

static void unix_broken_down_time_to_timestamp(struct tm* bdt, uint32 usec, uint8 stamp[8])
{
  nwos_insert_year_into_time_stamp(bdt->tm_year + 1900, stamp);
  nwos_insert_month_into_time_stamp(bdt->tm_mon + 1, stamp);
  nwos_insert_day_of_month_into_time_stamp(bdt->tm_mday, stamp);
  nwos_insert_day_of_week_into_time_stamp(bdt->tm_wday + 1, stamp);
  nwos_insert_hour_into_time_stamp(bdt->tm_hour, stamp);
  nwos_insert_minute_into_time_stamp(bdt->tm_min, stamp);
  nwos_insert_second_into_time_stamp(bdt->tm_sec, stamp);
  nwos_insert_microsecond_into_time_stamp(usec, stamp);
}


static void timestamp_to_unix_broken_down_time(uint8 stamp[8], struct tm* bdt, uint32* usec)
{
  /* insert an error to test test */
  /* if (day_of_month == 13 && year == 1999 && month == 7) second++; */

  *usec = nwos_extract_microsecond_from_time_stamp(stamp);

  bdt->tm_sec  = nwos_extract_second_from_time_stamp(stamp);
  bdt->tm_min  = nwos_extract_minute_from_time_stamp(stamp);
  bdt->tm_hour = nwos_extract_hour_from_time_stamp(stamp);
  bdt->tm_mday = nwos_extract_day_of_month_from_time_stamp(stamp);
  bdt->tm_mon  = nwos_extract_month_from_time_stamp(stamp) - 1;
  bdt->tm_year = nwos_extract_year_from_time_stamp(stamp) - 1900;
  bdt->tm_wday = nwos_extract_day_of_week_from_time_stamp(stamp) - 1;
}


void nwos_get_time_stamp(TimeStamp stamp)
{
  struct timespec ts;
#ifdef HAVE_CLOCK_GETTIME
  if (clock_gettime(CLOCK_REALTIME, &ts) != 0) {
    perror("reading real-time clock");
    exit(1);
  }
#else
  struct timeval tv;
  if (gettimeofday(&tv, NULL) != 0) {
    perror("reading real-time clock");
    exit(1);
  }
  ts.tv_sec = tv.tv_sec;
  ts.tv_nsec = tv.tv_usec * 1000;
#endif
  nwos_convert_timespec_to_time_stamp(&ts, stamp);
}


void nwos_convert_timespec_to_time_stamp(struct timespec* ts, TimeStamp stamp)
{
  uint32 usec;
  struct tm bdt;

  if (ts->tv_nsec < 0L || ts->tv_nsec > 1000000000L) {
    fprintf(stderr, "nanoseconds in timespec out of range: %ld\n",
	    ts->tv_nsec);
    exit(1);
  }

  usec = ts->tv_nsec / 1000;   /* convert to microseconds */

  if (gmtime_r(&ts->tv_sec, &bdt) == NULL) {
    perror("gmtime_r");
    exit(1);
  }

  unix_broken_down_time_to_timestamp(&bdt, usec, stamp);
}


void nwos_convert_time_stamp_to_timeval(TimeStamp time_stamp, struct timeval* tv)
{
  uint32 usec;
  struct tm bdt;

  timestamp_to_unix_broken_down_time(time_stamp, &bdt, &usec);

  tv->tv_sec = timegm(&bdt);  /* as an alternative change the tz environment variable to UTC and then call mktime, then set it back */
  tv->tv_usec = usec;
}


const char* nwos_time_stamp_to_string(TimeStamp ts)
{
    static char result[20];

    snprintf(result, sizeof(result), "%u-%02u-%02u %02u:%02u:%02u",
	       nwos_extract_year_from_time_stamp(ts),
	       nwos_extract_month_from_time_stamp(ts),
	       nwos_extract_day_of_month_from_time_stamp(ts),
	       nwos_extract_hour_from_time_stamp(ts),
	       nwos_extract_minute_from_time_stamp(ts),
	       nwos_extract_second_from_time_stamp(ts));

    return result;
}


/* Standalone test function - compile with -DTEST_ITER=n */

#ifdef TEST_ITER

#include <stdio.h>

void display_error(struct tm* bdt1, struct tm* bdt2, uint32 usec1, uint32 usec2, const char* name)
{
  printf("error in %s\n", name);
  printf("usec:    %lu %lu\n", usec1, usec2);
  printf("tm_sec:  %d %d\n", bdt1->tm_sec,  bdt2->tm_sec);
  printf("tm_min:  %d %d\n", bdt1->tm_min,  bdt2->tm_min);
  printf("tm_hour: %d %d\n", bdt1->tm_hour, bdt2->tm_hour);
  printf("tm_wday: %d %d\n", bdt1->tm_wday, bdt2->tm_wday);
  printf("tm_mday: %d %d\n", bdt1->tm_mday, bdt2->tm_mday);
  printf("tm_mon:  %d %d\n", bdt1->tm_mon,  bdt2->tm_mon);
  printf("tm_year: %d %d\n", bdt1->tm_year, bdt2->tm_year);
}

int main()
{
    int i;
    uint8 stamp[8];
    struct tm bdt1;
    struct tm bdt2;
    uint32 usec1;
    uint32 usec2;
    time_t sec;

    for (i = 0; i < TEST_ITER; i++)
    {
        /* generate a random time value */

	usec1 = rand() % 1000000;

	sec = rand() & 0x7fffffff;  /* always positive */

	if (gmtime_r(&sec, &bdt1) == NULL) {
	  perror("gmtime_r");
	  exit(1);
	}

	unix_broken_down_time_to_timestamp(&bdt1, usec1, stamp);
#if 0
	printf("%02x%02x%02x %02x%02x%02x%02x%02x\n",
	       stamp[0], stamp[1], stamp[2], stamp[3],
	       stamp[4], stamp[5], stamp[6], stamp[7]);
#endif
	timestamp_to_unix_broken_down_time(stamp, &bdt2, &usec2);

	if (usec1 != usec2) {
	  display_error(&bdt1, &bdt2, usec1, usec2, "usec");
	  exit(1);
	}
	if (bdt1.tm_sec != bdt2.tm_sec) {
	  display_error(&bdt1, &bdt2, usec1, usec2, "sec");
	  exit(1);
	}
	if (bdt1.tm_min != bdt2.tm_min) {
	  display_error(&bdt1, &bdt2, usec1, usec2, "min");
	  exit(1);
	}
	if (bdt1.tm_hour != bdt2.tm_hour) {
	  display_error(&bdt1, &bdt2, usec1, usec2, "hour");
	  exit(1);
	}
	if (bdt1.tm_wday != bdt2.tm_wday) {
	  display_error(&bdt1, &bdt2, usec1, usec2, "wday");
	  exit(1);
	}
	if (bdt1.tm_mday != bdt2.tm_mday) {
	  display_error(&bdt1, &bdt2, usec1, usec2, "mday");
	  exit(1);
	}
	if (bdt1.tm_mon != bdt2.tm_mon) {
	  display_error(&bdt1, &bdt2, usec1, usec2, "mon");
	  exit(1);
	}
	if (bdt1.tm_year != bdt2.tm_year) {
	  display_error(&bdt1, &bdt2, usec1, usec2, "year");
	  exit(1);
	}
    }

    return 0;
}

#endif



