/*
--          This file is part of the New World OS and Objectify projects
--                     Copyright (C) 2007, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-08-07 07:42:54 -0600 (Fri, 07 Aug 2009) $
--   $Revision: 4303 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
--
-- This program checks to see if two sequential chunks have no overlaps.
--
*/


#define _LARGEFILE64_SOURCE

#include <byteswap.h>
#include <endian.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "../objectify_private.h"

uint8 acc_map[BIT_MAP_BYTES];

static int8 bits_in_byte[256] = 
{
/*        0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f */
/* 0 */   0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4,
/* 1 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 2 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 3 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 4 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 5 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 6 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 7 */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* 8 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 9 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* a */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* b */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* c */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* d */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* e */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* f */   4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
};


static void print_usage(char *program)
{
    fprintf(stderr, "usage: %s \n", program);
}


static void clear_acc_map()
{
    int i;

    for (i = 0; i < sizeof(acc_map); i++) acc_map[i] = 0;
}


#define SIZE_COUNTS 16

int main(int argc, char* argv[])
{
    int obj_file_desc;
    off_t chunk;
    uint8 block_map[BIT_MAP_BYTES];
    uint8 block[FILE_BLOCK_SIZE];
    int i;
    size_t bytes_read;
    int num_blocks;
    uint32 total_private_blocks;
    uint32 used_private_blocks;
    uint32 chunk_block_offset;
    uint32 blocks_on_disk;
    char msg[128];
    Disk_Header disk_header;
    int chunk_num;
    int chunk_num_2;
    int used_blocks;
    uint32 used_chunks;
    uint32* chunk_index;
    int block_overlap;
    int num_overlap;
    uint8 used[16384];

    for (i = 0; i < sizeof(used); i++) used[i] = 0;

    if (argc > 1)
    {
	print_usage(argv[0]);
	exit(1);
    }

    /* Open the storage drive and verify the header info */

    obj_file_desc = open(DEFAULT_FILE, O_RDONLY | O_LARGEFILE);

    if (obj_file_desc < 0)
    {
	perror(DEFAULT_FILE);
	exit(1);
    }

    bytes_read = read(obj_file_desc, block, sizeof(block));

    if (bytes_read != sizeof(block))
    {
	perror("reading first block");
	exit(1);
    }

    if (memcmp(&block[0], MAGIC_NUMBER, 4) != 0)
    {
	fprintf(stderr, "Missing magic number in disk header\n");
	exit(1);
    }

    if (memcmp(&block[4], VERSION_STRING, 4) != 0)
    {
	fprintf(stderr, "Incorrect version string in disk header\n");
	exit(1);
    }

    memcpy(&disk_header, block, sizeof(disk_header));

    nwos_4_uint8_to_uint32(disk_header.total_blocks, &total_private_blocks);
    nwos_4_uint8_to_uint32(disk_header.used_blocks, &used_private_blocks);
    nwos_4_uint8_to_uint32(disk_header.block_offset_to_chunks, &chunk_block_offset);
    nwos_4_uint8_to_uint32(disk_header.used_chunks, &used_chunks);

    assert(used_chunks > 0);

    chunk_index = malloc(used_chunks * sizeof(uint32));
    assert(chunk_index != NULL);

    bytes_read = read(obj_file_desc, chunk_index, used_chunks * sizeof(uint32));

    if (bytes_read != used_chunks * sizeof(uint32))
    {
	perror("reading chunk index");
	exit(1);
    }

    /* fix the byte order on little endian machines */
#if __BYTE_ORDER == __LITTLE_ENDIAN
	{
	  int i;
	  for (i = 0; i < used_chunks; i++)
	  {
	      chunk_index[i] = bswap_32(chunk_index[i]);
	  }
	}
#endif

    blocks_on_disk = total_private_blocks; 

    printf("header: %c%c%c%c %c%c%c%c\n",
	   disk_header.magic_number[0], disk_header.magic_number[1], disk_header.magic_number[2], disk_header.magic_number[3], 
	   disk_header.version_string[0], disk_header.version_string[1], disk_header.version_string[2], disk_header.version_string[3]);

    printf("total blocks on disk: %08u  chunks_used: %d\n", blocks_on_disk, used_chunks);
    fflush(stdout);

    num_blocks = 0;

    clear_acc_map();

    // for now we can skip over public blocks because they should always stay the same
    for (chunk_num = 0; chunk_num < used_chunks; chunk_num++)
    {
	if (used[chunk_num]) continue;

	chunk = chunk_block_offset + chunk_num * BLOCKS_IN_CHUNK;

	if (lseek(obj_file_desc, chunk << 8, SEEK_SET) < 0)
	{
	    snprintf(msg, sizeof(msg), "lseek chunk:%08x", (uint32)chunk);
	    perror(msg);
	    exit(1);
	}

	bytes_read = read(obj_file_desc, acc_map, sizeof(acc_map));

	if (bytes_read != sizeof(acc_map))
	{
	    snprintf(msg, sizeof(msg), "reading block map: %u", (uint32)chunk);
	    perror(msg);
	    exit(1);
	}

	used_blocks = 0;

	for (i = 4; i < BIT_MAP_BYTES; i++)
	{
	    used_blocks += bits_in_byte[acc_map[i]];
	}

	printf("chunk %d used %d: ", chunk_num, used_blocks);
	fflush(stdout);

	num_overlap = 0;

	for (chunk_num_2 = chunk_num + 1; chunk_num_2 < used_chunks; chunk_num_2++)
	{
	    if (used[chunk_num_2]) continue;

	    chunk = chunk_block_offset + chunk_num_2 * BLOCKS_IN_CHUNK;

	    if (lseek(obj_file_desc, chunk << 8, SEEK_SET) < 0)
	    {
		snprintf(msg, sizeof(msg), "lseek chunk:%08x", (uint32)chunk);
		perror(msg);
		exit(1);
	    }

	    bytes_read = read(obj_file_desc, block_map, sizeof(block_map));

	    if (bytes_read != sizeof(block_map))
	    {
		snprintf(msg, sizeof(msg), "reading block map: %u", (uint32)chunk);
		perror(msg);
		exit(1);
	    }

	    block_overlap = 0;
	    used_blocks = 0;

	    for (i = 4; i < BIT_MAP_BYTES; i++)
	    {
		used_blocks += bits_in_byte[block_map[i]];
		block_overlap += bits_in_byte[acc_map[i] & block_map[i]];
	    }

	    //	    if (used_blocks < 10)
	    //{
	    //	printf("      %d used %d\n", chunk_num_2, used_blocks);
	    //	fflush(stdout);
	    //}


	    if (block_overlap == 0)
	    {
		printf("%d%c", chunk_num_2, num_overlap % 16 == 15 ? '\n' : ' ');
		fflush(stdout);

		for (i = 0; i < BIT_MAP_BYTES; i++)
		{
		    acc_map[i] |= block_map[i];
		}

		used[chunk_num_2] = 1;

		num_overlap++;
	    }
	}

	printf("num_overlap: %d\n", num_overlap);
    }

    printf("Number of blocks: %d\n", num_blocks);

    close(obj_file_desc);

    return 0;
}

