/*
--          This file is part of the New World OS and Objectify projects
--                  Copyright (C) 2006, 2007, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-09-12 06:17:06 -0600 (Sat, 12 Sep 2009) $
--   $Revision: 4353 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/


#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "disk_io.h"

static void print_usage(char *program)
{
    fprintf(stderr, "usage: %s [archive-1] archive-2\n", program);
    fprintf(stderr, "if archive-1 is not specified it uses the default archive.\n");
}


int main(int argc, char* argv[])
{
    char* archive1;
    char* archive2;
    size_t archive1_len;
    int arch1_file_desc;
    int arch2_file_desc;
    Disk_Header header1;
    Disk_Header header2;
    off_t chunk;
    uint8 block_map[BIT_MAP_BYTES];
    uint8 block1[FILE_BLOCK_SIZE];
    uint8 block2[FILE_BLOCK_SIZE];
    int i;
    size_t bytes_read;
    FILE* ofp;
    int num_blocks;
    uint32 public_blocks;
    uint32 blocks_on_disk;
    char msg[128];
    bool read1;
    bool read2;

    if (argc != 2 && argc != 3)
    {
	print_usage(argv[0]);
	exit(1);
    }

    if (*argv[1] == '-' || (argc == 3 && *argv[2] == '-'))
    {
	fprintf(stderr, "error: this program doesn't have any options\n");
	print_usage(argv[0]);
	exit(1);
    }

    if (argc == 2)
    {
	archive1_len = strlen(nwos_get_private_objects_path()) + 1;

	archive1 = nwos_malloc(archive1_len);

	strlcpy(archive1, nwos_get_private_objects_path(), archive1_len);

	archive2 = argv[1];
    }
    else
    {
	archive1 = argv[1];
	archive2 = argv[2];
    }

    if (strcmp(archive1, archive2) == 0)
    {
	fprintf(stderr, "Error: it doesn't make much sense to compare an archive with itself!\n");
	print_usage(argv[0]);
	exit(1);
    }



    /* Open the first archive and verify the header info */

    obj_file_desc = open(archive1, O_RDONLY);

    if (obj_file_desc < 0)
    {
	perror(DEFAULT_FILE);
	exit(1);
    }

    bytes_read = read(obj_file_desc, block1, sizeof(block1));

    if (bytes_read != sizeof(block1))
    {
	perror(archive1);
	exit(1);
    }

    error_msg = nwos_check_disk_header(&header1, false, TYPE_CODE_DISK);

    if (error_msg != NULL)
    {
	fprintf(stderr, "%s: %s\n", error_msg, archive1);
	fclose(fp1);
	exit(1);
    }


    public_blocks = ((uint32)block1[8] << 24) | ((uint32)block1[9]<< 16) | ((uint32)block1[10] << 8) | ((uint32)block1[11]);

    blocks_on_disk = ((uint32)block1[12] << 24) | ((uint32)block1[13]<< 16) | ((uint32)block1[14] << 8) | ((uint32)block1[15]);
    ofp = fopen(argv[1], "r");

    if (ofp == NULL)
    {
	perror(argv[1]);
	exit(1);
    }

    printf("header: %c%c%c%c %c%c%c%c\n",
	   block1[0], block1[1], block1[2], block1[3], 
	   block1[4], block1[5], block1[6], block1[7]);
    printf("total blocks on disk: %08u  public blocks: %08u\n",
	   blocks_on_disk, public_blocks);
    fflush(stdout);

    fread(block2, 1, FILE_BLOCK_SIZE, ofp);

    if (memcmp(block1, block2, FILE_BLOCK_SIZE) != 0)
    {
	printf("Header block is different:\n");
	for (i = 0; i < FILE_BLOCK_SIZE; i++)
	{
	    if (block1[i] != block2[i])
	    {
		printf("%d %02x-%02x\n", i, block1[i], block2[i]);
	    }
	}
    }

    num_blocks = 0;
    read1 = true;
    read2 = true;

    // for now we can skip over public blocks because they should always stay the same
    for (chunk = public_blocks; chunk < blocks_on_disk; chunk += BLOCKS_IN_CHUNK)
    {
	if (lseek(obj_file_desc, (off_t)chunk << 8, SEEK_SET) < 0)
	{
	    snprintf(msg, sizeof(msg), "lseek chunk:%08x", (uint32)chunk);
	    perror(msg);
	    exit(1);
	}

	bytes_read = read(obj_file_desc, block_map, sizeof(block_map));

	if (bytes_read != sizeof(block_map))
	{
	    snprintf(msg, sizeof(msg), "reading block map: %u", (uint32)chunk);
	    perror(msg);
	    exit(1);
	}

	/* scan block map (skip over the blocks for the block map itself) */
	for (i = BIT_MAP_BLOCKS; i < BLOCKS_IN_CHUNK; i++)
	{
	    if ((block_map[i/8] & (0x80 >> (i%8))) != 0)
	    {
		if (read1)
		{
		    if (lseek(obj_file_desc, (chunk + i) << 8, SEEK_SET) < 0)
		    {
			snprintf(msg, sizeof(msg), "lseek block:%08x", (uint32)(chunk + i));
			perror(msg);
			exit(1);
		    }

		    bytes_read = read(obj_file_desc, block1, sizeof(block1));

		    if (bytes_read != sizeof(block1))
		    {
			snprintf(msg, sizeof(msg), "reading block: %u", (uint32)(chunk + i));
			perror(msg);
			exit(1);
		    }
		}
		
		if (read2)
		{
		    if (fread(block2, 1, sizeof(block2), ofp) != sizeof(block2))
		    {
			perror(argv[1]);
			close(obj_file_desc);
			exit(1);
		    }
		}

		if (memcmp(&block1[4], &block2[4], 4) == 0)
		{
		    printf("id: %02x%02x%02x%02x - ", block1[4], block1[5], block1[6], block1[7]);
		    fflush(stdout);

		    if (memcmp(block1, block2, FILE_BLOCK_SIZE) == 0)
		    {
			printf("same\n");
		    }
		    else
		    {
			printf("different\n");
		    }
		}
		else
		{
		    printf("different blocks:\n");
		    printf("   %02x%02x%02x%02x\n", block1[4], block1[5], block1[6], block1[7]);
		    printf("   %02x%02x%02x%02x\n", block2[4], block2[5], block2[6], block2[7]);
		    fflush(stdout);
		    chunk = blocks_on_disk;   /* rather than try to re-sync right now just stop */
		}

		num_blocks++;
	    }
	}
    }

    printf("Number of blocks: %d\n", num_blocks);

    if (fclose(ofp) != 0)
    {
	perror(argv[1]);
	exit(1);
    }

    close(obj_file_desc);

    return 0;
}

