/*
--          This file is part of the New World OS and Objectify projects
--         Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-08-04 05:12:48 -0600 (Tue, 04 Aug 2009) $
--   $Revision: 4252 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#ifndef OBJECTIFY_H
#define OBJECTIFY_H

#include "config.h"
#include "gen_id.h"
#include "log.h"
#include "mem_alloc.h"
#include "objects.h"
#include "progress_bar.h"
#include "reference.h"
#include "reference_list.h"
#include "time_stamp.h"
#include "types.h"



typedef enum { CREATED_NEW, IMPORTED_DATA, CREATED_NEW_REVISION, FOUND_EXISTING, ERROR_OCCURRED, DUPLICATE_PATH, DUPLICATE_FILE, PATH_NOT_FOUND, MULTIPLE_FILES, CREATED_NEW_PATH_FOR_EXISTING_FILE, NUM_OBJ_RESULTS } ObjCreateResult;


extern void nwos_ref_to_name(ObjRef* ref, char name[]);

/* extern ObjRef store_object(EveryObject* obj, uint64 length); */

extern uint32 nwos_decode_variable_sized_count(uint8 count[4]);
extern void   nwos_encode_variable_sized_count(uint32 count, uint8 result[4]);


extern void nwos_initialize_objectify(AccessType type, const char* path);
extern void nwos_terminate_objectify();
extern void nwos_flush_to_disk();
extern void nwos_create_root(void);
extern bool nwos_set_root_object(ObjRef* ref);
extern bool nwos_in_public_mode();
extern bool nwos_is_ready(void);
extern bool nwos_is_writable(void);
extern bool nwos_backup_is_enabled(void);
extern const char* nwos_archive_version(void);

//extern bool nwos_find_class_definition(char* name, ObjRef* class_ref);
extern bool nwos_find_public_class_definition(char* name, ObjRef* class_ref);
extern bool nwos_find_private_class_definition(char* name, ObjRef* class_ref);
extern ObjCreateResult nwos_find_or_create_private_class_definition(char* name_of_class, ObjRef* ref);
extern bool nwos_find_public_feature_definition(ObjRef* type_ref, char* name, int count, ObjRef* feat_ref);
extern void nwos_find_or_create_public_feature_definition(ObjRef* type_ref, char* name, int count, ObjRef* feat_ref);
extern size_t nwos_get_object_size(void* object);
extern void nwos_read_class_definition(ObjRef* ref, C_struct_Class_Definition* class_def);


/* word stuff */

#define MAX_SPELLING_OBJ_SIZE  (sizeof(C_struct_Spelling) + 100)

extern bool nwos_find_language(char* name, ObjRef* ref);

extern bool nwos_find_public_spelling(char* spelling, ObjRef* ref);
extern bool nwos_find_private_spelling(char* spelling, ObjRef* ref);
extern ObjCreateResult nwos_create_public_spelling(char* spelling, ObjRef* ref);
extern ObjCreateResult nwos_create_private_spelling(char* spelling, ObjRef* ref);

extern bool nwos_find_word(char* word, ObjRef* thing_class, ObjRef* lang, ObjRef* ref);
extern ObjCreateResult nwos_create_word(char* word, ObjRef* thing, ObjRef* lang, ObjRef* ref);

extern bool nwos_find_abbreviation(char* abbr, ObjRef* thing, ObjRef* lang, ObjRef* ref);
extern ObjCreateResult nwos_create_abbreviation(char* abbr, ObjRef* thing, ObjRef* lang, ObjRef* ref);

extern bool nwos_find_public_acronym(char* acronym, ObjRef* ref);
extern bool nwos_acronym_to_string(ObjRef* ref, char* string, size_t size);

extern bool nwos_is_quit_command(char* command);
extern void nwos_ask_user(char* descr, char* buffer, size_t size);
extern bool nwos_ask_yes_or_no(char* statement, char* question);


/* name stuff */

#define MAX_NAME_OBJ_SIZE  (sizeof(C_struct_Name) + sizeof(ObjRef) * 10)

extern bool            nwos_find_public_name(const char* name, ObjRef* ref);
extern bool            nwos_find_private_name(const char* name, ObjRef* ref);
extern ObjCreateResult nwos_create_name(const char* name, ObjRef* ref);
extern bool            nwos_name_to_string(ObjRef* ref, char* string, size_t size);

/* date stuff */
extern bool            nwos_is_leap_year(int year);
extern void            nwos_get_month_ref(uint8 month, ObjRef* ref);
extern bool            nwos_find_public_year(uint16 year, ObjRef* ref);
extern bool            nwos_find_private_year(uint16 year, ObjRef* ref);
extern ObjCreateResult nwos_create_year(uint16 year, ObjRef* ref);
extern void            nwos_find_public_month(uint8 month, ObjRef* ref);
extern bool            nwos_find_private_month(uint8 month, ObjRef* ref);
extern bool            nwos_find_public_date(uint16 year, uint8 month, uint8 day, ObjRef* ref);
extern void            nwos_find_or_create_private_month_and_day(uint8 month, uint8 day, ObjRef* ref);
extern ObjCreateResult nwos_create_date(uint16 year, uint8 month, uint8 day, ObjRef* ref);
extern bool            nwos_year_to_string(ObjRef* year_obj_ref, char* string, size_t size_of_string);
extern bool            nwos_month_number_to_string(ObjRef* month_obj_ref, char* string, size_t size_of_string);
extern bool            nwos_date_to_string(ObjRef* date_obj_ref, char* string, size_t size_of_string);

/* numbers */

#define MAX_CARDINAL_NUMBER_DIGITS 127

#define MAX_CARDINAL_NUMBER_OBJ_SIZE (sizeof(C_struct_Cardinal_Number) + MAX_CARDINAL_NUMBER_DIGITS)

extern bool            nwos_find_cardinal_number(char* number, ObjRef* ref);
extern ObjCreateResult nwos_create_cardinal_number(char* number, ObjRef* ref);
extern void            nwos_cardinal_number_to_string(ObjRef* ref, char* string, size_t size);
extern bool            nwos_find_ordinal_number(ObjRef* cardinal_ref, ObjRef* ref);
extern ObjCreateResult nwos_create_ordinal_number(ObjRef* cardinal_ref, ObjRef* ref);
extern void            nwos_ordinal_number_to_string(ObjRef* ref, char* string, size_t size);

/* person stuff */
extern void nwos_add_person();
extern void nwos_add_birthday(char* name, char* date);
extern void nwos_list_persons();
extern void nwos_describe_person();
extern bool nwos_find_person();
extern bool nwos_any_persons_named(char* name);
extern void nwos_find_public_gender(char* gender, ObjRef* ref);

/* phone stuff */
extern bool nwos_find_public_area_code(char* area_code, ObjRef* ref);
extern bool nwos_find_private_area_code(char* area_code, ObjRef* ref);
extern ObjCreateResult nwos_create_area_code(char* area_code, char* state_postal_code, ObjRef* ref);
extern bool nwos_is_valid_phone_number(char* number);
extern const char* nwos_invalid_phone_number_msg(char* number);
extern bool nwos_find_phone_number(char* area_code, char* phone_number, ObjRef* ref);
extern ObjCreateResult nwos_create_phone_number(char* area_code, char* phone_number, ObjRef* ref);
extern bool nwos_phone_number_to_string(ObjRef* ref, char* string, size_t size);
extern void nwos_add_mobile_phone(char* name, char* phone_number);
extern void nwos_find_phone_for_person(char* name);

/* address stuff */
extern bool nwos_find_state_postal_code(char* state_postal_code, ObjRef* ref);
extern bool nwos_find_state_from_postal_code(char* postal_code, ObjRef* ref);
extern bool nwos_find_state_from_name(char* name, ObjRef* ref);
extern bool nwos_any_states_named(char* name);
extern void nwos_describe_state(char* name);
extern ObjCreateResult nwos_create_us_city(char* city, ObjRef* state_ref, ObjRef* city_ref);
extern void nwos_private_create_city(ObjRef* city_class_ref, ObjRef* name_ref, ObjRef* state_ref, ObjRef* city_ref);
extern void nwos_add_city(ObjRef* city_ref);
extern void nwos_list_cities_in_state(char* state);

/* financial stuff */

#define MAX_ACCOUNT_NUMBER_SIZE 18
#define MAX_CHECKING_ID_SIZE 4

#define MAX_ACCOUNT_NUMBER_OBJ_SIZE  (sizeof(C_struct_Account_Number) + MAX_ACCOUNT_NUMBER_SIZE)
#define MAX_CHECKING_OBJ_SIZE        (sizeof(C_struct_Checking) + MAX_CHECKING_ID_SIZE)

extern bool nwos_find_public_credit_union(char* name, ObjRef* ref);
extern void nwos_add_account();
extern void nwos_list_accounts();


/* device stuff */
extern bool nwos_find_public_manufacturer(char* name, ObjRef* ref);
extern void nwos_list_public_manufacturers();


/* file stuff */

#define MAX_PATH_OBJ_SIZE  (sizeof(C_struct_File_Path) + 4096)

typedef enum { File_Not_Found, Checksums_Match, MD5_Sum_Mismatch, SHA1_Sum_Mismatch, SHA256_Sum_Mismatch, SHA512_Sum_Mismatch, Checksums_Mismatch } CheckFileResult;

extern void nwos_list_public_credit_unions();

extern void nwos_normalize_path(char* dst, char* src, size_t size);
extern bool nwos_find_file_path(const char* path, ObjRef* ref);
extern ObjCreateResult nwos_create_file_path(const char* path, ObjRef* ref);
extern bool nwos_file_path_to_string(ObjRef* ref, char* string, size_t size);
extern int  nwos_number_of_files_for_path(ObjRef* path_ref);
extern uint32 nwos_estimate_blocks_for_file(const char* path);
extern bool nwos_file_path_to_path_and_file_association(ObjRef* path_ref, int index, ObjRef* assoc_ref);
extern bool nwos_find_md5(uint8 md5[16], ObjRef* ref);
extern bool nwos_find_path_and_file_association(ObjRef* path_ref, ObjRef* file_ref, TimeStamp mod_time, ObjRef* ref);
typedef enum { MatchAll, IgnoreTime } MatchCode;
extern bool nwos_find_matching_path_and_file_association(const char* path, const char* file, ObjRef* ref, MatchCode match);
extern CheckFileResult nwos_check_file_checksums(const char* path);
extern bool nwos_file_is_stored(ObjRef* association_ref);  /* returns true if data stored in system */
extern bool nwos_file_has_backup(ObjRef* association_ref);  /* returns true if file is on a backup disc */
extern ObjCreateResult nwos_create_file_with_only_md5(const char* file_name, uint8 md5_digest[MD5_DIGEST_SIZE], ObjRef* ref);
extern ObjCreateResult nwos_add_info_to_existing_file(const char* file_name, ObjRef* ref);
extern ObjCreateResult nwos_create_file_without_storing_data(const char* path, const char* file_name, ObjRef* ref);
extern ObjCreateResult nwos_create_file(const char* path, ObjRef* ref, bool allow_multiple_paths);
extern ObjCreateResult nwos_add_new_revision_of_file(const char* path, ObjRef* ref);
extern ObjCreateResult nwos_find_or_create_path_and_file_association(ObjRef* path_ref, ObjRef* file_ref, TimeStamp mod_time, ObjRef* ref);
extern bool nwos_restore_file(ObjRef* file_ref, const char* path);
extern bool nwos_file_is_identical(ObjRef* assoc_ref, const char* path, MatchCode match);
extern const char* nwos_check_invalid_type(mode_t mode);
extern void nwos_check_directory_is_all_ok(const char* path);

#define MAX_FILES_PER_DISC_LIST 1000000
#define MAX_SIZE_DISC_LIST (sizeof(C_struct_Disc_List) + sizeof(ObjRef) * MAX_FILES_PER_DISC_LIST)

/* the following defines are for the return value from the nwos_read_files_disc_list function */
#define DISC_LIST_NUM_FILES_MASK       0x0FFFFFFF
#define DISC_LIST_ERROR_SHIFT          28
#define DISC_LIST_EMPTY_DIR_FLAG       (1 << DISC_LIST_ERROR_SHIFT)
#define DISC_LIST_EMPTY_FILE_FLAG      (2 << DISC_LIST_ERROR_SHIFT)
#define DISC_LIST_INVALID_TYPE_FLAG    (4 << DISC_LIST_ERROR_SHIFT)
#define DISC_LIST_TOO_MANY_FILES_FLAG  (8 << DISC_LIST_ERROR_SHIFT)
#define DISC_LIST_ERROR_MASK           (DISC_LIST_EMPTY_DIR_FLAG | DISC_LIST_EMPTY_FILE_FLAG | DISC_LIST_INVALID_TYPE_FLAG | DISC_LIST_TOO_MANY_FILES_FLAG)

extern size_t nwos_get_disc_list_object_size(void* disc_list_obj);
extern bool nwos_find_disc_list(char id[12], ObjRef* ref);
extern uint32 nwos_read_files_disc_list(const char* root_dir, const char* subdirectory, char* file_names[MAX_FILES_PER_DISC_LIST], uint32 num_files);
extern bool nwos_find_matching_disc_list(ObjRef files[MAX_FILES_PER_DISC_LIST], ObjRef* ref);
extern ObjCreateResult nwos_create_disc_list(char id[12], ObjRef disc_list[MAX_FILES_PER_DISC_LIST], ObjRef* ref);
extern bool nwos_find_disc_copy(ObjRef* disc_list, int copy_num, ObjRef* ref);
extern ObjCreateResult nwos_create_disc_copy(ObjRef* disc_list, int copy_num, ObjRef* location, ObjRef* ref);
extern bool nwos_find_storage_location(const char* location, ObjRef* ref);
extern ObjCreateResult nwos_create_storage_location(const char* location, ObjRef* ref);

/* uniform resource locator stuff */

#define MAX_URL_OBJ_SIZE   (sizeof(C_struct_Uniform_Resource_Locator) + 160)
extern bool nwos_url_to_string(ObjRef* ref, char* string, size_t size);

extern ObjCreateResult nwos_create_url(char* url, ObjRef* ref);

extern void nwos_create_public_acronym(char* acronym, ObjRef* ref);

extern void nwos_create_public_manufacturer(char* brand_name, char* acronym, char* url_list[], ObjRef* ref);

extern bool nwos_is_valid_ssn(const char* ssn);

#endif

