/*
--          This file is part of the New World OS and Objectify projects
--            Copyright (C) 2005, 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-07-25 17:25:15 -0600 (Sat, 25 Jul 2009) $
--   $Revision: 4184 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/


#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>   /* define memset */

#include "crc32.h"
#include "objectify.h"



bool nwos_find_public_area_code(char* area_code, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Area_Code area_code_obj;
    ObjRef reference;
    ReferenceList* ref_list;
    int num_area_codes;
    int i;

    assert(strlen(area_code) == 3);

    assert(isdigit(area_code[0]) && isdigit(area_code[1]) && isdigit(area_code[2]));

    /* first find out if we already have this area_code */

    assert(nwos_find_public_class_definition("AREA CODE", &reference));

    nwos_read_class_definition(&reference, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_area_codes = ref_list->common_header.num_refs;

    /* printf("num_area_codes: %d\n", num_area_codes); */

    for (i = 0; i < num_area_codes; i++)
    {
	assert(nwos_read_object_from_disk(&ref_list->references[i], &area_code_obj, sizeof(area_code_obj)));

	if (strncmp(area_code_obj.storage, area_code, 3) == 0)   /* found a match */
	{
	    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
	    break;
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i < num_area_codes);  /* return true if we found it */
}


bool nwos_find_private_area_code(char* area_code, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Area_Code area_code_obj;
    ObjRef reference;
    ReferenceList* ref_list;
    int num_area_codes;
    int i;

    assert(strlen(area_code) == 3);

    assert(isdigit(area_code[0]) && isdigit(area_code[1]) && isdigit(area_code[2]));

    /* first find out if we already have this area_code */

    nwos_find_or_create_private_class_definition("AREA CODE", &reference);

    nwos_read_class_definition(&reference, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_area_codes = ref_list->common_header.num_refs;

    /* printf("num_area_codes: %d\n", num_area_codes); */

    for (i = 0; i < num_area_codes; i++)
    {
	assert(nwos_read_object_from_disk(&ref_list->references[i], &area_code_obj, sizeof(area_code_obj)));

	if (strncmp(area_code_obj.storage, area_code, 3) == 0)   /* found a match */
	{
	    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
	    break;
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i < num_area_codes);  /* return true if we found it */
}


#ifdef PUBLIC_MODE

ObjCreateResult nwos_create_public_area_code(char* area_code, char* state_code, ObjRef* ref)
{
    C_struct_Area_Code area_code_obj;
    ObjRef area_code_class_ref;

    assert(strlen(area_code) == 3);
    assert(isdigit(area_code[0]) && isdigit(area_code[1]) && isdigit(area_code[2]));
    assert(strlen(state_code) == 2);
    assert(isalpha(state_code[0]) && isalpha(state_code[1]));

    assert(nwos_find_public_class_definition("AREA CODE", &area_code_class_ref));

    assert(!nwos_find_public_area_code(area_code, ref));

    memset(&area_code_obj, 0, sizeof(area_code_obj));  /* zero it out */

    nwos_generate_new_id(ref);

    nwos_fill_in_common_header(&area_code_obj.header.common, ref, &area_code_class_ref);

    memcpy(area_code_obj.storage, area_code, 3);

    nwos_create_reference_list(ref, &area_code_obj.header.object.references);

    nwos_find_state_from_postal_code(state_code, &area_code_obj.state);
    nwos_add_to_references(ref, &area_code_obj.state);

    nwos_crc32_calculate((uint8*) &area_code_obj.header.object, sizeof(ObjectHeader), area_code_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &area_code_obj.state, sizeof(area_code_obj) - sizeof(EveryObject), area_code_obj.header.common.data_chksum);

    nwos_write_object_to_disk(ref, &area_code_obj, sizeof(area_code_obj));

    nwos_add_to_references(ref, &area_code_class_ref);

    return CREATED_NEW;
}
#else


/* Returns false if area code does not exist in public objects */

bool nwos_find_or_create_private_area_code(char* area_code, ObjRef* ref)
{
    C_struct_Area_Code area_code_obj;
    ObjRef area_code_class_ref;
    ObjRef public_ref;
    bool result = false;

    assert(strlen(area_code) == 3);
    assert(isdigit(area_code[0]) && isdigit(area_code[1]) && isdigit(area_code[2]));

    if (nwos_find_private_area_code(area_code, ref))   /* already existed */
    {
	result = true;
    }
    else   /* didn't find it */
    {
	if (nwos_find_public_area_code(area_code, &public_ref))
	{
	    assert(nwos_read_object_from_disk(&public_ref, &area_code_obj, sizeof(area_code_obj)));

	    nwos_find_or_create_private_class_definition("AREA CODE", &area_code_class_ref);

	    nwos_generate_new_id(ref);

	    copy_reference(&area_code_obj.header.common.id, ref);
	    copy_reference(&area_code_obj.header.common.class_definition, &area_code_class_ref);

	    copy_reference(&area_code_obj.header.object.clone_of, &public_ref);

	    nwos_create_reference_list(ref, &area_code_obj.header.object.references);

	    nwos_crc32_calculate((uint8*) &area_code_obj.header.object, sizeof(ObjectHeader), area_code_obj.header.common.header_chksum);

	    nwos_write_object_to_disk(ref, &area_code_obj, sizeof(area_code_obj));

	    nwos_add_to_references(ref, &area_code_class_ref);

	    result = true;
	}
    }

    return result;
}
#endif


bool nwos_is_valid_phone_number(char* number)
{
    return nwos_invalid_phone_number_msg(number) == NULL;
}

const char* nwos_invalid_phone_number_msg(char* number)
{
    static char msg[64];
    int i;
    size_t length = strlen(number);
    bool separator = false;

    for (i = 0; number[i] != '\0'; i++)
    {
	if (!isdigit(number[i]))
	{
	    /* ok to have hyphen or period in the fourth place*/
	    if (number[i] == '-' || number[i] == '.')
	    {
		if (separator)  /* already have one separator */
		{
		    return "multiple separators (- or .)";
		}

		if (i != 3)
		{
		    snprintf(msg, sizeof(msg), "separator '%c' in the wrong place", number[i]);
		    return msg;
		}

		separator = true;
	    }
	    else
	    {
		if (isgraph(number[i]))
		{
		    snprintf(msg, sizeof(msg), "unexpected character: %c\n", number[i]);
		}
		else
		{
		    snprintf(msg, sizeof(msg), "unexpected character: 0x%02x\n", (uint8)number[i]);
		}

		return msg;
	    }
	}
    }

    if (separator)
    {
	if (length < 8) return "not enough digits";
	if (length > 8) return "too many digits";
    }
    else
    {
	if (length < 7) return "not enough digits";
	if (length > 7) return "too many digits";
    }

    return NULL;   /* no errors */
}


static void copy_phone_number(char number[7], char* phone_number)
{
    assert(nwos_is_valid_phone_number(phone_number));

    memcpy(number, phone_number, 3);   /* copy first 3 */
    if (!isdigit(phone_number[3]))
    {
	memcpy(number+3, phone_number+4, 4);
    }
    else
    {
	memcpy(number+3, phone_number+3, 4);
    }
}


bool nwos_find_phone_number(char* area_code, char* phone_number, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Phone_Number phone_number_obj;
    ObjRef reference;
    ObjRef area_code_ref;
    ReferenceList* ref_list;
    int num_phone_numbers;
    int i;
    char number[7];
    bool result = true;

    assert(nwos_is_valid_phone_number(phone_number));

    copy_phone_number(number, phone_number);

    if (area_code != NULL)
    {
#ifdef PUBLIC_MODE
	result = nwos_find_public_area_code(area_code, &area_code_ref);
#else
	result = nwos_find_private_area_code(area_code, &area_code_ref);
#endif
    }

    if (result)
    {
	assert(nwos_find_private_class_definition("PHONE NUMBER", &reference));

	nwos_read_class_definition(&reference, &class_def_obj);

	ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

	num_phone_numbers = ref_list->common_header.num_refs;

	/* printf("num_phone_numbers: %d\n", num_phone_numbers); */

	result = false;

	for (i = 0; i < num_phone_numbers; i++)
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &phone_number_obj, sizeof(phone_number_obj)));

	    if (area_code == NULL || is_same_object(&phone_number_obj.area_code, &area_code_ref))
	    {
		if (memcmp(phone_number_obj.storage, number, 7) == 0)   /* found a match */
		{
		    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		    result = true;
		    break;
		}
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }

    return result;
}


ObjCreateResult nwos_create_phone_number(char* area_code, char* phone_number, ObjRef* ref)
{
    C_struct_Phone_Number phone_number_obj;
    ObjRef reference;

    assert(nwos_is_valid_phone_number(phone_number));

    assert(!nwos_find_phone_number(area_code, phone_number, ref));

    memset(&phone_number_obj, 0, sizeof(phone_number_obj));  /* zero it out */

    nwos_find_or_create_private_class_definition("PHONE NUMBER", &reference);

    nwos_generate_new_id(ref);

    nwos_fill_in_common_header(&phone_number_obj.header.common, ref, &reference);

    nwos_create_reference_list(ref, &phone_number_obj.header.object.references);

    copy_phone_number(phone_number_obj.storage, phone_number);

    if (!nwos_find_or_create_private_area_code(area_code, &phone_number_obj.area_code))   /* didn't find it */
    {
	fprintf(stderr, "Unknown area code: %s\n", area_code);
	exit(1);
    }
    nwos_add_to_references(ref, &phone_number_obj.area_code);

    nwos_crc32_calculate((uint8*) &phone_number_obj.header.object, sizeof(ObjectHeader), phone_number_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &phone_number_obj.country, sizeof(phone_number_obj) - sizeof(EveryObject), phone_number_obj.header.common.data_chksum);

    nwos_write_object_to_disk(ref, &phone_number_obj, sizeof(phone_number_obj));

    nwos_add_to_references(ref, &reference);

    return CREATED_NEW;
}


bool nwos_phone_number_to_string(ObjRef* ref, char* string, size_t size)
{
    C_struct_Phone_Number number_obj;
    C_struct_Area_Code area_code_obj;
    int i;
    int j;

    assert(nwos_read_object_from_disk(ref, &number_obj, sizeof(number_obj)));

    assert(nwos_read_object_from_disk(&number_obj.area_code, &area_code_obj, sizeof(area_code_obj)));

    i = 0;

    assert(i < size);
    string[i++] = '(';

    for (j = 0; j < 3; j++) 
    {
	assert(i < size);
	string[i++] = area_code_obj.storage[j];
    }

    assert(i < size);
    string[i++] = ')';

    assert(i < size);
    string[i++] = ' ';

    for (j = 0; j < 3; j++) 
    {
	assert(i < size);
	string[i++] = number_obj.storage[j];
    }

    assert(i < size);
    string[i++] = '-';

    for (j = 3; j < 7; j++) 
    {
	assert(i < size);
	string[i++] = number_obj.storage[j];
    }

    assert(i < size);
    string[i++] = '\0';

    return true;
}



static bool parse_phone_number(char* phone_number, char area_code[], char number[])
{
    static char* patterns[] =
    {
	"(AAA) NNN-NNNN",
	"AAA-NNN-NNNN",
	"AAA.NNN.NNNN",
	"AAA NNN NNNN",
    };
    const int num_patterns = sizeof(patterns) / sizeof(char*);
    char* ptrpat;
    char* ptrin;
    char* ptrac;
    char* ptrnum;
    int i;

    for (i = 0; i < num_patterns; i++)
    {
	ptrpat = patterns[i];
	ptrin = phone_number;
	ptrac = area_code;
	ptrnum = number;

	while (ptrpat != NULL && *ptrpat != '\0' && *ptrin != '\0')
	{
	    switch (*ptrpat)
	    {
	      case 'A':
		if (isdigit(*ptrin))
		{
		    *ptrac = *ptrin;
		    ptrin++;
		    ptrac++;
		    ptrpat++;
		}
		else    /* no match */
		{
		    ptrpat = NULL;
		}
		break;

	      case ' ':   /* spaces are optional */
		while (*ptrin == ' ') ptrin++;
		ptrpat++;
		break;

	      case 'N':
		if (isalnum(*ptrin))
		{
		    *ptrnum = *ptrin;
		    ptrin++;
		    ptrnum++;
		    ptrpat++;
		}
		else    /* no match */
		{
		    ptrpat = NULL;
		}
		break;

	      default:    /* verify it matches the pattern exactly */
		if (*ptrin == *ptrpat)
		{
		    ptrin++;
		    ptrpat++;
		}
		else   /* no match */
		{
		    ptrpat = NULL;
		}
		break;
	    }
	}

	if (ptrpat != NULL && *ptrpat == '\0' && *ptrin == '\0')    /* it matched */
	{
	    *ptrac = '\0';
	    *ptrnum = '\0';
	    break;
	}
    }

    assert(i == num_patterns || strlen(area_code) == 3);
    assert(i == num_patterns || strlen(number) == 7);

    return (i < num_patterns);
}


static bool parse_phone_number_wo_area_code(char* phone_number, char number[])
{
    static char* patterns[] =
    {
	"NNN-NNNN",
	"NNN.NNNN",
	"NNN NNNN",
    };
    const int num_patterns = sizeof(patterns) / sizeof(char*);
    char* ptrpat;
    char* ptrin;
    char* ptrnum;
    int i;

    for (i = 0; i < num_patterns; i++)
    {
	ptrpat = patterns[i];
	ptrin = phone_number;
	ptrnum = number;

	while (ptrpat != NULL && *ptrpat != '\0' && *ptrin != '\0')
	{
	    switch (*ptrpat)
	    {
	      case ' ':   /* spaces are optional */
		while (*ptrin == ' ') ptrin++;
		ptrpat++;
		break;

	      case 'N':
		if (isalnum(*ptrin))
		{
		    *ptrnum = *ptrin;
		    ptrin++;
		    ptrnum++;
		    ptrpat++;
		}
		else    /* no match */
		{
		    ptrpat = NULL;
		}
		break;

	      default:    /* verify it matches the pattern exactly */
		if (*ptrin == *ptrpat)
		{
		    ptrin++;
		    ptrpat++;
		}
		else   /* no match */
		{
		    ptrpat = NULL;
		}
		break;
	    }
	}

	if (ptrpat != NULL && *ptrpat == '\0' && *ptrin == '\0')    /* it matched */
	{
	    *ptrnum = '\0';
	    break;
	}
    }

    assert(i == num_patterns || strlen(number) == 7);

    return (i < num_patterns);
}


#ifndef PUBLIC_MODE
void nwos_add_mobile_phone(char* name, char* phone_number)
{
    ObjRef reference;
    char area_code[16];
    char number[16];
    C_struct_Area_Code area_code_obj;
    C_struct_US_State state_obj;
    C_struct_Mobile_Phone phone_obj;
    ObjRef area_code_ref;
    ObjRef phone_class_ref;
    ObjRef person_ref;
    char temp[64];
    bool number_ok;
    int i;

    /* parse the number first in case there is something wrong with it */

    number_ok = parse_phone_number(phone_number, area_code, number);

    if (!number_ok)    /* try parsing without the area code */
    {
	if (parse_phone_number_wo_area_code(phone_number, number))
	{
	    while (!number_ok)
	    {
		nwos_ask_user("What is the area code (enter `quit' to return to main command)", area_code, 16);

		printf("\n");

		if (nwos_is_quit_command(area_code))    /* want's to bail out */
		{
		    printf("Ok, returning to main command.\n");
		    return;
		}

		if (isdigit(area_code[0]) && isdigit(area_code[1]) && isdigit(area_code[2]) && area_code[3] == '\0')
		{
		    number_ok = true;
		}
		else
		{
		    printf("Invalid area code: %s.  Please re-enter.\n\n", area_code);
		}
	    }
	}
    }

    while (number_ok && !nwos_find_public_area_code(area_code, &area_code_ref))
    {
	char* unimplemented_area_codes[] = { "800", "866", "877", "900" };
	const int num_unimplemented_area_codes = sizeof(unimplemented_area_codes) / sizeof(char*);
	for (i = 0; i < num_unimplemented_area_codes; i++)
	{
	    if (strcmp(area_code, unimplemented_area_codes[i]) == 0)
	    {
		printf("I'm sorry, this version of software cannot deal with toll free or pay\n");
		printf("numbers such as %s.  Returning to main command.\n", area_code);
		return;
	    }
	}

	printf("That area code is not in the database.  Unfortunately this version of the\n");
	printf("software does not have the capability to add an area code.  If it was a\n");
	printf("mistake, please re-enter the correct area code.  If not please check for\n");
	printf("a newer version of this software at http://sourceforge.net/projects/nwos/.\n");
	printf("\n");

	number_ok = false;

	while (!number_ok)
	{
	    nwos_ask_user("What is the area code (enter `quit' to return to main command)", area_code, 16);

	    printf("\n");

	    if (nwos_is_quit_command(area_code))    /* want's to bail out */
	    {
		printf("Ok, returning to main command.\n");
		return;
	    }

	    if (isdigit(area_code[0]) && isdigit(area_code[1]) && isdigit(area_code[2]) && area_code[3] == '\0')
	    {
		number_ok = true;
	    }
	    else
	    {
		printf("Invalid area code: %s.  Please re-enter.\n\n", area_code);
	    }
	}
    }

    /* Not sure about the phone number at this point, let's see about the person */

    if (!nwos_find_person(name, &person_ref))
    {
	if (!number_ok)
	{
	    printf("I'm sorry I can't find %s in the database and there is something\n", name);
	    printf("wrong with the phone number given: %s.  Please try again.\n", phone_number);
	}

	return;
    }

    /* At this point we have a person, so make sure the phone number is sane */

    if (!number_ok)
    {
	if (!nwos_ask_yes_or_no("I can't figure out the phone number given", "Would you like to re-enter it"))
	{
	    printf("Ok, returning to main command.\n");
	    return;
	}

	while (!number_ok)
	{
	    nwos_ask_user("What is the area code", area_code, 16);

	    printf("\n");

	    if (nwos_is_quit_command(area_code))    /* want's to bail out */
	    {
		printf("Ok, returning to main command.\n");
		return;
	    }

	    if (isdigit(area_code[0]) && isdigit(area_code[1]) && isdigit(area_code[2]) && area_code[3] == '\0')
	    {
		number_ok = true;
	    }
	    else
	    {
		printf("Invalid area code: %s.  Please re-enter.\n\n", area_code);
	    }
	}

	number_ok = false;

	while (!number_ok)
	{
	    nwos_ask_user("What is the phone number", number, 16);

	    number_ok = nwos_is_valid_phone_number(number);

	    if (!number_ok)
	    {
		printf("Invalid phone number: %s - %s\n", number, nwos_invalid_phone_number_msg(number));
		printf("Please try again.\n");
	    }
	}
    }

    assert(nwos_read_object_from_disk(&area_code_ref, &area_code_obj, sizeof(area_code_obj)));

    assert(nwos_read_object_from_disk(&area_code_obj.state, &state_obj, sizeof(state_obj)));

    nwos_name_to_string(&state_obj.name, temp, 64);

    printf("Area code is in state: %s\n", temp);


    nwos_find_or_create_private_class_definition("MOBILE PHONE", &phone_class_ref);

    memset(&phone_obj, 0, sizeof(phone_obj));

    nwos_generate_new_id(&reference);

    nwos_fill_in_common_header(&phone_obj.header.common, &reference, &phone_class_ref);

    memcpy(&phone_obj.person, &person_ref, sizeof(ObjRef));

    if (nwos_find_phone_number(area_code, number, &phone_obj.number))
    {
	printf("Sorry, that phone number already exists!\n");
    }
    else
      {
	nwos_create_phone_number(area_code, number, &phone_obj.number);

	nwos_crc32_calculate((uint8*) &phone_obj.header.object, sizeof(ObjectHeader), phone_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &phone_obj.person, sizeof(phone_obj) - sizeof(EveryObject), phone_obj.header.common.data_chksum);

	nwos_write_object_to_disk(&reference, &phone_obj, sizeof(phone_obj));

	nwos_create_reference_list(&reference, &phone_obj.header.object.references);

	nwos_add_to_references(&reference, &phone_obj.number);
	nwos_add_to_references(&reference, &phone_obj.person);
	    
	nwos_add_to_references(&reference, &phone_class_ref);
	
	printf("Mobile phone created: %02x%02x%02x%02x\n", reference.id[0], reference.id[1], reference.id[2], reference.id[3]);
    }
}


void nwos_find_phone_for_person(char* name)
{
    ObjRef person_ref;
    ObjRef phone_class_ref;
    ObjRef object_class;
    EveryObject header;
    C_struct_Mobile_Phone phone_obj;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    char temp[20];

    if (!nwos_find_person(name, &person_ref))
    {
	printf("I'm sorry I can't find anyone with the name %s.\n", name);
	return;
    }

    /**********************************************/
    /* Now look for a phone in the reference list */
    /**********************************************/

    nwos_read_object_headers_from_disk(&person_ref, &header);

    ref_list = nwos_malloc_reference_list(&header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("person num refs: %d\n", num_refs); */

    assert(nwos_find_private_class_definition("MOBILE PHONE", &phone_class_ref));

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	if (is_same_object(&object_class, &phone_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &phone_obj, sizeof(phone_obj)));

	    nwos_phone_number_to_string(&phone_obj.number, temp, 20);
	    printf("%s ", temp);
	}

	printf("\n");
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;
}
#endif

