/*
--          This file is part of the New World OS and Objectify projects
--                  Copyright (C) 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-07-25 17:25:15 -0600 (Sat, 25 Jul 2009) $
--   $Revision: 4184 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>    /* define sleep() */

#include "header.h"


uint32 convert_4_uint8_to_uint32(uint8 byte[4])
{
  return ((uint32)byte[0] << 24) | ((uint32)byte[1] << 16) | ((uint32)byte[2] << 8) | (uint32)byte[3];
}


static int version_string_to_int(char string[4])
{
    int i;
    int result = 0;

    for (i = 0; i < 4; i++) if (string[i] != '0') break;

    while (i < 4)
    {
	if (!isdigit(string[i])) return -1;

	result = result * 10 + string[i] - '0';

	i++;
    }

    return result;
}

#if 0
static void convert_0022_to_0029_header_to_0030_header(Disk_Header_0022_to_0029* header_0029, Disk_Header* header_0030)
{
    uint32 total_blocks;
    uint32 total_chunks;

    /* NOTE: this won't work because the sizes of the variables are different (32 bits vs. 64 bits) */
    /* But I am not sure this is even necessary because the diff file should have the correct headers */

    memcpy(header_0030->last_prep_disk,         header_0029->last_prep_disk,         sizeof(header_0030->last_prep_disk));
    memcpy(header_0030->last_change,            header_0029->last_change,            sizeof(header_0030->last_change));
    memcpy(header_0030->block_offset_to_chunks, header_0029->block_offset_to_chunks, sizeof(header_0030->block_offset_to_chunks));
    memcpy(header_0030->used_blocks,            header_0029->used_blocks,            sizeof(header_0030->used_blocks));
    memcpy(header_0030->used_chunks,            header_0029->used_chunks,            sizeof(header_0030->used_chunks));

    nwos_4_uint8_to_uint32(header_0029->total_blocks, &total_blocks);
    total_chunks = total_blocks / 65536;
    nwos_uint32_to_4_uint8(&total_chunks, header_0030->total_chunks);
}
#endif

int main(int argc, char* argv[])
{
    FILE* fp1;
    FILE* fp2;
    FILE* fp3 = NULL;
    uint8 buf1[FILE_BLOCK_SIZE];
    uint8 buf2[FILE_BLOCK_SIZE];
    int block1;
    int block2;
    size_t read1;
    size_t read2;
    size_t write3;
    Disk_Header header1;
    Disk_Header header2;
    bool wrapped_file = false;     /* true if older file that wasn't sequencial */
    bool bad_upper_ref;
    int header1_version;
    int header2_version;
    int in1;
    int in2;
    int out3;
    bool ignore_times;
    bool mismatch = false;
    const char* error_msg;


    if (argc == 4)
    {
	in1 = 1;
	in2 = 2;
	out3 = 3;
	ignore_times = false;
    }
    else if (argc == 5 && strcmp(argv[1], "--ignore-times") == 0)
    {
	in1 = 2;
	in2 = 3;
	out3 = 4;
	ignore_times = true;
    }
      else
    {
	fprintf(stderr, "usage: %s old_file patch_file new_file\n", argv[0]);
	exit(1);
    }


    /* Open the old file and check it */

    fp1 = fopen(argv[in1], "r");
    if (fp1 == NULL)
    {
	perror(argv[in1]);
	exit(1);
    }

    read1 = fread(buf1, 1, sizeof(buf1), fp1);

    if (read1 != FILE_BLOCK_SIZE)
    {
	if (ferror(fp1))
	{
	    perror(argv[in1]);
	}
	else
	{
	    fprintf(stderr, "Unexpected end of file: %s\n", argv[in1]);
	}
	fclose(fp1);
	exit(1);
    }

    memcpy(&header1, buf1, sizeof(header1));

    error_msg = nwos_check_disk_header(&header1, true, TYPE_CODE_COMPRESSED);   /* allow older versions */

    if (error_msg != NULL)
    {
	fprintf(stderr, "%s: %s\n", error_msg, argv[in1]);
	fclose(fp1);
	exit(1);
    }

    header1_version = version_string_to_int(header1.version_string);

    if (header1_version < 1)
    {
	fprintf(stderr, "Invalid version string in file: %s\n", argv[in1]);
	exit(1);
    }


    /* Open the patch file and check it */

    fp2 = fopen(argv[in2], "r");
    if (fp2 == NULL)
    {
	perror(argv[in2]);
	exit(1);
    }

    read2 = fread(buf2, 1, sizeof(buf2), fp2);

    if (read2 != FILE_BLOCK_SIZE)
    {
	if (ferror(fp2))
	{
	    perror(argv[in2]);
	}
	else
	{
	    fprintf(stderr, "Unexpected end of file: %s\n", argv[in2]);
	}
	fclose(fp1);
	fclose(fp2);
	exit(1);
    }


    /* Verify the old file header and the patch file header are exactly the same */

    memcpy(&header2, buf2, sizeof(header2));

    error_msg = nwos_check_disk_header(&header2, true, TYPE_CODE_DIFF);   /* allow older versions */

    if (error_msg != NULL)
    {
	fprintf(stderr, "%s: %s\n", error_msg, argv[in2]);
	fclose(fp1);
	fclose(fp2);
	exit(1);
    }

    header2_version = version_string_to_int(header2.version_string);

    if (header2_version < 1)
    {
	fprintf(stderr, "Invalid version string in file: %s\n", argv[in2]);
	fclose(fp1);
	fclose(fp2);
	exit(1);
    }

    if (header1_version != header2_version)
    {
	fprintf(stderr, "Old file version: %d doesn't match patch file version: %d!\n", header1_version, header2_version);
	mismatch = true;
    }

    if (!ignore_times)
    {
	if (memcmp(header1.last_prep_disk, header2.last_prep_disk, sizeof(header1.last_prep_disk)) != 0)
	{
	    fprintf(stderr, "Old file prep time stamp doesn't match patch file prep time stamp!\n");
	    mismatch = true;
	}

	if (memcmp(header1.last_change, header2.last_change, sizeof(header1.last_change)) != 0)
	{
	    fprintf(stderr, "Old file last change time stamp doesn't match patch file last change time stamp!\n");
	    mismatch = true;
	}
    }

    if (memcmp(header1.total_chunks, header2.total_chunks, sizeof(header1.total_chunks)) != 0)
    {
	fprintf(stderr, "Old file total chunks doesn't match patch file total chunks!\n");
	mismatch = true;
    }

    if (memcmp(header1.used_blocks, header2.used_blocks, sizeof(header1.used_blocks)) != 0)
    {
	uint32 old_blocks = (header1.used_blocks[0] << 24) | (header1.used_blocks[1] << 16) |
				(header1.used_blocks[2] << 8) | header1.used_blocks[3];
	uint32 new_blocks = (header2.used_blocks[0] << 24) | (header2.used_blocks[1] << 16) |
				(header2.used_blocks[2] << 8) | header2.used_blocks[3];
	fprintf(stderr, "Old file used blocks (%u) doesn't match patch file used blocks (%u)!\n", old_blocks, new_blocks);
	mismatch = true;
    }

    if (memcmp(header1.block_offset_to_chunks, header2.block_offset_to_chunks, sizeof(header1.block_offset_to_chunks)) != 0)
    {
	fprintf(stderr, "Old file block offset to chunks doesn't match patch file block offset to chunks!\n");
	mismatch = true;
    }

    if (memcmp(header1.used_chunks, header2.used_chunks, sizeof(header1.used_chunks)) != 0)
    {
	fprintf(stderr, "Old file used chunks doesn't match patch file used chunks!\n");
	mismatch = true;
    }

    if (mismatch)
    {
	fclose(fp1);
	fclose(fp2);
	exit(1);
    }


    /* Now read in the new file header */

    read2 = fread(buf2, 1, sizeof(buf2), fp2);

    if (read2 != FILE_BLOCK_SIZE)
    {
	if (ferror(fp2))
	{
	    perror(argv[in2]);
	}
	else
	{
	    fprintf(stderr, "Unexpected end of file: %s\n", argv[in2]);
	}
	fclose(fp1);
	fclose(fp2);
	exit(1);
    }


    memcpy(&header2, buf2, sizeof(header2));

    if (memcmp(header2.magic_number, "NWOS", 4) != 0)
    {
	fprintf(stderr, "Not an Objectify patch file: %s\n", argv[in2]);
	fclose(fp1);
	fclose(fp2);
	exit(1);
    }


    header2_version = version_string_to_int(header2.version_string);

    if (header2_version < 1)
    {
	fprintf(stderr, "Invalid version string in file: %s\n", argv[in2]);
	exit(1);
    }

    if (header1_version < 23 && header2_version < 23)
    {
	wrapped_file = true;   /* it's an old file */
    }
    else if (header1_version < 23 || header2_version < 23)
    {
	fprintf(stderr, "ERROR: '%s' is an old wrapped file, but file '%s' is not, cannot diff!\n",
		header1_version < 23 ? argv[in1] : argv[in2],
		header1_version < 23 ? argv[in2] : argv[in1]);
	exit(1);
    }


    fp3 = fopen(argv[out3], "w");

    if (fp3 == NULL)
    {
	perror(argv[out3]);
	fclose(fp1);
	fclose(fp2);
	exit(1);
    }

    if (header2_version >= 30)
    {
	memcpy(&(((Disk_Header*)buf2)->type_code), TYPE_CODE_COMPRESSED, sizeof((((Disk_Header*)buf2)->type_code)));
    }

    write3 = fwrite(buf2, 1, sizeof(buf2), fp3);

    if (write3 != FILE_BLOCK_SIZE)
    {
	perror(argv[out3]);
	fclose(fp1);
	fclose(fp2);
	fclose(fp3);
	exit(1);
    }

    block1 = 1;
    block2 = 1;

    read1 = fread(buf1, 1, sizeof(buf1), fp1);
    read2 = fread(buf2, 1, sizeof(buf2), fp2);

    while (!feof(fp1) && !feof(fp2) && read1 == FILE_BLOCK_SIZE && read2 == FILE_BLOCK_SIZE)
    {
	bad_upper_ref = false;

	if (header1_version < 30)
	{
	    if (buf1[0] != 0 || buf1[1] != 0 || buf1[2] != 0 || buf1[3] != 0)
	    {
		fprintf(stderr, "\n%s block %d - first four bytes not zero: %02x%02x%02x%02x\n",
			argv[in1], block1, buf1[0], buf1[1], buf1[2], buf1[3]);
		bad_upper_ref = true;
	    }

	    if (header2_version >= 30)
	    {
		buf1[0] = 0xff;
		buf1[1] = 0xff;
		buf1[2] = 0xff;
		buf1[3] = 0xfe;
	    }
	}
	else 
	{
	    if (buf1[0] != 0xff || buf1[1] != 0xff || buf1[2] != 0xff || buf1[3] != 0xfe)
	    {
		fprintf(stderr, "\n%s block %d - first four bytes not fffffffe: %02x%02x%02x%02x\n",
			argv[in1], block1, buf1[0], buf1[1], buf1[2], buf1[3]);
		bad_upper_ref = true;
	    }
	}

	if (header2_version < 30)
	{
	    if (buf2[0] != 0 || buf2[1] != 0 || buf2[2] != 0 || buf2[3] != 0)
	    {
		fprintf(stderr, "\n%s block %d - first four bytes not zero: %02x%02x%02x%02x\n",
			argv[in2], block2, buf2[0], buf2[1], buf2[2], buf2[3]);
		bad_upper_ref = true;
	    }
	}
	else
	{
	    if (buf2[0] != 0xff || buf2[1] != 0xff || buf2[2] != 0xff || buf2[3] != 0xfe)
	    {
		fprintf(stderr, "\n%s block %d - first four bytes not fffffffe: %02x%02x%02x%02x\n",
			argv[in2], block2, buf2[0], buf2[1], buf2[2], buf2[3]);
		bad_upper_ref = true;
	    }
	}

	if (bad_upper_ref)
	{
	    fclose(fp1);
	    fclose(fp2);
	    fclose(fp3);

	    exit(1);
	}

	if (buf1[4] == buf2[4] && buf1[5] == buf2[5] && buf1[6] == buf2[6] && buf1[7] == buf2[7])
	{
	    /* blocks have the same id number, must have changed */

	    if (memcmp(buf1, buf2, FILE_BLOCK_SIZE) != 0)
	    {
		printf("%02x%02x%02x%02x: changed\n",
		       buf1[4], buf1[5], buf1[6], buf1[7]);

		write3 = fwrite(buf2, 1, sizeof(buf2), fp3);
	    }
	    else   /* block stayed the same write the old block */
	    {
		write3 = fwrite(buf1, 1, sizeof(buf1), fp3);
	    }

	    if (write3 != FILE_BLOCK_SIZE)
	    {
		perror(argv[in2]);
		fclose(fp1);
		fclose(fp2);
		fclose(fp3);
		exit(1);
	    }

	    read1 = fread(buf1, 1, sizeof(buf1), fp1);
	    block1++;
	    read2 = fread(buf2, 1, sizeof(buf2), fp2);
	    block2++;
	}
	else   /* id is different, assume this an add, since right now can't delete */
	{
	    if (wrapped_file)
	    {
		/* if we've found the proper place to insert it */
		if (buf1[4] == buf2[0] && buf1[5] == buf2[1] && buf1[6] == buf2[2] && buf1[7] == buf2[3])
		{
		    printf("%02x%02x%02x%02x: added\n",
			   buf2[4], buf2[5], buf2[6], buf2[7]);

		    buf2[0] = 0;
		    buf2[1] = 0;
		    buf2[2] = 0;
		    buf2[3] = 0;

		    write3 = fwrite(buf2, 1, sizeof(buf2), fp3);

		    read2 = fread(buf2, 1, sizeof(buf2), fp2);
		    block2++;
		}
		else  /* not time to insert yet, just write the old block */
		{
		    write3 = fwrite(buf1, 1, sizeof(buf1), fp3);

		    read1 = fread(buf1, 1, sizeof(buf1), fp1);
		    block1++;
		}
	    }
	    else   /* new non-wrapped file that is sequential */
	    {
		/* if we've found the proper place to insert it */
		if (buf1[4] > buf2[4] || 
		    (buf1[4] == buf2[4] && buf1[5] > buf2[5]) ||
		    (buf1[4] == buf2[4] && buf1[5] == buf2[5] && buf1[6] > buf2[6]) ||
		    (buf1[4] == buf2[4] && buf1[5] == buf2[5] && buf1[6] == buf2[6] && buf1[7] > buf2[7])
		    )
		{
		    printf("%02x%02x%02x%02x: added\n",
			   buf2[4], buf2[5], buf2[6], buf2[7]);

		    write3 = fwrite(buf2, 1, sizeof(buf2), fp3);

		    read2 = fread(buf2, 1, sizeof(buf2), fp2);
		    block2++;

		    /* Change the upper 4 bytes back because we are going through the loop again */
		    if (header1_version < 30 && header2_version >= 30)
		    {
			buf1[0] = 0;
			buf1[1] = 0;
			buf1[2] = 0;
			buf1[3] = 0;
		    }
		}
		else  /* not time to insert yet, just write the old block */
		{
		    write3 = fwrite(buf1, 1, sizeof(buf1), fp3);

		    read1 = fread(buf1, 1, sizeof(buf1), fp1);
		    block1++;
		}
	    }

	    if (write3 != FILE_BLOCK_SIZE)
	    {
		perror(argv[out3]);
		fclose(fp1);
		fclose(fp2);
		fclose(fp3);
		exit(1);
	    }
	}

    }

    /* finish writing output */

    while (!feof(fp1) && read1 == FILE_BLOCK_SIZE)
    {
	bad_upper_ref = false;

	if (header1_version < 30)
	{
	    if (buf1[0] != 0 || buf1[1] != 0 || buf1[2] != 0 || buf1[3] != 0)
	    {
		fprintf(stderr, "\n%s block %d - first four bytes not zero: %02x%02x%02x%02x\n",
			argv[in1], block1, buf1[0], buf1[1], buf1[2], buf1[3]);
		bad_upper_ref = true;
	    }

	    if (header2_version >= 30)
	    {
		buf1[0] = 0xff;
		buf1[1] = 0xff;
		buf1[2] = 0xff;
		buf1[3] = 0xfe;
	    }
	}
	else 
	{
	    if (buf1[0] != 0xff || buf1[1] != 0xff || buf1[2] != 0xff || buf1[3] != 0xfe)
	    {
		fprintf(stderr, "\n%s block %d - first four bytes not fffffffe: %02x%02x%02x%02x\n",
			argv[in1], block1, buf1[0], buf1[1], buf1[2], buf1[3]);
		bad_upper_ref = true;
	    }
	}

	if (bad_upper_ref)
	{
	    fclose(fp1);
	    fclose(fp2);
	    fclose(fp3);

	    exit(1);
	}

	write3 = fwrite(buf1, 1, sizeof(buf1), fp3);

	if (write3 != FILE_BLOCK_SIZE)
	{
	    perror(argv[out3]);
	    fclose(fp1);
	    fclose(fp2);
	    fclose(fp3);
	    exit(1);
	}

	read1 = fread(buf1, 1, sizeof(buf1), fp1);
	block1++;
    }

    /* more in the patch file */

    while (!feof(fp2) && read2 == FILE_BLOCK_SIZE)
    {
	bad_upper_ref = false;

	if (header2_version < 30)
	{
	    if (buf2[0] != 0 || buf2[1] != 0 || buf2[2] != 0 || buf2[3] != 0)
	    {
		fprintf(stderr, "\n%s block %d - first four bytes not zero: %02x%02x%02x%02x\n",
			argv[in2], block2, buf2[0], buf2[1], buf2[2], buf2[3]);
		bad_upper_ref = true;
	    }
	}
	else
	{
	    if (buf2[0] != 0xff || buf2[1] != 0xff || buf2[2] != 0xff || buf2[3] != 0xfe)
	    {
		fprintf(stderr, "\n%s block %d - first four bytes not fffffffe: %02x%02x%02x%02x\n",
			argv[in2], block2, buf2[0], buf2[1], buf2[2], buf2[3]);
		bad_upper_ref = true;
	    }
	}

	if (bad_upper_ref)
	{
	    fclose(fp1);
	    fclose(fp2);
	    fclose(fp3);

	    exit(1);
	}

	printf("%02x%02x%02x%02x: added\n", buf2[4], buf2[5], buf2[6], buf2[7]);

	write3 = fwrite(buf2, 1, sizeof(buf2), fp3);

	if (write3 != FILE_BLOCK_SIZE)
	{
	    perror(argv[out3]);
	    fclose(fp1);
	    fclose(fp2);
	    fclose(fp3);
	    exit(1);
	}

	read2 = fread(buf2, 1, sizeof(buf2), fp2);
	block2++;
    }

    printf("\n");


    if (memcmp(header1.version_string, header2.version_string, 4) != 0)
    {
	fprintf(stderr, "WARNING, versions don't match - %s: %c%c%c%c  %s: %c%c%c%c\n",
		argv[in1], 
		header1.version_string[0], header1.version_string[1],
		header1.version_string[2], header1.version_string[3],
		argv[in2], 
		header2.version_string[0], header2.version_string[1],
		header2.version_string[2], header2.version_string[3]);
    }


    if (!feof(fp1) && read1 != FILE_BLOCK_SIZE)
    {
	perror(argv[in1]);
    }

    fclose(fp1);

    if (!feof(fp2) && read2 != FILE_BLOCK_SIZE)
    {
	perror(argv[in2]);
    }

    fclose(fp2);

    if (fclose(fp3) != 0)
    {
	perror(argv[out3]);
    }

    return 0;
}
