/*
--          This file is part of the New World OS and Objectify projects
--               Copyright (C) 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-07-25 17:25:15 -0600 (Sat, 25 Jul 2009) $
--   $Revision: 4184 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "crc32.h"
#include "objectify.h"



static size_t get_cardinal_object_size(void* cardinal_obj)
{
    int count = ((C_struct_Cardinal_Number*)cardinal_obj)->count;

    assert(0 < count && count <= MAX_CARDINAL_NUMBER_DIGITS);

    return sizeof(C_struct_Cardinal_Number) + count;
}


/* returns true if all characters in string are ascii decimal digits */
static bool is_number(char* string)
{
    char *p;

    if (*string == '\0') return false;  /* can't be an empty string */

    for (p = string; *p != '\0'; p++) if (!isdigit(*p)) break;

    return *p == '\0';
}


static char* skip_leading_zeros(char* number)
{
    char* result;

    assert(is_number(number));

    result = number;

    while (*result == '0') result++;

    if (*result == '\0') result--;

    return result;
}


bool nwos_find_cardinal_number(char* number, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    uint8 kludge[MAX_CARDINAL_NUMBER_OBJ_SIZE];
    C_struct_Cardinal_Number* cardinal_obj_ptr = (C_struct_Cardinal_Number*)kludge;
    ObjRef cardinal_class_ref;
    ObjRef class_ref;
    ReferenceList* ref_list;
    int num_digits;
    int num_numbers;
    int i;
    char* ptr_first_digit;

    assert(is_number(number));

    ptr_first_digit = skip_leading_zeros(number);

    num_digits = strlen(ptr_first_digit);

    assert(num_digits <= MAX_CARDINAL_NUMBER_DIGITS);


    assert(nwos_find_public_class_definition("CARDINAL NUMBER", &cardinal_class_ref));

    nwos_read_class_definition(&cardinal_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_numbers = ref_list->common_header.num_refs;

    /* printf("num_months: %d\n", num_months); */

    for (i = 0; i < num_numbers; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &class_ref);   /* find it's class */

	if (is_same_object(&cardinal_class_ref, &class_ref))
	{
	    assert(nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &get_cardinal_object_size));

	    if (cardinal_obj_ptr->count == num_digits)   /* this could be it */
	    {
		if (memcmp(cardinal_obj_ptr->digits, ptr_first_digit, num_digits) == 0)   /* found a match */
		{
		    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		    break;
		}
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return i < num_numbers;
}


/* Find existing cardinal_number or create new */

ObjCreateResult nwos_create_cardinal_number(char* number, ObjRef* ref)
{
    uint8 kludge[MAX_CARDINAL_NUMBER_OBJ_SIZE];
    C_struct_Cardinal_Number* ptr_cardinal_obj = (C_struct_Cardinal_Number*)kludge;
    ObjCreateResult result = FOUND_EXISTING;
    ObjRef class_ref;
    int num_digits;
    char* ptr_first_digit;

    assert(is_number(number));

    ptr_first_digit = skip_leading_zeros(number);

    num_digits = strlen(ptr_first_digit);

    assert(num_digits <= MAX_CARDINAL_NUMBER_DIGITS);


    /* first find out if we already have this cardinal_number */

    if (!nwos_find_cardinal_number(ptr_first_digit, ref))   /* didn't find it */
    {
	assert(nwos_find_public_class_definition("CARDINAL NUMBER", &class_ref));

	memset(kludge, 0, sizeof(kludge));  /* zero it out */

	/* remember ptr_cardinal_obj points to the kludge buffer */

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&ptr_cardinal_obj->header.common, ref, &class_ref);

	ptr_cardinal_obj->count = num_digits;

	memcpy(ptr_cardinal_obj->digits, ptr_first_digit, num_digits);

	nwos_create_reference_list(ref, &ptr_cardinal_obj->header.object.references);

	nwos_crc32_calculate((uint8*) &ptr_cardinal_obj->header.object, sizeof(ObjectHeader), ptr_cardinal_obj->header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &ptr_cardinal_obj->count, get_cardinal_object_size(kludge) - sizeof(EveryObject), ptr_cardinal_obj->header.common.data_chksum);

	nwos_write_object_to_disk(ref, kludge, get_cardinal_object_size(kludge));

	nwos_add_to_references(ref, &class_ref);

#ifdef VERIFY_WRITE
	ptr_cardinal_obj = malloc(sizeof(C_struct_Cardinal_Number) + num_digits);
	assert(nwos_read_object_from_disk(ref, ptr_cardinal_obj, sizeof(C_struct_Cardinal_Number) + num_digits));
	assert(memcmp(kludge, ptr_cardinal_obj, sizeof(C_struct_Cardinal_Number) + num_digits) == 0);

	memset(kludge, 0, sizeof(kludge));  /* clear it */
	assert(nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_cardinal_object_size));  /* read the other way */
	assert(memcmp(ptr_cardinal_obj, kludge, sizeof(C_struct_Cardinal_Number) + num_digits) == 0);

	free(ptr_cardinal_obj);
	ptr_cardinal_obj = NULL;
#endif

	result = CREATED_NEW;
    }

    return result;
}


/* returns true if the object passed in is a cardinal number */
static bool is_cardinal_number(ObjRef* ref)
{
    ObjRef cardinal_class_ref;
    ObjRef class_ref;

    assert(nwos_find_public_class_definition("CARDINAL NUMBER", &cardinal_class_ref));
    nwos_get_object_class(ref, &class_ref);   /* find it's class */

    return is_same_object(&cardinal_class_ref, &class_ref);
}


void nwos_cardinal_number_to_string(ObjRef* ref, char* string, size_t size)
{
    uint8 kludge[MAX_CARDINAL_NUMBER_OBJ_SIZE];
    C_struct_Cardinal_Number* cardinal_obj_ptr = (C_struct_Cardinal_Number*)kludge;
    int i;

    assert(is_cardinal_number(ref));

    assert(nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_cardinal_object_size));
    assert(cardinal_obj_ptr->count < size);

    for (i = 0; i < cardinal_obj_ptr->count; i++) string[i] = cardinal_obj_ptr->digits[i];
    string[i] = '\0';
}



/*******************/
/* Ordinal numbers */
/*******************/

bool nwos_find_ordinal_number(ObjRef* cardinal_ref, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Ordinal_Number ordinal_obj;
    ObjRef ordinal_class_ref;
    ObjRef class_ref;
    ReferenceList* ref_list;
    int num_numbers;
    int i;

    assert(!is_void_reference(cardinal_ref));
    assert(is_cardinal_number(cardinal_ref));

    assert(nwos_find_public_class_definition("ORDINAL NUMBER", &ordinal_class_ref));

    nwos_read_class_definition(&ordinal_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_numbers = ref_list->common_header.num_refs;

    /* printf("num_months: %d\n", num_months); */

    for (i = 0; i < num_numbers; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &class_ref);   /* find it's class */

	if (is_same_object(&ordinal_class_ref, &class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &ordinal_obj, sizeof(ordinal_obj)));

	    if (is_same_object(&ordinal_obj.cardinal, cardinal_ref))   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return i < num_numbers;
}


/* Find existing ordinal_number or create new */

ObjCreateResult nwos_create_ordinal_number(ObjRef* cardinal_ref, ObjRef* ref)
{
    C_struct_Ordinal_Number ordinal_obj;
    ObjCreateResult result = FOUND_EXISTING;
    ObjRef class_ref;

    assert(!is_void_reference(cardinal_ref));
    assert(is_cardinal_number(cardinal_ref));


    /* first find out if we already have this ordinal_number */

    if (!nwos_find_ordinal_number(cardinal_ref, ref))   /* didn't find it */
    {
	assert(nwos_find_public_class_definition("ORDINAL NUMBER", &class_ref));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&ordinal_obj.header.common, ref, &class_ref);

	copy_reference(&ordinal_obj.cardinal, cardinal_ref);

	nwos_create_reference_list(ref, &ordinal_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &ordinal_obj.header.object, sizeof(ObjectHeader), ordinal_obj.header.common.header_chksum);
	nwos_crc32_calculate((uint8*) &ordinal_obj.cardinal, sizeof(ordinal_obj) - sizeof(EveryObject), ordinal_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &ordinal_obj, sizeof(ordinal_obj));

#ifdef VERIFY_WRITE
	{
	C_struct_Ordinal_Number* ptr_ordinal_obj;
	ptr_ordinal_obj = malloc(sizeof(C_struct_Ordinal_Number));
	assert(nwos_read_object_from_disk(ref, ptr_ordinal_obj, sizeof(C_struct_Ordinal_Number)));
	assert(memcmp(&ordinal_obj, ptr_ordinal_obj, sizeof(C_struct_Ordinal_Number)) == 0);
	free(ptr_ordinal_obj);
	ptr_ordinal_obj = NULL;
	}
#endif

	nwos_add_to_references(ref, &class_ref);
	nwos_add_to_references(ref, cardinal_ref);

	result = CREATED_NEW;
    }

    return result;
}


void nwos_ordinal_number_to_string(ObjRef* ref, char* string, size_t size)
{
    C_struct_Ordinal_Number ordinal_obj;

    assert(nwos_read_object_from_disk(ref, &ordinal_obj, sizeof(ordinal_obj)));

    nwos_cardinal_number_to_string(&ordinal_obj.cardinal, string, size);
}


