/*
--          This file is part of the New World OS and Objectify projects
--               Copyright (C) 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-07-25 17:25:15 -0600 (Sat, 25 Jul 2009) $
--   $Revision: 4184 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/


#include <assert.h>
#include <ctype.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "class_definition.h"


char* current_filename;   /* kludgy, but needed for printing */

FILE* ref_list_fp;

#define MAX_LINE_LENGTH 256
#define MAX_REF_LIST_LINE_LENGTH 8192


static size_t get_class_object_size(void* class_obj)
{
    return sizeof(C_struct_Class_Definition) + (((C_struct_Class_Definition*)class_obj)->count * sizeof(ObjRef));
}


char *class_to_type(ObjRef* class_ref)
{
    uint8 kludge[MAX_CLASS_DEFINITION_OBJ_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*)kludge;
    char name[32];

    assert(nwos_read_variable_sized_object_from_disk(class_ref, kludge, sizeof(kludge), &get_class_object_size));
    nwos_name_to_string(&class_def_obj_ptr->name, name, sizeof(name));

    if (strcmp(name, "Object Reference") == 0) return "ObjRef";

    if (strcmp(name, "Byte") == 0) return "uint8";

    if (strcmp(name, "Character") == 0) return "char";

    if (strcmp(name, "Time Stamp") == 0) return "TimeStamp";

    /* sprintf(result, "%02x%02x%02x%02x",
               class_ref->id[0], class_ref->id[1], class_ref->id[2], class_ref->id[3]); */

    return "** ERROR **";
}


void output_feature(ObjRef* feature_ref)
{
    C_struct_Feature_Definition feature_obj;
    char label[32];
    int i;

    assert(nwos_read_object_from_disk(feature_ref, &feature_obj, sizeof(feature_obj)));

    nwos_name_to_string(&feature_obj.label, label, sizeof(label));

    for (i = 0; label[i] != '\0'; i++)
    {
	if (label[i] == ' ')
	{
	    label[i] = '_';
	}
	else if (isupper(label[i]))
	{
	    label[i] = tolower(label[i]);
	}
    }

    if (feature_obj.count == 1)
    {
	printf("    %s %s;\n", class_to_type(&feature_obj.class), label);
    }
    else
    {
	printf("    %s %s[%d];\n", class_to_type(&feature_obj.class), label, feature_obj.count);
    }
}

void output_class(ObjRef* class_ref)
{
    uint8 kludge[MAX_CLASS_DEFINITION_OBJ_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*)kludge;
    char name[32];
    int i;
    int revision;

    assert(nwos_read_variable_sized_object_from_disk(class_ref, kludge, sizeof(kludge), &get_class_object_size));
    nwos_name_to_string(&class_def_obj_ptr->name, name, sizeof(name));

    /* don't create structures for the built in types */

    if (strcmp(name, "Byte") != 0 &&
	strcmp(name, "Character") != 0 &&
	strcmp(name, "Object Reference") != 0 &&
	strcmp(name, "Reference List") != 0 &&
	strcmp(name, "Time Stamp") != 0)
    {
	for (i = 0; name[i] != '\0'; i++) if (name[i] == ' ') name[i] = '_';

	if (name[0] == 'U' && name[1] == 's' && name[2] == '_')
	{
	    name[1] = 'S';
	}
	else if (name[0] == 'C' && name[1] == 'd' && name[2] == '_')
	{
	    name[1] = 'D';
	}
	else if (name[0] == 'M' && name[1] == 'd' && name[2] == '5')
	{
	    name[1] = 'D';
	}
	else if (name[0] == 'S' && name[1] == 'h' && name[2] == 'a' && isdigit(name[3]))
	{
	    name[1] = 'H';
	    name[2] = 'A';
	}

	printf("typedef struct {\n");

	printf("    EveryObject header;\n");

	for (i = 0; i < class_def_obj_ptr->count; i++)
	{
	    output_feature(&class_def_obj_ptr->feature[i]);
	}

	if (is_void_reference(&class_def_obj_ptr->header.object.next_version))  /* no newer version */
	{
	    printf("} C_struct_%s;\n\n", name);
	}
	else   /* newer version of this class exists */
	{
	    revision = 1;
	    while (!is_void_reference(&class_def_obj_ptr->header.object.prev_version))
	    {
		revision++;

		assert(nwos_read_variable_sized_object_from_disk(&class_def_obj_ptr->header.object.prev_version, kludge, sizeof(kludge), &get_class_object_size));
	    }

	    printf("} C_struct_%s_%03d;\n\n", name, revision);
	}
    }
}

int main(int argc, char* argv[])
{
    uint8 kludge[MAX_CLASS_DEFINITION_OBJ_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*)kludge;
    ObjRef obj_class;
    ReferenceList* ref_list;
    int num_refs;
    int i;

    nwos_log_arguments(0, NULL);   /* disable logging */

    if (argc == 1)
    {
	nwos_initialize_objectify(PUBLIC, NULL);    /* public only */ 
    }
    else
    {
	fprintf(stderr, "usage: %s [public.objects.file]\n", argv[0]);
    }


    printf("/* This file was created by the 'export_c_structs' program from the objectify */\n");
    printf("/* public objects.  It should not be edited to make additions or changes, the */\n");
    printf("/* objectify classes should be changed and then regenerate this file.         */\n");
    printf("\n\n");


    assert(nwos_read_variable_sized_object_from_disk(&nwos_public_class_definition_class_ref, kludge, sizeof(kludge), &get_class_object_size));

    ref_list = nwos_malloc_reference_list(&class_def_obj_ptr->header.object.references);

    num_refs = ref_list->common_header.num_refs;

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &obj_class);

	if (is_same_object(&obj_class, &nwos_public_class_definition_class_ref))
	{
	    output_class(&ref_list->references[i]);
	}
    }

    nwos_free_reference_list(ref_list);
	 
    nwos_terminate_objectify();

    return 0;
}

