/*
--          This file is part of the New World OS and Objectify projects
--               Copyright (C) 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-07-25 17:25:15 -0600 (Sat, 25 Jul 2009) $
--   $Revision: 4184 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file and the file.c file which these
--   functions were taken from.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <assert.h>
#include <dirent.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "crc32.h"
#include "objectify.h"
#include "strlcxx.h"           /* in case strlcpy and strlcat are not provided by the system */


/*-------------------------------------------------------------------------------------------------------------------*/
/* Disc List object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

size_t nwos_get_disc_list_object_size(void* disc_list_obj)
{
    uint32 count = nwos_decode_variable_sized_count(((C_struct_Disc_List*)disc_list_obj)->count);

    assert(0 < count && count <= MAX_FILES_PER_DISC_LIST);

    return sizeof(C_struct_Disc_List) + count * sizeof(ObjRef);
}


bool nwos_find_disc_list(char id[12], ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    uint8 kludge[MAX_SIZE_DISC_LIST];
    C_struct_Disc_List* ptr_disc_obj = (C_struct_Disc_List*)kludge;
    ObjRef disc_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    int i;

    void_reference(ref);  /* in case we don't find it */


    /* for now return false if the private class doesn't exist */

    if (!nwos_find_private_class_definition("DISC LIST", &disc_class_ref)) return false;


    nwos_read_class_definition(&disc_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &disc_class_ref))
	{
	    assert(nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &nwos_get_disc_list_object_size));

	    /* remember ptr_disc_obj points to the kludge buffer */

	    if (memcmp(ptr_disc_obj->id, id, sizeof(ptr_disc_obj->id)) == 0)   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}


/* returns number of files added (zero if none) in bits 0-27                  */
/*   if an error occurred one of the upper bits is set:                       */
/*   (see DISC_LIST_*_FLAG defintions in objectify.h)                         */
/*                                                                            */
/*     bit 28 - if empty directories were found.                              */
/*     bit 29 - if empty files were found,                                    */
/*     bit 30 - if invalid files (symbolic links, devices, etc.) were found,  */
/*     bit 31 - if maximum files were reached,                                */

uint32 nwos_read_files_disc_list(const char* root_dir, const char* subdirectory, char* file_names[MAX_FILES_PER_DISC_LIST], uint32 num_files)
{
    DIR* dp;
    struct dirent *dir_entry;
    char path[PATH_MAX];
    char* subpath;
    size_t path_len;
    size_t subpath_size;
    struct stat stat_struct;
    const char* invalid_type;
    uint32 call_result;
    uint32 total_files;
    uint32 new_files = 0;
    uint32 error_code = 0;

    assert(root_dir != NULL && root_dir[0] != '\0');

    path_len = strlcpy(path, root_dir, sizeof(path));

    if (path_len + 1 >= sizeof(path))
    {
	fprintf(stderr, "root directory: %s is too long for buffer size: %d\n", root_dir, PATH_MAX);
	exit(1);
    }

    if (path[path_len - 1] != '/')   /* if there wasn't a slash at the end of the root directory, add one */
    {
	path[path_len] = '/';
	path_len++;
	path[path_len] = '\0';
    }

    subpath = &path[path_len];   /* save a pointer to just the file or subdirectory */

    if (subdirectory != NULL)
    {
	path_len = strlcat(path, subdirectory, sizeof(path));

	if (path_len >= sizeof(path))
	{
	    fprintf(stderr, "combined root directory: %s and subdirectory: %s are too long for buffer size: %d\n",
		    root_dir, subdirectory, PATH_MAX);
	}
    }

    dp = opendir(path);
    if (dp == NULL)
    {
	perror(path);
	exit(1);
    }

    dir_entry = readdir(dp);
    while (dir_entry != NULL)
    {
	if (strcmp(dir_entry->d_name, ".") != 0 && strcmp(dir_entry->d_name, "..") != 0)
	{
	    assert(path_len == strlen(path));

	    if (path[path_len - 1] != '/')   /* if there isn't a slash at the end of the path, add one */
	    {
		path[path_len] = '/';
		path[path_len + 1] = '\0';
	    }

	    if (strlcat(path, dir_entry->d_name, sizeof(path)) >= sizeof(path))
	    {
		fprintf(stderr, "combined path: %s and %s are too long for buffer size: %d\n",
			subdirectory, dir_entry->d_name, PATH_MAX);
	    }

	    if (lstat(path, &stat_struct) != 0)
	    {
		perror(path);
		exit(1);
	    }

	    invalid_type = nwos_check_invalid_type(stat_struct.st_mode);

	    if (invalid_type != NULL)
	    {
		fprintf(stderr, "ERROR: '%s' is a %s and cannot be in a disc list!\n", path, invalid_type);

		error_code |= DISC_LIST_INVALID_TYPE_FLAG;
	    }
	    else if (S_ISDIR(stat_struct.st_mode))
	    {
	      call_result = nwos_read_files_disc_list(root_dir, subpath, file_names, (num_files + new_files) | error_code);

		new_files  += (call_result & DISC_LIST_NUM_FILES_MASK);  /* add files in */
		error_code |= (call_result & DISC_LIST_ERROR_MASK);      /* combine the errors */

		if ((call_result & DISC_LIST_NUM_FILES_MASK) == 0)       /* empty directory */
		{
		    fprintf(stderr, "ERROR: subdirectory '%s' is empty!\n", path);

		    error_code |= DISC_LIST_EMPTY_DIR_FLAG;                  /* merge in empty directory error (1) */
		}
	    }
	    else   /* must be a regular file */
	    {
		assert(S_ISREG(stat_struct.st_mode));

		/* first check for an empty file */

		if (stat_struct.st_size == 0)
		{
		    fprintf(stderr, "ERROR: file '%s' is empty, this version cannot handle empty files.\n", path);

		    error_code |= DISC_LIST_EMPTY_FILE_FLAG;  /* merge in a 2 */
		}
		else
		{
		    total_files = (num_files + new_files) & DISC_LIST_NUM_FILES_MASK;

		    if (total_files < MAX_FILES_PER_DISC_LIST)
		    {
			assert(file_names[total_files] == NULL);

			subpath_size = strlen(subpath) + 1;

			file_names[total_files] = malloc(subpath_size);

			if (file_names[total_files] == NULL)
			{
			    perror("allocating memory for file name");
			    exit(1);
			}

			strlcpy(file_names[total_files], subpath, subpath_size);

			/* printf("%s\n", file_names[num_files + new_files]); */
		    }

		    new_files++;
		}
	    }

	    path[path_len] = '\0';    /* truncate back to just the subdirectory */
	}

	dir_entry = readdir(dp);
    }

    if (closedir(dp) != 0)
    {
	perror(path);
	exit(1);
    }

    if (subdirectory == NULL)   /* this is the root invocation */
    {
	total_files = (num_files + new_files) & DISC_LIST_NUM_FILES_MASK;

	if (total_files >= MAX_FILES_PER_DISC_LIST)
	{
	    fprintf(stderr, "ERROR: Too many files: %d in directory '%s', this version can only handle a maximum of %d files.\n",
		    total_files, root_dir, MAX_FILES_PER_DISC_LIST);

	    error_code |= DISC_LIST_TOO_MANY_FILES_FLAG;
	}
    }

    return error_code | new_files;
}



bool nwos_find_matching_disc_list(ObjRef files[MAX_FILES_PER_DISC_LIST], ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    uint8 kludge[MAX_SIZE_DISC_LIST];
    C_struct_Disc_List* ptr_disc_obj = (C_struct_Disc_List*)kludge;
    ObjRef disc_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    int j;
    int k;
    int count = 0;

    void_reference(ref);  /* in case we don't find it */


    /* first count the files */

    for (i = 0; i < MAX_FILES_PER_DISC_LIST; i++)
    {
	if (!is_void_reference(&files[i])) count++;
    }

    assert(count > 0);

    /* for now return false if the private disc list class doesn't exist */

    if (!nwos_find_private_class_definition("DISC LIST", &disc_class_ref)) return false;


    nwos_read_class_definition(&disc_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &disc_class_ref))
	{
	    assert(nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &nwos_get_disc_list_object_size));

	    /* remember ptr_disc_obj points to the kludge buffer */

	    if (nwos_decode_variable_sized_count(ptr_disc_obj->count) == count)   /* found a possible match */
	    {
		for (j = 0; j < MAX_FILES_PER_DISC_LIST; j++)
		{
		    if (!is_void_reference(&files[j]))
		    {
			for (k = 0; k < count; k++)
			{
			    if (is_same_object(&files[j], &ptr_disc_obj->files[k])) break;
			}

			if (k == count) break;   /* didn't find a match this can't be the list */
		    }
		}

		if (j == MAX_FILES_PER_DISC_LIST)   /* found a match for each one */
		{
		    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		    break;
		}
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}



/* Find existing disc list or create new */

ObjCreateResult nwos_create_disc_list(char id[12], ObjRef files[MAX_FILES_PER_DISC_LIST], ObjRef* ref)
{
    uint8 kludge[MAX_SIZE_DISC_LIST];
    C_struct_Disc_List* ptr_disc_obj = (C_struct_Disc_List*)kludge;
    ObjRef disc_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
    int i;
    int count = 0;


    /* first find out if we already have this disc */

    if (!nwos_find_disc_list(id, ref))   /* didn't find it */
    {
	memset(kludge, 0, sizeof(kludge));  /* zero it out */

	/* remember ptr_path_obj points to the kludge buffer */

	nwos_find_or_create_private_class_definition("DISC LIST", &disc_class_ref);

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&ptr_disc_obj->header.common, ref, &disc_class_ref);

	memcpy(&ptr_disc_obj->id, id, sizeof(ptr_disc_obj->id));

	for (i = 0; i < MAX_FILES_PER_DISC_LIST; i++)
	{
	    if (!is_void_reference(&files[i]))
	    {
		copy_reference(&ptr_disc_obj->files[count], &files[i]);
		count++;

		nwos_add_to_references(ref, &files[i]);
	    }
	}

	nwos_encode_variable_sized_count(count, ptr_disc_obj->count);

	assert(nwos_get_disc_list_object_size(ptr_disc_obj) == sizeof(C_struct_Disc_List) + count * sizeof(ObjRef));

	nwos_create_reference_list(ref, &ptr_disc_obj->header.object.references);

	nwos_crc32_calculate((uint8*) &ptr_disc_obj->header.object, sizeof(ObjectHeader), ptr_disc_obj->header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &ptr_disc_obj->id, nwos_get_disc_list_object_size(ptr_disc_obj) - sizeof(EveryObject), ptr_disc_obj->header.common.data_chksum);

	nwos_write_object_to_disk(ref, kludge, nwos_get_disc_list_object_size(ptr_disc_obj));

	nwos_add_to_references(ref, &disc_class_ref);

#ifdef VERIFY_WRITE
	ptr_disc_obj = malloc(sizeof(C_struct_Disc_List) + count * sizeof(ObjRef));
	assert(nwos_read_object_from_disk(ref, ptr_disc_obj, sizeof(C_struct_Disc_List) + count * sizeof(ObjRef)));
	assert(memcmp(kludge, ptr_disc_obj, sizeof(C_struct_Disc_List) + count * sizeof(ObjRef)) == 0);

	memset(kludge, 0, sizeof(kludge));  /* clear it */
	assert(nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &nwos_get_disc_list_object_size));  /* read the other way */
	assert(memcmp(ptr_disc_obj, kludge, sizeof(C_struct_Disc_List) + count * sizeof(ObjRef)) == 0);

	free(ptr_disc_obj);
	ptr_disc_obj = NULL;
#endif

	result = CREATED_NEW;
    }

    return result;
}


/*-------------------------------------------------------------------------------------------------------------------*/
/* Disc Copy object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/


/* find existing disc copy */

bool nwos_find_disc_copy(ObjRef* disc_list, int copy_num, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Disc_Copy disc_obj;
    ObjRef disc_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    int i;

    void_reference(ref);  /* in case we don't find it */


    /* for now return false if there isn't a disc copy private class definition */

    if (!nwos_find_private_class_definition("DISC COPY", &disc_class_ref)) return false;

    nwos_read_class_definition(&disc_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &disc_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &disc_obj, sizeof(disc_obj)));

	    if (is_same_object(&disc_obj.disc_list, disc_list) && disc_obj.copy_number == copy_num)   /* found a match */
	    {
		/* if there is a newer version get it */
		while (!is_void_reference(&disc_obj.header.object.next_version))
		{
		    assert(nwos_read_object_from_disk(&disc_obj.header.object.next_version, &disc_obj, sizeof(disc_obj)));
		    assert(is_same_object(&disc_obj.disc_list, disc_list));
		    assert(disc_obj.copy_number == copy_num);
		}

		copy_reference(ref, &disc_obj.header.common.id);
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}



/* Find existing disc copy or create new */

ObjCreateResult nwos_create_disc_copy(ObjRef* disc_list, int copy_num, ObjRef* location, ObjRef* ref)
{
    C_struct_Disc_Copy disc_copy_obj;
#ifdef VERIFY_WRITE
    C_struct_Disc_Copy* ptr_disc_obj;
#endif
    ObjRef disc_class_ref;
    ObjCreateResult result = FOUND_EXISTING;


    /* first find out if we already have this disc copy */

    if (nwos_find_disc_copy(disc_list, copy_num, ref))   /* already exists */
    {
	assert(nwos_read_object_from_disk(ref, &disc_copy_obj, sizeof(disc_copy_obj)));

	if (is_same_object(&disc_copy_obj.location, location))
	{
	    /* add verification here */
	}
	else  /* location changed, create new object */
	{
	    nwos_find_or_create_private_class_definition("DISC COPY", &disc_class_ref);

	    /* verify ref is == to object ID in header */
	    assert(is_same_object(ref, &disc_copy_obj.header.common.id));
	    assert(is_void_reference(&disc_copy_obj.header.object.next_version));

	    nwos_generate_new_id(ref);

	    copy_reference(&disc_copy_obj.header.object.next_version, ref);

	    nwos_crc32_calculate((uint8*) &disc_copy_obj.header.object,
				 sizeof(ObjectHeader),
				 disc_copy_obj.header.common.header_chksum);

	    nwos_overwrite_object_to_disk(&disc_copy_obj.header.common.id, &disc_copy_obj, sizeof(disc_copy_obj));

	    /* clear out next pointer that we just wrote */
	    void_reference(&disc_copy_obj.header.object.next_version);

	    /* copy the old object into the previous pointer */
	    copy_reference(&disc_copy_obj.header.object.prev_version, &disc_copy_obj.header.common.id);

	    /* put the new id in */
	    copy_reference(&disc_copy_obj.header.common.id, ref);

	    /* change the location */
	    copy_reference(&disc_copy_obj.location, location);

	    nwos_crc32_calculate((uint8*) &disc_copy_obj.header.object,
				 sizeof(ObjectHeader),
				 disc_copy_obj.header.common.header_chksum);

	    nwos_crc32_calculate((uint8*) &disc_copy_obj.disc_list,
				 sizeof(disc_copy_obj) - sizeof(EveryObject),
				 disc_copy_obj.header.common.data_chksum);

	    nwos_write_object_to_disk(ref, &disc_copy_obj, sizeof(disc_copy_obj));

	    nwos_add_to_references(ref, &disc_class_ref);
	    nwos_add_to_references(ref, &disc_copy_obj.disc_list);


	    nwos_add_to_references(ref, location);
	}
    }
    else   /* didn't find it */
    {
	nwos_find_or_create_private_class_definition("DISC COPY", &disc_class_ref);

	memset(&disc_copy_obj, 0, sizeof(disc_copy_obj));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&disc_copy_obj.header.common, ref, &disc_class_ref);

	copy_reference(&disc_copy_obj.disc_list, disc_list);
	copy_reference(&disc_copy_obj.location, location);
	disc_copy_obj.copy_number = copy_num;

	nwos_create_reference_list(ref, &disc_copy_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &disc_copy_obj.header.object, sizeof(ObjectHeader), disc_copy_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &disc_copy_obj.disc_list, sizeof(disc_copy_obj) - sizeof(EveryObject), disc_copy_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &disc_copy_obj, sizeof(disc_copy_obj));

	nwos_add_to_references(ref, &disc_class_ref);

	nwos_add_to_references(ref, disc_list);
	nwos_add_to_references(ref, location);

#ifdef VERIFY_WRITE
	ptr_disc_obj = malloc(sizeof(C_struct_Disc_Copy));
	assert(nwos_read_object_from_disk(ref, ptr_disc_obj, sizeof(C_struct_Disc_Copy)));
	assert(memcmp(&disc_copy_obj, ptr_disc_obj, sizeof(C_struct_Disc_Copy)) == 0);

	free(ptr_disc_obj);
	ptr_disc_obj = NULL;
#endif

	result = CREATED_NEW;
    }

    return result;
}



/*-------------------------------------------------------------------------------------------------------------------*/
/* Storage Location object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/


/* find existing storage location */

bool nwos_find_storage_location(const char* location, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Storage_Location location_obj;
    ObjRef location_class_ref;
    ObjRef object_class;
    ObjRef name_ref;
    ReferenceList* ref_list;
    int num_refs;
    int i;

    void_reference(ref);  /* in case we don't find it */


    if (!nwos_find_private_name(location, &name_ref))
    {
	return false;
    }


    /* for now return false if storage location private class definition doesn't exist */

    if (!nwos_find_private_class_definition("STORAGE LOCATION", &location_class_ref)) return false;


    nwos_read_class_definition(&location_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &location_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &location_obj, sizeof(location_obj)));

	    if (is_same_object(&location_obj.name, &name_ref))   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}



/* Find existing disc copy or create new */

ObjCreateResult nwos_create_storage_location(const char* location, ObjRef* ref)
{
    C_struct_Storage_Location location_obj;
#ifdef VERIFY_WRITE
    C_struct_Storage_Location* ptr_loc_obj;
#endif
    ObjRef location_class_ref;
    ObjRef name_ref;
    ObjCreateResult result = FOUND_EXISTING;


    /* first find out if we already have this location */

    if (!nwos_find_storage_location(location, ref))   /* didn't find it */
    {
	nwos_create_name(location, &name_ref);

	nwos_find_or_create_private_class_definition("STORAGE LOCATION", &location_class_ref);

	memset(&location_obj, 0, sizeof(location_obj));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&location_obj.header.common, ref, &location_class_ref);

	copy_reference(&location_obj.name, &name_ref);

	nwos_create_reference_list(ref, &location_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &location_obj.header.object, sizeof(ObjectHeader), location_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &location_obj.name, sizeof(location_obj) - sizeof(EveryObject), location_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &location_obj, sizeof(location_obj));

	nwos_add_to_references(ref, &location_class_ref);

	nwos_add_to_references(ref, &name_ref);

#ifdef VERIFY_WRITE
	ptr_loc_obj = malloc(sizeof(C_struct_Storage_Location));
	assert(nwos_read_object_from_disk(ref, ptr_loc_obj, sizeof(C_struct_Storage_Location)));
	assert(memcmp(&location_obj, ptr_loc_obj, sizeof(C_struct_Storage_Location)) == 0);

	free(ptr_loc_obj);
	ptr_loc_obj = NULL;
#endif

	result = CREATED_NEW;
    }

    return result;
}


