/*
--          This file is part of the New World OS and Objectify projects
--                  Copyright (C) 2006, 2007, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-07-25 17:25:15 -0600 (Sat, 25 Jul 2009) $
--   $Revision: 4184 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/


#include <assert.h>
#include <ctype.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "../crc32.h"
#include "../objectify_private.h"
#include "c_structs_0014.h"


#define NUM_PUBLIC_IDS 268435456

#define NUM_PRIVATE_IDS ((uint32) (4294967296LL - NUM_PUBLIC_IDS))


char* current_filename;   /* kludgy, but needed for printing */

FILE* ref_list_fp;

ObjRef nwos_root_object_ref;


#define MAX_LINE_LENGTH 256
#define MAX_REF_LIST_LINE_LENGTH 8192


static size_t get_spelling_object_size(void* spelling_obj)
{
    assert(((C_struct_Spelling*)spelling_obj)->count > 0);
    return sizeof(C_struct_Spelling) + ((C_struct_Spelling*)spelling_obj)->count;
}

static size_t get_name_object_size(void* name_obj)
{
    assert(((C_struct_Name*)name_obj)->count > 0);
    return sizeof(C_struct_Name) + (((C_struct_Name*)name_obj)->count * sizeof(ObjRef));
}

static size_t get_year_object_size(void* year_obj)
{
    assert(((C_struct_Year_0014*)year_obj)->count > 0);
    return sizeof(C_struct_Year_0014) + ((C_struct_Year_0014*)year_obj)->count;
}


void shift_line(char* line, int index)
{
    int i;

    assert(strlen(line) < (MAX_REF_LIST_LINE_LENGTH - index));

    for (i = 0; line[i + index] != '\0'; i++)
    {
	line[i] = line[i + index];
    }

    line[i] = '\0';
}


void fput_char(char c, FILE* fp)
{
    if (fputc(c, fp) == EOF)
    {
	perror(current_filename);
	exit(1);
    }
}


void check_comma(char line[MAX_LINE_LENGTH])
{
    if (line[0] != ',')
    {
	printf("%s: comma\n", current_filename);
	printf("remainder of line: %s\n", line);
	exit(1);
    }

    shift_line(line, 1);
}


void check_double_quote(char line[MAX_LINE_LENGTH])
{
    if (line[0] != '"')
    {
	printf("%s: missing double quote\n", current_filename);
	printf("remainder of line: %s\n", line);
	exit(1);
    }

    shift_line(line, 1);
}


void check_newline(char line[MAX_LINE_LENGTH])
{
    if (line[0] != '\n')
    {
	printf("%s: missing end of line\n", current_filename);
	printf("remainder of line: %s\n", line);
	exit(1);
    }

    shift_line(line, 1);
}

void get_string(char line[MAX_LINE_LENGTH], char string[], int size)
{
    int i;

    check_double_quote(line);

    for (i = 0; line[i] != '"' && line[i] != '\0'; i++)
    {
	if (i >= (size - 1))
	{
	    printf("%s: string too long\n", current_filename);
	    printf("remainder of line: %s\n", line);
	    exit(1);
	}

	string[i] = line[i];
    }

    string[i] = '\0';

    shift_line(line, i);

    check_double_quote(line);
}


void get_uint8(char line[MAX_LINE_LENGTH], uint8* ptr_uint8)
{
    uint8 digit;

    if (!isdigit(line[0]))
    {
	printf("no digit for reading uint8\n");
	printf("remainder of line: %s\n", line);
	exit(1);
    }

    *ptr_uint8 = line[0] - '0';

    shift_line(line, 1);

    if (isdigit(line[0]))
    {
	digit = line[0] - '0';

	shift_line(line, 1);

	*ptr_uint8 = *ptr_uint8 * 10 + digit;
    }

    if (isdigit(line[0]))
    {
	digit = line[0] - '0';

	shift_line(line, 1);

	if (*ptr_uint8 > 25 || (*ptr_uint8 == 25 && line[0] > '6'))
	{
	    printf("value too large for uint8: %u%u\n", *ptr_uint8, digit);
	    printf("remainder of line: %s\n", line);
	    exit(1);
	}

	*ptr_uint8 = *ptr_uint8 * 10 + digit;
    }

    if (isdigit(line[0]))
    {
	printf("too many digits for uint8: %u\n", *ptr_uint8);
	printf("remainder of line: %s\n", line);
	exit(1);
    }
}


int hex_char_to_value(char line[MAX_LINE_LENGTH])
{
    char hex_digit[17] = "0123456789abcdef";
    char* p = strchr(hex_digit, line[0]);

    if (p == NULL)
    {
	printf("%s: invalid hexadecimal character\n", current_filename);
	printf("remainder of line: %s\n", line);
	exit(1);
    }

    shift_line(line, 1);

    return p - hex_digit;
}
    

void get_hex4(char line[MAX_LINE_LENGTH], uint8 hex[4])
{
    int i;
    uint8 upper;
    uint8 lower;

    if (line[0] != '0' || line[1] != 'x')
    {
	printf("%s: missing 0x for hexadecimal value\n", current_filename);
	printf("remainder of line: %s\n", line);
	exit(1);
    }

    shift_line(line, 2);

    for (i = 0; i < 4; i++)
    {
	upper = hex_char_to_value(line);
	lower = hex_char_to_value(line);
	hex[i] = (upper << 4) | lower;
    }
}


void get_objref(char line[MAX_LINE_LENGTH], ObjRef* ref)
{
    get_hex4(line, ref->id);
}


void get_timestamp(char line[MAX_LINE_LENGTH], TimeStamp ts)
{
    char string[32];
    uint year;
    uint month;
    uint day_of_month;
    uint day_of_week;
    uint hour;
    uint minute;
    uint second;
    uint microsecond;
    int count;

    get_string(line, string, sizeof(string));


    count = sscanf(string, "%04u-%02u-%02u (%u) %02u:%02u:%02u.%06u",
		   &year, &month, &day_of_month,  &day_of_week,  &hour, &minute, &second, &microsecond);

    if (count != 8)
    {
	printf("count mismatch when reading time stamp: %s - %d\n", string, count);
	exit(1);
    }

    nwos_insert_year_into_time_stamp(year, ts);
    nwos_insert_month_into_time_stamp(month, ts);
    nwos_insert_day_of_month_into_time_stamp(day_of_month, ts);

    nwos_insert_day_of_week_into_time_stamp(day_of_week, ts);

    nwos_insert_hour_into_time_stamp(hour, ts);
    nwos_insert_minute_into_time_stamp(minute, ts);
    nwos_insert_second_into_time_stamp(second, ts);
    nwos_insert_microsecond_into_time_stamp(microsecond, ts);
}


#if 0
/* this routine updates all references to flatten them into 1 space instead of all 32 bit numbers */
/* objects versions 0012 to 0015 allowed any 32 bit value and wrapped them to fit in the disk space */

#define MAX_XLAT 8192

#define OLD_DISK_BLOCKS 0x3a38b2e0
#define NEW_DISK_BLOCKS 0x2a38b2e0

static struct {
    ObjRef old_id;
    ObjRef new_id;
} xlat_old_new_tbl[MAX_XLAT];

bool update_objref(ObjRef* ref)
{
    int i;
    uint32 block;

    if (!is_void_reference(ref) && nwos_ref_to_word(ref) > OLD_DISK_BLOCKS)     /* it needs to be changed */
    {
	for (i = 0; !is_void_reference(&xlat_old_new_tbl[i].old_id); i++)     /* find out if it has already been encountered */
	{
	    if (is_same_object(ref, &xlat_old_new_tbl[i].old_id))
	    {
		break;
	    }
	}

	if (is_void_reference(&xlat_old_new_tbl[i].old_id))    /* first time for this one */
	{
	    assert(i < MAX_XLAT);    /* not very graceful */

	    memcpy(&xlat_old_new_tbl[i].old_id, ref, sizeof(ObjRef));    /* save the old one in the table */

	    block = (nwos_ref_to_word(ref) - NUM_PUBLIC_IDS) % NEW_DISK_BLOCKS + NUM_PUBLIC_IDS;

	    xlat_old_new_tbl[i].new_id.id[0] = block >> 24;
	    xlat_old_new_tbl[i].new_id.id[1] = block >> 16;
	    xlat_old_new_tbl[i].new_id.id[2] = block >> 8;
	    xlat_old_new_tbl[i].new_id.id[3] = block;

	    printf("Change %02x%02x%02x%02x to %02x%02x%02x%02x\n",
		   xlat_old_new_tbl[i].old_id.id[0],
		   xlat_old_new_tbl[i].old_id.id[1],
		   xlat_old_new_tbl[i].old_id.id[2],
		   xlat_old_new_tbl[i].old_id.id[3],
		   xlat_old_new_tbl[i].new_id.id[0],
		   xlat_old_new_tbl[i].new_id.id[1],
		   xlat_old_new_tbl[i].new_id.id[2],
		   xlat_old_new_tbl[i].new_id.id[3]);
	}

	memcpy(ref, &xlat_old_new_tbl[i].new_id, sizeof(ObjRef));

	return true;
    }

    return false;    /* no change needed */
}
#endif


void import_common_header(char line[MAX_LINE_LENGTH], CommonHeader* ch)
{
    get_hex4(line, ch->header_chksum);

    check_comma(line);

    get_hex4(line, ch->data_chksum);

    check_comma(line);

    get_timestamp(line, ch->creation_time);

    check_comma(line);

    get_objref(line, &ch->class_definition);
}


void import_object_header(char line[MAX_LINE_LENGTH], ObjectHeader* oh)
{
    get_objref(line, &oh->references);
}


void import_every_object(char line[MAX_LINE_LENGTH], EveryObject* eo)
{
    uint8 chksum[4];

    get_objref(line, &eo->common.id);

//    update_objref(&eo->common.id);    /* can just update this now because it isn't part of a checksum */

    check_comma(line);

    import_common_header(line, &eo->common);

    check_comma(line);

    import_object_header(line, &eo->object);

    check_comma(line);

    nwos_crc32_calculate((uint8*) &eo->object, sizeof(eo->object), chksum);

    assert(memcmp(chksum, eo->common.header_chksum, sizeof(chksum)) == 0);

//    if (update_objref(&eo->object.references))   /* it had to be changed, so recompute header checksum */
//    {
//	nwos_crc32_calculate((uint8*) &eo->object, sizeof(eo->object), eo->common.header_chksum);
//    }
}
    
#if 0
void import_root(char line[MAX_LINE_LENGTH])
{
    C_struct_Root new_root_obj;
    uint8 chksum[4];
    struct {
        EveryObject header;
        ObjRef      class_definition_class;
    } old_root_obj;

    assert(strlen(line) < MAX_LINE_LENGTH);

    memset(&old_root_obj, 0, sizeof(old_root_obj));

    import_every_object(line, &old_root_obj.header);

    printf("Old Root Object %02x%02x%02x%02x: ", 
	   old_root_obj.header.common.id.id[0],
	   old_root_obj.header.common.id.id[1],
	   old_root_obj.header.common.id.id[2],
	   old_root_obj.header.common.id.id[3]);

    get_objref(line, &old_root_obj.class_definition_class);

    check_newline(line);

    printf("class_definition_class: %02x%02x%02x%02x\n",
	   old_root_obj.class_definition_class.id[0],
	   old_root_obj.class_definition_class.id[1],
	   old_root_obj.class_definition_class.id[2],
	   old_root_obj.class_definition_class.id[3]);

    nwos_crc32_calculate((uint8*) &old_root_obj.class_definition_class, sizeof(old_root_obj) - sizeof(EveryObject), chksum);

    assert(memcmp(chksum, old_root_obj.header.common.data_chksum, sizeof(chksum)) == 0);

    memcpy(&nwos_class_definition_class_ref, &old_root_obj.class_definition_class, sizeof(ObjRef));

    assert(!is_void_reference(&nwos_reference_list_class_ref));
    assert(!is_void_reference(&nwos_root_object_ref));

    nwos_create_root(&nwos_root_object_ref);

    printf("New Root Object %02x%02x%02x%02x: ", 
	   new_root_obj.header.common.id.id[0],
	   new_root_obj.header.common.id.id[1],
	   new_root_obj.header.common.id.id[2],
	   new_root_obj.header.common.id.id[3]);
    fflush(stdout);

    nwos_read_object_from_disk(&nwos_root_object_ref, &new_root_obj, sizeof(new_root_obj));

    printf("class_definition_class: %02x%02x%02x%02x  reference_list_class: %02x%02x%02x%02x\n",
	   new_root_obj.class_definition_class.id[0],
	   new_root_obj.class_definition_class.id[1],
	   new_root_obj.class_definition_class.id[2],
	   new_root_obj.class_definition_class.id[3],
	   new_root_obj.reference_list_class.id[0],
	   new_root_obj.reference_list_class.id[1],
	   new_root_obj.reference_list_class.id[2],
	   new_root_obj.reference_list_class.id[3]);
}


/* The old private class definitions are converted to clones of the public class definitions */

void import_class_definition(char line[MAX_LINE_LENGTH])
{
    ObjRef public_class_ref;
    C_struct_Class_Definition class_def_obj;
    uint8 chksum[4];

    assert(strlen(line) < MAX_LINE_LENGTH);

    memset(&class_def_obj, 0, sizeof(class_def_obj));

    import_every_object(line, &class_def_obj.header);

    printf("Class Def %02x%02x%02x%02x: ", 
	   class_def_obj.header.common.id.id[0],
	   class_def_obj.header.common.id.id[1],
	   class_def_obj.header.common.id.id[2],
	   class_def_obj.header.common.id.id[3]);

    get_objref(line, &class_def_obj.name);

    check_newline(line);

    nwos_crc32_calculate((uint8*) &class_def_obj.name, sizeof(class_def_obj) - sizeof(EveryObject), chksum);

    assert(memcmp(chksum, class_def_obj.header.common.data_chksum, sizeof(chksum)) == 0);

    if (update_objref(&class_def_obj.name))      /* if name reference changed, recompute data checksum */
    {
	nwos_crc32_calculate((uint8*) &class_def_obj.name,
			     sizeof(class_def_obj) - sizeof(EveryObject), 
			     class_def_obj.header.common.data_chksum);
    }

    printf("%02x%02x%02x%02x\n",
	   class_def_obj.name.id[0],
	   class_def_obj.name.id[1],
	   class_def_obj.name.id[2],
	   class_def_obj.name.id[3]);

#if 0
    /* kludgy thing to find reference list class */
    if (is_void_reference(&class_def_obj.header.object.references))   /* this must be the reference list class */
    {
	assert(is_void_reference(&nwos_reference_list_class_ref));
	memcpy(&nwos_reference_list_class_ref, &class_def_obj.header.common.id, sizeof(ObjRef));
    }
#endif
}
#endif


C_struct_Spelling* spellings[500];
int num_spellings;

void import_spelling(char line[MAX_LINE_LENGTH])
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Spelling* spelling_obj_ptr = (C_struct_Spelling*) kludge;
    int i;
    uint8 chksum[4];

    assert(strlen(line) < MAX_LINE_LENGTH);

    memset(kludge, 0, sizeof(kludge));

    import_every_object(line, &spelling_obj_ptr->header);

    printf("Spelling %02x%02x%02x%02x: ", 
	   spelling_obj_ptr->header.common.id.id[0],
	   spelling_obj_ptr->header.common.id.id[1],
	   spelling_obj_ptr->header.common.id.id[2],
	   spelling_obj_ptr->header.common.id.id[3]);

    if (nwos_block_used(&spelling_obj_ptr->header.common.id)) printf("  *** block used ***  ");

    check_double_quote(line);

    for (i = 0; line[i] != '"' && line[i] != '\0'; i++)
    {
	assert(i < 100);                          /* there shouldn't be any words this long */
	spelling_obj_ptr->storage[i] = line[i];
	printf("%c", spelling_obj_ptr->storage[i]);
    }

    shift_line(line, i);

    spelling_obj_ptr->count = i;

    check_double_quote(line);

    check_newline(line);

    printf("\n");

    nwos_crc32_calculate((uint8*) &spelling_obj_ptr->character_set, 
			 get_spelling_object_size(spelling_obj_ptr) - sizeof(EveryObject), 
			 chksum);

    assert(memcmp(chksum, spelling_obj_ptr->header.common.data_chksum, sizeof(chksum)) == 0);

    spellings[num_spellings] = malloc(get_spelling_object_size(spelling_obj_ptr));
    assert(spellings[num_spellings] != NULL);
    memcpy(spellings[num_spellings], spelling_obj_ptr, get_spelling_object_size(spelling_obj_ptr));
    num_spellings++;
}


void process_spelling(ObjRef* spelling_ref, char* string)
{
    int i;
#ifndef TEST_BLOCKS_USED
    ObjRef spelling_class_ref;
    ObjRef public_ref;
    char msg[128];
#endif

    for (i = 0; i < num_spellings; i++)
    {
	if (is_same_object(spelling_ref, &spellings[i]->header.common.id)) break;
    }

    assert(i < num_spellings);

    memcpy(string, spellings[i]->storage, spellings[i]->count);
    string[spellings[i]->count] = '\0';

    printf("Found spelling: %02x%02x%02x%02x - %s\n",
	   spelling_ref->id[0], spelling_ref->id[1], spelling_ref->id[2], spelling_ref->id[3],
	   string);

#ifndef TEST_BLOCKS_USED
    if (spellings[i]->header.common.flags == 0)
    {
	nwos_find_or_create_private_class_definition("SPELLING", &spelling_class_ref);

	snprintf(msg, sizeof(msg), "storing old spelling %02x%02x%02x%02x\n",
		 spelling_ref->id[0], spelling_ref->id[1], spelling_ref->id[2], spelling_ref->id[3]);
	nwos_log(msg);

	copy_reference(&spellings[i]->header.common.class_definition, &spelling_class_ref);

	nwos_create_reference_list(spelling_ref, &spellings[i]->header.object.references);

	if (nwos_find_public_spelling(string, &public_ref))
	{
	    printf("Found public spelling: %02x%02x%02x%02x\n",
		   public_ref.id[0], public_ref.id[1], public_ref.id[2], public_ref.id[3]);

	    copy_reference(&spellings[i]->header.object.clone_of, &public_ref);
	}

	nwos_crc32_calculate((uint8*) &spellings[i]->header.object, sizeof(ObjectHeader), spellings[i]->header.common.header_chksum);

	nwos_set_bit_in_map(nwos_hash_ref(spelling_ref));
	nwos_write_object_to_disk(spelling_ref, spellings[i], get_spelling_object_size(spellings[i]));

	nwos_add_to_references(spelling_ref, &spelling_class_ref);

	spellings[i]->header.common.flags = 1;    /* flag that we already processed this spelling */
    }
#endif
}


C_struct_Name* names[333];
int num_names;

void import_name(char line[MAX_LINE_LENGTH])
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Name* name_obj_ptr = (C_struct_Name*) kludge;
    int i;
    uint8 chksum[4];
    bool changed;

    assert(strlen(line) < MAX_LINE_LENGTH);

    memset(kludge, 0, sizeof(kludge));

    import_every_object(line, &name_obj_ptr->header);

    printf("Name %02x%02x%02x%02x:", 
	   name_obj_ptr->header.common.id.id[0],
	   name_obj_ptr->header.common.id.id[1],
	   name_obj_ptr->header.common.id.id[2],
	   name_obj_ptr->header.common.id.id[3]);

    if (nwos_block_used(&name_obj_ptr->header.common.id)) printf("  *** block used ***  ");

    i = 0;
    do
    {
	if (i > 0) check_comma(line);

	get_objref(line, &name_obj_ptr->spelling[i]);

	i++;
    }
    while (line[0] == ',');

    name_obj_ptr->count = i;

    check_newline(line);

    nwos_crc32_calculate((uint8*) &name_obj_ptr->count, get_name_object_size(name_obj_ptr) - sizeof(EveryObject), chksum);

    assert(memcmp(chksum, name_obj_ptr->header.common.data_chksum, sizeof(chksum)) == 0);

    changed = false;
   
    for (i = 0; i < name_obj_ptr->count; i++)    /* go back and check all references are legal */
    {
      /*	changed |= update_objref(&name_obj_ptr->spelling[i]); */

	printf(" %02x%02x%02x%02x", 
	   name_obj_ptr->spelling[i].id[0],
	   name_obj_ptr->spelling[i].id[1],
	   name_obj_ptr->spelling[i].id[2],
	   name_obj_ptr->spelling[i].id[3]);
    }

    printf("\n");
#if 0
    if (changed)      /* if any of the spelling references changed, recompute data checksum */
    {
	nwos_crc32_calculate((uint8*) &name_obj_ptr->count,
			     get_name_object_size(name_obj_ptr) - sizeof(EveryObject), 
			     name_obj_ptr->header.common.data_chksum);
    }
#endif

    names[num_names] = malloc(get_name_object_size(name_obj_ptr));
    assert(names[num_names] != NULL);
    memcpy(names[num_names], name_obj_ptr, get_name_object_size(name_obj_ptr));
    num_names++;
}


void process_name(ObjRef* name_ref)
{
    int i;
    int j;
    char spelling[64];
    char name[128];
#ifndef TEST_BLOCKS_USED
    ObjRef name_class_ref;
    ObjRef public_ref;
    char msg[128];
#endif

    for (i = 0; i < num_names; i++)
    {
	if (is_same_object(name_ref, &names[i]->header.common.id)) break;
    }

    assert(i < num_names);

    name[0] = '\0';
    for (j = 0; j < names[i]->count; j++)
    {
	process_spelling(&names[i]->spelling[j], spelling);

	spelling[0] = toupper(spelling[0]);

	if (name[0] == '\0')
	{
	    strcpy(name, spelling);
	}
	else
	{
	    strcat(name, " ");
	    strcpy(name, spelling);
	}
    }

    printf("Found name %02x%02x%02x%02x: %s\n",
	   name_ref->id[0], name_ref->id[1], name_ref->id[2], name_ref->id[3], name);

#ifndef TEST_BLOCKS_USED
    if (names[i]->header.common.flags == 0)
    {
	nwos_find_or_create_private_class_definition("NAME", &name_class_ref);

	snprintf(msg, sizeof(msg), "storing old name %02x%02x%02x%02x\n",
		 name_ref->id[0], name_ref->id[1], name_ref->id[2], name_ref->id[3]);
	nwos_log(msg);

	copy_reference(&names[i]->header.common.class_definition, &name_class_ref);

	nwos_create_reference_list(name_ref, &names[i]->header.object.references);

	if (nwos_find_public_name(name, &public_ref))
	{
	    printf("Found public name: %02x%02x%02x%02x\n",
		   public_ref.id[0], public_ref.id[1], public_ref.id[2], public_ref.id[3]);

	    copy_reference(&names[i]->header.object.clone_of, &public_ref);
	}

	nwos_crc32_calculate((uint8*) &names[i]->header.object, sizeof(ObjectHeader), names[i]->header.common.header_chksum);

	nwos_set_bit_in_map(nwos_hash_ref(name_ref));
	nwos_write_object_to_disk(name_ref, names[i], get_name_object_size(names[i]));

	nwos_add_to_references(name_ref, &name_class_ref);

	for (j = 0; j < names[i]->count; j++)
	{
	    nwos_add_to_references(name_ref, &names[i]->spelling[j]);
	}

	names[i]->header.common.flags = 1;    /* flag that we already processed this name */
    }
#endif
}


#if 0
void import_word(char line[MAX_LINE_LENGTH])
{
    C_struct_Word word_obj;
    uint8 chksum[4];

    assert(strlen(line) < MAX_LINE_LENGTH);

    memset(&word_obj, 0, sizeof(word_obj));

    import_every_object(line, &word_obj.header);

    printf("Word %02x%02x%02x%02x: ", 
	   word_obj.header.common.id.id[0],
	   word_obj.header.common.id.id[1],
	   word_obj.header.common.id.id[2],
	   word_obj.header.common.id.id[3]);

    get_objref(line, &word_obj.thing);

    check_comma(line);

    get_objref(line, &word_obj.language);

    check_comma(line);

    get_objref(line, &word_obj.spelling);

    check_newline(line);

    nwos_crc32_calculate((uint8*) &word_obj.thing, sizeof(word_obj) - sizeof(EveryObject), chksum);

    assert(memcmp(chksum, word_obj.header.common.data_chksum, sizeof(chksum)) == 0);

    /* if any of the references changed, recompute the data checksum */
    if (update_objref(&word_obj.thing) | update_objref(&word_obj.language) | update_objref(&word_obj.spelling))
    {
	nwos_crc32_calculate((uint8*) &word_obj.thing,
			     sizeof(word_obj) - sizeof(EveryObject),
			     word_obj.header.common.data_chksum);
    }

    printf("thing: %02x%02x%02x%02x  language: %02x%02x%02x%02x  spelling: %02x%02x%02x%02x\n",
	   word_obj.thing.id[0],
	   word_obj.thing.id[1],
	   word_obj.thing.id[2],
	   word_obj.thing.id[3],
	   word_obj.language.id[0],
	   word_obj.language.id[1],
	   word_obj.language.id[2],
	   word_obj.language.id[3],
	   word_obj.spelling.id[0],
	   word_obj.spelling.id[1],
	   word_obj.spelling.id[2],
	   word_obj.spelling.id[3]);
}
    

void import_language(char line[MAX_LINE_LENGTH])
{
    C_struct_Language language_obj;
    uint8 chksum[4];

    assert(strlen(line) < MAX_LINE_LENGTH);

    memset(&language_obj, 0, sizeof(language_obj));

    import_every_object(line, &language_obj.header);

    printf("Language %02x%02x%02x%02x: ", 
	   language_obj.header.common.id.id[0],
	   language_obj.header.common.id.id[1],
	   language_obj.header.common.id.id[2],
	   language_obj.header.common.id.id[3]);

    get_objref(line, &language_obj.definition);

    check_newline(line);

    nwos_crc32_calculate((uint8*) &language_obj.definition, sizeof(language_obj) - sizeof(EveryObject), chksum);

    assert(memcmp(chksum, language_obj.header.common.data_chksum, sizeof(chksum)) == 0);

    if (update_objref(&language_obj.definition))
    {
	nwos_crc32_calculate((uint8*) &language_obj.definition,
			     sizeof(language_obj) - sizeof(EveryObject),
			     language_obj.header.common.data_chksum);
    }

    printf("%02x%02x%02x%02x\n",
	   language_obj.definition.id[0],
	   language_obj.definition.id[1],
	   language_obj.definition.id[2],
	   language_obj.definition.id[3]);
}
    

void import_us_state(char line[MAX_LINE_LENGTH])
{
    C_struct_US_State_0014 state_obj;
    uint8 chksum[4];

    assert(strlen(line) < MAX_LINE_LENGTH);

    memset(&state_obj, 0, sizeof(state_obj));

    import_every_object(line, &state_obj.header);

    printf("State %02x%02x%02x%02x: ", 
	   state_obj.header.common.id.id[0],
	   state_obj.header.common.id.id[1],
	   state_obj.header.common.id.id[2],
	   state_obj.header.common.id.id[3]);

    get_objref(line, &state_obj.name);

    check_comma(line);

    get_objref(line, &state_obj.date);

    check_comma(line);

    get_objref(line, &state_obj.capital);

    check_comma(line);

    get_objref(line, &state_obj.postal_code);

    check_comma(line);

    get_uint8(line, &state_obj.number);

    check_newline(line);

    nwos_crc32_calculate((uint8*) &state_obj.name, sizeof(state_obj) - sizeof(EveryObject), chksum);

    assert(memcmp(chksum, state_obj.header.common.data_chksum, sizeof(chksum)) == 0);

    if (update_objref(&state_obj.name) | update_objref(&state_obj.date) |
	update_objref(&state_obj.capital) | update_objref(&state_obj.postal_code))
    {
	nwos_crc32_calculate((uint8*) &state_obj.name,
			     sizeof(state_obj) - sizeof(EveryObject),
			     state_obj.header.common.data_chksum);
    }

    printf("name: %02x%02x%02x%02x  date: %02x%02x%02x%02x  capital: %02x%02x%02x%02x  abbr: %02x%02x%02x%02x  num:%d\n",
	   state_obj.name.id[0], state_obj.name.id[1], state_obj.name.id[2], state_obj.name.id[3],
	   state_obj.date.id[0], state_obj.date.id[1], state_obj.date.id[2], state_obj.date.id[3],
	   state_obj.capital.id[0], state_obj.capital.id[1], state_obj.capital.id[2], state_obj.capital.id[3],
	   state_obj.postal_code.id[0], state_obj.postal_code.id[1], state_obj.postal_code.id[2], state_obj.postal_code.id[3],
	   state_obj.number);
}
    

void import_us_city(char line[MAX_LINE_LENGTH])
{
    C_struct_US_City city_obj;
    uint8 chksum[4];

    assert(strlen(line) < MAX_LINE_LENGTH);

    memset(&city_obj, 0, sizeof(city_obj));

    import_every_object(line, &city_obj.header);

    printf("City %02x%02x%02x%02x: ", 
	   city_obj.header.common.id.id[0],
	   city_obj.header.common.id.id[1],
	   city_obj.header.common.id.id[2],
	   city_obj.header.common.id.id[3]);

    get_objref(line, &city_obj.name);

    check_comma(line);

    get_objref(line, &city_obj.state);

    check_newline(line);

    nwos_crc32_calculate((uint8*) &city_obj.name, sizeof(city_obj) - sizeof(EveryObject), chksum);

    assert(memcmp(chksum, city_obj.header.common.data_chksum, sizeof(chksum)) == 0);

    if (update_objref(&city_obj.name) | update_objref(&city_obj.state))
    {
	nwos_crc32_calculate((uint8*) &city_obj.name, sizeof(city_obj) - sizeof(EveryObject), city_obj.header.common.data_chksum);
    }

    printf("name: %02x%02x%02x%02x  state: %02x%02x%02x%02x\n",
	   city_obj.name.id[0], city_obj.name.id[1], city_obj.name.id[2], city_obj.name.id[3],
	   city_obj.state.id[0], city_obj.state.id[1], city_obj.state.id[2], city_obj.state.id[3]);
}
#endif


void import_area_code(char line[MAX_LINE_LENGTH])
{
    C_struct_Area_Code area_code_obj;
    int i;
    uint8 chksum[4];
    char area_code[4];
#ifndef TEST_BLOCKS_USED
    ObjRef area_code_ref;
    ObjRef area_code_class_ref;
    ObjRef public_ref;
    ObjRef public_state_ref;
    char msg[128];
#endif

    assert(strlen(line) < MAX_LINE_LENGTH);

    memset(&area_code_obj, 0, sizeof(area_code_obj));

    import_every_object(line, &area_code_obj.header);

    printf("Area Code %02x%02x%02x%02x: ", 
	   area_code_obj.header.common.id.id[0],
	   area_code_obj.header.common.id.id[1],
	   area_code_obj.header.common.id.id[2],
	   area_code_obj.header.common.id.id[3]);

    if (nwos_block_used(&area_code_obj.header.common.id)) printf("  *** block used ***  ");

    get_objref(line, &area_code_obj.state);

    check_comma(line);

    get_string(line, area_code, sizeof(area_code));

    if (strlen(area_code) != sizeof(area_code_obj.storage))
    {
	printf("area code too short: %s\n", area_code);
	exit(0);
    }

    for (i = 0; i < sizeof(area_code_obj.storage); i++) 
    {
	if (!isdigit(area_code[i]))
	{
	    printf("invalid character in area code: %s\n", area_code);
	    exit(0);
	}
	area_code_obj.storage[i] = area_code[i];
    }

    check_newline(line);

    nwos_crc32_calculate((uint8*) &area_code_obj.state, sizeof(area_code_obj) - sizeof(EveryObject), chksum);

    assert(memcmp(chksum, area_code_obj.header.common.data_chksum, sizeof(chksum)) == 0);

#if 0
    if (update_objref(&area_code_obj.state))      /* if state reference changed, update data checksum */
    {
	nwos_crc32_calculate((uint8*) &area_code_obj.state,
			     sizeof(area_code_obj) - sizeof(EveryObject),
			     area_code_obj.header.common.data_chksum);
    }
#endif

    printf("state: %02x%02x%02x%02x  storage: %c%c%c\n",
	   area_code_obj.state.id[0], area_code_obj.state.id[1], area_code_obj.state.id[2], area_code_obj.state.id[3],
	   area_code_obj.storage[0], area_code_obj.storage[1], area_code_obj.storage[2]);

#ifndef TEST_BLOCKS_USED
    if (area_code_obj.storage[0] == '8' && area_code_obj.storage[1] == '0' && area_code_obj.storage[2] == '1')
    {
	copy_reference(&area_code_ref, &area_code_obj.header.common.id);

	nwos_find_or_create_private_class_definition("AREA CODE", &area_code_class_ref);

	snprintf(msg, sizeof(msg), "storing old area_code %02x%02x%02x%02x\n",
		 area_code_ref.id[0], area_code_ref.id[1], area_code_ref.id[2], area_code_ref.id[3]);
	nwos_log(msg);

	copy_reference(&area_code_obj.header.common.class_definition, &area_code_class_ref);

	nwos_create_reference_list(&area_code_ref, &area_code_obj.header.object.references);

	assert(nwos_find_public_area_code(area_code, &public_ref));

	printf("Found public area_code: %02x%02x%02x%02x\n",
	       public_ref.id[0], public_ref.id[1], public_ref.id[2], public_ref.id[3]);

	copy_reference(&area_code_obj.header.object.clone_of, &public_ref);

	assert(nwos_find_state_from_name("Utah", &public_state_ref));  /* replace with public state */

	printf("Found public state: %02x%02x%02x%02x\n",
	       public_state_ref.id[0], public_state_ref.id[1], public_state_ref.id[2], public_state_ref.id[3]);

	copy_reference(&area_code_obj.state, &public_state_ref);

	nwos_crc32_calculate((uint8*) &area_code_obj.header.object,
			     sizeof(ObjectHeader),
			     area_code_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &area_code_obj.state,
			     sizeof(area_code_obj) - sizeof(EveryObject),
			     area_code_obj.header.common.data_chksum);

	nwos_set_bit_in_map(nwos_hash_ref(&area_code_ref));
	nwos_write_object_to_disk(&area_code_ref, &area_code_obj, sizeof(area_code_obj));

	nwos_add_to_references(&area_code_ref, &area_code_class_ref);
    }
#endif
}


void import_phone_number(char line[MAX_LINE_LENGTH])
{
    C_struct_Phone_Number phone_num_obj;
    int i;
    uint8 chksum[4];
    char number[8];
#ifndef TEST_BLOCKS_USED
    ObjRef phone_number_ref;
    ObjRef phone_number_class_ref;
    char msg[128];
#endif

    assert(strlen(line) < MAX_LINE_LENGTH);

    memset(&phone_num_obj, 0, sizeof(phone_num_obj));

    import_every_object(line, &phone_num_obj.header);

    printf("Phone Number %02x%02x%02x%02x: ", 
	   phone_num_obj.header.common.id.id[0],
	   phone_num_obj.header.common.id.id[1],
	   phone_num_obj.header.common.id.id[2],
	   phone_num_obj.header.common.id.id[3]);

    if (nwos_block_used(&phone_num_obj.header.common.id)) printf("  *** block used ***  ");

    get_objref(line, &phone_num_obj.area_code);

    check_comma(line);

    get_string(line, number, sizeof(number));

    if (strlen(number) != sizeof(phone_num_obj.storage))
    {
	printf("phone number too short: %s\n", number);
	exit(0);
    }

    for (i = 0; i < sizeof(phone_num_obj.storage); i++) 
    {
	if (!isdigit(number[i]))
	{
	    printf("invalid character in phone number: %s\n", number);
	    exit(0);
	}
	phone_num_obj.storage[i] = number[i];
    }

    check_newline(line);

    nwos_crc32_calculate((uint8*) &phone_num_obj.country, sizeof(phone_num_obj) - sizeof(EveryObject), chksum);

    assert(memcmp(chksum, phone_num_obj.header.common.data_chksum, sizeof(chksum)) == 0);

#if 0
    if (update_objref(&phone_num_obj.area_code))   /* if phone number reference changed, update data checksum */
    {
	nwos_crc32_calculate((uint8*) &phone_num_obj.country,
			     sizeof(phone_num_obj) - sizeof(EveryObject),
			     phone_num_obj.header.common.data_chksum);
    }
#endif

    printf("area_code: %02x%02x%02x%02x  storage: %c%c%c%c%c%c%c\n",
	   phone_num_obj.area_code.id[0], phone_num_obj.area_code.id[1], 
	   phone_num_obj.area_code.id[2], phone_num_obj.area_code.id[3],
	   phone_num_obj.storage[0], phone_num_obj.storage[1], phone_num_obj.storage[2],
	   phone_num_obj.storage[3], phone_num_obj.storage[4], phone_num_obj.storage[5], phone_num_obj.storage[6]);

#ifndef TEST_BLOCKS_USED
    copy_reference(&phone_number_ref, &phone_num_obj.header.common.id);

    nwos_find_or_create_private_class_definition("PHONE NUMBER", &phone_number_class_ref);

    snprintf(msg, sizeof(msg), "storing old phone_number %02x%02x%02x%02x\n",
	     phone_number_ref.id[0], phone_number_ref.id[1], phone_number_ref.id[2], phone_number_ref.id[3]);
    nwos_log(msg);

    copy_reference(&phone_num_obj.header.common.class_definition, &phone_number_class_ref);

    nwos_create_reference_list(&phone_number_ref, &phone_num_obj.header.object.references);

    nwos_crc32_calculate((uint8*) &phone_num_obj.header.object,
			 sizeof(ObjectHeader),
			 phone_num_obj.header.common.header_chksum);

    nwos_set_bit_in_map(nwos_hash_ref(&phone_number_ref));
    nwos_write_object_to_disk(&phone_number_ref, &phone_num_obj, sizeof(phone_num_obj));

    nwos_add_to_references(&phone_number_ref, &phone_number_class_ref);
    nwos_add_to_references(&phone_number_ref, &phone_num_obj.area_code);
#endif
}


void import_mobile_phone(char line[MAX_LINE_LENGTH])
{
    C_struct_Mobile_Phone mobile_phone_obj;
    uint8 chksum[4];
#ifndef TEST_BLOCKS_USED
    ObjRef mobile_phone_ref;
    ObjRef mobile_phone_class_ref;
    char msg[128];
#endif

    assert(strlen(line) < MAX_LINE_LENGTH);

    memset(&mobile_phone_obj, 0, sizeof(mobile_phone_obj));

    import_every_object(line, &mobile_phone_obj.header);

    printf("Mobile Phone %02x%02x%02x%02x: ", 
	   mobile_phone_obj.header.common.id.id[0],
	   mobile_phone_obj.header.common.id.id[1],
	   mobile_phone_obj.header.common.id.id[2],
	   mobile_phone_obj.header.common.id.id[3]);

    if (nwos_block_used(&mobile_phone_obj.header.common.id)) printf("  *** block used ***  ");

    get_objref(line, &mobile_phone_obj.person);

    check_comma(line);

    get_objref(line, &mobile_phone_obj.number);

    check_newline(line);

    nwos_crc32_calculate((uint8*) &mobile_phone_obj.person, sizeof(mobile_phone_obj) - sizeof(EveryObject), chksum);

    assert(memcmp(chksum, mobile_phone_obj.header.common.data_chksum, sizeof(chksum)) == 0);

#if 0
    if (update_objref(&mobile_phone_obj.person) | update_objref(&mobile_phone_obj.number))
    {
	nwos_crc32_calculate((uint8*) &mobile_phone_obj.person,
			     sizeof(mobile_phone_obj) - sizeof(EveryObject),
			     mobile_phone_obj.header.common.data_chksum);
    }
#endif

    printf("person: %02x%02x%02x%02x  number: %02x%02x%02x%02x\n",
	   mobile_phone_obj.person.id[0], mobile_phone_obj.person.id[1], 
	   mobile_phone_obj.person.id[2], mobile_phone_obj.person.id[3],
	   mobile_phone_obj.number.id[0], mobile_phone_obj.number.id[1], 
	   mobile_phone_obj.number.id[2], mobile_phone_obj.number.id[3]);

#ifndef TEST_BLOCKS_USED
    copy_reference(&mobile_phone_ref, &mobile_phone_obj.header.common.id);

    nwos_find_or_create_private_class_definition("MOBILE PHONE", &mobile_phone_class_ref);

    snprintf(msg, sizeof(msg), "storing old mobile_phone %02x%02x%02x%02x\n",
	     mobile_phone_ref.id[0], mobile_phone_ref.id[1], mobile_phone_ref.id[2], mobile_phone_ref.id[3]);
    nwos_log(msg);

    copy_reference(&mobile_phone_obj.header.common.class_definition, &mobile_phone_class_ref);

    nwos_create_reference_list(&mobile_phone_ref, &mobile_phone_obj.header.object.references);

    nwos_crc32_calculate((uint8*) &mobile_phone_obj.header.object,
			 sizeof(ObjectHeader),
			 mobile_phone_obj.header.common.header_chksum);

    nwos_set_bit_in_map(nwos_hash_ref(&mobile_phone_ref));
    nwos_write_object_to_disk(&mobile_phone_ref, &mobile_phone_obj, sizeof(mobile_phone_obj));

    nwos_add_to_references(&mobile_phone_ref, &mobile_phone_class_ref);
    nwos_add_to_references(&mobile_phone_ref, &mobile_phone_obj.number);
    nwos_add_to_references(&mobile_phone_ref, &mobile_phone_obj.person);
#endif
}


C_struct_Year_0014* years[50];
int num_years;    

void import_year(char line[MAX_LINE_LENGTH])
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Year_0014* year_obj_ptr = (C_struct_Year_0014*) kludge;
    int i;
    uint8 chksum[4];

    assert(strlen(line) < MAX_LINE_LENGTH);

    memset(kludge, 0, sizeof(kludge));

    import_every_object(line, &year_obj_ptr->header);

    printf("Year %02x%02x%02x%02x: ", 
	   year_obj_ptr->header.common.id.id[0],
	   year_obj_ptr->header.common.id.id[1],
	   year_obj_ptr->header.common.id.id[2],
	   year_obj_ptr->header.common.id.id[3]);

    if (nwos_block_used(&year_obj_ptr->header.common.id)) printf("  *** block used ***  ");

    check_double_quote(line);

    for (i = 0; line[i] != '"' && line[i] != '\0'; i++)
    {
	assert(i < 12);                          /* there shouldn't be any ones this long */
	year_obj_ptr->storage[i] = line[i];
	printf("%c", year_obj_ptr->storage[i]);
    }

    shift_line(line, i);

    year_obj_ptr->count = i;

    check_double_quote(line);

    check_newline(line);

    printf("\n");

    nwos_crc32_calculate((uint8*) &year_obj_ptr->count, 
			 get_year_object_size(year_obj_ptr) - sizeof(EveryObject), 
			 chksum);

    assert(memcmp(chksum, year_obj_ptr->header.common.data_chksum, sizeof(chksum)) == 0);

    years[num_years] = malloc(get_year_object_size(year_obj_ptr));
    assert(years[num_years] != NULL);
    memcpy(years[num_years], year_obj_ptr, get_year_object_size(year_obj_ptr));
    num_years++;
}


void process_year(ObjRef* year_ref)
{
    int i;
    int j;
    uint16 year = 0;
    ObjRef test_ref;
    ObjRef public_year_ref;
#ifndef TEST_BLOCKS_USED
    ObjRef private_year_class_ref;
    C_struct_Year year_obj;
    char msg[128];
#endif

    for (i = 0; i < num_years; i++)
    {
	if (is_same_object(year_ref, &years[i]->header.common.id)) break;
    }

    assert(i < num_years);

    for (j = 0; j < years[i]->count; j++) year = year * 10 + years[i]->storage[j] - '0';

    printf("Found year: %d - %u\n", i, year);

    assert(1582 <= year && year <= 2020);   /* for now limit it to 4 digit years */

    assert(!nwos_find_private_year(year, &test_ref));

    assert(nwos_find_public_year(year, &public_year_ref));

#ifndef TEST_BLOCKS_USED
    nwos_find_or_create_private_class_definition("YEAR", &private_year_class_ref);

    nwos_read_object_from_disk(&public_year_ref, &year_obj, sizeof(year_obj));

    snprintf(msg, sizeof(msg), "clone_year(%02x%02x%02x%02x) -> %02x%02x%02x%02x\n",
	     public_year_ref.id[0], public_year_ref.id[1], public_year_ref.id[2], public_year_ref.id[3],
	     year_ref->id[0], year_ref->id[1], year_ref->id[2], year_ref->id[3]);
    nwos_log(msg);

    copy_reference(&year_obj.header.common.id, year_ref);

    memcpy(year_obj.header.common.creation_time, years[i]->header.common.creation_time, sizeof(TimeStamp));

    copy_reference(&year_obj.header.common.class_definition, &private_year_class_ref);

    copy_reference(&year_obj.header.object.clone_of, &public_year_ref);

    nwos_create_reference_list(year_ref, &year_obj.header.object.references);

    nwos_crc32_calculate((uint8*) &year_obj.header.object, sizeof(ObjectHeader), year_obj.header.common.header_chksum);

    nwos_set_bit_in_map(nwos_hash_ref(year_ref));
    nwos_write_object_to_disk(year_ref, &year_obj, sizeof(year_obj));

    nwos_add_to_references(year_ref, &private_year_class_ref);
#endif
}


ObjRef old_month_refs[12];

void import_month(char line[MAX_LINE_LENGTH])
{
    C_struct_Month_0014 old_month_obj;
    uint8 chksum[4];
    char month_str[3];
    char* valid_months[12] = { "01", "02", "03", "04", "05", "06", "07", "08", "09", "10", "11", "12" };
    int i;
    uint8 month;
    ObjRef month_ref;
#ifndef TEST_BLOCKS_USED
    ObjRef public_month_ref;
    ObjRef private_month_class_ref;
    C_struct_Month month_obj;
    char msg[128];
#endif


    assert(strlen(line) < MAX_LINE_LENGTH);

    memset(&old_month_obj, 0, sizeof(old_month_obj));

    import_every_object(line, &old_month_obj.header);

    printf("Month %02x%02x%02x%02x: ", 
	   old_month_obj.header.common.id.id[0],
	   old_month_obj.header.common.id.id[1],
	   old_month_obj.header.common.id.id[2],
	   old_month_obj.header.common.id.id[3]);

    if (nwos_block_used(&old_month_obj.header.common.id)) printf("  *** block used ***  ");

    get_string(line, month_str, sizeof(month_str));

    for (i = 0; i < 12; i++) if (strcmp(month_str, valid_months[i]) == 0) break;

    if (i == 12)
    {
	printf("Invalid month: %s\n", month_str);
	exit(1);
    }

    old_month_obj.storage[0] = month_str[0];
    old_month_obj.storage[1] = month_str[1];

    check_comma(line);

    get_uint8(line, &old_month_obj.minimum_days);

    check_comma(line);

    get_uint8(line, &old_month_obj.maximum_days);

    check_newline(line);

    nwos_crc32_calculate((uint8*) &old_month_obj.definition, sizeof(old_month_obj) - sizeof(EveryObject), chksum);

    assert(memcmp(chksum, old_month_obj.header.common.data_chksum, sizeof(chksum)) == 0);

    printf("month: %c%c  minimum: %u  maximum: %u\n",
	   old_month_obj.storage[0],
	   old_month_obj.storage[1],
	   old_month_obj.minimum_days,
	   old_month_obj.maximum_days);

    month = (old_month_obj.storage[0] - '0') * 10 + old_month_obj.storage[1] - '0';

    printf("Found month: %d - %u\n", i, month);

    assert(1 <= month && month <= 12);

    copy_reference(&month_ref, &old_month_obj.header.common.id);

    assert(is_void_reference(&old_month_refs[month-1]));

    copy_reference(&old_month_refs[month-1], &month_ref);  /* save it for making month_and_day */

    assert(!nwos_find_private_month(month, &month_ref));

#ifndef TEST_BLOCKS_USED
    nwos_find_public_month(month, &public_month_ref);

    nwos_find_or_create_private_class_definition("MONTH", &private_month_class_ref);

    nwos_read_object_from_disk(&public_month_ref, &month_obj, sizeof(month_obj));

    snprintf(msg, sizeof(msg), "clone_month(%02x%02x%02x%02x) -> %02x%02x%02x%02x\n",
	     public_month_ref.id[0], public_month_ref.id[1], public_month_ref.id[2], public_month_ref.id[3],
	     month_ref.id[0], month_ref.id[1], month_ref.id[2], month_ref.id[3]);
    nwos_log(msg);

    copy_reference(&month_obj.header.common.id, &month_ref);

    memcpy(month_obj.header.common.creation_time, month_obj.header.common.creation_time, sizeof(TimeStamp));

    copy_reference(&month_obj.header.common.class_definition, &private_month_class_ref);

    copy_reference(&month_obj.header.object.clone_of, &public_month_ref);

    nwos_create_reference_list(&month_ref, &month_obj.header.object.references);

    nwos_crc32_calculate((uint8*) &month_obj.header.object, sizeof(ObjectHeader), month_obj.header.common.header_chksum);

    nwos_set_bit_in_map(nwos_hash_ref(&month_ref));
    nwos_write_object_to_disk(&month_ref, &month_obj, sizeof(month_obj));

    nwos_add_to_references(&month_ref, &private_month_class_ref);
#endif
}


C_struct_Date_0014* dates[60];
int num_dates;    

void import_date(char line[MAX_LINE_LENGTH])
{
    C_struct_Date_0014 date_obj;
    uint8 chksum[4];

    assert(strlen(line) < MAX_LINE_LENGTH);

    memset(&date_obj, 0, sizeof(date_obj));

    import_every_object(line, &date_obj.header);

    printf("Date %02x%02x%02x%02x: ", 
	   date_obj.header.common.id.id[0],
	   date_obj.header.common.id.id[1],
	   date_obj.header.common.id.id[2],
	   date_obj.header.common.id.id[3]);

    if (nwos_block_used(&date_obj.header.common.id)) printf("  *** block used ***  ");

    get_objref(line, &date_obj.year);

    check_comma(line);

    get_objref(line, &date_obj.month);

    check_comma(line);

    get_uint8(line, &date_obj.day_of_month);

    check_newline(line);

    nwos_crc32_calculate((uint8*) &date_obj.year, sizeof(date_obj) - sizeof(EveryObject), chksum);

    assert(memcmp(chksum, date_obj.header.common.data_chksum, sizeof(chksum)) == 0);

//    if (update_objref(&date_obj.year) | update_objref(&date_obj.month))
//    {
//	nwos_crc32_calculate((uint8*) &date_obj.year, sizeof(date_obj) - sizeof(EveryObject), date_obj.header.common.data_chksum);
//    }

    printf("year: %02x%02x%02x%02x  month: %02x%02x%02x%02x  day: %u\n",
	   date_obj.year.id[0], date_obj.year.id[1], date_obj.year.id[2], date_obj.year.id[3],
	   date_obj.month.id[0], date_obj.month.id[1], date_obj.month.id[2], date_obj.month.id[3],
	   date_obj.day_of_month);

    dates[num_dates] = malloc(sizeof(date_obj));
    assert(dates[num_dates] != NULL);
    memcpy(dates[num_dates], &date_obj, sizeof(date_obj));
    num_dates++;
}


void process_date(ObjRef* date_ref)
{
    int i;
    uint8 month;
    uint8 day;
#ifndef TEST_BLOCKS_USED
    C_struct_Date date_obj;
    ObjRef date_class_ref;
    ObjRef month_and_day_ref;
    char msg[128];
#endif

    for (i = 0; i < num_dates; i++)
    {
	if (is_same_object(date_ref, &dates[i]->header.common.id)) break;
    }

    assert(i < num_dates);

    printf("Found birthdate: %d\n", i);

    process_year(&dates[i]->year);

    /* process_month(&dates[i]->month);  -- don't need to process months, they should all be done */

    /* convert old month reference to a number */

    for (month = 0; month < 12; month++)
    {
	if (is_same_object(&dates[i]->month, &old_month_refs[month])) break;
    }

    month++;   /* make 1-12 instead of 0-11 */

    day = dates[i]->day_of_month;

    assert(1 <= month && month <= 12);
#if 0
    assert(1 <= day && month <= days_per_month[month-1]);
    assert(1582 <= year && year <= 2099);
    if (month == 2 && day == 29)  /* more checking required */
    {
	assert(nwos_is_leap_year(year));
    }
#endif

    /* need to make a routine to convert month and year to a number... */
    /* assert(!nwos_find_private_date(year, month, day, ref)); */

#ifndef TEST_BLOCKS_USED
    nwos_find_or_create_private_class_definition("DATE", &date_class_ref);
    nwos_find_or_create_private_month_and_day(month, day, &month_and_day_ref);

    memset(&date_obj, 0, sizeof(date_obj));  /* zero it out */

    snprintf(msg, sizeof(msg), "copy_date %02x%02x%02x%02x\n",
	     date_ref->id[0], date_ref->id[1], date_ref->id[2], date_ref->id[3]);
    nwos_log(msg);

    nwos_fill_in_common_header(&date_obj.header.common, date_ref, &date_class_ref);

    memcpy(date_obj.header.common.creation_time, dates[i]->header.common.creation_time, sizeof(TimeStamp));

    copy_reference(&date_obj.year, &dates[i]->year);
    copy_reference(&date_obj.month_and_day, &month_and_day_ref);

    nwos_create_reference_list(date_ref, &date_obj.header.object.references);

    nwos_crc32_calculate((uint8*) &date_obj.header.object, sizeof(ObjectHeader), date_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &date_obj.year, sizeof(date_obj) - sizeof(EveryObject), date_obj.header.common.data_chksum);

    nwos_set_bit_in_map(nwos_hash_ref(date_ref));
    nwos_write_object_to_disk(date_ref, &date_obj, sizeof(date_obj));

    nwos_add_to_references(date_ref, &date_class_ref);
    nwos_add_to_references(date_ref, &dates[i]->year);
    nwos_add_to_references(date_ref, &month_and_day_ref);
#endif
}


void import_gender(char line[MAX_LINE_LENGTH])
{
    C_struct_Gender gender_obj;
    uint8 chksum[4];
    char* gender;
    ObjRef gender_ref;
#ifndef TEST_BLOCKS_USED
    ObjRef public_gender_ref;
    ObjRef private_gender_class_ref;
    char msg[128];
#endif

    const uint32 Female_Ref = 0xb0a7a730;
    const uint32 Male_Ref = 0xeceeeb0b;

    assert(strlen(line) < MAX_LINE_LENGTH);

    memset(&gender_obj, 0, sizeof(gender_obj));

    import_every_object(line, &gender_obj.header);

    printf("Gender %02x%02x%02x%02x: ", 
	   gender_obj.header.common.id.id[0],
	   gender_obj.header.common.id.id[1],
	   gender_obj.header.common.id.id[2],
	   gender_obj.header.common.id.id[3]);

    copy_reference(&gender_ref, &gender_obj.header.common.id);

    assert(nwos_ref_to_word(&gender_ref) == Female_Ref || nwos_ref_to_word(&gender_ref) == Male_Ref);

    if (nwos_ref_to_word(&gender_ref) == Female_Ref)
    {
	gender = "F";
    }
    else
    {
	gender = "m";
    }
	
    if (nwos_block_used(&gender_obj.header.common.id)) printf("  *** block used ***  ");

    get_objref(line, &gender_obj.definition);

    check_newline(line);

    nwos_crc32_calculate((uint8*) &gender_obj.definition, sizeof(gender_obj) - sizeof(EveryObject), chksum);

    assert(memcmp(chksum, gender_obj.header.common.data_chksum, sizeof(chksum)) == 0);

//    if (update_objref(&gender_obj.definition))
//    {
//	nwos_crc32_calculate((uint8*) &gender_obj.definition,
//			     sizeof(gender_obj) - sizeof(EveryObject),
//			     gender_obj.header.common.data_chksum);
//    }

    printf("definition: %02x%02x%02x%02x\n",
	   gender_obj.definition.id[0], gender_obj.definition.id[1], gender_obj.definition.id[2], gender_obj.definition.id[3]);

#ifndef TEST_BLOCKS_USED
    nwos_find_public_gender(gender, &public_gender_ref);

    nwos_find_or_create_private_class_definition("GENDER", &private_gender_class_ref);

//    nwos_read_object_from_disk(&public_gender_ref, &public_gender_obj, sizeof(gender_obj));

    snprintf(msg, sizeof(msg), "create_gender(%02x%02x%02x%02x) -> %02x%02x%02x%02x\n",
		 public_gender_ref.id[0], public_gender_ref.id[1], public_gender_ref.id[2], public_gender_ref.id[3],
		 gender_ref.id[0], gender_ref.id[1], gender_ref.id[2], gender_ref.id[3]);
    nwos_log(msg);

    copy_reference(&gender_obj.header.common.class_definition, &private_gender_class_ref);

    copy_reference(&gender_obj.header.object.clone_of, &public_gender_ref);

    nwos_create_reference_list(&gender_ref, &gender_obj.header.object.references);

    nwos_crc32_calculate((uint8*) &gender_obj.header.object, sizeof(ObjectHeader), gender_obj.header.common.header_chksum);

    nwos_set_bit_in_map(nwos_hash_ref(&gender_ref));
    nwos_write_object_to_disk(&gender_ref, &gender_obj, sizeof(gender_obj));

    nwos_add_to_references(&gender_ref, &private_gender_class_ref);
#endif
}

    
void import_person(char line[MAX_LINE_LENGTH])
{
    C_struct_Person_0014 old_person_obj;
    uint8 chksum[4];
//    bool changed;
    ObjRef person_ref;
#ifndef TEST_BLOCKS_USED
    ObjRef person_class_ref;
    C_struct_Person new_person_obj;
    char msg[128];
#endif

    assert(strlen(line) < MAX_LINE_LENGTH);

    memset(&old_person_obj, 0, sizeof(old_person_obj));

    import_every_object(line, &old_person_obj.header);

    printf("Person %02x%02x%02x%02x: ", 
	   old_person_obj.header.common.id.id[0],
	   old_person_obj.header.common.id.id[1],
	   old_person_obj.header.common.id.id[2],
	   old_person_obj.header.common.id.id[3]);

    copy_reference(&person_ref, &old_person_obj.header.common.id);

    if (nwos_block_used(&old_person_obj.header.common.id)) printf("  *** block used ***  ");

    get_objref(line, &old_person_obj.gender);

    check_comma(line);

    get_objref(line, &old_person_obj.first_name);

    check_comma(line);

    get_objref(line, &old_person_obj.middle_name);

    check_comma(line);

    get_objref(line, &old_person_obj.last_name);

    check_comma(line);

    get_objref(line, &old_person_obj.maiden_name);

    check_comma(line);
    
    get_objref(line, &old_person_obj.goes_by);

    check_comma(line);

    get_objref(line, &old_person_obj.nickname);

    check_comma(line);

    get_objref(line, &old_person_obj.birth_date);

    check_newline(line);

    nwos_crc32_calculate((uint8*) &old_person_obj.gender, sizeof(old_person_obj) - sizeof(EveryObject), chksum);

    assert(memcmp(chksum, old_person_obj.header.common.data_chksum, sizeof(chksum)) == 0);
#if 0
    changed = update_objref(&old_person_obj.gender);
    changed |= update_objref(&old_person_obj.first_name);
    changed |= update_objref(&old_person_obj.middle_name);
    changed |= update_objref(&old_person_obj.last_name);
    changed |= update_objref(&old_person_obj.maiden_name);
    changed |= update_objref(&old_person_obj.goes_by);
    changed |= update_objref(&old_person_obj.nickname);
    changed |= update_objref(&old_person_obj.birth_date);

    if (changed)    // if any of the references changed, recompute the data checksum */
    {
	nwos_crc32_calculate((uint8*) &old_person_obj.gender,
			     sizeof(old_person_obj) - sizeof(EveryObject),
			     old_person_obj.header.common.data_chksum);
    }
#endif
    printf("gender: %02x%02x%02x%02x  first:   %02x%02x%02x%02x  middle: %02x%02x%02x%02x  last:  %02x%02x%02x%02x\n",
	   old_person_obj.gender.id[0], old_person_obj.gender.id[1],
	   old_person_obj.gender.id[2], old_person_obj.gender.id[3],
	   old_person_obj.first_name.id[0], old_person_obj.first_name.id[1],
	   old_person_obj.first_name.id[2], old_person_obj.first_name.id[3],
	   old_person_obj.middle_name.id[0], old_person_obj.middle_name.id[1], 
	   old_person_obj.middle_name.id[2], old_person_obj.middle_name.id[3],
	   old_person_obj.last_name.id[0], old_person_obj.last_name.id[1],
	   old_person_obj.last_name.id[2], old_person_obj.last_name.id[3]);

    printf("                 "); 
    printf("maiden: %02x%02x%02x%02x  goes_by: %02x%02x%02x%02x  nick:   %02x%02x%02x%02x  birth: %02x%02x%02x%02x\n",
	   old_person_obj.maiden_name.id[0], old_person_obj.maiden_name.id[1], 
	   old_person_obj.maiden_name.id[2], old_person_obj.maiden_name.id[3],
	   old_person_obj.goes_by.id[0], old_person_obj.goes_by.id[1],
	   old_person_obj.goes_by.id[2], old_person_obj.goes_by.id[3],
	   old_person_obj.nickname.id[0], old_person_obj.nickname.id[1],
	   old_person_obj.nickname.id[2], old_person_obj.nickname.id[3],
	   old_person_obj.birth_date.id[0], old_person_obj.birth_date.id[1],
	   old_person_obj.birth_date.id[2], old_person_obj.birth_date.id[3]);

    if (!is_void_reference(&old_person_obj.birth_date))
    {
	process_date(&old_person_obj.birth_date);
    }

    process_name(&old_person_obj.last_name);
    process_name(&old_person_obj.first_name);
    if (!is_void_reference(&old_person_obj.maiden_name)) process_name(&old_person_obj.maiden_name);
    if (!is_void_reference(&old_person_obj.middle_name)) process_name(&old_person_obj.middle_name);
    if (!is_void_reference(&old_person_obj.goes_by)) process_name(&old_person_obj.goes_by);
    if (!is_void_reference(&old_person_obj.nickname)) process_name(&old_person_obj.nickname);

#ifndef TEST_BLOCKS_USED
    nwos_find_or_create_private_class_definition("PERSON", &person_class_ref);

    snprintf(msg, sizeof(msg), "storing old person %02x%02x%02x%02x\n",
	     person_ref.id[0], person_ref.id[1], person_ref.id[2], person_ref.id[3]);
    nwos_log(msg);

    nwos_fill_in_common_header(&new_person_obj.header.common, &person_ref, &person_class_ref);

    nwos_create_reference_list(&person_ref, &new_person_obj.header.object.references);

    copy_reference(&new_person_obj.gender,      &old_person_obj.gender);
    copy_reference(&new_person_obj.first_name,  &old_person_obj.first_name);
    copy_reference(&new_person_obj.middle_name, &old_person_obj.middle_name);
    copy_reference(&new_person_obj.last_name,   &old_person_obj.last_name);
    copy_reference(&new_person_obj.maiden_name, &old_person_obj.maiden_name);
    copy_reference(&new_person_obj.goes_by,     &old_person_obj.goes_by);
    copy_reference(&new_person_obj.nickname,    &old_person_obj.nickname);
    copy_reference(&new_person_obj.birth_date,  &old_person_obj.birth_date);

    nwos_crc32_calculate((uint8*) &new_person_obj.header.object, sizeof(ObjectHeader), new_person_obj.header.common.header_chksum);
    nwos_crc32_calculate((uint8*) &new_person_obj.gender, sizeof(new_person_obj) - sizeof(EveryObject), new_person_obj.header.common.data_chksum);

    nwos_set_bit_in_map(nwos_hash_ref(&person_ref));
    nwos_write_object_to_disk(&person_ref, &new_person_obj, sizeof(new_person_obj));

    nwos_add_to_references(&person_ref, &person_class_ref);
    nwos_add_to_references(&person_ref, &new_person_obj.gender);
    nwos_add_to_references(&person_ref, &new_person_obj.last_name);
    nwos_add_to_references(&person_ref, &new_person_obj.first_name);

    if (!is_void_reference(&new_person_obj.birth_date))
    {
	nwos_add_to_references(&person_ref, &new_person_obj.birth_date);
    }

    if (!is_void_reference(&new_person_obj.maiden_name))
    {
	nwos_add_to_references(&person_ref, &new_person_obj.maiden_name);
    }

    if (!is_void_reference(&new_person_obj.middle_name))
    {
	nwos_add_to_references(&person_ref, &new_person_obj.middle_name);
    }

    if (!is_void_reference(&new_person_obj.goes_by))
    {
	nwos_add_to_references(&person_ref, &new_person_obj.goes_by);
    }

    if (!is_void_reference(&new_person_obj.nickname))
    {
	nwos_add_to_references(&person_ref, &new_person_obj.nickname);
    }
#endif
}


#if 0
void import_reference_list(char *line)
{
    ObjRef ref;
    CommonHeader common_header;
    int num_on_this_line;

    get_objref(line, &common_header.id);

    check_comma(line);

    import_common_header(line, &common_header);

    /* note there may not be a comma after the class definition because the reference list could be empty */

    printf("Reference List %02x%02x%02x%02x:", 
	   common_header.id.id[0],
	   common_header.id.id[1],
	   common_header.id.id[2],
	   common_header.id.id[3]);

    update_objref(&common_header.id);

    nwos_create_reference_list_with_existing_id(NULL, &common_header.id);

    num_on_this_line = 1;

    while (line[0] == ',')
    {
	check_comma(line);

	get_objref(line, &ref);

	update_objref(&ref);

	printf(" %02x%02x%02x%02x", ref.id[0], ref.id[1], ref.id[2], ref.id[3]);

	if (num_on_this_line > 13)
	{
	    printf("\n");
	    num_on_this_line = 0;
	}

	nwos_add_to_reference_list(&ref, &common_header.id);
    }

    if (num_on_this_line > 0) printf("\n");

    check_newline(line);
}
#endif    

void import_class(char* name)
{
    char filename[64];
    char line[MAX_LINE_LENGTH];      /* this is large enough for all files except reference_lists */
    int i;
    FILE* fp = NULL;
//    char* ref_list_line;

    filename[0] = 'c';
    filename[1] = 's';
    filename[2] = 'v';
    filename[3] = '/';

    for (i = 4; name[i-4] != '\0'; i++)
    {
	if (name[i-4] == ' ')
	{
	    filename[i] = '_';
	}
	else
	{
	    filename[i] = tolower(name[i-4]);
	}
    }

    filename[i++] = '.';
    filename[i++] = 'c';
    filename[i++] = 's';
    filename[i++] = 'v';
    filename[i] = '\0';

    fp = fopen(filename, "r");

    if (fp == NULL)
    {
	perror(filename);
	exit(1);
    }

    current_filename = filename;

#if 0
    if (strcmp(name, "Class Definition") == 0)
    {
	while (fgets(line, sizeof(line), fp) != NULL) import_class_definition(line);
    }
    else if (strcmp(name, "Root") == 0)
    {
	while (fgets(line, sizeof(line), fp) != NULL) import_root(line);
    }
#endif
    /* else */ if (strcmp(name, "Spelling") == 0)
    {
	while (fgets(line, sizeof(line), fp) != NULL) import_spelling(line);
    }
    else if (strcmp(name, "Name") == 0)
    {
	while (fgets(line, sizeof(line), fp) != NULL) import_name(line);
    }
#if 0
    else if (strcmp(name, "Word") == 0)
    {
	while (fgets(line, sizeof(line), fp) != NULL) import_word(line);
    }
    else if (strcmp(name, "Language") == 0)
    {
	while (fgets(line, sizeof(line), fp) != NULL) import_language(line);
    }
    else if (strcmp(name, "Us State") == 0)
    {
	while (fgets(line, sizeof(line), fp) != NULL) import_us_state(line);
    }
    else if (strcmp(name, "Us City") == 0)
    {
	while (fgets(line, sizeof(line), fp) != NULL) import_us_city(line);
    }
#endif
    else if (strcmp(name, "Area Code") == 0)
    {
	while (fgets(line, sizeof(line), fp) != NULL) import_area_code(line);
    }
    else if (strcmp(name, "Phone Number") == 0)
    {
	while (fgets(line, sizeof(line), fp) != NULL) import_phone_number(line);
    }
    else if (strcmp(name, "Mobile Phone") == 0)
    {
	while (fgets(line, sizeof(line), fp) != NULL) import_mobile_phone(line);
    }
    else if (strcmp(name, "Year") == 0)
    {
	while (fgets(line, sizeof(line), fp) != NULL) import_year(line);
    }
    else if (strcmp(name, "Month") == 0)
    {
	while (fgets(line, sizeof(line), fp) != NULL) import_month(line);
    }
    else if (strcmp(name, "Date") == 0)
    {
	while (fgets(line, sizeof(line), fp) != NULL) import_date(line);
    }
    else if (strcmp(name, "Gender") == 0)
    {
	while (fgets(line, sizeof(line), fp) != NULL) import_gender(line);
    }
    else if (strcmp(name, "Person") == 0)
    {
	while (fgets(line, sizeof(line), fp) != NULL) import_person(line);
    }
#if 0
    else if (strcmp(name, "Reference List") == 0)
    {
	ref_list_line = malloc(MAX_REF_LIST_LINE_LENGTH);
	assert(ref_list_line != NULL);
	while (fgets(ref_list_line, MAX_REF_LIST_LINE_LENGTH, fp) != NULL) import_reference_list(ref_list_line);
	free(ref_list_line);
    }
#endif
    else
    {
	printf("Class not handled: %s\n", name);
    }
}


int main(int argc, char* argv[])
{
    uint8 big_key[16 + 8 + 4];
    uint8 bf_key[16];
    uint32 linear;
    uint32 serial;
    ObjRef root_object_ref;

    printf("\n");

    nwos_get_key_from_password(big_key, sizeof(big_key));

    printf("\n");

    memcpy(bf_key, big_key, 16);
    linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
    memcpy(root_object_ref.id, big_key+20, 4);
    serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

    nwos_log_arguments(argc, argv);

#ifdef TEST_BLOCKS_USED   // open read only if we are testing for block overlap
    nwos_initialize_objectify(bf_key, linear, serial, DEFAULT_TYPE_RO, DEFAULT_FILE);
#else
    nwos_initialize_objectify(bf_key, linear, serial, DEFAULT_TYPE_RW, DEFAULT_FILE);
#endif

    nwos_set_root_object(&root_object_ref);

    nwos_set_encryption_level(Encryption_Extreme);

#if 0
    import_class("Class Definition");

    import_class("Word");
    import_class("Language");

    import_class("Us State");
    import_class("Us City");
#endif

    import_class("Year");
    import_class("Month");
    import_class("Date");

    import_class("Spelling");
    import_class("Name");

    import_class("Gender");
    import_class("Person");

    /* do the mobile phone thing after persons so they can be linked */
    /* these have to be done in this specific order */
    import_class("Area Code");
    import_class("Phone Number");
    import_class("Mobile Phone");

#if 0
    import_class("Reference List");

    import_class("Root");   /* this has to be last because it depends upon everything else existing */
#endif

    nwos_terminate_objectify();

    return 0;
}

