/*
--          This file is part of the New World OS and Objectify projects
--                  Copyright (C) 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-07-25 17:25:15 -0600 (Sat, 25 Jul 2009) $
--   $Revision: 4184 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.  I have retained the CVS log from
--   this file before it was renamed.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
--
-- Revision 1.2  2007/06/09 17:16:18  jsedwards
-- Fix to correctly calculate disk factor for determining wrap.
--
-- Revision 1.1  2007/06/09 15:48:11  jsedwards
-- Initial version.
--
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>    /* define sleep() */

#include "../objectify_private.h"

#define USABLE_BLOCKS_PER_CHUNK (BLOCKS_IN_CHUNK - BIT_MAP_BLOCKS)

uint32 convert_4_uint8_to_uint32(uint8 byte[4])
{
  return ((uint32)byte[0] << 24) | ((uint32)byte[1] << 16) | ((uint32)byte[2] << 8) | (uint32)byte[3];
}


int main(int argc, char* argv[])
{
    FILE* fp1;
    FILE* fp2;
    uint8 buf1[FILE_BLOCK_SIZE];
    uint32 lower;
    uint32 upper;
    uint32 chunks;
    uint32 blocks;
    uint32 ref;
    size_t read1;
    size_t write2;
    Disk_Header header;


    if (argc != 3)
    {
	fprintf(stderr, "usage: %s 0022_file 0023_file\n", argv[0]);
	exit(1);
    }


    /* Open the old file and check it */

    fp1 = fopen(argv[1], "r");
    if (fp1 == NULL)
    {
	perror(argv[1]);
	exit(1);
    }

    read1 = fread(buf1, 1, sizeof(buf1), fp1);

    if (read1 != FILE_BLOCK_SIZE)
    {
	if (ferror(fp1))
	{
	    perror(argv[1]);
	}
	else
	{
	    fprintf(stderr, "Unexpected end of file: %s\n", argv[1]);
	}
	fclose(fp1);
	exit(1);
    }

    memcpy(&header, buf1, sizeof(header));

    if (memcmp(header.magic_number, "NWOS", 4) != 0)
    {
	fprintf(stderr, "Not an Objectify file: %s\n", argv[1]);
	fclose(fp1);
	exit(1);
    }

    if (memcmp(header.version_string, "0022", 4) != 0)
    {
	fprintf(stderr, "Not an version 0022 file: %s\n", argv[1]);
	fclose(fp1);
	exit(1);
    }


    /* Open the output file */

    fp2 = fopen(argv[2], "w");

    if (fp2 == NULL)
    {
	perror(argv[3]);
	fclose(fp1);
	fclose(fp2);
	exit(1);
    }

    header.version_string[3] = '3';  /* change the version to 0022 */

    /* Adjust used_blocks to 0023 representation, in 0023 blocks used */
    /* for the bit map are not included in the used_blocks count.     */

    chunks = (convert_4_uint8_to_uint32(header.total_blocks) + BLOCKS_IN_CHUNK - 1) / BLOCKS_IN_CHUNK;
    blocks = convert_4_uint8_to_uint32(header.used_blocks) - chunks * BIT_MAP_BLOCKS;
    nwos_uint32_to_4_uint8(&blocks, header.used_blocks);

    memcpy(buf1, &header, sizeof(header));

    write2 = fwrite(buf1, 1, sizeof(buf1), fp2);

    if (write2 != FILE_BLOCK_SIZE)
    {
	perror(argv[3]);
	fclose(fp1);
	fclose(fp2);
	exit(1);
    }

    chunks = convert_4_uint8_to_uint32(header.total_blocks) / BLOCKS_IN_CHUNK;

    blocks = chunks * USABLE_BLOCKS_PER_CHUNK;

    lower = RESERVED_PUBLIC_BLOCKS;
    upper = lower + blocks;

    while (lower != 0xffffffff)
    {
	printf("Doing range %08x to %08x\n", lower, upper);

	read1 = fread(buf1, 1, sizeof(buf1), fp1);

	while (!feof(fp1) && read1 == FILE_BLOCK_SIZE)
	{
	    if (buf1[0] != 0 || buf1[1] != 0 || buf1[2] != 0 || buf1[3] != 0)
	    {
		fprintf(stderr, "\n%s - first four bytes not zero: %02x%02x%02x%02x\n",
		       argv[1], buf1[0], buf1[1], buf1[2], buf1[3]);
		exit(1);
	    }

	    ref = convert_4_uint8_to_uint32(&buf1[4]);

	    if (ref < RESERVED_PUBLIC_BLOCKS)
	    {
		fprintf(stderr, "\n%s - public block found: %02x%02x%02x%02x\n",
		       argv[1], buf1[0], buf1[1], buf1[2], buf1[3]);
		exit(1);
	    }

	    if (lower <= ref && ref < upper)
	    {
		write2 = fwrite(buf1, 1, sizeof(buf1), fp2);
	    }

	    if (write2 != FILE_BLOCK_SIZE)
	    {
		perror(argv[2]);
		fclose(fp1);
		fclose(fp2);
		exit(1);
	    }

	    read1 = fread(buf1, 1, sizeof(buf1), fp1);
	}

	if (ferror(fp1))
	{
	    perror(argv[1]);
	    exit(1);
	}

	if (!feof(fp1) && read1 != FILE_BLOCK_SIZE)
	{
	    fprintf(stderr, "Short read: %zd bytes\n", read1);
	    exit(1);
	}

	lower = upper;

	if ((0xffffffff - upper) < blocks)   /* this is the last pass */
	{
	    upper = 0xffffffff;
	}
	else
	{
	    upper += blocks;
	}

	if (lower != 0xffffffff)
	{
	    fseek(fp1, FILE_BLOCK_SIZE, SEEK_SET);
	}
    }

    fclose(fp1);

    if (fclose(fp2) != 0)
    {
	perror(argv[2]);
    }

    return 0;
}
