/*
--          This file is part of the New World OS and Objectify projects
--                  Copyright (C) 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-07-25 17:25:15 -0600 (Sat, 25 Jul 2009) $
--   $Revision: 4184 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "../objectify_private.h"
#include "../crc32.h"

#define MAX_NAME_SIZE 96
#define MAX_URL_SIZE 64
#define MAX_ACRONYM_SIZE 8


static void get_input(char* descr, char* buffer, size_t size)
{
    char *ptr;

    while (1)
    {
	printf("%s: ", descr);
	fflush(stdout);
	fgets(buffer, size, stdin);
	ptr = strchr(buffer, '\n');
	if (ptr != NULL)
	{
	    *ptr = '\0';
	    break;
	}
	do { fgets(buffer, size, stdin); } while (strchr(buffer, '\n') == NULL);
	printf("input too long - try again!\n");
    }
}

extern ObjCreateResult nwos_create_url(char* url, ObjRef* ref);

int main(int argc, char* argv[])
{
    ObjRef root_ref;
    ObjRef reference;
    char full_name[MAX_NAME_SIZE];
    char short_name[MAX_NAME_SIZE];
    char acronym[MAX_ACRONYM_SIZE];
    char url[MAX_URL_SIZE];
    ObjRef cu_class_ref;
    ObjRef url_ref;
    C_struct_Credit_Union cu_obj;
    C_struct_Credit_Union test_cu_obj;
    char test_string[128];
    ReferenceList* ref_list;
    int num_refs;


    nwos_log_arguments(argc, argv);

    nwos_initialize_objectify(NULL, 0, 0, DEFAULT_TYPE_RW, DEFAULT_FILE);

    nwos_set_encryption_level(Encryption_None);

    printf("Adding credit union on: %s\n", DEFAULT_FILE);

    printf("next_reference: %02x%02x%02x%02x\n", 
	   nwos_next_public_ref.id[0],
	   nwos_next_public_ref.id[1],
	   nwos_next_public_ref.id[2],
	   nwos_next_public_ref.id[3]);

    root_ref.id[0] = 0;
    root_ref.id[1] = 0;
    root_ref.id[2] = 0;
    root_ref.id[3] = 1;

    nwos_set_root_object(&root_ref);


    assert(nwos_find_public_class_definition("CREDIT UNION", &cu_class_ref));


    get_input("full name", full_name, sizeof(full_name));
    get_input("short name", short_name, sizeof(short_name));
    get_input("acronym", acronym, sizeof(acronym));
    get_input("url", url, sizeof(url));


    if (nwos_ask_yes_or_no(NULL, "Create?"))
    {
	memset(&cu_obj, 0, sizeof(cu_obj));

	nwos_generate_new_id(&reference);

	nwos_fill_in_common_header(&cu_obj.header.common, &reference, &cu_class_ref);

	nwos_create_name(full_name, &cu_obj.full_name);

	if (*short_name != '\0')
	{
	    nwos_create_name(short_name, &cu_obj.short_name);
	}

	if (*url != '\0')
	{
	    nwos_create_reference_list(&reference, &cu_obj.url_list);
	    nwos_create_url(url, &url_ref);
	    nwos_add_to_reference_list(&url_ref, &cu_obj.url_list);
	}

	if (*acronym != '\0')
	{
	    nwos_create_public_acronym(acronym, &cu_obj.acronym);
	}

	nwos_create_reference_list(&reference, &cu_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &cu_obj.header.object, sizeof(ObjectHeader), cu_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &cu_obj.full_name, sizeof(cu_obj) - sizeof(EveryObject), cu_obj.header.common.data_chksum);

	nwos_write_object_to_disk(&reference, &cu_obj, sizeof(cu_obj));

        nwos_add_to_references(&reference, &cu_class_ref);
	nwos_add_to_references(&reference, &cu_obj.full_name);

	if (!is_void_reference(&cu_obj.short_name))
	{
	    nwos_add_to_references(&reference, &cu_obj.short_name);
	}

	if (!is_void_reference(&cu_obj.url_list))
	{
	    nwos_add_to_references(&reference, &url_ref);
	}

	if (!is_void_reference(&cu_obj.acronym))
	{
	    nwos_add_to_references(&reference, &cu_obj.acronym);
	}

	printf("Object created: %02x%02x%02x%02x\n", reference.id[0], reference.id[1], reference.id[2], reference.id[3]);
    }

    assert(nwos_read_object_from_disk(&reference, &test_cu_obj, sizeof(test_cu_obj)));

    if (memcmp(&test_cu_obj, &cu_obj, sizeof(cu_obj)) != 0)
    {
	printf("Error: object read back doesn't match original object\n");
    }
    else
    {
	assert(nwos_name_to_string(&test_cu_obj.full_name, test_string, sizeof(test_string)));
	printf("full name: %s\n", test_string);

	if (!is_void_reference(&test_cu_obj.short_name))
	{
	    assert(nwos_name_to_string(&test_cu_obj.short_name, test_string, sizeof(test_string)));
	    printf("short name: %s\n", test_string);
	}

	if (!is_void_reference(&test_cu_obj.url_list))
	{
	    ref_list = nwos_malloc_reference_list(&test_cu_obj.url_list);

	    num_refs = ref_list->common_header.num_refs;

	    if (num_refs != 1)
	    {
		printf("Error: number of URLs is not 1: %d\n", num_refs);
	    }
	    else
	    {
		assert(nwos_url_to_string(&ref_list->references[0], test_string, sizeof(test_string)));
		printf("url: %s\n", test_string);
	    }

	    nwos_free_reference_list(ref_list);
	    ref_list = NULL;
	}
    }

    nwos_terminate_objectify();

    return 0;
}


