/*
--             This file is part of the New World OS project
--                 Copyright (C) 2007-2009  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: diff_compressed.c,v $
-- Revision 1.25  2009/07/18 15:40:43  jsedwards
-- Added code to set the upper 4 bytes of the reference in blocks from a 0029
-- or older file to 0xfffffffe and back to zero if the block has to go around
-- again.
--
-- Revision 1.24  2009/07/17 14:01:51  jsedwards
-- Fixed to deal with having 0xfffffffe in the upper word of the reference in
-- each block if the version > 29.
--
-- Revision 1.23  2009/07/17 13:23:19  jsedwards
-- Fixed added blocks error upper bytes print statement to print to stderr too.
--
-- Revision 1.22  2009/07/17 13:10:01  jsedwards
-- Fixed to report error and exit if upper four bytes of reference are not zero.
--
-- Revision 1.21  2009/07/11 13:30:30  jsedwards
-- Changed to pass TYPE_CODE_COMPRESSED in the calls to check headers.
--
-- Revision 1.20  2009/07/10 15:05:59  jsedwards
-- Changed to write "diff" into the headers of the output file if they are
-- version 0030 or above.
--
-- Revision 1.19  2009/07/02 13:47:01  jsedwards
-- Changed to call the new nwos_check_disk_header function instead of testing
-- the header inline.
--
-- Revision 1.18  2009/04/06 14:21:14  jsedwards
-- Merged in changes from CVS branch_0030_new_chunk_info branch, see revisions
-- 1.15.2.1, 1.15.2.2, and 1.15.2.3 below.
--
-- Revision 1.17  2009/03/08 00:06:18  jsedwards
-- Changed include objectify_private.h to disk_io.h.
--
-- Revision 1.16  2008/09/01 00:10:50  jsedwards
-- Fix copyright year.  NO code changes.
--
-- Revision 1.15.2.3  2008/08/16 02:26:14  jsedwards
-- Changed to work with new Disk_Header with total_chunks instead of
-- total_blocks and still allow diff'ing with 0022 to 0029 headers.
--
-- Revision 1.15.2.2  2008/08/15 13:48:54  jsedwards
-- Changed for new Disk_Header with total_chunks instead of total_blocks.
--
-- Revision 1.15.2.1  2008/08/15 13:39:16  jsedwards
-- Removed old code to upgrade from version 0021 to 0022.
--
-- Revision 1.15  2008/01/04 04:25:08  jsedwards
-- Added call to close the output file (fp3).
--
-- Revision 1.14  2007/11/17 21:26:57  jsedwards
-- Fix increment in version_string_to_int function so it isn't an infinite loop.
--
-- Revision 1.13  2007/11/17 20:00:09  jsedwards
-- Changed to detect if the files are older than version 0023 (where the
-- reference id's wrapped around the storage) and insert the next block in
-- bytes 0 to 3 of the block.  If they are not old leave zeros in bytes 0 to 3.
--
-- Revision 1.12  2007/07/01 19:44:11  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.11  2007/06/22 22:07:17  jsedwards
-- Add code to correctly deal with 0021 to 0022 conversion.
--
-- Revision 1.10  2007/06/19 19:06:30  jsedwards
-- Changed feature names in Disk Header because it was change (0022).
--
-- Revision 1.9  2007/03/24 14:08:17  jsedwards
-- Change to not write the next block info into bytes 0-3 when file 1 has
-- reached the end of file, because it tricks the patch program.
--
-- Revision 1.8  2007/03/24 12:32:39  jsedwards
-- Moved the increment of blocks added count outside the if writing output
-- file, so it gets incremented even when not writing an output file.
--
-- Revision 1.7  2007/03/24 12:29:13  jsedwards
-- Fixed to finish adding blocks from file 2 after file file 1 has reached
-- the end of the file.
--
-- Revision 1.6  2007/03/15 13:44:04  jsedwards
-- Changed to zero out the bytes before writing and added counts.
--
-- Revision 1.5  2007/03/15 12:50:25  jsedwards
-- Changed to write both the old header and the new header into the patch file.
--
-- Revision 1.4  2007/03/14 23:53:38  jsedwards
-- Added code to write the differences to a file.
--
-- Revision 1.3  2007/03/14 23:42:01  jsedwards
-- Changed to just look for changed blocks and blocks added in file 2 (assume
-- that no blocks were deleted).
--
-- Revision 1.2  2007/03/14 14:38:12  jsedwards
-- Changed to verify the headers are the same.
--
-- Revision 1.1  2007/02/25 16:00:53  jsedwards
-- This is a quick hack to compare the compressed archives of version 0017
-- and the updated (moved root objects) 0018 archive.
--
*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>    /* define sleep() */

#include "header.h"


uint32 convert_4_uint8_to_uint32(uint8 byte[4])
{
  return ((uint32)byte[0] << 24) | ((uint32)byte[1] << 16) | ((uint32)byte[2] << 8) | (uint32)byte[3];
}


static int version_string_to_int(char string[4])
{
    int i;
    int result = 0;

    for (i = 0; i < 4; i++) if (string[i] != '0') break;

    while (i < 4)
    {
	if (!isdigit(string[i])) return -1;

	result = result * 10 + string[i] - '0';

	i++;
    }

    return result;
}


int main(int argc, char* argv[])
{
    FILE* fp1;
    FILE* fp2;
    FILE* fp3 = NULL;
    unsigned char buf1a[FILE_BLOCK_SIZE];
    unsigned char buf2a[FILE_BLOCK_SIZE];
    int block1;
    int block2;
    int changed = 0;
    int added = 0;
    size_t read1;
    size_t read2;
    size_t write3;
    Disk_Header header1;
    Disk_Header header2;
    Disk_Header_0022_to_0029 old_header1;
    Disk_Header_0022_to_0029 old_header2;
    bool wrapped_file = false;     /* true if older file that wasn't sequencial */
    bool bad_upper_ref;
    int header1_version;
    int header2_version;
    uint32 header1_chunks;
    uint32 header2_chunks;
    const char* error_msg;

    if (argc != 3 && argc != 4)
    {
	fprintf(stderr, "usage: %s old_file new_file [outfile]\n", argv[0]);
	exit(1);
    }


    /* Open the first file and check it */

    fp1 = fopen(argv[1], "r");
    if (fp1 == NULL)
    {
	perror(argv[1]);
	exit(1);
    }

    read1 = fread(buf1a, 1, sizeof(buf1a), fp1);

    if (read1 != FILE_BLOCK_SIZE)
    {
	if (ferror(fp1))
	{
	    perror(argv[1]);
	}
	else
	{
	    fprintf(stderr, "Unexpected end of file: %s\n", argv[1]);
	}
	fclose(fp1);
	exit(1);
    }

    memcpy(&header1, buf1a, sizeof(header1));

    error_msg = nwos_check_disk_header(&header1, true, TYPE_CODE_COMPRESSED);  /* allow older versions */

    if (error_msg != NULL)
    {
	fprintf(stderr, "%s: %s\n", error_msg, argv[1]);
	fclose(fp1);
	exit(1);
    }


    /* Open the second file and check it */

    fp2 = fopen(argv[2], "r");
    if (fp2 == NULL)
    {
	perror(argv[2]);
	exit(1);
    }

    read2 = fread(buf2a, 1, sizeof(buf2a), fp2);

    if (read2 != FILE_BLOCK_SIZE)
    {
	if (ferror(fp2))
	{
	    perror(argv[2]);
	}
	else
	{
	    fprintf(stderr, "Unexpected end of file: %s\n", argv[2]);
	}
	fclose(fp1);
	fclose(fp2);
	exit(1);
    }

    memcpy(&header2, buf2a, sizeof(header2));

    error_msg = nwos_check_disk_header(&header2, true, TYPE_CODE_COMPRESSED);  /* allow older versions */

    if (error_msg != NULL)
    {
	fprintf(stderr, "%s: %s\n", error_msg, argv[1]);
	fclose(fp1);
	exit(1);
    }


    /* Verify they are the same version and disk size */

    if (memcmp(header1.version_string, header2.version_string, 4) != 0)
    {
	fprintf(stderr, "WARNING, versions don't match - %s: %c%c%c%c  %s: %c%c%c%c\n",
		argv[1], 
		header1.version_string[0], header1.version_string[1],
		header1.version_string[2], header1.version_string[3],
		argv[2], 
		header2.version_string[0], header2.version_string[1],
		header2.version_string[2], header2.version_string[3]);
	sleep(5);
    }

    header1_version = version_string_to_int(header1.version_string);

    if (header1_version < 1)
    {
	fprintf(stderr, "Invalid version string in file: %s\n", argv[1]);
	exit(1);
    }

    header2_version = version_string_to_int(header2.version_string);

    if (header2_version < 1)
    {
	fprintf(stderr, "Invalid version string in file: %s\n", argv[2]);
	exit(1);
    }

    if (header1_version > header2_version)
    {
	fprintf(stderr, "ERROR: '%s' is a newer version than '%s', cannot diff!\n", argv[1], argv[2]);
	exit(1);
    }

    if (header1_version < 23 && header2_version < 23)
    {
	wrapped_file = true;   /* it's an old file */
    }
    else if (header1_version < 23)
    {
	fprintf(stderr, "ERROR: '%s' is an old wrapped file, but file '%s' is not, cannot diff!\n", argv[1], argv[2]);
	exit(1);
    }

    if (header1_version < 30)
    {
	memcpy(&old_header1, buf1a, sizeof(old_header1));
	header1_chunks = convert_4_uint8_to_uint32(old_header1.total_blocks) / 65536; /* old chunks were 65536 blocks including bit map */
    }
    else
    {
	header1_chunks = convert_4_uint8_to_uint32(header1.total_chunks);
    }

    if (header2_version < 30)
    {
	memcpy(&old_header2, buf2a, sizeof(old_header2));
	header2_chunks = convert_4_uint8_to_uint32(old_header2.total_blocks) / 65536; /* old chunks were 65536 blocks including bit map */
    }
    else
    {
	header2_chunks = convert_4_uint8_to_uint32(header2.total_chunks);
    }

    if (header1_chunks != header2_chunks)
    {
	fprintf(stderr, "Warning: disk total chunks size different - %s: %u  %s: %u\n", argv[1], header1_chunks, argv[2], header2_chunks);
    }

    if (argc == 4)
    {
	fp3 = fopen(argv[3], "w");

	if (fp3 == NULL)
	{
	    perror(argv[3]);
	    fclose(fp1);
	    fclose(fp2);
	    exit(1);
	}
    }

    if (fp3 != NULL)   /* write both headers into the patch file */
    {
	if (header1_version >= 30)
	{
	    memcpy(&(((Disk_Header*)buf1a)->type_code), TYPE_CODE_DIFF, sizeof((((Disk_Header*)buf1a)->type_code)));
	}
	    
	write3 = fwrite(buf1a, 1, sizeof(buf1a), fp3);

	if (write3 != FILE_BLOCK_SIZE)
	{
	    perror(argv[3]);
	    fclose(fp1);
	    fclose(fp2);
	    fclose(fp3);
	    exit(1);
	}

	if (header2_version >= 30)
	{
	    memcpy(&(((Disk_Header*)buf2a)->type_code), TYPE_CODE_DIFF, sizeof((((Disk_Header*)buf2a)->type_code)));
	}
	    
	write3 = fwrite(buf2a, 1, sizeof(buf2a), fp3);

	if (write3 != FILE_BLOCK_SIZE)
	{
	    perror(argv[3]);
	    fclose(fp1);
	    fclose(fp2);
	    fclose(fp3);
	    exit(1);
	}
    }

    block1 = 1;
    block2 = 1;

    read1 = fread(buf1a, 1, sizeof(buf1a), fp1);
    read2 = fread(buf2a, 1, sizeof(buf2a), fp2);

    while (!feof(fp1) && !feof(fp2) && read1 == FILE_BLOCK_SIZE && read2 == FILE_BLOCK_SIZE)
    {
	bad_upper_ref = false;

	if (header1_version < 30)
	{
	    if (buf1a[0] != 0 || buf1a[1] != 0 || buf1a[2] != 0 || buf1a[3] != 0)
	    {
		fprintf(stderr, "\n%s block %d - first four bytes not zero: %02x%02x%02x%02x\n",
			argv[1], block1, buf1a[0], buf1a[1], buf1a[2], buf1a[3]);
		bad_upper_ref = true;
	    }

	    if (header2_version >= 30)
	    {
		buf1a[0] = 0xff;
		buf1a[1] = 0xff;
		buf1a[2] = 0xff;
		buf1a[3] = 0xfe;
	    }
	}
	else 
	{
	    if (buf1a[0] != 0xff || buf1a[1] != 0xff || buf1a[2] != 0xff || buf1a[3] != 0xfe)
	    {
		fprintf(stderr, "\n%s block %d - first four bytes not fffffffe: %02x%02x%02x%02x\n",
			argv[1], block1, buf1a[0], buf1a[1], buf1a[2], buf1a[3]);
		bad_upper_ref = true;
	    }
	}

	if (header2_version < 30)
	{
	    if (buf2a[0] != 0 || buf2a[1] != 0 || buf2a[2] != 0 || buf2a[3] != 0)
	    {
		fprintf(stderr, "\n%s block %d - first four bytes not zero: %02x%02x%02x%02x\n",
			argv[2], block2, buf2a[0], buf2a[1], buf2a[2], buf2a[3]);
		bad_upper_ref = true;
	    }
	}
	else
	{
	    if (buf2a[0] != 0xff || buf2a[1] != 0xff || buf2a[2] != 0xff || buf2a[3] != 0xfe)
	    {
		fprintf(stderr, "\n%s block %d - first four bytes not fffffffe: %02x%02x%02x%02x\n",
			argv[2], block2, buf2a[0], buf2a[1], buf2a[2], buf2a[3]);
		bad_upper_ref = true;
	    }
	}

	if (bad_upper_ref)
	{
	    fclose(fp1);
	    fclose(fp2);
	    if (fp3 != NULL) fclose(fp3);

	    exit(1);
	}

	if (buf1a[4] == buf2a[4] && buf1a[5] == buf2a[5] && buf1a[6] == buf2a[6] && buf1a[7] == buf2a[7])
	{
	    /* blocks have the same id number, just compare them */

	    if (memcmp(buf1a, buf2a, FILE_BLOCK_SIZE) != 0)
	    {
		printf("%02x%02x%02x%02x: changed\n",
		       buf1a[4], buf1a[5], buf1a[6], buf1a[7]);

		if (fp3 != NULL)   /* write the changed block */
		{
		    write3 = fwrite(buf2a, 1, sizeof(buf2a), fp3);

		    if (write3 != FILE_BLOCK_SIZE)
		    {
			perror(argv[2]);
			fclose(fp1);
			fclose(fp2);
			fclose(fp3);
			exit(1);
		    }
		}

		changed++;
	    }

	    read1 = fread(buf1a, 1, sizeof(buf1a), fp1);
	    block1++;
	}
	else   /* id is different, assume this an add, since right now can't delete */
	{
	    printf("%02x%02x%02x%02x: added\n",
		   buf2a[4], buf2a[5], buf2a[6], buf2a[7]);

	    if (fp3 != NULL)   /* write the new block */
	    {
		if (wrapped_file)   /* write the id of the next old block into bytes 0-3 of the new block */
		{
		    buf2a[0] = buf1a[4];
		    buf2a[1] = buf1a[5];
		    buf2a[2] = buf1a[6];
		    buf2a[3] = buf1a[7];
		}

		write3 = fwrite(buf2a, 1, sizeof(buf2a), fp3);

		if (write3 != FILE_BLOCK_SIZE)
		{
		    perror(argv[3]);
		    fclose(fp1);
		    fclose(fp2);
		    fclose(fp3);
		    exit(1);
		}
	    }

	    added++;

	    /* Change the upper 4 bytes back because we are going through the loop again */
	    if (header1_version < 30 && header2_version >= 30)
	    {
		buf1a[0] = 0;
		buf1a[1] = 0;
		buf1a[2] = 0;
		buf1a[3] = 0;
	    }
	}

	read2 = fread(buf2a, 1, sizeof(buf2a), fp2);
	block2++;
    }

    if (feof(fp2) && !feof(fp1))
    {
	fprintf(stderr, "WARNING: end of file reached on %s before %s\n",
		argv[2], argv[1]);
    }

    while (!feof(fp2) && read2 == FILE_BLOCK_SIZE)
    {
	bad_upper_ref = false;

	if (header2_version < 30)
	{
	    if (buf2a[0] != 0 || buf2a[1] != 0 || buf2a[2] != 0 || buf2a[3] != 0)
	    {
		fprintf(stderr, "\n%s block %d - first four bytes not zero: %02x%02x%02x%02x\n",
			argv[2], block2, buf2a[0], buf2a[1], buf2a[2], buf2a[3]);
		bad_upper_ref = true;
	    }
	}
	else
	{
	    if (buf2a[0] != 0xff || buf2a[1] != 0xff || buf2a[2] != 0xff || buf2a[3] != 0xfe)
	    {
		fprintf(stderr, "\n%s block %d - first four bytes not fffffffe: %02x%02x%02x%02x\n",
			argv[2], block2, buf2a[0], buf2a[1], buf2a[2], buf2a[3]);
		bad_upper_ref = true;
	    }
	}

	if (bad_upper_ref)
	{
	    fclose(fp1);
	    fclose(fp2);
	    if (fp3 != NULL) fclose(fp3);

	    exit(1);
	}

	printf("%02x%02x%02x%02x: added\n",
	       buf2a[4], buf2a[5], buf2a[6], buf2a[7]);

	if (fp3 != NULL)   /* write the new block */
	{
	    write3 = fwrite(buf2a, 1, sizeof(buf2a), fp3);

	    if (write3 != FILE_BLOCK_SIZE)
	    {
		perror(argv[3]);
		fclose(fp1);
		fclose(fp2);
		fclose(fp3);
		exit(1);
	    }
	}

	added++;

	read2 = fread(buf2a, 1, sizeof(buf2a), fp2);
	block2++;
    }

    printf("\n");


    if (memcmp(header1.version_string, header2.version_string, 4) != 0)
    {
	fprintf(stderr, "WARNING, versions don't match - %s: %c%c%c%c  %s: %c%c%c%c\n",
		argv[1], 
		header1.version_string[0], header1.version_string[1],
		header1.version_string[2], header1.version_string[3],
		argv[2], 
		header2.version_string[0], header2.version_string[1],
		header2.version_string[2], header2.version_string[3]);
    }

    printf("Added: %d\n", added);
    printf("Changed: %d\n", changed);

    if (!feof(fp1) && read1 != FILE_BLOCK_SIZE)
    {
	perror(argv[1]);
    }

    fclose(fp1);

    if (!feof(fp2) && read2 != FILE_BLOCK_SIZE)
    {
	perror(argv[2]);
    }

    fclose(fp2);
    
    if (fp3 != NULL)   /* close the output file */
    {
	if (fclose(fp3) != 0)
	{
	    perror(argv[3]);
	}
    }

    return 0;
}
