/*
--             This file is part of the New World OS project
--                 Copyright (C) 2005-2009  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: spelling.c,v $
-- Revision 1.29  2009/03/10 10:39:39  jsedwards
-- Moved MAX_SPELLING_SIZE to the objectify.h file and renamed to
-- MAX_SPELLING_OBJ_SIZE.
--
-- Revision 1.28  2009/03/10 02:23:02  jsedwards
-- Eliminated the use of FILE_BLOCK_SIZE for kludge buffers.
--
-- Revision 1.27  2009/03/07 20:17:19  jsedwards
-- Removed #include of objectify_private.h, not needed.
--
-- Revision 1.26  2008/09/01 00:10:55  jsedwards
-- Fix copyright year.  NO code changes.
--
-- Revision 1.25  2008/08/31 21:53:54  jsedwards
-- Added an assert around calls to nwos_read_variable_sized_object_from_disk
-- and nwos_read_object_from_disk because now they return false when they fail
-- instead of asserting themselves.
--
-- Revision 1.24  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.23  2007/05/22 02:38:42  jsedwards
-- Change 'create_spelling' to 'create_public_spelling'.
--
-- Revision 1.22  2007/04/22 13:51:29  jsedwards
-- Remove old debugging code and a superfluous line of code.
--
-- Revision 1.21  2007/04/02 04:08:27  jsedwards
-- Fixed to operate in private mode.
--
-- Revision 1.20  2007/01/08 14:26:23  jsedwards
-- Added code to change class of cloned spelling objects to private class.
--
-- Revision 1.19  2007/01/06 19:32:08  jsedwards
-- Split find_spelling into find_public_spelling and find_private_spelling.
-- Added create_private_spelling function.
--
-- Revision 1.18  2006/12/21 12:59:47  jsedwards
-- Hack to make it compile with new split public and private classes,
-- NON-FUNCTIONAL!
--
-- Revision 1.17  2006/12/09 16:02:45  jsedwards
-- Removed old upgrade from 0004 to 0005 routine.
--
-- Revision 1.16  2006/12/01 14:31:31  jsedwards
-- Changed to use new malloc_reference_list and free_reference_list functions
-- instead of inlining the code.
--
-- Revision 1.15  2006/11/18 15:09:10  jsedwards
-- Added "max_size" parameter to read_variable_sized_object_from_disk because
-- objects are no longer limited to one file block.
--
-- Revision 1.14  2006/11/11 12:01:07  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.13  2006/10/26 01:51:29  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.12.2.5  2006/10/25 12:22:30  jsedwards
-- Changed C_struct_class_definition to C_struct_Class_Definition so the case
-- is consistent with all the other C_struct objects.
--
-- Revision 1.12.2.4  2006/09/03 11:05:45  jsedwards
-- Changed name of spelling_class_ref function to get_spelling_class_ref and
-- class_ref variable to spelling_class_ref.
--
-- Revision 1.12.2.3  2006/09/01 13:27:20  jsedwards
-- Changed "nwos_object_size" to "nwos_reference_list_size" and added the
-- object reference to "nwos_fill_in_common_header" so it can put the "id"
-- in the header now.
--
-- Revision 1.12.2.2  2006/08/18 12:53:48  jsedwards
-- Changed hard coded 512 for block size to FILE_BLOCK_SIZE.
--
-- Revision 1.12.2.1  2006/08/16 12:51:24  jsedwards
-- Added a cast so new versions of GCC won't complain about the signedness of
-- a pointer.
--
-- Revision 1.12  2006/01/12 02:53:45  jsedwards
-- Changed so letters are always stored in lower case.  Added upgrade routine
-- to change all letters in a 0004 object to lower case.
--
-- Revision 1.11  2005/12/29 17:50:53  jsedwards
-- Commented out printf debugging statements, that aren't useful now.
--
-- Revision 1.10  2005/12/29 17:28:16  jsedwards
-- Changed so that a spelling object is variable sized depending upon it's
-- contents.  That way when it is written to disk the write routine can fill
-- in the remaining data up to 512 bytes with random data.
--
-- Revision 1.9  2005/12/27 18:26:57  jsedwards
-- Added special routine for the "big_bang" to set the class reference.
-- Changed to look up class definition instead of using a fixed file name.
-- Also changed so that object id is random instead of based upon contents.
--
-- Revision 1.8  2005/12/24 16:18:26  jsedwards
-- Removed "host" id from object references (ObjRef).  Host redirection will
-- be done using a "redirection" object in the future.
--
-- Revision 1.7  2005/12/21 03:59:02  jsedwards
-- Added check to make sure the counts match before comparing the characters.
--
-- Revision 1.6  2005/12/11 16:34:56  jsedwards
-- Fixed error message to say "spelling" instead of "year".
-- Split "find_spelling" function out of "create_spelling" function.
--
-- Revision 1.5  2005/12/10 15:03:36  jsedwards
-- Fixed header to say the GPL is in the LICENSE file instead of COPYING.
--
-- Revision 1.4  2005/12/05 05:22:46  jsedwards
-- Change to call read_reference_list_from_disk for reference lists instead
-- of read_object_from_disk.  Also moved calculating checksums down to just
-- before writing object, so that no changes occur after computing them.
--
-- Revision 1.3  2005/12/04 04:13:03  jsedwards
-- Added 'nwos' prefix to create_xxxx function names and eliminated the
-- 'referring' object parameter from all of them.
--
-- Revision 1.2  2005/12/03 22:13:28  jsedwards
-- Changed to create a spelling object.
--
-- Revision 1.1  2005/12/03 22:11:26  jsedwards
-- Initial version created from word.c.
--
*/


#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>   /* define memset */

#include "crc32.h"
#include "objectify.h"



static ObjRef* get_public_spelling_class_ref()
{
    static ObjRef public_spelling_class_ref;

    if (is_void_reference(&public_spelling_class_ref))
    {
	nwos_find_public_class_definition("SPELLING", &public_spelling_class_ref);
    }

    assert(!is_void_reference(&public_spelling_class_ref));

    return &public_spelling_class_ref;
}


static ObjRef* get_private_spelling_class_ref()
{
    static ObjRef private_spelling_class_ref;

    if (is_void_reference(&private_spelling_class_ref))
    {
	if (!nwos_find_private_class_definition("SPELLING", &private_spelling_class_ref))
	{
	    return NULL;
	}
    }

    return &private_spelling_class_ref;
}


static size_t get_spelling_object_size(void* spelling_obj)
{
    assert(((C_struct_Spelling*)spelling_obj)->count > 0);

    return sizeof(C_struct_Spelling) + ((C_struct_Spelling*)spelling_obj)->count;
}


bool nwos_find_public_spelling(char* spelling, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    uint8 kludge[MAX_SPELLING_OBJ_SIZE];
    C_struct_Spelling* ptr_spelling_obj = (C_struct_Spelling*)kludge;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    size_t length;
    int i;


    length = strlen(spelling);

    assert(length <= 255);
    
    nwos_read_class_definition(get_public_spelling_class_ref(), &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, get_public_spelling_class_ref()))
	{
	    assert(nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &get_spelling_object_size));

	    /* remember ptr_spelling_obj points to the kludge buffer */

	    if (ptr_spelling_obj->count == length && strncasecmp((char*)ptr_spelling_obj->storage, spelling, length) == 0)   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}


bool nwos_find_private_spelling(char* spelling, ObjRef* ref)
{
    ObjRef* spelling_class_ref_ptr;
    C_struct_Class_Definition class_def_obj;
    uint8 kludge[MAX_SPELLING_OBJ_SIZE];
    C_struct_Spelling* ptr_spelling_obj = (C_struct_Spelling*)kludge;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    size_t length;
    int i;


    length = strlen(spelling);

    assert(length <= 255);

    spelling_class_ref_ptr = get_private_spelling_class_ref();

    if (spelling_class_ref_ptr == NULL) return false;

    nwos_read_class_definition(spelling_class_ref_ptr, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, spelling_class_ref_ptr))
	{
	    assert(nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &get_spelling_object_size));

	    /* remember ptr_spelling_obj points to the kludge buffer */

	    if (ptr_spelling_obj->count == length && strncasecmp((char*)ptr_spelling_obj->storage, spelling, length) == 0)   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}



/* Find existing spelling or create new */

ObjCreateResult nwos_create_public_spelling(char* spelling, ObjRef* ref)
{
    uint8 kludge[MAX_SPELLING_OBJ_SIZE];
    C_struct_Spelling* ptr_spelling_obj = (C_struct_Spelling*)kludge;
    size_t length;
    ObjCreateResult result = FOUND_EXISTING;
    int i;
    C_struct_Class_Definition class_def_obj;

    length = strlen(spelling);

    assert(length <= 255);

    /* first find out if we already have this spelling */

    if (!nwos_find_public_spelling(spelling, ref))   /* didn't find it */
    {
	memset(kludge, 0, sizeof(kludge));  /* zero it out */

	/* remember ptr_spelling_obj points to the kludge buffer */

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&ptr_spelling_obj->header.common, ref, get_public_spelling_class_ref());

	nwos_read_class_definition(get_public_spelling_class_ref(), &class_def_obj);

	ptr_spelling_obj->count = length;
	for (i = 0; i < length; i++) ptr_spelling_obj->storage[i] = tolower(spelling[i]);

	nwos_create_reference_list(ref, &ptr_spelling_obj->header.object.references);

	nwos_crc32_calculate((uint8*) &ptr_spelling_obj->header.object, sizeof(ObjectHeader), ptr_spelling_obj->header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &ptr_spelling_obj->character_set, sizeof(C_struct_Spelling) + length - sizeof(EveryObject), ptr_spelling_obj->header.common.data_chksum);

	nwos_write_object_to_disk(ref, kludge, sizeof(C_struct_Spelling) + length);

	nwos_add_to_references(ref, get_public_spelling_class_ref());

	ptr_spelling_obj = malloc(sizeof(C_struct_Spelling) + length);
	assert(nwos_read_object_from_disk(ref, ptr_spelling_obj, sizeof(C_struct_Spelling) + length));
	assert(memcmp(kludge, ptr_spelling_obj, sizeof(C_struct_Spelling) + length) == 0);

	memset(kludge, 0, sizeof(kludge));  /* clear it */
	assert(nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_spelling_object_size));  /* read the other way */
	assert(memcmp(ptr_spelling_obj, kludge, sizeof(C_struct_Spelling) + length) == 0);

	free(ptr_spelling_obj);
	ptr_spelling_obj = NULL;

	result = CREATED_NEW;
    }

    return result;
}


/* Find existing private spelling or clone the public object */

ObjCreateResult nwos_create_private_spelling(char* spelling, ObjRef* ref)
{
    ObjRef spelling_class_ref;
    ObjRef public_ref;
    uint8 kludge[MAX_SPELLING_OBJ_SIZE];
    C_struct_Spelling* ptr_spelling_obj = (C_struct_Spelling*)kludge;
    size_t length;
    ObjCreateResult result = FOUND_EXISTING;

    if (!nwos_find_private_spelling(spelling, ref))
    {
	nwos_find_or_create_private_class_definition("SPELLING", &spelling_class_ref);

	assert(nwos_find_public_spelling(spelling, &public_ref));

	assert(nwos_read_variable_sized_object_from_disk(&public_ref, kludge, sizeof(kludge), &get_spelling_object_size));

	nwos_generate_new_id(ref);

	/* copy the public id into the clone_of field */
	copy_reference(&ptr_spelling_obj->header.object.clone_of, &public_ref);

	copy_reference(&ptr_spelling_obj->header.common.id, ref);

	copy_reference(&ptr_spelling_obj->header.common.class_definition, &spelling_class_ref);

	nwos_create_reference_list(ref, &ptr_spelling_obj->header.object.references);

	nwos_crc32_calculate((uint8*) &ptr_spelling_obj->header.object, sizeof(ObjectHeader), ptr_spelling_obj->header.common.header_chksum);

	length = ptr_spelling_obj->count;

	nwos_write_object_to_disk(ref, kludge, sizeof(C_struct_Spelling) + length);

	nwos_add_to_references(ref, &spelling_class_ref);

	ptr_spelling_obj = malloc(sizeof(C_struct_Spelling) + length);
	assert(nwos_read_object_from_disk(ref, ptr_spelling_obj, sizeof(C_struct_Spelling) + length));
	assert(memcmp(kludge, ptr_spelling_obj, sizeof(C_struct_Spelling) + length) == 0);

	memset(kludge, 0, sizeof(kludge));  /* clear it */
	assert(nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_spelling_object_size));  /* read the other way */
	assert(memcmp(ptr_spelling_obj, kludge, sizeof(C_struct_Spelling) + length) == 0);

	free(ptr_spelling_obj);
	ptr_spelling_obj = NULL;

	result = CREATED_NEW;
    }

    return result;
}

