/*
--             This file is part of the New World OS project
--                 Copyright (C) 2004-2009  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: bit_map.c,v $
-- Revision 1.6  2009/03/14 23:14:04  jsedwards
-- Added include of the new chunk_info.h file.
--
-- Revision 1.5  2009/03/14 11:46:46  jsedwards
-- Added include of gen_id.h file.
--
-- Revision 1.4  2009/03/14 03:31:24  jsedwards
-- Added include of bit_map.h file.
--
-- Revision 1.3  2009/03/08 00:06:18  jsedwards
-- Changed include objectify_private.h to disk_io.h.
--
-- Revision 1.2  2009/03/07 13:24:30  jsedwards
-- Fixed --minimal-security assert bugs and simplified find next block.
--
-- Revision 1.1  2009/03/06 04:01:11  jsedwards
-- Initial version created from functions taken from the disk_io.c file.
--
*/

#include <stdlib.h>    /* define NULL */

#include "bit_map.h"
#include "chunk_info.h"
#include "disk_io.h"
#include "gen_id.h"


#define BIT_MAP_CACHE_SIZE 256

typedef struct {
  bool8 in_use;
  bool8 dirty;
  int chunk_index;    /* index into chunk (NOT chunk_info index) */
  int age;
  uint8* map;
} Bit_Map_Cache_Entry;

static Bit_Map_Cache_Entry bit_map_cache[BIT_MAP_CACHE_SIZE];

static int bit_map_tick = 0;

static void write_bit_map(Bit_Map_Cache_Entry* entry);


/*********************************************/
/* Functions to manipulate the bit map cache */
/*********************************************/

static int8 bits_in_byte[256] = 
{
/*        0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f */
/* 0 */   0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4,
/* 1 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 2 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 3 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 4 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 5 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 6 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 7 */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* 8 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 9 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* a */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* b */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* c */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* d */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* e */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* f */   4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
};


static void write_bit_map(Bit_Map_Cache_Entry* entry)
{
    int i;
    int info_index;
    uint32 blocks_used = 0;
#ifdef LOG_BIT_MAP
    char log_msg[80];
#endif

    assert(0 <= entry->chunk_index && entry->chunk_index < nwos_used_private_chunks);

    info_index = nwos_chunk_index_to_info_index(entry->chunk_index);

    assert(0 <= info_index && info_index < nwos_used_private_chunks);
    assert(nwos_chunk_info[info_index].index == entry->chunk_index);

#ifdef LOG_BIT_MAP
    snprintf(log_msg, sizeof(log_msg), "write bit_map - info_index: %d  chunk: %d - %08x", info_index, entry->chunk_index, chunk);
    nwos_log(log_msg);
#endif

    /* don't count the blocks used by the bit map itself */
    for (i = BIT_MAP_BLOCKS / 8; i < BIT_MAP_BYTES; i++)
    {
	blocks_used += bits_in_byte[entry->map[i]];
    }

    assert(blocks_used <= BLOCKS_IN_CHUNK - BIT_MAP_BLOCKS);

    if (blocks_used != nwos_chunk_info[info_index].used)
    {
	nwos_reset_chunk_info_blocks_used(info_index, blocks_used);
    }

    nwos_write_bit_map(entry->chunk_index, entry->map);

    entry->dirty = false;
}


static void read_bit_map_into_cache(Bit_Map_Cache_Entry* entry)
{
    int i;
    int info_index;
    uint32 blocks_used = 0;
#ifdef LOG_BIT_MAP
    char log_msg[80];

    snprintf(log_msg, sizeof(log_msg), "read bit_map: %08x\n", entry->chunk);
    nwos_log(log_msg);
#endif

    assert(0 <= entry->chunk_index && entry->chunk_index < nwos_used_private_chunks);

    info_index = nwos_chunk_index_to_info_index(entry->chunk_index);

    assert(0 <= info_index && info_index < nwos_used_private_chunks);
    assert(nwos_chunk_info[info_index].index == entry->chunk_index);

    if (entry->map == NULL)
    {
	entry->map = malloc(BIT_MAP_BYTES);
	assert(entry->map != NULL);
    }

    nwos_read_bit_map(entry->chunk_index, entry->map);

    /* don't count the blocks used by the bit map itself */
    for (i = BIT_MAP_BLOCKS / 8; i < BIT_MAP_BYTES; i++)
    {
	blocks_used += bits_in_byte[entry->map[i]];
    }

    assert(blocks_used <= BLOCKS_IN_CHUNK - BIT_MAP_BLOCKS);


    if (blocks_used != nwos_chunk_info[info_index].used)
    {
	nwos_reset_chunk_info_blocks_used(info_index, blocks_used);
    }

    entry->in_use = true;
    entry->dirty = false;
}


static Bit_Map_Cache_Entry* find_bit_map_in_cache(uint32 block)
{
    int i;
    uint16 chunk_index;
    Bit_Map_Cache_Entry* result = NULL;

    assert(nwos_block_offset_to_chunks <= block && block < nwos_block_offset_to_chunks + nwos_total_private_blocks);

    /* assert((block % CHUNK_SIZE) != 0);  what was this for? */

    chunk_index = (block - nwos_block_offset_to_chunks) / BLOCKS_IN_CHUNK;


    /* see if it's already in the cache */
    for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
    {
	if (bit_map_cache[i].in_use && bit_map_cache[i].chunk_index == chunk_index) break;
    }

    if (i < BIT_MAP_CACHE_SIZE)   /* found it */
    {
	result = &bit_map_cache[i];
    }
    else                           /* didn't find it */
    {
	/* find an empty one or find the oldest */

	result = bit_map_cache;   /* so we have an age to compare to */

	for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
	{
	    if (!bit_map_cache[i].in_use)
	    {
		result = &bit_map_cache[i];
		break;
	    }

	    if (bit_map_cache[i].age < result->age)
	    {
		result = &bit_map_cache[i];
	    }
	}

	if (i == BIT_MAP_CACHE_SIZE && result->dirty)    /* didn't find an empty one, write the oldest one out */
	{
	    write_bit_map(result);
	}

	result->chunk_index = chunk_index;

	read_bit_map_into_cache(result);
    }

    bit_map_tick++;
    result->age = bit_map_tick;

    assert(result != NULL);

    return result;
}


void nwos_set_bit_in_map(uint32 block)
{
    int byte_num;
    int bit_num;
    int info_index;
    Bit_Map_Cache_Entry* entry;

    entry = find_bit_map_in_cache(block);

    byte_num = (block % BLOCKS_IN_CHUNK) / 8;
    assert(byte_num < BIT_MAP_BYTES);
    bit_num = block % 8;

    if ((entry->map[byte_num] & (0x80 >> bit_num)) == 0)    /* don't count a block that was already used */
    {
	entry->map[byte_num] |= (0x80 >> bit_num);

/* printf("set_bit_in_map block: %08x  chunk: %08x  byte: %d  bit: %d\n", block, entry->chunk, byte_num, bit_num); */

	entry->dirty = true;

	info_index = nwos_chunk_index_to_info_index(entry->chunk_index);

	assert(0 <= info_index && info_index < nwos_used_private_chunks);
	assert(nwos_chunk_info[info_index].index == entry->chunk_index);

	nwos_increment_chunk_info_block_used(info_index);

	nwos_used_private_blocks++;
    }
}


void nwos_clear_bit_in_map(uint32 block)
{
    int byte_num;
    int bit_num;
    int info_index;
    Bit_Map_Cache_Entry* entry;

    entry = find_bit_map_in_cache(block);

    byte_num = (block % BLOCKS_IN_CHUNK) / 8;
    assert(byte_num < BIT_MAP_BYTES);
    bit_num = block % 8;

    if ((entry->map[byte_num] & (0x80 >> bit_num)) != 0)    /* don't count a block that was already clear */
    {
	entry->map[byte_num] &= ~(0x80 >> bit_num);

/* printf("clear_bit_in_map block: %08x  chunk: %08x  byte: %d  bit: %d\n", block, entry->chunk, byte_num, bit_num); */

	entry->dirty = true;

	info_index = nwos_chunk_index_to_info_index(entry->chunk_index);

	assert(0 <= info_index && info_index < nwos_used_private_chunks);
	assert(nwos_chunk_info[info_index].index == entry->chunk_index);

	nwos_decrement_chunk_info_block_used(info_index);

	nwos_used_private_blocks--;
    }
}


bool nwos_test_bit_in_map(uint32 block)
{
    int byte_num;
    int bit_num;
    Bit_Map_Cache_Entry* entry;

    entry = find_bit_map_in_cache(block);

    byte_num = (block % BLOCKS_IN_CHUNK) / 8;
    assert(byte_num < BIT_MAP_BYTES);
    bit_num = block % 8;

    return (entry->map[byte_num] & (0x80 >> bit_num)) != 0;
}


bool nwos_block_used(ObjRef* ref)
{
#ifdef PUBLIC_MODE
    uint8 block[FILE_BLOCK_SIZE];

    return  nwos_read_block(ref, block) && !is_void_reference((ObjRef*)&block[4]);
#else
    uint32 block;
    int byte_num;
    int bit_num;
    Bit_Map_Cache_Entry* entry;

    block = nwos_hash_ref(ref);

    entry = find_bit_map_in_cache(block);

    byte_num = (block % BLOCKS_IN_CHUNK) / 8;
    assert(byte_num < BIT_MAP_BYTES);
    bit_num = block % 8;

    return ((entry->map[byte_num] & (0x80 >> bit_num)) != 0);
#endif
}


uint32 nwos_find_random_block_in_chunk(int info_index)
{
    int i;
    int bit;
    int byte;
    Bit_Map_Cache_Entry* entry;

    entry = find_bit_map_in_cache(nwos_block_offset_to_chunks + ((uint32)nwos_chunk_info[info_index].index * BLOCKS_IN_CHUNK) + BIT_MAP_BLOCKS);

    /* don't pick one of the bytes (currently 0-3) used by the bit map */
    byte = (random() % (BIT_MAP_BYTES - (BIT_MAP_BLOCKS / 8))) + (BIT_MAP_BLOCKS / 8);

    if (entry->map[byte] == 0xff)   /* didn't get lucky */
    {
	for (i = 1; i < BIT_MAP_BYTES; i++)   /* find the nearest empty */
	{
	    if (byte - i >= 0 && entry->map[byte - i] != 0xff)
	    {
		byte -= i;
		break;
	    }

	    if (byte + i < BIT_MAP_BYTES && entry->map[byte + i] != 0xff)
	    {
		byte += i;
		break;
	    }

	    assert(0 <= byte - i || byte + i < BIT_MAP_BYTES);
	}
    }

    assert(BIT_MAP_BLOCKS / 8 <= byte && byte < BIT_MAP_BYTES);
    assert(entry->map[byte] != 0xff);

    bit = random() % 8;

    if ((entry->map[byte] & (0x80 >> bit)) != 0)
    {
	for (i = 1; i < 8; i++)   /* find the nearest empty */
	{
	    if (bit - i >= 0 && (entry->map[byte] & (0x80 >> (bit - i))) == 0)
	    {
		bit -= i;
		break;
	    }

	    if (bit + i < 8 && (entry->map[byte] & (0x80 >> (bit + i))) == 0)
	    {
		bit += i;
		break;
	    }

	    assert(0 <= bit - i || bit + i < 8);
	}
    }

    assert(0 <= bit && bit < 8);

    return nwos_chunk_info[info_index].ref + byte * 8 - BIT_MAP_BLOCKS + bit;
}


int nwos_find_low_density_in_chunk(int info_index, int density)
{
    Bit_Map_Cache_Entry* entry = NULL;
    int byte_num;
    int bits_used = 0;

    entry = find_bit_map_in_cache(nwos_block_offset_to_chunks + nwos_chunk_info[info_index].index * BLOCKS_IN_CHUNK + BIT_MAP_BLOCKS);

    for (byte_num = BIT_MAP_BYTES - 1; byte_num > 0; byte_num--)
    {
	bits_used += bits_in_byte[entry->map[byte_num]];

	if (bits_used / (BIT_MAP_BYTES - byte_num) > density) break;
    }

    return byte_num;
}


uint32 nwos_find_next_empty_block(uint32 start_block, int density)
{
    Bit_Map_Cache_Entry* entry;
    uint32 offset;
    uint32 chunk;
    int    byte_num;
    int    bit_num;
    uint8  byte;
    int    info_index;
    int    skipped = 0;

    offset = (start_block - nwos_block_offset_to_chunks) % BLOCKS_IN_CHUNK;
    chunk = start_block - offset;

    assert(BIT_MAP_BLOCKS <= offset);

    entry = find_bit_map_in_cache(chunk);
    info_index = nwos_chunk_index_to_info_index(entry->chunk_index);

    assert(entry->chunk_index == (chunk - nwos_block_offset_to_chunks) / BLOCKS_IN_CHUNK);

    byte_num = offset / 8;
    bit_num = offset % 8;

    while (byte_num < BIT_MAP_BYTES)
    {
	assert(bit_num < 8);
	assert(0 <= info_index && info_index < nwos_used_private_chunks);
	assert(nwos_chunk_info[info_index].index == entry->chunk_index);
	assert(nwos_chunk_info[info_index].index == (chunk - nwos_block_offset_to_chunks) / BLOCKS_IN_CHUNK);

	byte = entry->map[byte_num];

	if ((byte & (0x80 >> bit_num)) == 0)  /* this block is empty */
	{
	    if ((random() % 8) < density + skipped)   /* randomly skip some blocks */
	    {
		break;                                   /* use this empty */
	    }

	    skipped++;
	}

	bit_num++;                                /* check the next bit */

	if (bit_num == 8)
	{
	    bit_num = 0;
	    byte_num++;
	}
    }

    offset = (byte_num * 8) + bit_num;

    assert(BIT_MAP_BLOCKS <= offset && offset <= BLOCKS_IN_CHUNK);

    return chunk + offset;
}


void nwos_flush_bit_maps()
{
    int i;

    for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
    {
	if (bit_map_cache[i].in_use && bit_map_cache[i].dirty)
	{
	    write_bit_map(&bit_map_cache[i]);
	}
    }
}


void nwos_terminate_bit_map(void)
{
    int i;

    for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
    {
	if (bit_map_cache[i].in_use && bit_map_cache[i].dirty)
	{
	    write_bit_map(&bit_map_cache[i]);

	    bit_map_cache[i].in_use = false;

	    free(bit_map_cache[i].map);
	    bit_map_cache[i].map = NULL;
	}
    }
}


