/*
--             This file is part of the New World OS project
--                    Copyright (C) 2009  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: check_files.c,v $
-- Revision 1.7  2009/01/31 15:46:06  jsedwards
-- Changed printing of num_refs slightly.
--
-- Revision 1.6  2009/01/31 15:28:35  jsedwards
-- Added test to verify the file object is in the correct reference list.
--
-- Revision 1.5  2009/01/31 15:12:17  jsedwards
-- Changed to do file class definition types 001 and 002.
--
-- Revision 1.4  2009/01/31 12:59:23  jsedwards
-- Change to use array for old file class definition references.
--
-- Revision 1.3  2009/01/31 12:48:31  jsedwards
-- Added printing of old file class references.
--
-- Revision 1.2  2009/01/26 16:34:46  jsedwards
-- Changed to print file number and added call to nwos_restore_file to read
-- and verify checksums of file.
--
-- Revision 1.1  2009/01/25 16:07:28  jsedwards
-- Initial version - only prints information, no checking yet.
--
*/


#include <assert.h>
#include <limits.h>    /* define PATH_MAX */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "objectify_private.h"


static void display_usage(char* prog_name)
{
    fprintf(stderr, "usage: %s\n", prog_name);
    fprintf(stderr, "   or: %s --compressed file\n", prog_name);
    fprintf(stderr, "\n");
    fprintf(stderr, "  --compressed file\tuse file for archive.\n");
    fprintf(stderr, "\n");
}


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}



int main(int argc, char* argv[])
{
    char* path = DEFAULT_FILE;
    ObjRef file_class_refs[NUM_FILE_CLASS_DEFINITIONS];
    ObjRef file_class_ref;
    ObjRef assoc_class_ref;
    ObjRef object_class;
    ObjRef real_object_class;
    ObjRef assoc_ref;
    C_struct_Class_Definition class_def_obj;
    C_struct_File file_obj;
    C_struct_Path_And_File_Association assoc_obj;
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    ReferenceList* ref_list;
    ReferenceList* file_ref_list;
    int num_refs;
    int num_file_refs;
    int real_type;
    int def;
    int i;
    int j;
    int k;
    char name[PATH_MAX];
    uint64 file_size;
    int num_files = 0;
    int num_failures = 0;
    bool result = true;

    printf("\n");

    if (argc == 3 && strcmp(argv[1], "--compressed") == 0)
    {
	path = argv[2];
    }
    else if (argc != 1)
    {
	display_usage(argv[0]);
	exit(1);
    }

    nwos_log_arguments(0, NULL);

    nwos_initialize_objectify(READ_ONLY, path);

    if (!nwos_find_private_class_definition("FILE", &file_class_ref))
    {
	printf("No files are stored in the system!\n\n");
	exit(1);
    }

    if (!nwos_find_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref))
    {
	printf("NOTE: No path and file associations are stored in the system!\n\n");
    }

    assert(NUM_FILE_CLASS_DEFINITIONS == 3);

    copy_reference(&file_class_refs[0], nwos_get_file_001_reference());
    copy_reference(&file_class_refs[1], nwos_get_file_002_reference());
    copy_reference(&file_class_refs[2], nwos_get_file_class_reference());

    assert(is_same_object(&file_class_ref, &file_class_refs[2]));

    for (def = 0; def < NUM_FILE_CLASS_DEFINITIONS; def++)
    {
	printf("File class %03d reference: %08x\n\n", def + 1, nwos_ref_to_word(&file_class_refs[def]));

	if (nwos_is_public_reference(&file_class_refs[def]))
	{
	    printf("  NO private files of file class %03d exist\n", def + 1);
	}
	else
	{
	    assert(nwos_is_private_reference(&file_class_refs[def]));

	    nwos_read_object_headers_from_disk(&file_class_refs[def], &class_def_obj.header);

	    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

	    num_refs = ref_list->common_header.num_refs;

	    printf("  num_refs: %d\n\n", num_refs);

	    for (i = 0; i < num_refs; i++)
	    {
		nwos_get_object_class(&ref_list->references[i], &object_class);

		if (is_same_object(&object_class, &file_class_ref))
		{
		    num_files++;

		    printf("file: %d\n", num_files);

		    /* verify it is in the correct list */
		    nwos_get_object_class_without_update(&ref_list->references[i], &real_object_class);

		    if (!is_same_object(&real_object_class, &file_class_refs[def]))
		    {
			real_type = 0;

			if (is_same_object(&real_object_class, &file_class_refs[0]))
			{
			    real_type = 1;
			}
			else if (is_same_object(&real_object_class, &file_class_refs[1]))
			{
			    real_type = 2;
			}
			else if (is_same_object(&real_object_class, &file_class_refs[2]))
			{
			    real_type = 3;
			}

			assert(1 <= real_type && real_type <= 3);

			printf("Warning: file object is in reference list of %03d but is of type %03d\n", def + 1, real_type);
		    }

		    nwos_read_object_from_disk(&ref_list->references[i], &file_obj, sizeof(file_obj));

		    file_size = ((uint64)file_obj.size[0] << 32) |
				((uint64)file_obj.size[1] << 24) |
				((uint64)file_obj.size[2] << 16) |
				((uint64)file_obj.size[3] << 8)  |
				 (uint64)file_obj.size[4];

		    printf("         id: %08x\n", nwos_ref_to_word(&file_obj.header.common.id));
		    printf("       size: %llu\n", file_size);
		    printf("     md5sum: %08x\n", nwos_ref_to_word(&file_obj.md5sum));
		    printf("    sha1sum: %08x\n", nwos_ref_to_word(&file_obj.sha1sum));
		    printf("  sha256sum: %08x\n", nwos_ref_to_word(&file_obj.sha256sum));
		    printf("  sha512sum: %08x\n", nwos_ref_to_word(&file_obj.sha512sum));
//    ObjRef media;
		    printf(" block_list: %08x\n", nwos_ref_to_word(&file_obj.block_list));

		    if (!is_void_reference(&assoc_class_ref))
		    {
			file_ref_list = nwos_malloc_reference_list(&file_obj.header.object.references);

			num_file_refs = file_ref_list->common_header.num_refs;

			void_reference(&assoc_ref);   /* no association yet */

			for (j = 0; j < num_file_refs; j++)
			{
			    nwos_get_object_class(&file_ref_list->references[j], &object_class);

			    if (is_same_object(&object_class, &assoc_class_ref))
			    {
				printf("      assoc: %08x\n", nwos_ref_to_word(&file_ref_list->references[j]));

				nwos_read_object_from_disk(&file_ref_list->references[j], &assoc_obj, sizeof(assoc_obj));

				printf("          time: %s\n", nwos_time_stamp_to_string(assoc_obj.modification_time));

				assert(nwos_read_variable_sized_object_from_disk(&assoc_obj.path, kludge, sizeof(kludge), &get_path_object_size));

				/* remember ptr_path_obj points to the kludge buffer */

				for (k = 0; k < ptr_path_obj->count; k++) name[k] = ptr_path_obj->storage[k];

				name[k] = '\0';

				printf("          path: %s\n", name);

				if (is_same_object(&assoc_obj.file, &file_obj.header.common.id))
				{
				    copy_reference(&assoc_ref, &file_ref_list->references[j]);
				}
				else
				{
				    printf("   ERROR: association object points to a different file: %08x\n", nwos_ref_to_word(&assoc_obj.file));
				    num_failures++;
				    result = false;
				}
			    }
			}

			if (is_void_reference(&assoc_ref))
			{
			    printf("   ERROR: no association objects found for this file!\n");
			    num_failures++;
			    result = false;
			}
			else
			{
			    if (!nwos_restore_file(&assoc_ref, NULL))     /* calling restore_file with no file name just verfies the file */
			    {
				num_failures++;
			    }
			}

			nwos_free_reference_list(file_ref_list);
			file_ref_list = NULL;
		    }

		    printf("\n");
		}
	    }    

	    nwos_free_reference_list(ref_list);
	    ref_list = NULL;
	}

	printf("\n");
    }

    nwos_terminate_objectify();

    printf("Number of files checked: %d\n", num_files);
    printf("Number of files failed:  %d\n", num_failures);

    return !result;    /* return 0 means success */
}

