/*
--             This file is part of the New World OS project
--                 Copyright (C) 2006-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: disk_usage.c,v $
-- Revision 1.32  2008/12/27 14:34:23  jsedwards
-- Merged from alpha_29_5_branch back into main trunk.
--
-- Revision 1.27.2.1  2008/12/21 04:12:47  jsedwards
-- Merged in main truck to get bug fixes for graph.
--
-- Revision 1.31  2008/12/05 13:00:09  jsedwards
-- Fix Bug #2392474 - overflow in graph calculation.  (Note: this was fixed
-- inadvertantly by the previous change to display gigabytes instead of 100%,
-- but decided to fix the overflow anyway, just in case.)
--
-- Revision 1.30  2008/11/28 16:57:46  jsedwards
-- Changed to display graph in gigabytes instead of percent.
--
-- Revision 1.29  2008/11/28 16:17:57  jsedwards
-- Added printing of allocated space in graph with a '+'.
--
-- Revision 1.28  2008/11/28 15:27:08  jsedwards
-- Fix Bug #2354269 - was only using allocated blocks for divisor instead of
-- all possible blocks in the chunks.
--
-- Revision 1.27  2008/10/20 02:20:24  jsedwards
-- Fix Bug #2117258 - Seg fault running disk_usage.  Changed for loop to only
-- do used_private_chunks instead of total_private_chunks because only space
-- for used_private_chunks is malloc'd.
--
-- Revision 1.26  2008/09/17 10:58:25  jsedwards
-- Added display_graph function to print bar graph of usage.
--
-- Revision 1.25  2008/09/16 13:20:37  jsedwards
-- Change to print a progress bar when not verbose instead of blocks used.
--
-- Revision 1.24  2008/08/10 15:23:25  jsedwards
-- Added path to "Missing magic number" and "Incorrect version in header" error
-- messages.
--
-- Revision 1.23  2008/04/02 03:06:21  jsedwards
-- Added file locking.
--
-- Revision 1.22  2008/02/03 01:15:40  jsedwards
-- Change to use nwos_get_private_objects_path function instead of DEFAULT_FILE.
--
-- Revision 1.21  2007/11/03 14:12:31  jsedwards
-- Rearranged chunk/block output table yet again.
--
-- Revision 1.20  2007/11/03 13:58:23  jsedwards
-- Changed to use chunk_info table to get to bit maps and also verify block
-- used count in chunk_info table.
--
-- Revision 1.19  2007/11/03 13:36:48  jsedwards
-- Added --verbose option, which prints the usage for each chunk.
--
-- Revision 1.18  2007/11/03 13:22:27  jsedwards
-- Removed old commented out and #ifdef'd out code.
--
-- Revision 1.17  2007/11/03 12:35:37  jsedwards
-- Renamed variable 'ref' to 'block' in calculate_blocks_used function to be
-- consistent.
--
-- Revision 1.16  2007/11/03 12:29:26  jsedwards
-- Added code to read in chunk_info table.
--
-- Revision 1.15  2007/11/02 14:24:21  jsedwards
-- Changed output of chunks and blocks into a table.
--
-- Revision 1.14  2007/11/02 13:21:07  jsedwards
-- Added printing of chunks (system, allocated, free) and blocks used by system.
--
-- Revision 1.13  2007/11/01 14:01:35  jsedwards
-- Changed printing of results to display in megabytes or gigabytes depending
-- on total size.
--
-- Revision 1.12  2007/11/01 13:56:00  jsedwards
-- Changed printing of blocks to include megabytes and available space.
--
-- Revision 1.11  2007/08/10 00:03:36  jsedwards
-- Removed defintion of _LARGEFILE64_SOURCE, now using _FILE_OFFSET_BITS=64.
-- Also removed using O_LARGEFILE from open call.
--
-- Revision 1.10  2007/07/01 19:44:11  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.9  2007/06/20 22:42:09  jsedwards
-- Updated to work with the new 0022 disk header.
--
-- Revision 1.8  2007/04/15 17:09:53  jsedwards
-- Fixed bug in percentage print out.
--
-- Revision 1.7  2007/02/11 15:15:20  jsedwards
-- Change 'sprintf' calls to 'snprintf' calls so the OpenBSD linker will stop
-- whining.
--
-- Revision 1.6  2007/02/11 14:41:26  jsedwards
-- Change all 'off64_t' and 'lseek64' references to 'off_t' and 'lseek',
-- because BSD doesn't dig the whole brain damaged 64 bit thing.
--
-- Revision 1.5  2006/12/03 21:43:03  jsedwards
-- Changed for the new disk layout: first section contains public objects,
-- second section private unencrypted objects, and third section private
-- encrypted objects.
--
-- Revision 1.4  2006/11/11 12:01:03  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.3  2006/11/02 11:49:28  jsedwards
-- Fixed all cases where 'z' was used as a format for 'off64_t' values because
-- the older compiler complains.
--
-- Revision 1.2  2006/10/26 01:51:26  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.10  2006/10/22 16:01:42  jsedwards
-- Change so number of blocks and reserved public blocks are read from disk
-- instead of being define constants.
--
-- Revision 1.1.2.9  2006/10/19 01:45:48  jsedwards
-- Fixed format specifiers for off64_t type.
--
-- Revision 1.1.2.8  2006/10/17 01:38:02  jsedwards
-- Added code to keep track of then number of blocks used in each chunk.
--
-- Revision 1.1.2.7  2006/10/15 12:30:03  jsedwards
-- Change to print blocks used in each print instead of a running total.
--
-- Revision 1.1.2.6  2006/10/15 12:17:54  jsedwards
-- Change to print output every 16 gigabytes instead of 10 gigabytes.
--
-- Revision 1.1.2.5  2006/10/14 12:33:58  jsedwards
-- Added in blocks skipped over at the beginning and stop when reached
-- the capacity of the disk.
--
-- Revision 1.1.2.4  2006/10/10 07:32:36  jsedwards
-- Move bit map defines to objectify_private.h instead of spread all over.
--
-- Revision 1.1.2.3  2006/10/10 07:02:37  jsedwards
-- Removed code that verified the bits_in_byte table (since it has been
-- verified there is no longer a need for it) and added printing blocks
-- used for every gigabyte printed.
--
-- Revision 1.1.2.2  2006/10/10 06:56:08  jsedwards
-- Changed to 8192 byte bit maps to speed up this program (with 256 byte bit
-- maps this program took around an hour to run on a 250 GB disk).
--
-- Revision 1.1.2.1  2006/10/08 13:50:02  jsedwards
-- Program to count the number of blocks used on the disk using the
-- block bit maps.
--
*/


#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "objectify_private.h"

static void print_usage(char *program)
{
    fprintf(stderr, "usage: %s [--verbose]\n", program);
}


bool verbose;

static int bits_in_byte[256] = 
{
/*        0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f */
/* 0 */   0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4,
/* 1 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 2 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 3 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 4 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 5 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 6 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 7 */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* 8 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 9 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* a */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* b */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* c */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* d */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* e */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* f */   4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
};


uint32 calculate_blocks_used(int obj_fd, uint32 start, Chunk_Info chunk_info[], uint32 num_chunks)
{
    char ref_str[128];
    off_t block;
    size_t bytes_read;
    uint8 map[BIT_MAP_BYTES];
    uint32 blocks_used;
    uint32 result = 0;
    int i;
    int j;


    if (!verbose)
    {
	nwos_start_progress_bar();
    }

    for (i = 0; i < num_chunks; i++)
    {
	block = start + chunk_info[i].index * BLOCKS_IN_CHUNK;

	if (lseek(obj_fd, block << 8, SEEK_SET) < 0)
	{
	    snprintf(ref_str, sizeof(ref_str), "lseek ref:%08x", (uint32)block);
	    perror(ref_str);
	    exit(1);
	}

	bytes_read = read(obj_fd, map, sizeof(map));

	if (bytes_read != sizeof(map))
	{
	    printf("error: incomplete bit map: %08x bytes_read: %zd\n", (uint32)(block), bytes_read);
	    break;
	}

	blocks_used = 0;

	for (j = 0; j < bytes_read; j++)
	{
	    blocks_used += bits_in_byte[map[j]];
	}

	result += blocks_used;

	if (verbose)
	{
	    printf("Chunk %d  ref: %08x  index: %u  used: %u\n",
		   i, chunk_info[i].ref, chunk_info[i].index, chunk_info[i].used);
	}
	else
	{
	    nwos_update_progress_bar((float)i / (float)num_chunks);
	}

	if ((blocks_used - BIT_MAP_BLOCKS) != chunk_info[i].used)
	{
	    if (!verbose)
	    {
		nwos_erase_progress_bar();
	    }

	    fprintf(stderr, "Error: chunk %d index %u stored count (%u) doesn't match bit map count (%u)\n",
		    i, chunk_info[i].index, chunk_info[i].used, blocks_used - BIT_MAP_BLOCKS);

	    if (!verbose)
	    {
		nwos_start_progress_bar();
	    }
	}
    }

    if (!verbose)
    {
	nwos_finish_progress_bar();
    }

    return result;
}


void display_graph(Chunk_Info chunk_info[], uint32 num_chunks)
{
    const int graph_bits = 6;
    const int divisor = 1 << (graph_bits - 4);
    uint32 used_blocks[1 << graph_bits];
    uint32 alloc_blocks[1 << graph_bits];
    uint32 total_blocks[1 << graph_bits];
    uint32 chunk;
    int segment;
    int gigabytes;
    
    int i;


    memset(used_blocks, 0, sizeof(used_blocks));
    memset(alloc_blocks, 0, sizeof(alloc_blocks));
    memset(total_blocks, 0, sizeof(total_blocks));

    for (chunk = MINIMUM_PRIVATE_REFERENCE; chunk < MAXIMUM_PRIVATE_REFERENCE; chunk += USABLE_BLOCKS_PER_CHUNK)
    {
	segment = chunk >> (32 - graph_bits);

	assert((RESERVED_PUBLIC_BLOCKS >> (32 - graph_bits)) <= segment && segment < (1 << graph_bits));

	total_blocks[segment] += USABLE_BLOCKS_PER_CHUNK;
    }

    for (i = 0; i < num_chunks; i++)
    {
	segment = chunk_info[i].ref >> (32 - graph_bits);

	assert((RESERVED_PUBLIC_BLOCKS >> (32 - graph_bits)) <= segment && segment < (1 << graph_bits));

	used_blocks[segment] += chunk_info[i].used;
	alloc_blocks[segment] += USABLE_BLOCKS_PER_CHUNK;
    }

    for (gigabytes = (1024 / (1 << graph_bits)); gigabytes >= 0; gigabytes--)
    {
	printf(" %2d GB |", gigabytes);

	for (i = (RESERVED_PUBLIC_BLOCKS >> (32 - graph_bits)); i < (1 << graph_bits); i++)
	{
	    if (used_blocks[i] > 0 && (((uint64)used_blocks[i] * (1024 / (1 << graph_bits))) / total_blocks[i]) >= (uint64)gigabytes)
	    {
		putchar('*');
	    }
	    else if (alloc_blocks[i] > 0 && (((uint64)alloc_blocks[i] * (1024 / (1 << graph_bits))) / total_blocks[i]) >= (uint64)gigabytes)
	    {
		putchar('+');
	    }
	    else
	    {
		putchar(' ');
	    }
	}

	putchar('\n');
    }

    printf("       +");
    for (i = (RESERVED_PUBLIC_BLOCKS >> (32 - graph_bits)); i < (1 << graph_bits); i++) putchar('-');
    putchar('\n');

    printf("        ");
    for (i = (RESERVED_PUBLIC_BLOCKS >> (32 - graph_bits)); i < (1 << graph_bits); i++)
    {
	if (i % divisor == 0)
	{
	    if (i / divisor < 10)
	    {
		putchar('0' + i / divisor);
	    }
	    else
	    {
		putchar('A' + i / divisor - 10);
	    }
	}
	else
	{
	    putchar(' ');
	}
    }
    putchar('\n');
    putchar('\n');

    printf("     * = used\n");
    printf("     + = allocated, but unused\n");
}


int main(int argc, char* argv[])
{
    int obj_file_desc;
    struct flock lock;
    const char* obj_file_path;
    char ref_str[128];
    uint32 reserved_private_blocks;
    uint32 used_private_blocks;
    uint32 available_private_blocks;
    uint32 chunk_offset;
    uint32 system_blocks;
    uint32 used_blocks;    /* includes blocks used for bit maps */
    uint32 allocated_blocks;
    uint32 unallocated_blocks;
    uint32 total_blocks;
    uint32 total_private_chunks;
    uint32 used_private_chunks;
    uint8 block[FILE_BLOCK_SIZE];
    float divisor;
    char* range;
    Disk_Header disk_header;
    Chunk_Info* chunk_info;
    size_t sizeof_chunk_info;
    int i;


    if (argc > 2)
    {
	print_usage(argv[0]);
	exit(1);
    }

    if (argc > 1)
    {
	if (strcmp(argv[1], "--verbose") == 0)
	{
	    verbose = true;
	}
	else
	{
	    print_usage(argv[0]);
	    exit(1);
	}
    }

    obj_file_path = nwos_get_private_objects_path();

    obj_file_desc = open(obj_file_path, O_RDONLY);

    if (obj_file_desc < 0)
    {
	perror(obj_file_path);
	exit(1);
    }

    lock.l_type = F_RDLCK;
    lock.l_whence = SEEK_SET;
    lock.l_start = 0;
    lock.l_len = 0;

    if (fcntl(obj_file_desc, F_SETLK, &lock) != 0)
    {
	perror(obj_file_path);
	exit(1);
    }

    if (read(obj_file_desc, block, sizeof(block)) != sizeof(block))
    {
	snprintf(ref_str, sizeof(ref_str), "reading disk header from: %s", obj_file_path);
	perror(ref_str);
	exit(1);
    }

    memcpy(&disk_header, block, sizeof(disk_header));

    if (memcmp(disk_header.magic_number, MAGIC_NUMBER, 4) != 0)
    {
	fprintf(stderr, "Missing magic number in disk header: %s\n", obj_file_path);
	exit(1);
    }

    if (memcmp(disk_header.version_string, VERSION_STRING, 4) != 0)
    {
	fprintf(stderr, "Incorrect version string in disk header: %s\n", obj_file_path);
	exit(1);
    }

    printf("\n");
    printf("Last change: %u-%02u-%02u %02u:%02u:%02u  version: %c%c%c%c\n",
	   nwos_extract_year_from_time_stamp(disk_header.last_change),
	   nwos_extract_month_from_time_stamp(disk_header.last_change),
	   nwos_extract_day_of_month_from_time_stamp(disk_header.last_change),
	   nwos_extract_hour_from_time_stamp(disk_header.last_change),
	   nwos_extract_minute_from_time_stamp(disk_header.last_change),
	   nwos_extract_second_from_time_stamp(disk_header.last_change),
	   disk_header.version_string[0], disk_header.version_string[1],
	   disk_header.version_string[2], disk_header.version_string[3]);

    /* get what the disk thinks */
    nwos_4_uint8_to_uint32(disk_header.total_blocks, &reserved_private_blocks);
    nwos_4_uint8_to_uint32(disk_header.used_blocks, &used_private_blocks);
    nwos_4_uint8_to_uint32(disk_header.block_offset_to_chunks, &chunk_offset);
    nwos_4_uint8_to_uint32(disk_header.used_chunks, &used_private_chunks);

    sizeof_chunk_info = used_private_chunks * sizeof(Chunk_Info);
    chunk_info = malloc(sizeof_chunk_info);
    assert(chunk_info != NULL);

    if (read(obj_file_desc, chunk_info, sizeof_chunk_info) != sizeof_chunk_info)
    {
	snprintf(ref_str, sizeof(ref_str), "reading chunk_info from: %s", obj_file_path);
	perror(ref_str);
	exit(1);
    }

    assert(reserved_private_blocks % BLOCKS_IN_CHUNK == 0);

    total_private_chunks = reserved_private_blocks / BLOCKS_IN_CHUNK;

    used_blocks = 0;
    for (i = 0; i < (int)used_private_chunks; i++)
    {
#ifndef WORDS_BIGENDIAN
	chunk_info[i].ref = byteswap_uint32(chunk_info[i].ref);
	chunk_info[i].used = byteswap_uint16(chunk_info[i].used);
	chunk_info[i].index = byteswap_uint16(chunk_info[i].index);
#endif
	used_blocks += chunk_info[i].used;
    }

    if (used_blocks != used_private_blocks)
    {
	/* snprintf(log_msg, sizeof(log_msg), */
	fprintf(stderr,
		 "Warning: calculated sum of used blocks (%u) doesn't match stored (%u)",
		 used_blocks, used_private_blocks);
	/* nwos_log(log_msg); */
	/* fprintf(stderr, "%s\n", log_msg); */
    }

    printf("\n");
    printf("Calculating private blocks used...\n");


    used_blocks = calculate_blocks_used(obj_file_desc, chunk_offset, chunk_info, used_private_chunks);

    close(obj_file_desc);


    if (reserved_private_blocks < 4194304)    /* print in megabytes */
    {
	divisor = 1048576.0f / 256.0f;
	range = "Megabytes";
    }
    else  /* print in gigabytes */
    {
	divisor = 1073741824.0f / 256.0f;
	range = "Gigabytes";
    }

    available_private_blocks = reserved_private_blocks - used_private_blocks;

    system_blocks = BLOCKS_IN_CHUNK + total_private_chunks * BIT_MAP_BLOCKS;

    allocated_blocks = used_private_chunks * USABLE_BLOCKS_PER_CHUNK;
    unallocated_blocks = (total_private_chunks - used_private_chunks) * USABLE_BLOCKS_PER_CHUNK,

    total_blocks = reserved_private_blocks + BLOCKS_IN_CHUNK;

    printf("\n");

    printf("              Chunks   Blocks   %s\n", range);

    printf("System:        %5u  %9u    %5.1f  %5.1f%%\n",
	   1,
	   system_blocks,
	   (float)system_blocks / divisor,
	   100.0f * (float)system_blocks / (float)total_blocks);

    printf("Unallocated:   %5u  %9u    %5.1f  %5.1f%%\n",
	   total_private_chunks - used_private_chunks,
	   unallocated_blocks,
	   (float)unallocated_blocks / divisor,
	   100.0f * (float)unallocated_blocks / (float)total_blocks);

    printf("Allocated:     %5u\n", used_private_chunks);

    printf("  Used:               %9u    %5.1f  %5.1f%%\n",
	   used_private_blocks,
	   (float)used_private_blocks / divisor,
	   100.0f * (float)used_private_blocks / (float)total_blocks);

    printf("  Free:               %9u    %5.1f  %5.1f%%\n",
	   allocated_blocks - used_private_blocks,
	   (float)(allocated_blocks - used_private_blocks) / divisor,
	   100.0f * (float)(allocated_blocks - used_private_blocks) / (float)total_blocks);

    printf("  Total:              %9u    %5.1f  %5.1f%%\n",
	   allocated_blocks,
	   (float)allocated_blocks / divisor,
	   100.0f * (float)allocated_blocks / (float)total_blocks);

    printf("\n");


    display_graph(chunk_info, used_private_chunks);


    printf("\n");

    fflush(stdout);

    return 0;
}

