/*
--             This file is part of the New World OS project
--                 Copyright (C) 2006-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: prep_disk.c,v $
-- Revision 1.62  2008/09/06 15:34:23  jsedwards
-- Changed to use TEST_ENVIRONMENT_VARIABLE.
--
-- Revision 1.61  2008/09/06 13:36:50  jsedwards
-- Change pass phrase from TEST_PASS_PHRASE define to environment variable.
--
-- Revision 1.60  2008/09/01 18:24:14  jsedwards
-- Moved almost all of the code that prepared to create the root object into
-- the nwos_create_root function in objectify.c.
--
-- Revision 1.59  2008/08/31 17:58:26  jsedwards
-- Add assert around call to nwos_read_object_from_disk because it now returns
-- false when it fails, whereas before it would assert itself.
--
-- Revision 1.58  2008/08/30 12:46:16  jsedwards
-- Removed code and variables to read pass phrase and pass it to initialize,
-- and change parameters passed to initialize.
--
-- Revision 1.57  2008/07/19 15:40:20  jsedwards
-- Removed the #ifndef TEST_PASS_PHRASE around the verifying it that it is
-- okay to write over the storage.  Too dangerous!!
--
-- Revision 1.56  2008/07/19 11:17:08  jsedwards
-- Moved start_progress_bar, update_progress_bar, and finish_progress_bar
-- functions to their own file: progress_bar.c and renamed with nwos_ prefix.
--
-- Revision 1.55  2008/04/02 03:06:21  jsedwards
-- Added file locking.
--
-- Revision 1.54  2008/03/12 14:47:07  jsedwards
-- Added code to create all subdirectories in the path before writing the
-- private objects file.
--
-- Revision 1.53  2008/02/03 20:38:22  jsedwards
-- Change to pass DEFAULT_FILE to nwos_initialize_objectify instead of
-- private_path because passing a file now to initialize tells it that it
-- is a compressed file which isn't the case.
--
-- Revision 1.52  2008/02/03 01:15:40  jsedwards
-- Change to use nwos_get_private_objects_path function instead of DEFAULT_FILE.
--
-- Revision 1.51  2008/01/17 15:21:25  jsedwards
-- Changed to use progress bar when writing chunks to sparse file instead of
-- printing blocks written every now and then.
--
-- Revision 1.50  2008/01/17 15:07:22  jsedwards
-- Moved progress bar code here from disk_io.c and made into functions.  Added
-- a function pointer parameter to the allocate_all_hack function which is
-- called with progress.
--
-- Revision 1.49  2007/11/03 14:20:31  jsedwards
-- Added --allocate-all option that calls nwos_allocate_all_chunks_hack in
-- disk_io.c to allocate all available chunks, until I can fix the allocation
-- the right way.
--
-- Revision 1.48  2007/10/07 03:53:24  jsedwards
-- Renamed 'nwos_set_block_estimate' to 'nwos_check_blocks_available' and
-- changed to assert if it returs false (which should never happen).
--
-- Revision 1.47  2007/09/15 14:31:11  jsedwards
-- Added ifndef TEST_PASS_PHRASE around code requiring user input so it will
-- blindly plow ahead without it.
--
-- Revision 1.46  2007/09/02 19:47:25  jsedwards
-- Added call to set the block estimate.
--
-- Revision 1.45  2007/08/12 20:45:01  jsedwards
-- Change all of the "Encryption Level" stuff to "Security Level" because it
-- doesn't really change the encryption at all, all it does is change the
-- randomization of where objects are stored.
--
-- Revision 1.44  2007/08/10 00:03:36  jsedwards
-- Removed defintion of _LARGEFILE64_SOURCE, now using _FILE_OFFSET_BITS=64.
-- Also removed using O_LARGEFILE from open call.
--
-- Revision 1.43  2007/07/17 12:58:31  jsedwards
-- Changed print statement with off_t parameter to cast it to int64 so
-- compilers won't complain so much.
--
-- Revision 1.42  2007/07/13 21:01:52  jsedwards
-- Changed RELEASE_VERSION (which was defined in objectify.h) to PACKAGE_VERSION
-- which is generated by ./configure script in config.h.
--
-- Revision 1.41  2007/07/11 14:11:35  jsedwards
-- Added line to clear errno before writing chunks, to avoid it being non-zero
-- because of some previous call.
--
-- Revision 1.40  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.39  2007/06/28 14:09:15  jsedwards
-- Fix seg fault if no parameters were given on the command line.
--
-- Revision 1.38  2007/06/26 20:04:18  jsedwards
-- Add call to allocate the chunk for the root object.
--
-- Revision 1.37  2007/06/26 11:55:05  jsedwards
-- Changes for new non-wrapped, chunk indexed format (0023).
--
-- Revision 1.36  2007/06/25 15:07:11  jsedwards
-- Move call to agree with warranty after parameters are checked, so that it
-- doesn't ask about the warranty if the parameters are bad.
--
-- Revision 1.35  2007/06/25 15:02:15  jsedwards
-- Fix bug in convert_size so it correctly returns SIZE_ERROR when there is an
-- error.
--
-- Revision 1.34  2007/06/23 14:38:37  jsedwards
-- Removed #ifndef DONT_COPY_PUBLIC_BLOCKS statements that are no longer
-- necessary since we don't copy the public blocks into the private storage
-- anymore.
--
-- Revision 1.33  2007/06/20 13:35:08  jsedwards
-- Remove #if 0 and #endif accidentally left in.
--
-- Revision 1.32  2007/06/20 03:48:44  jsedwards
-- Added function to figure out how many blocks are available on a disk or
-- disk partition and use it to calculate the number of blocks up front.
--
-- Revision 1.31  2007/06/19 18:52:32  jsedwards
-- Removed all stuff related to public objects which are now stored separately
-- from the private objects.
--
-- Revision 1.30  2007/04/15 16:17:39  jsedwards
-- Fix bug which occurred when only allocating 1 chunk to public data.  The
-- buffer didn't get zeroed out so it wrote the public data into the private
-- area which trashed the bit maps, etc.
--
-- Revision 1.29  2007/04/14 15:42:25  jsedwards
-- Changed to handle errors in size specification more gracefully.
--
-- Revision 1.28  2007/04/14 12:53:38  jsedwards
-- Change to create or turncate the storage when it is opened for writing,
-- if it is a file.
--
-- Revision 1.27  2007/04/14 12:50:10  jsedwards
-- Removed commented out variable.
--
-- Revision 1.26  2007/04/14 12:49:21  jsedwards
-- Add code to break out of write_chunk loop if an error occurred, otherwise
-- the warning about a chunk not being a multiple of 512 will be triggered by
-- an error.
--
-- Revision 1.25  2007/04/14 12:44:52  jsedwards
-- Changed name of 'extra_argument' boolean variable to 'is_file' which is
-- hopefully more descriptive of what it means.
--
-- Revision 1.24  2007/04/14 12:39:05  jsedwards
-- Only write the remaining public blocks if there are more than 1 chunk.
--
-- Revision 1.23  2007/04/14 12:36:49  jsedwards
-- Move storage open after asking yes or no to write it.
--
-- Revision 1.22  2007/03/27 11:36:41  jsedwards
-- Change to have separate public and private last change times.
--
-- Revision 1.21  2007/03/14 13:15:12  jsedwards
-- Added LARGEFILE define for Linux.
--
-- Revision 1.20  2007/02/11 16:58:26  jsedwards
-- Changed so DEFAULT_TYPE has to specify RO (Read-Only) or RW (Read-Write).
--
-- Revision 1.19  2007/01/02 11:20:23  jsedwards
-- Changed to not create a private root object class because I can't think of
-- any reason to create one right now.  There isn't a list of root objects
-- in the private area.
--
-- Revision 1.18  2006/12/30 03:50:13  jsedwards
-- Removed assert that verified the NUM_PUBLIC_IDS was a multiple of
-- BLOCKS_IN_CHUNK.  NUM_PUBLIC_IDS no longer used.
--
-- Revision 1.17  2006/12/28 23:15:54  jsedwards
-- Added logging of block counts.
--
-- Revision 1.16  2006/12/27 14:16:42  jsedwards
-- Added some log messages.
--
-- Revision 1.15  2006/12/25 12:20:54  jsedwards
-- Ifdef out set_public_root_object function when compiled "don't copy public"
-- mode.
--
-- Revision 1.14  2006/12/21 13:08:36  jsedwards
-- Hack to make it compile with new split public and private classes,
-- NON-FUNCTIONAL!
--
-- Revision 1.13  2006/12/20 12:33:07  jsedwards
-- Changed to have two class definition refs, public and private.
--
-- Revision 1.12  2006/12/16 13:08:23  jsedwards
-- Added option to not create the private root and class definition objects.
--
-- Revision 1.11  2006/12/16 12:39:42  jsedwards
-- Added disclaimers back in.
--
-- Revision 1.10  2006/12/15 15:02:47  jsedwards
-- Changed back to only having two areas on the disk, public and private
-- encrypted.
--
-- Revision 1.9  2006/12/14 22:30:43  jsedwards
-- Add ifdef to build version that doesn't copy the public blocks or setup
-- the private root and class definition objects.
--
-- Revision 1.8  2006/12/14 14:21:31  jsedwards
-- Added code to copy public objects from file and setup private root and
-- class defintion.
--
-- Revision 1.7  2006/12/05 14:57:14  jsedwards
-- Moved root object creation code from big_bang.c (also #if'd out).
--
-- Revision 1.6  2006/12/05 14:23:49  jsedwards
-- Moved user warnings and security stuff from big_bang.c to here (#if'd out).
--
-- Revision 1.5  2006/12/04 14:55:44  jsedwards
-- Moved next available private chunk to bytes 52-55 in header.
--
-- Revision 1.4  2006/12/04 04:44:23  jsedwards
-- Changed for new header layout.
--
-- Revision 1.3  2006/12/03 17:56:56  jsedwards
-- Changed for the new disk layout: first section contains public objects,
-- second section private unencrypted objects, and third section private
-- encrypted objects.
--
-- Revision 1.2  2006/10/26 01:51:28  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.5  2006/10/22 12:32:30  jsedwards
-- Added code to handle variable number of reserved public blocks.
--
-- Revision 1.1.2.4  2006/10/09 13:10:35  jsedwards
-- Change bit map size from 256 to 8192 (to improve speed, disk_usage program
-- took over an hour to run with 256 byte bit maps).
--
-- Revision 1.1.2.3  2006/10/07 22:28:40  jsedwards
-- Changed to write block bit map into the first block of every 2048 block
-- chunk.
--
-- Revision 1.1.2.2  2006/09/26 20:15:51  jsedwards
-- Added warning and verification before nuking the disk or partition.
--
-- Revision 1.1.2.1  2006/09/26 14:03:50  jsedwards
-- Kludged program to prepare a disk or partition for being used with
-- objectify.
--
*/

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "objectify_private.h"



/**********************************************************/
/* Determine if path is a regular file or a block device. */
/**********************************************************/

bool is_device(const char* path)
{
    assert(path[0] == '/');   /* must be a full path specified */

    return path[1] == 'd' && path[2] == 'e' && path[3] == 'v' && path[4] == '/';
}


/***************************************************************/
/* figure out how many blocks are available on a block device. */
/***************************************************************/

unsigned max_blocks_on_device(const char* path)
{
    int fd;
    off_t lower = 0LL;
    off_t upper = 1099511627776LL;
    off_t middle;
    off_t result;
    unsigned num_sectors;


    fd = open(path, O_RDONLY);

    if (fd < 0)
    {
	perror(path);
	exit(1);
    }

    while ((upper - lower) > 512)
    {
	middle = lower + (upper - lower) / 2;

	result = lseek(fd, middle, SEEK_SET);

	/* printf("%zd: %zd\n", middle, result); */

	if (result < 0)
	{
	    upper = middle;
	}
	else
	{
	    lower = middle;
	}
    }

    close(fd);

    num_sectors = (unsigned) (lower / 512);

    if (((off_t)num_sectors * 512) != lower)
    {
	printf("something is wrong, not a multiple of 512 bytes: %lld\n", (int64)lower);
    }

    return num_sectors * 2;
}


/****************************************************/
/* Print the usage information, is can be different */
/* depending upon if writing to a file or device.   */
/****************************************************/

void print_usage(char* name, bool specify_private_size)
{
    assert((CHUNK_SIZE % 1048576) == 0);  /* chunks must be a multiple of 1M */

    if (specify_private_size == false)
    {
	fprintf(stderr, "usage: %s [--no-private]\n\n", name);
    }
    else
    {
	fprintf(stderr, "usage: %s [--no-private] size\n\n", name);
    }

    fprintf(stderr, "  --no-private: don't create the initial private blocks (if importing old data)\n");

    if (specify_private_size == true)
    {
	fprintf(stderr, "  size can be specified in %dM chunks, megabytes 'M', or gigabytes 'G'.\n", CHUNK_SIZE / 1048576);
	fprintf(stderr, "  it must be a multiple of %dM.\n\n", CHUNK_SIZE / 1048576);
    }
}


void goodbye()
{
    fprintf(stderr, "\n");
    fprintf(stderr, "I'm sorry you don't wish to use this software.  If you have any comments,\n");
    fprintf(stderr, "questions, or concerns please e-mail me at `j.scott.edwards.nwos@gmail.com'.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "-------->   Program terminated without preparing disk!   <--------\n");
    fprintf(stderr, "\n");
    exit(1);
}


void agree_no_warranty()
{
    char buffer[16];

                  /*          1         2         3         4         5         6         7         8 */
                  /* 12345678901234567890123456789012345678901234567890123456789012345678901234567890 */
    fprintf(stderr, "\n");
    fprintf(stderr, "Objectify version %s\n", PACKAGE_VERSION);
    fprintf(stderr, "\n");
    fprintf(stderr, "Copyright (C) 2005-2008  J. Scott Edwards - QRW Software\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "This software comes with ABSOLUTELY NO WARRANTY; THE ENTIRE RISK AS TO THE\n");
    fprintf(stderr, "QUALITY AND PERFORMANCE OF THIS PROGRAM IS WITH YOU.  For details read the\n");
    fprintf(stderr, "`GNU General Public License' described in the file `LICENSE', which should\n");
    fprintf(stderr, "have been distributed with this software.  If it is not included, it can be\n");
    fprintf(stderr, "read online at `http://www.gnu.org/copyleft/gpl.html'.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "This is free software, and you are welcome to redistribute it under certain\n");
    fprintf(stderr, "conditions; also see the `LICENSE' file for details.  If you have any\n");
    fprintf(stderr, "comments or questions you can e-mail me at `j.scott.edwards.nwos@gmail.com'\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "If you wish to use this program, enter \"I agree\" at the prompt.  By doing\n");
    fprintf(stderr, "so you agree to the terms specified in the GNU General Public License.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Do you agree with the license terms of this software? ");
    fflush(stderr);

    fgets(buffer, sizeof(buffer), stdin);

    if (strcmp(buffer, "I agree\n") != 0) goodbye();

    fprintf(stderr, "\n");
    fprintf(stderr, "*******************************************************************************\n");
}


void understand_security()
{
    char buffer[16];

    fprintf(stderr, "\n");
    fprintf(stderr, "*******************************************************************************\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "In addition to the above there are two things you should be aware of before\n");
    fprintf(stderr, "using this software:\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "1) This software attempts to make your data secure by encrypting the data\n");
    fprintf(stderr, "when it is written to the disk.  This should make your data secure, but I\n");
    fprintf(stderr, "am NOT a security expert and I CANNOT guarantee your data cannot be read\n");
    fprintf(stderr, "by someone if they can access your disk drive.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "2) This version of Objectify uses a long pass phrase (a minimum of 34\n");
    fprintf(stderr, "characters) to encrypt the objects.  Be aware that if you forget or lose\n");
    fprintf(stderr, "the pass phrase, it is UNLIKELY you will be able to recover your data.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "If you still wish to use this program, enter \"I understand\" at the prompt.\n");
    fprintf(stderr, "By doing so you acknowledge that you understand that the security methods in\n");
    fprintf(stderr, "this program are not guaranteed.  You also acknowledge that if you forget or\n");
    fprintf(stderr, "otherwise lose the pass phrase, YOUR DATA WILL BE LOST FOREVER.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Do you understand this? ");
    fflush(stderr);

    fgets(buffer, sizeof(buffer), stdin);

    if (strcmp(buffer, "I understand\n") != 0) goodbye();

    fprintf(stderr, "\n");
    fprintf(stderr, "*******************************************************************************\n");
}


/********************************************/
/* Convert size string to number of blocks. */
/* Must be a multiple of chunk_size.        */
/********************************************/

#define SIZE_ERROR 0xffffffff      /* return value if error occurred */

uint32 convert_size(char* size)
{
    char* p;
    uint32 result = 0;

    assert(strlen(size) > 0);   /* bug if called with empty string */

    for (p = size; isdigit(*p); p++)
    {
	result = result * 10 + (*p - '0');
    }

    if (toupper(*p) == 'M')
    {
	p++;   /* after this point we want the value in megabytes so just ignore it */
    }
    else if (toupper(*p) == 'G')
    {
	result = result * 1024;    /* change gigabytes to megabytes */
	p++;
    }
    else
    {
	result = result * (CHUNK_SIZE / 1048576);  /* change chunks to megabytes */
	p++;
    }

    /* at this point the result in in megabytes */

    if (!isdigit(*size) || *p != 0)
    {
	fprintf(stderr, "Invalid size: %s\n", size);
	return SIZE_ERROR;
    }

    if (result == 0 || (result % (CHUNK_SIZE / 1048576)) != 0)
    {
	fprintf(stderr, "Size must be a multiple of: %dM\n", CHUNK_SIZE / 1048576);
	return SIZE_ERROR;
    }

    if (result < 2 * CHUNK_SIZE / 1048576)
    {
	fprintf(stderr, "Size must be equal to, or larger than %dM\n", 2 * CHUNK_SIZE / 1048576);
	return SIZE_ERROR;
    }

    if (result > 1048576)
    {
	fprintf(stderr, "Size must be less than a 1024 gigabytes\n");
	return SIZE_ERROR;
    }

    return result * (1048576 / FILE_BLOCK_SIZE);   /* convert megabytes to blocks */
}


/********************************************************************/
/* Write the given number of chunks to the device or file.          */
/* If num_chunks == 0 it means write until EOF.  Returns the number */
/* blocks written (NOT chunks) because the last may not be even.    */
/********************************************************************/

uint32 write_chunks(int fd, uint8* chunk, uint32 num_chunks)
{
    size_t bytes_written;
    uint32 blocks = 0;

    nwos_start_progress_bar();

    errno = 0;

    while (num_chunks == 0 || blocks < (num_chunks * BLOCKS_IN_CHUNK))
    {
	nwos_update_progress_bar((float)blocks / (float)(num_chunks * BLOCKS_IN_CHUNK));

	bytes_written = write(fd, chunk, CHUNK_SIZE);

	if (bytes_written < 0) break;    /* exit if error occurred */

	if (bytes_written % 512 != 0)
	{
	    fprintf(stderr, "bytes written not a multiple of 512: %zd\n", bytes_written);
	    close(fd);
	    exit(1);
	}

	blocks += bytes_written / 256;

	if (bytes_written < CHUNK_SIZE) break;
    }

    if (errno != 0)
    {
	printf("\n");  /* don't finish progress bar */
	fflush(stdout);

	perror("after last write");
	exit(1);
    }

    nwos_finish_progress_bar();

    return blocks;
}


/***************************************************************************/
/* Program to prepare a disk or file for NWOS storage.                     */
/* Now there is a bit map of used blocks as the first block in every chunk */
/* 8192 * 8 = 65536 blocks * 256 bytes per block = 16777216 bytes in chunk */
/***************************************************************************/

int main(int argc, char* argv[])
{
    int obj_fd;
    const char* private_path;
    struct flock lock;
    uint8 *chunk;
    char answer[8];
    int arg_index = 1;
    bool is_file = false;
    bool do_private_blocks = true;

    uint32 total_blocks;
    uint32 reserved_private_blocks;
    uint32 blocks = 0;
    uint32 blocks_written;
    uint32 num_chunks;
    Disk_Header disk_header;
    char msg[128];
    char* p;
#if 0
    ObjRef root_object_reference;
    bool upgrade = false;
//    struct stat stat_struct;
#endif

    printf("\n");

    nwos_log_arguments(argc, argv);


    /**************************/
    /* sort out the arguments */
    /**************************/

    private_path = nwos_get_private_objects_path();

    if (!is_device(private_path))
    {
	is_file = true;

	/* NOTE: this is a bad thing to do, it temporarily modifies */
	/* the string returned by nwos_get_private_objects_path.    */
        /* We cheat the system by calling strchr.                   */

	p = strchr(private_path, '/');
	assert(p != NULL);
	p = strchr(p + 1, '/');
	while (p != NULL)
	{
	    *p = '\0';
	    if (mkdir(private_path, 0755) != 0 && errno != EEXIST)
	    {
		perror(p);
		exit(1);
	    }
	    *p = '/';

	    p = strchr(p + 1, '/');
	}
    }

    if (argc > 1 && *argv[arg_index] == '-')   /* option specified */
    {
	if (strcmp(argv[arg_index], "--no-private") == 0)
	{
	    do_private_blocks = false;
	}
	else
	{
	    fprintf(stderr, "\nunknown option: %s\n\n", argv[arg_index]);
	    print_usage(argv[0], is_file);
	    exit(1);
	}


	arg_index++;
    }


    /* must have 0 or 1 arguments */

    if (is_file)
    {
	if (argc - arg_index != 1)
	{
	    print_usage(argv[0], true);
	    exit(1);
	}
    }
    else
    {
	if (argc - arg_index != 0)
	{
	    print_usage(argv[0], false);
	    exit(1);
	}
    }


    /* convert arguments */


    if (!is_file)
    {
	total_blocks = max_blocks_on_device(private_path);   /* zero means unlimited */

	num_chunks = total_blocks / BLOCKS_IN_CHUNK;

	/* round it to an even number of chunks */
	total_blocks = num_chunks * BLOCKS_IN_CHUNK;
    }
    else
    {
	/* use one chunk for header and block map */
	total_blocks = convert_size(argv[arg_index]);

	if (total_blocks == 0 || total_blocks == SIZE_ERROR)
	{
	    print_usage(argv[0], is_file);
	    exit(1);
	}
    }

    if (getenv(TEST_ENVIRONMENT_VARIABLE) == NULL)
    {
	agree_no_warranty();   /* make sure he or she agrees with the licensing */
    }

    reserved_private_blocks = total_blocks - BLOCKS_IN_CHUNK;  /* first chunk is for header */

    printf("\ntotal chunks: %u  private chunks: %u\n",
	   total_blocks / BLOCKS_IN_CHUNK, reserved_private_blocks / BLOCKS_IN_CHUNK);


    chunk = malloc(CHUNK_SIZE);

    if (chunk == NULL)
    {
	perror("allocating chunk");
	exit(1);
    }

    printf("\n");
    printf("WARNING: all of the contents of %s will be ERASED!!!\n", private_path);
    printf("\n");
    printf("Do you want to continue? (enter `yes' to erase %s) ", private_path);

    fflush(stdout);

    fgets(answer, sizeof(answer), stdin);

    printf("\n");

    if (strcmp(answer, "yes\n") != 0)
    {
	exit(1);
    }

    /***************************/
    /* Open the storage device */
    /***************************/

    if (is_file)
    {
	obj_fd = open(private_path, O_WRONLY | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR);
    }
    else
    {
	obj_fd = open(private_path, O_WRONLY);    /* don't need large file if not seeking? */
    }

    if (obj_fd < 0)
    {
	perror(private_path);
	exit(1);
    }


    lock.l_type = F_WRLCK;
    lock.l_whence = SEEK_SET;
    lock.l_start = 0;
    lock.l_len = 0;

    if (fcntl(obj_fd, F_SETLK, &lock) != 0)
    {
	perror(private_path);
	exit(1);
    }



    /*********************************/
    /* finally write the disk header */
    /*********************************/

    memset(&disk_header, 0, sizeof(disk_header));
    
    memcpy(disk_header.magic_number, MAGIC_NUMBER, 4);
    memcpy(disk_header.version_string, VERSION_STRING, 4);

    nwos_get_time_stamp(disk_header.last_prep_disk);

    memcpy(disk_header.last_change, disk_header.last_prep_disk, sizeof(disk_header.last_change));

    nwos_uint32_to_4_uint8(&reserved_private_blocks, disk_header.total_blocks);


    /* compute the number of private blocks used */

    assert(reserved_private_blocks % BLOCKS_IN_CHUNK == 0);

    num_chunks = reserved_private_blocks / BLOCKS_IN_CHUNK;  /* round up */

    /* set the number of used blocks to 0 */
    blocks = 0;
    nwos_uint32_to_4_uint8(&blocks, disk_header.used_blocks);

    /* set the number of used chunks to 0 */
    blocks = 0;
    nwos_uint32_to_4_uint8(&blocks, disk_header.used_chunks);

    /* for now set the chunk offset to 1 chunk (for the header) */
    blocks = BLOCKS_IN_CHUNK;
    nwos_uint32_to_4_uint8(&blocks, disk_header.block_offset_to_chunks);

    memset(chunk, 0, CHUNK_SIZE);

    memcpy(chunk, &disk_header, sizeof(disk_header));

    /* if (write(obj_fd, chunk, FILE_BLOCK_SIZE) != FILE_BLOCK_SIZE) */
    if (write(obj_fd, chunk, CHUNK_SIZE) != CHUNK_SIZE)
    {
	perror(private_path);
	exit(1);
    }


    /*************************************/
    /* write private portion of the disk */
    /*************************************/

    if (is_file)
    {
	printf("Writing private chunks...\n");
	fflush(stdout);

	memset(chunk, 0, CHUNK_SIZE);

	/* private blocks have bit maps so mark off 32 bits for the bit map itself */
	chunk[0] = 0xff;
	chunk[1] = 0xff;
	chunk[2] = 0xff;
	chunk[3] = 0xff;

	blocks_written = write_chunks(obj_fd, chunk, reserved_private_blocks / BLOCKS_IN_CHUNK);

	if (blocks_written < reserved_private_blocks)
	{
	    fprintf(stderr, "Error: not enough space for private chunks\n");
	    close(obj_fd);
	    exit(1);
	}

	blocks += blocks_written;
	printf("blocks: %u  blocks_written: %u  reserved_private_blocks: %u\n", 
	       blocks, blocks_written, reserved_private_blocks);

	assert(blocks == reserved_private_blocks + BLOCKS_IN_CHUNK);
    }

    if (close(obj_fd) != 0)
    {
	perror(private_path);
	exit(1);
    }

    printf("Total chunks: %u\n", num_chunks);


    snprintf(msg, sizeof(msg), "   private blocks: %9u  used: %9u", reserved_private_blocks, blocks);
    nwos_log(msg);


    if (do_private_blocks)
    {
	if (getenv(TEST_ENVIRONMENT_VARIABLE) == NULL)
	{
	    understand_security();   /* make sure he or she understands the risks */
	}

	nwos_create_root();
    }

    return 0;
}

