/*
--             This file is part of the New World OS project
--                 Copyright (C) 2007-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: acronym.c,v $
-- Revision 1.5  2008/09/01 00:10:48  jsedwards
-- Fix copyright year.  NO code changes.
--
-- Revision 1.4  2008/08/31 21:53:51  jsedwards
-- Added an assert around calls to nwos_read_variable_sized_object_from_disk
-- and nwos_read_object_from_disk because now they return false when they fail
-- instead of asserting themselves.
--
-- Revision 1.3  2007/07/01 19:56:15  jsedwards
-- Remove spurious line accidentally left in.
--
-- Revision 1.2  2007/07/01 19:44:11  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.1  2007/05/24 11:46:21  jsedwards
-- Initial version.
--
*/


#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>   /* define memset */

#include "crc32.h"
#include "objectify.h"
#include "objectify_private.h"


static ObjRef* get_public_acronym_class_ref()
{
    static ObjRef class_ref;

    if (is_void_reference(&class_ref))
    {
	nwos_find_public_class_definition("ACRONYM", &class_ref);

	assert(!is_void_reference(&class_ref));
    }

    return &class_ref;
}


/*****************************************************************************/

bool nwos_find_public_acronym(char* acronym, ObjRef* ref)
{
    EveryObject spelling_header;
    ObjRef* acronym_class_ref_ptr;
    ObjRef spelling_ref;
    ObjRef object_class;
    ReferenceList* spelling_ref_list;
    int num_spellings;
    int j;
    bool result = false;

    if (nwos_find_public_spelling(acronym, &spelling_ref))
    {
	acronym_class_ref_ptr = get_public_acronym_class_ref();

	nwos_read_object_headers_from_disk(&spelling_ref, &spelling_header);

	spelling_ref_list = nwos_malloc_reference_list(&spelling_header.object.references);

	num_spellings = spelling_ref_list->common_header.num_refs;

	for (j = 0; j < num_spellings; j++)
	{
	    nwos_get_object_class(&spelling_ref_list->references[j], &object_class);   /* find out what kind of object it is */
	    if (is_same_object(&object_class, acronym_class_ref_ptr))   /* it is a name object */
	    {
		copy_reference(ref, &spelling_ref_list->references[j]);
		result = true;
		break;
	    }
	}

	nwos_free_reference_list(spelling_ref_list);
	spelling_ref_list = NULL;
    }

    return result;
}


/*****************************************************************************/

#ifdef PUBLIC_MODE
void nwos_create_public_acronym(char* acronym, ObjRef* ref)
{
    C_struct_Acronym acronym_obj;
    ObjRef* acronym_class_ref_ptr;
    ObjRef spelling_ref;

    assert(!nwos_find_public_acronym(acronym, ref));

    acronym_class_ref_ptr = get_public_acronym_class_ref();

    nwos_create_public_spelling(acronym, &spelling_ref);

    nwos_generate_new_id(ref);

    nwos_fill_in_common_header(&acronym_obj.header.common, ref, acronym_class_ref_ptr);

    copy_reference(&acronym_obj.spelling, &spelling_ref);
    nwos_add_to_references(ref, &spelling_ref);

    nwos_create_reference_list(ref, &acronym_obj.header.object.references);

    nwos_crc32_calculate((uint8*) &acronym_obj.header.object, sizeof(ObjectHeader), acronym_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &acronym_obj.spelling, sizeof(acronym_obj) - sizeof(EveryObject), acronym_obj.header.common.data_chksum);


    nwos_write_object_to_disk(ref, &acronym_obj, sizeof(acronym_obj));

    nwos_add_to_references(ref, acronym_class_ref_ptr);
}
#endif


/*****************************************************************************/

/* This is horrible - FIX THIS!! */

static size_t get_spelling_object_size(void* spelling_obj)
{
    assert(((C_struct_Spelling*)spelling_obj)->count > 0);

    return sizeof(C_struct_Spelling) + ((C_struct_Spelling*)spelling_obj)->count;
}


bool nwos_acronym_to_string(ObjRef* ref, char* string, size_t size)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Spelling* ptr_spelling_obj = (C_struct_Spelling*) kludge;
    C_struct_Acronym acronym_obj;
    int j;

    assert(nwos_read_object_from_disk(ref, &acronym_obj, sizeof(acronym_obj)));

    assert(nwos_read_variable_sized_object_from_disk(&acronym_obj.spelling, kludge, sizeof(kludge), &get_spelling_object_size));

    if (size < ptr_spelling_obj->count + 1) return false;

    for (j = 0; j < ptr_spelling_obj->count; j++) 
    {
	string[j] = toupper(ptr_spelling_obj->storage[j]);
    }

    string[j++] = '\0';

    return true;
}


