/*
--             This file is part of the New World OS project
--                 Copyright (C) 2004-2009  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: disk_io.c,v $
-- Revision 1.134  2009/04/04 15:12:46  jsedwards
-- Moved call to nwos_terminate_chunk_info so that it is always done, even
-- if the archive hasn't been modified.
--
-- Revision 1.133  2009/04/03 13:03:40  jsedwards
-- Added logging of public objects path and private objects path.
--
-- Revision 1.132  2009/03/14 22:57:13  jsedwards
-- Added includes for backup.h and chunk_info.h files.
-- Moved nwos_next_public_ref to gen_id.c file.
--
-- Revision 1.131  2009/03/14 13:45:08  jsedwards
-- Added assert to nwos_read_block to verify that the reference id of the
-- block read matches the reference id requested.
--
-- Revision 1.130  2009/03/14 11:43:58  jsedwards
-- Added includes for gen_id.h and user_config.h files.
--
-- Revision 1.129  2009/03/14 03:31:24  jsedwards
-- Added include of bit_map.h file.
--
-- Revision 1.128  2009/03/13 12:27:54  jsedwards
-- Added includes for config.h, log.h, mem_alloc.h and strlcxx.h files. Also
-- renamed nwos_remove_object function and changed to call test_bit_in_map and
-- read_block instead of nwos_object_exists function.
--
-- Revision 1.127  2009/03/09 01:05:56  jsedwards
-- Moved block_estimate, last_used_blocks, last_ref, chunk_density_target,
-- chunk_skip_forward, log_estimate_results variables and
-- initialize_random_number_generator, generate_new_completely_random_id
-- nwos_generate_new_completely_random_id, nwos_restart_id_generation
-- nwos_set_allocation_parameters, nwos_check_blocks_available
-- nwos_generate_new_closely_spaced_id, nwos_generate_new_1_in_N_id
-- nwos_generate_new_public_id functions to gen_id.c file.
--
-- Revision 1.126  2009/03/08 21:18:38  jsedwards
-- Fixed to compile correctly in PUBLIC_MODE again.
--
-- Revision 1.125  2009/03/08 02:04:47  jsedwards
-- Changed include objectify_private.h to disk_io.h and added new
-- nwos_archive_version and nwos_backup_is_enabled functions.
--
-- Revision 1.124  2009/03/06 05:23:15  jsedwards
-- Moved bit map functions to new bit_map.c file.
--
-- Revision 1.123  2009/03/04 15:24:31  jsedwards
-- Moved nwos_hash_uint32_ref, nwos_hash_ref, and ref_to_offset functions to
-- the gen_id.c file.
--
-- Revision 1.122  2009/03/04 14:58:19  jsedwards
-- Moved uint32_ref_to_info_index function to chunk_info.c.
--
-- Revision 1.121  2009/03/02 15:22:41  jsedwards
-- Changed so write_empty_chunk function adjusts the total chunks and blocks
-- if the archive is in a file (instead of a block device).
--
-- Revision 1.120  2009/03/02 13:46:40  jsedwards
-- Renamed archive_is_block_device function to nwos_archive_is_block_device and
-- archive_is_file function to nwos_archive_is_file.
--
-- Revision 1.119  2009/03/02 13:08:55  jsedwards
-- Deleted find_first_low_density_in_chunk and find_starting_chunk functions,
-- and bytes_written variable that weren't used.
--
-- Revision 1.118  2009/03/02 12:30:46  jsedwards
-- Changed to use write_empty_chunk to expand file archive instead of inline
-- code.
--
-- Revision 1.117  2009/03/01 17:33:03  jsedwards
-- Moved functions and variables related to the chunk_info table into a new
-- chunk_info.c file.
--
-- Revision 1.116  2009/02/28 03:55:36  jsedwards
-- Changed to expand size of archive file if need more chunks.  Untested!!
--
-- Revision 1.115  2009/02/25 14:46:31  jsedwards
-- Deleted chunk_skip_backward global variable which was not used anywhere and
-- define of UBPC which also wasn't used anywhere.
--
-- Revision 1.114  2008/12/30 13:15:15  jsedwards
-- Put the log entries back in for 1.109, 1.110, and 1.111 which I accidentally
-- left out in yesterdays check in.  NO code changes.
--
-- Revision 1.113  2008/12/29 13:49:17  jsedwards
-- Removed the changes made in revisions 1.109, 1.110, and 1.111 which were
-- experimental and in retrospect should have been done in a branch.  This
-- revision is identical to revision 1.108.2.3 from the alpha_29_5_branch.
--
-- Revision 1.112  2008/12/27 16:10:16  jsedwards
-- Merged from alpha_29_5_branch back into main trunk.
--
-- Revision 1.108.2.3  2008/12/26 16:24:24  jsedwards
-- Changed nwos_read_block so offset is set to -1 when it is declared and
-- removed code that specifically set it to -1 in different cases.
--
-- Revision 1.108.2.2  2008/12/26 16:10:25  jsedwards
-- Fix Bug #2354266 - Somtimes it aborts when entering wrong pass phrase.
-- Added code to nwos_read_block so that it returns false if the block is
-- empty (I.E. the bit isn't set in the bit map).
--
-- Revision 1.108.2.1  2008/12/23 11:03:15  jsedwards
-- Changed to use strlcpy instead of strcpy.
--
-- Revision 1.111  2008/11/04 14:07:15  jsedwards
-- EXPERIMENTAL - Does not work! - Added check_chunk_density function, incomplete.
--
-- Revision 1.110  2008/11/03 10:39:13  jsedwards
-- EXPERIMENTAL - Does not work! - Deleted analyze_chunk_density, etc.
--
-- Revision 1.109  2008/11/02 14:00:58  jsedwards
-- EXPERIMENTAL - Does not work! - Added analyze_chunk_density, etc.
--
-- Revision 1.108  2008/11/01 10:22:09  jsedwards
-- Missed one change of RESERVED_PUBLIC_BLOCKS to MAXIMUM_PUBLIC_REFERENCE in
-- previous check in.
--
-- Revision 1.107  2008/10/31 13:43:04  jsedwards
-- Moved nwos_reference_type and is_valid_private_reference functions from
-- objectify.c and renamed the latter to nwos_is_private_reference.  Added new
-- nwos_is_public_reference and nwos_is_temporary_reference functions.  Changed
-- all references to RESERVED_PUBLIC_BLOCKS to MAXIMUM_PUBLIC_REFERENCE or
-- MINIMUM_PRIVATE_REFERENCE and MAXIMUM_VALID_PRIVATE_REFERENCE to
-- MAXIMUM_PRIVATE_REFERENCE.
--
-- Revision 1.106  2008/10/28 10:23:52  jsedwards
-- Changed skip_forward and skip_backwards parameters to unsigned.
--
-- Revision 1.105  2008/10/27 14:17:36  jsedwards
-- Changed so that if backup directory is an empty string (as it is if the
-- environment variable is defined but empty) it does not make a backup.
--
-- Revision 1.104  2008/10/27 04:07:42  jsedwards
-- Remove new line character from log message.
--
-- Revision 1.103  2008/10/27 04:06:39  jsedwards
-- Changed nwos_check_blocks_available to search backwards for a new chunk
-- when it reaches the end of the private chunk space.
--
-- Revision 1.102  2008/10/27 01:48:18  jsedwards
-- Changed 'index' in Bit_Map_Cache_Entry structure to 'chunk_index' so that it
-- is a chunk index instead an index into the chunk_info table.  This was
-- necessary because if the chunk_info table has a new entry inserted the bit
-- map cache will have the wrong indexes.
--
-- Revision 1.101  2008/10/27 00:48:21  jsedwards
-- Fixed spacing that was screwed up when the chunk_index_to_info_index was
-- moved in the previous revision.  NO code changes.
--
-- Revision 1.100  2008/10/27 00:20:38  jsedwards
-- Moved chunk_index_to_info_index function so that it appears before the
-- write_bit_map function that will need it after the next change.
--
-- Revision 1.99  2008/10/27 00:09:20  jsedwards
-- Doh!  Put the Bug # back into the log entry for revision 1.96.  This version
-- is the same as revision 1.97 with only the log entries for 1.98 and 1.99
-- added.  NO code changes!
--
-- Revision 1.98  2008/10/26 23:49:01  jsedwards
-- Removed line added to log in last revision, I had a brain cramp and added it
-- but then realized that change did not fix the bug.  This version is the same
-- as version 1.96 except for the logs for 1.97 and 1.98.  NO code changes.
--
-- Revision 1.97  2008/10/26 18:07:53  jsedwards
-- Added bug number to previous log entry (Revision 1.96).  NO code changes.
--
-- Revision 1.96  2008/10/26 17:11:10  jsedwards
-- Fix Bug #2184020 - Assert when adding files and chunk is near upper limit.
-- Fixed test to see if it is going to allocate a block beyond the maximum
-- valid private reference.
--
-- Revision 1.95  2008/09/25 13:46:41  jsedwards
-- Fix Bug #2126403 - changed to not process chunk_info table when using a
-- compressed file.
--
-- Revision 1.94  2008/09/19 13:11:55  jsedwards
-- Added a warning if any chunks are located above MAXIMUM_VALID_PRIVATE_REFERENCE.
--
-- Revision 1.93  2008/09/01 18:08:19  jsedwards
-- Added chunk_info_reverse_index to get from chunk number back to an index
-- into the chunk_info table.
--
-- Revision 1.92  2008/09/01 16:18:10  jsedwards
-- Removed nwos_allocate_all_chunks_hack, no longer used.  Removing the
-- --allocate-all option from prep_disk.
--
-- Revision 1.91  2008/09/01 15:44:01  jsedwards
-- Fix bug in nwos_allocate_new_chunk where it wasn't searching the chunk_info
-- table to insert new chunk.
--
-- Revision 1.90  2008/09/01 02:57:40  jsedwards
-- Fix so that if hash isn't found or if block isn't in compressed index
-- nwos_read_block returns false.
--
-- Revision 1.89  2008/08/10 15:16:38  jsedwards
-- Added printing of path in "Missing magic number" error message.
--
-- Revision 1.88  2008/08/10 15:07:26  jsedwards
-- Added path in which error occurred to "Incorrect version string" message.
--
-- Revision 1.87  2008/07/19 13:34:12  jsedwards
-- Added new disk_io_is_read_only and disk_io_is_read_write functions.
--
-- Revision 1.86  2008/07/16 14:23:47  jsedwards
-- Fixed copy and paste error, where chunk_info .used and .index were written
-- into .ref when moving entry.
--
-- Revision 1.85  2008/06/26 14:54:30  jsedwards
-- Change so that if initialize backup fails, it shuts disk_io down again and
-- exits.
--
-- Revision 1.84  2008/06/12 02:16:29  jsedwards
-- Fixed so that read_only flag is not set in PUBLIC_MODE, because if it is
-- set no backup file gets written.
--
-- Revision 1.83  2008/06/07 16:26:21  jsedwards
-- Fix Bug #1987367 - moved setting of 'read_only' variable outside of if
-- private objects statement so that it is always set, even if there are no
-- private objects.
--
-- Revision 1.82  2008/05/22 11:33:51  jsedwards
-- Made disk_header global and renamed to nwos_disk_header.
--
-- Revision 1.81  2008/05/05 12:41:15  jsedwards
-- Changed so that if fopen of index file fails that it just prints a warning
-- and continues without writing the index file.
--
-- Revision 1.80  2008/05/05 12:15:33  jsedwards
-- Fix Bug #1957451 - added test to make sure fopen didn't return NULL before
-- calling fwrite for index file.
--
-- Revision 1.79  2008/04/13 23:06:57  jsedwards
-- Changed to conditionally use gettimeofday instead of clock_gettime if the
-- OS doesn't support it (like Mac OS X).
--
-- Revision 1.78  2008/04/02 03:06:21  jsedwards
-- Added file locking.
--
-- Revision 1.77  2008/03/24 05:04:10  jsedwards
-- Change to not close private (or public) private descriptor if it is not
-- open.
--
-- Revision 1.76  2008/02/25 14:31:20  jsedwards
-- Changed nwos_allocate_new_chunk function to allow creating a new chunk for
-- any reference and not just greater than the existing references.
--
-- Revision 1.75  2008/02/24 21:00:18  jsedwards
-- Added code to allocate more chunks if the existing chunks are more than 90%
-- full.
--
-- Revision 1.74  2008/02/24 20:30:44  jsedwards
-- Fix Bug #1900855 - changed to just pick a block at random and if that fails
-- to just do a linear search.
--
-- Revision 1.73  2008/02/11 04:33:37  jsedwards
-- Changed to set 'read_only' flag for PUBLIC AccessType too.
--
-- Revision 1.72  2008/02/11 03:59:57  jsedwards
-- Added "PUBLIC" AccessType for opening only the Public Objects.
--
-- Revision 1.71  2008/02/09 17:43:19  jsedwards
-- Moved backup code out of "if (private..." so that backup files will be made
-- in PUBLIC_MODE as well.
--
-- Revision 1.70  2008/02/06 03:25:58  jsedwards
-- Added initializer for compressed_file boolean variable so compiler won't
-- complain about it being uninitialized.
--
-- Revision 1.69  2008/02/04 03:10:19  jsedwards
-- Changed to get backup directory path from user_config.
--
-- Revision 1.68  2008/02/03 02:11:22  jsedwards
-- Change to use the new nwos_get_private_objects_path function that checks
-- for environment variable and configuration file.
--
-- Revision 1.67  2008/02/03 02:05:42  jsedwards
-- Fixed wrong path for public objects broken in previous check in.
--
-- Revision 1.66  2008/02/03 01:46:34  jsedwards
-- Reverted some of the changes made in revision 1.64 for the nwos_public_path
-- back to the way they were in 1.63.
--
-- Revision 1.65  2008/02/03 01:39:50  jsedwards
-- Fix to compile without warnings in PUBLIC_MODE.
--
-- Revision 1.64  2008/01/22 14:55:25  jsedwards
-- Changed to call nwos_get_public_objects_path() in new user_config.c file,
-- instead of having a global variable.
--
-- Revision 1.63  2008/01/17 15:04:02  jsedwards
-- Moved progress bar code to prep_disk.c and instead pass a function pointer
-- in that is called (if non-null) with progress.
--
-- Revision 1.62  2008/01/17 14:42:59  jsedwards
-- Fixed the progress bar so it works correctly even for small sizes.
--
-- Revision 1.61  2008/01/17 14:33:36  jsedwards
-- Took code added in previous revision back out because in the sparse file
-- case the entire file has already been written, all we have to do is
-- allocate the chunks.  This revision should be the same as Revision 1.59.
--
-- Revision 1.60  2008/01/17 13:44:45  jsedwards
-- Added code to allocate all hack that checks to see if it is a sparse file
-- or a partition and writes the whole chunk if it is a sparse file.  This is
-- necessary for the time being because if we only write the bit map to a
-- sparse file it really will create a sparse file which wastes a lot of disk
-- space.
--
-- Revision 1.59  2008/01/17 05:13:55  jsedwards
-- Added call to flush data after each bit map block write and usleep 10 mS.
--
-- Revision 1.58  2008/01/17 05:05:44  jsedwards
-- Changed to just write the bit map bytes into each chunk allocated instead
-- of writing the entire chunk with zeros.  Should no longer be necessary to
-- write all the zeros because the it map determines if the block is used or
-- not.
--
-- Revision 1.57  2008/01/17 04:55:59  jsedwards
-- Added code to print the progress in the allocate_all_hack function.
--
-- Revision 1.56  2007/12/09 16:56:14  jsedwards
-- Changed assert to more descriptive prints, when block isn't found in
-- storage_index.
--
-- Revision 1.55  2007/12/09 16:49:29  jsedwards
-- Comment out statement in terminate_disk_io that nulled out the public_path,
-- so we can start objectify back up again if necessary.
--
-- Revision 1.54  2007/11/13 15:01:33  jsedwards
-- Added #ifndef NO_BACKUP around calls to backup module.  This way we don't
-- have to include it when compiling export_c_structs.
--
-- Revision 1.53  2007/11/09 15:53:47  jsedwards
-- Moved variables and code that deal with backup file to new file backup.c.
--
-- Revision 1.52  2007/11/08 12:32:57  jsedwards
-- Split code to finish writing the backup file out into a separate function.
--
-- Revision 1.51  2007/11/08 05:23:17  jsedwards
-- Added code to create and write backup file.
--
-- Revision 1.50  2007/11/03 15:15:10  jsedwards
-- Fix to handle the case where the block_estimate is smaller than what will
-- fit in one chunk.
--
-- Revision 1.49  2007/11/03 15:11:03  jsedwards
-- Renamed 'max_usable' variable in nwos_generate_closely_spaced_id to
-- 'max_used' because it makes more sense.
--
-- Revision 1.48  2007/11/03 14:18:46  jsedwards
-- Added nwos_allocate_all_chunks_hack function for prep_disk to allocate all
-- available space for chunks, until I can get the real allocation working.
--
-- Revision 1.47  2007/10/25 12:35:53  jsedwards
-- Moved the nwos_restart_id_generation and nwos_check_blocks_available
-- functions from the beginning to a location farther into the file.  NO code
-- changes.
--
-- Revision 1.46  2007/10/25 04:26:45  jsedwards
-- Moved last_ref and nwos_restart_id_generation from objectify.c.
--
-- Revision 1.45  2007/10/21 13:21:40  jsedwards
-- Added code to nwos_check_blocks_available to allocate more chunks (at the
-- end) if there are any available.  This version is limited to chunks after
-- chunks that are already allocated.
--
-- Revision 1.44  2007/10/19 01:56:06  jsedwards
-- Added global variables to store parameters to control chunk allocation and
-- a function to set them.
--
-- Revision 1.43  2007/10/13 18:20:00  jsedwards
-- Changed the 'calculate_chunk_density' function to just search back from the
-- end and renamed to 'find_first_low_density_in_chunk'.
--
-- Revision 1.42  2007/10/13 11:06:39  jsedwards
-- Changed the name of the CHUNK_FACTOR define to BYTES_PER_SLICE.
--
-- Revision 1.41  2007/10/11 13:05:07  jsedwards
-- Added code to allow older compatible compressed files to be read in when
-- the version doesn't match exactly.
--
-- Revision 1.40  2007/10/08 14:12:39  jsedwards
-- Added 'calculate_chunk_density' and 'find_starting_chunk' functions, not
-- used yet.
--
-- Revision 1.39  2007/10/07 03:31:21  jsedwards
-- Changed 'nwos_set_block_estimate' function to 'nwos_check_blocks_available'
-- and to return a boolean value if there is room or not.
--
-- Revision 1.38  2007/09/02 19:36:31  jsedwards
-- Added variables and functions to save and log the block estimate.
--
-- Revision 1.37  2007/08/29 15:38:17  jsedwards
-- Added #ifdef LOG_BIT_MAP around log messages for the bit map read and
-- writes, so by default they are not logged.
--
-- Revision 1.36  2007/08/14 00:43:14  jsedwards
-- Fix value passed to find bit map to include offset to chunks.
--
-- Revision 1.35  2007/08/13 19:18:42  jsedwards
-- Fixed two bugs in calculating start block in generating closely spaced id.
--
-- Revision 1.34  2007/08/13 18:01:11  jsedwards
-- Removed a couple more debugging print statements accidentally left in.
--
-- Revision 1.33  2007/08/13 00:25:58  jsedwards
-- Fix bug in the case where more than one chunk is allocated and the first
-- chunk is not used.
--
-- Revision 1.32  2007/08/12 20:05:30  jsedwards
-- Removed debugging print statements accidentally left in last check in.
--
-- Revision 1.31  2007/08/12 19:37:04  jsedwards
-- Fixed previous log entry which was completely wrong (NO code changes).
--
-- Revision 1.30  2007/08/12 19:31:39  jsedwards
-- Rewrite generate_new_closely_spaced_id to work with new chunk stuff.
--
-- Revision 1.29  2007/08/10 00:03:35  jsedwards
-- Removed defintion of _LARGEFILE64_SOURCE, now using _FILE_OFFSET_BITS=64.
-- Also removed using O_LARGEFILE from open call.
--
-- Revision 1.28  2007/08/07 05:09:22  jsedwards
-- Changed to store index into chunk_info table in bit map cache instead of
-- the chunk block number.
--
-- Revision 1.27  2007/08/07 02:34:50  jsedwards
-- Changed variable name from 'block' to 'ref' because block was misleading.
--
-- Revision 1.26  2007/08/07 01:52:37  jsedwards
-- Added 'in_use' element to bit map cache to determine if an entry is in use
-- instead of testing for 'chunk' being non-zero.
--
-- Revision 1.25  2007/08/06 04:12:19  jsedwards
-- Moved binary search of chunk_info table for reference into separate function.
--
-- Revision 1.24  2007/08/06 02:15:57  jsedwards
-- Changed 1_in_N algorithm to pick a random incrment between 1 and N, instead
-- of picking 1 in N number of blocks.
--
-- Revision 1.23  2007/08/02 18:41:06  jsedwards
-- Change to use the new index in the chunk_info table to comupute the address
-- of the chunk in storage.
--
-- Revision 1.22  2007/08/01 00:50:09  jsedwards
-- Fixed to byte swap chunk_info index values on little endian machines when
-- the chunk_info table is read and written.  Also added code to allocate chunk
-- function to calculate index and store it in chunk_info table (not used yet).
--
-- Revision 1.21  2007/07/15 20:32:36  jsedwards
-- Added chunk_used_index and code to sort chunk_info[].used and put indexes
-- into chunk_used_index.
--
-- Revision 1.20  2007/07/15 17:19:49  jsedwards
-- Changed to use WORDS_BIGENDING instead of __BYTE_ORDER == __LITTLE_ENDIAN
-- to determine endianess and byteswap_uint16 and 32 functions in objectify.h
-- instead of bswap_16 and 32 to make more platform independant.  Also added
-- initializer to 'i' variable so compiler wouldn't complaing that it was
-- possibly uninitialized.
--
-- Revision 1.19  2007/07/06 15:26:13  jsedwards
-- Changes to work in PUBLIC_MODE with new separate public objects file.
--
-- Revision 1.18  2007/07/04 12:42:38  jsedwards
-- Fix bug in read_block searching in index table, doesn't need to use hash
-- anymore to search storage_index, just convert the reference to uint32.
--
-- Revision 1.17  2007/07/03 01:19:18  jsedwards
-- Changed from chunk_index which was a uint32 to chunk_info which is a
-- structure containing the old uint32 reference and a uint32 used blocks
-- count.
--
-- Revision 1.16  2007/07/02 15:06:26  jsedwards
-- Removed special_update routine that was used to import 0021 compressed into
-- 0022 storage.  No longer needed.
--
-- Revision 1.15  2007/07/01 19:44:11  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.14  2007/06/30 04:28:59  jsedwards
-- Added #ifdef around log message for write block, now off by default.
--
-- Revision 1.13  2007/06/30 00:49:04  jsedwards
-- Fix bug in generate_new_completely_random_id where the byte offset into
-- the bit map was off by 4 (BIT_MAP_BLOCKS / 8 bits), so it was testing the
-- wrong byte in the bit map for used or not.  Assert failed because sometimes
-- the block was already used.
--
-- Revision 1.12  2007/06/28 19:06:09  jsedwards
-- Add log message when generating a random ID or 1 in N ID.
--
-- Revision 1.11  2007/06/28 13:26:06  jsedwards
-- Removed line in allocate_new_chunk that added the blocks used for the bit
-- map to the used_blocks count.  In 0023 the blocks used count doesn't
-- include the bit map blocks.
--
-- Revision 1.10  2007/06/28 04:37:32  jsedwards
-- Fix the stupid binary search thing yet again.
--
-- Revision 1.9  2007/06/28 02:43:41  jsedwards
-- Tweak the binary search for id a bit more.
--
-- Revision 1.8  2007/06/28 02:36:37  jsedwards
-- Make binary search for reference more robust.
--
-- Revision 1.7  2007/06/27 01:13:42  jsedwards
-- Fix nwos_generate_new_1_in_N_id to generate new id if current one doesn't
-- fit in a chunk.
--
-- Revision 1.6  2007/06/26 20:02:37  jsedwards
-- Changed algorithm to generate random id to generate id in existing chunk
-- instead of completely random.
--
-- Revision 1.5  2007/06/26 16:46:06  jsedwards
-- Moved code to compute number of blocks used in an entry to the read
-- cache entry function.
--
-- Revision 1.4  2007/06/26 14:31:44  jsedwards
-- Rename private_generate_new_completely_random_id to
-- generate_new_completely_random_id and make it static.
--
-- Revision 1.3  2007/06/26 14:21:34  jsedwards
-- Moved random id generation functions from objectify.c.
--
-- Revision 1.2  2007/06/26 13:20:53  jsedwards
-- Changed to use new chunk index to convert reference ids to block numbers on
-- disk, instead of wrapping them around the disk size.
--
-- Revision 1.1  2007/06/25 05:28:54  jsedwards
-- New file created from variables, functions and code related to disk I/O,
--  taken from the objectfy.c file.
--
*/

#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>

#include "backup.h"
#include "bit_map.h"
#include "chunk_info.h"
#include "config.h"
#include "disk_io.h"
#include "gen_id.h"
#include "log.h"
#include "mem_alloc.h"
#include "strlcxx.h"           /* in case strlcpy and strlcat are not provided by the system */
#include "user_config.h"


Disk_Header nwos_disk_header;  /* this should NOT be accessed by normal programs */

const char*  nwos_public_path;
const char*  nwos_private_path;

static char version_string[5];

uint32 nwos_total_private_blocks;
uint32 nwos_used_public_blocks;
uint32 nwos_used_private_blocks;

uint32 nwos_block_offset_to_chunks;
uint32 nwos_total_private_chunks = 0;
uint32 nwos_used_private_chunks;

static int    public_file_desc;
static int    private_file_desc;
static bool    read_only;
static bool    modified;
static ObjRef* storage_index;
static int     storage_count;



/*****************/
/* Archive type. */
/*****************/

bool nwos_archive_is_block_device(void)
{
    assert(nwos_private_path[0] == '/');   /* must be a full path specified */

    return nwos_private_path[1] == 'd' && nwos_private_path[2] == 'e' && nwos_private_path[3] == 'v' && nwos_private_path[4] == '/';
}

bool nwos_archive_is_file(void)
{
    return !nwos_archive_is_block_device();
}

const char* nwos_archive_version()
{
    return version_string;
}

bool nwos_backup_is_enabled(void)
{
    const char *backup_dir;

    backup_dir = nwos_get_backup_directory_path();

    return backup_dir != NULL && *backup_dir != '\0';
}


/**********************/
/* Reference testing. */
/**********************/

bool nwos_is_public_reference(ObjRef* ref)
{
    return nwos_ref_to_word(ref) <= MAXIMUM_PUBLIC_REFERENCE;
}


bool nwos_is_private_reference(ObjRef* ref)
{
    return MINIMUM_PRIVATE_REFERENCE <= nwos_ref_to_word(ref) && nwos_ref_to_word(ref) <= MAXIMUM_PRIVATE_REFERENCE;
}


bool nwos_is_temporary_reference(ObjRef* ref)
{
    return MINIMUM_TEMPORARY_REFERENCE <= nwos_ref_to_word(ref) && nwos_ref_to_word(ref) <= MAXIMUM_TEMPORARY_REFERENCE;
}


Reference_Type nwos_reference_type(ObjRef* ref)
{
    if (nwos_is_public_reference(ref))
    {
	return Public_Reference;
    }

    if (nwos_is_private_reference(ref))
    {
	return Private_Reference;
    }

    if (nwos_is_temporary_reference(ref))
    {
	return Temporary_Reference;
    }

    return Unknown_Reference;     /* this should never happen */
}


/*****************************************/
/* Function that indexes compressed file */
/*****************************************/

#define INDEX_ALLOC_SIZE 262144 

void index_compressed_file()
{
    int i;
    int size_of_index = INDEX_ALLOC_SIZE;
    size_t bytes_read;
    uint8 buffer[FILE_BLOCK_SIZE];
    uint8 header[FILE_BLOCK_SIZE];
    char* index_path = NULL;
    FILE* index_fp;

#ifdef NO_INDEX_FILE
    index_fp = NULL;
#else
    assert(private_file_desc > 0);

    /* first see if there is an existing index file */

    i = strlen(nwos_private_path);
    index_path = nwos_malloc(i+5);  /* 4 for .ndx and 1 for null */
    strlcpy(index_path, nwos_private_path, i+5);

    if (index_path[i-4] == '.' && index_path[i-3] == 'o' && index_path[i-2] == 'b' && index_path[i-1] == 'j')
    {
	i = i - 4;   /* write over the old .obj */
    }

    index_path[i++] = '.';
    index_path[i++] = 'n';
    index_path[i++] = 'd';
    index_path[i++] = 'x';
    index_path[i++] = '\0';

    fprintf(stderr, "index_path: %s\n", index_path);

    index_fp = fopen(index_path, "r");
#endif

    storage_index = malloc(size_of_index * sizeof(ObjRef));   /* allocate space for the first batch */

    assert(storage_index != NULL);

    /* read the first block, it contains the header */
    bytes_read = read(private_file_desc, header, sizeof(header));

    assert(bytes_read == sizeof(header));

    /* if there is an index file see if the header matches */
    if (index_fp != NULL)
    {
	bytes_read = fread(buffer, 1, FILE_BLOCK_SIZE, index_fp);
	assert(bytes_read == sizeof(buffer));

	fprintf(stderr, "Index file found: ");
	fflush(stderr);

	if (memcmp(header, buffer, FILE_BLOCK_SIZE) == 0)   /* we have a winner */
	{
	    fprintf(stderr, "match\n");
	}
	else
	{
	    fprintf(stderr, "no match\n");
	    fclose(index_fp);
	    index_fp = NULL;
	}
    }
    else /* no index file */
    {
	errno = 0;   /* clear errno for later */
    }

    fprintf(stderr, "Indexing");
    fflush(stderr);

    i = 1;
    while (1)
    {
	if (i % 262144 == 1)
	{
	    fprintf(stderr, ".");
	    fflush(stderr);
	}

	if (i == size_of_index)   /* need to allocate more space */
	{
	    size_of_index += INDEX_ALLOC_SIZE;
	    storage_index = realloc(storage_index, size_of_index * sizeof(ObjRef));
	    assert(storage_index != NULL);
	}

	assert(i < size_of_index);

	if (index_fp != NULL)   /* read index from file */
	{
	    bytes_read = fread(&storage_index[i], sizeof(ObjRef), 1, index_fp);

	    if (bytes_read != 1) break;
	}
	else
	{
	    bytes_read = read(private_file_desc, buffer, sizeof(buffer));

	    if (bytes_read != sizeof(buffer)) break;

	    memcpy(&storage_index[i], &buffer[4], sizeof(ObjRef));
	}

	i++;
    }

    storage_count = i;

    fprintf(stderr, "\nblocks: %d\n", i);

    if (index_fp != NULL)
    {
	if (ferror(index_fp))
	{
	    perror(index_path);
	    exit(1);
	}
	fclose(index_fp);
    }
    else
    {
	if (errno != 0)
	{
	    perror(nwos_private_path);
	    exit(1);
	}

#ifndef NO_INDEX_FILE
	index_fp = fopen(index_path, "w");

	if (index_fp == NULL)
	{
	    fprintf(stderr, "Could not create index file: ");
	    perror(index_path);
	    fprintf(stderr, "Index will not be saved!\n");
	}
	else
	{
	    fprintf(stderr, "Writing index file: %s\n", index_path);

	    if (fwrite(header, 1, sizeof(header), index_fp) != sizeof(header))
	    {
		perror(index_path);
		exit(1);
	    }

	    if (fwrite(&storage_index[1], sizeof(ObjRef), i, index_fp) != i)
	    {
		perror(index_path);
		exit(1);
	    }

	    if (fclose(index_fp))
	    {
		perror(index_path);
		exit(1);
	    }
	}
#endif
    }

    assert(bytes_read == 0);

    if (index_path != NULL)
    {
	nwos_free(index_path);
    }
}



/**********************************************/
/* Function to initialize the disk I/O module */
/**********************************************/

void nwos_initialize_disk_io(AccessType type, const char* path)
{
    char log_msg[256];
    bool compressed_file = false;
    struct flock lock;
#ifndef NO_BACKUP
    const char* backup_dir;
#endif

    /* make sure the storage is something we can deal with */
#ifdef PUBLIC_MODE
    assert(type == PUBLIC);
#else
    assert(type == PUBLIC || type == READ_ONLY || type == READ_WRITE);
#endif

    assert(nwos_private_path == NULL);   /* make sure this is only called once */
    assert(private_file_desc == 0);
    assert(public_file_desc == 0);

    nwos_initialize_gen_id();

    nwos_public_path = nwos_get_public_objects_path();

    if (type == PUBLIC)
    {
	assert(path == NULL);
    }
    else
    {
	if (path == NULL)    /* no compressed file was passed in */
	{
	    nwos_private_path = nwos_get_private_objects_path();
	    compressed_file = (strstr(nwos_private_path, "compressed") != NULL);
	}
	else
	{
	    assert(type == READ_ONLY);
	    nwos_private_path = path;
	    compressed_file = true;
	}
    }

    /********************************/
    /* open the public objects file */
    /********************************/

#ifdef PUBLIC_MODE
    public_file_desc = open(nwos_public_path, O_RDWR);
#else
    public_file_desc = open(nwos_public_path, O_RDONLY);
#endif

    if (public_file_desc < 0)
    {
	perror(nwos_public_path);
	exit(1);
    }

#ifdef PUBLIC_MODE
    lock.l_type = F_WRLCK;
    lock.l_whence = SEEK_SET;
    lock.l_start = 0;
    lock.l_len = 0;

    if (fcntl(public_file_desc, F_SETLK, &lock) != 0)
    {
	perror(nwos_public_path);
	exit(1);
    }
#endif

    snprintf(log_msg, sizeof(log_msg), " public_path: %s", nwos_public_path);
    nwos_log(log_msg);

    if (read(public_file_desc, &nwos_disk_header, sizeof(nwos_disk_header)) != sizeof(nwos_disk_header))
    {
	snprintf(log_msg, sizeof(log_msg), "reading disk header from: %s", nwos_public_path);
	perror(log_msg);
	exit(1);
    }

    if (memcmp(nwos_disk_header.magic_number, MAGIC_NUMBER, 4) != 0)
    {
	fprintf(stderr, "Missing magic number in disk header: %s\n", nwos_public_path);
	exit(1);
    }

    if (memcmp(nwos_disk_header.version_string, VERSION_STRING, 4) != 0)
    {
	fprintf(stderr, "Incorrect version string in disk header: %s\n", nwos_public_path);
	exit(1);
    }

    assert(sizeof(version_string) == 5);
    memcpy(version_string, nwos_disk_header.version_string, 4);
    version_string[4] = '\0';

    nwos_4_uint8_to_uint32(nwos_disk_header.used_blocks, &nwos_used_public_blocks);

    snprintf(log_msg, sizeof(log_msg), " used public blocks: %9u", nwos_used_public_blocks);
    nwos_log(log_msg);

#ifdef PUBLIC_MODE
    nwos_word_to_ref(nwos_used_public_blocks, &nwos_next_public_ref);

    /* don't set read_only flag if in PUBLIC_MODE */
#else
    read_only = (type == PUBLIC || type == READ_ONLY);


    /***************************************************/
    /* if a private objects file was specified open it */
    /***************************************************/

    if (nwos_private_path != NULL)
    {
	if (read_only)
	{
	    private_file_desc = open(nwos_private_path, O_RDONLY);
	}
	else
	{
	    private_file_desc = open(nwos_private_path, O_RDWR);
	}

	if (private_file_desc < 0)
	{
	    perror(nwos_private_path);
	    exit(1);
	}

	lock.l_type = read_only ? F_RDLCK : F_WRLCK;
	lock.l_whence = SEEK_SET;
	lock.l_start = 0;
	lock.l_len = 0;

	if (fcntl(private_file_desc, F_SETLK, &lock) != 0)
	{
	    perror(nwos_private_path);
	    exit(1);
	}

	snprintf(log_msg, sizeof(log_msg), " private_path: %s", nwos_private_path);
	nwos_log(log_msg);

	if (compressed_file)
	{
	    index_compressed_file();   /* build an index into the compressed file */

	    if (lseek(private_file_desc, 0LL, SEEK_SET) < 0)
	    {
		perror("rewinding after indexing");
		exit(1);
	    }
	}

	if (read(private_file_desc, &nwos_disk_header, sizeof(nwos_disk_header)) != sizeof(nwos_disk_header))
	{
	    snprintf(log_msg, sizeof(log_msg), "reading disk header from: %s", nwos_private_path);
	    perror(log_msg);
	    exit(1);
	}

	if (memcmp(nwos_disk_header.magic_number, MAGIC_NUMBER, 4) != 0)
	{
	    fprintf(stderr, "Missing magic number in disk header: %s\n", nwos_private_path);
	    exit(1);
	}

	if (memcmp(nwos_disk_header.version_string, VERSION_STRING, 4) != 0)
	{
	    // allow more compressed files to be other compatible versions

	    if ((compressed_file && memcmp(nwos_disk_header.version_string, OLDEST_COMPATIBLE_COMPRESSED_VERSION, 4) < 0) ||
		(!compressed_file && memcmp(nwos_disk_header.version_string, VERSION_STRING, 4) != 0))
	    {
		fprintf(stderr, "Incorrect version string in disk header: %s\n", nwos_private_path);
		exit(1);
	    }
	}

	nwos_4_uint8_to_uint32(nwos_disk_header.block_offset_to_chunks, &nwos_block_offset_to_chunks);
	nwos_4_uint8_to_uint32(nwos_disk_header.total_blocks, &nwos_total_private_blocks);
	nwos_4_uint8_to_uint32(nwos_disk_header.used_blocks, &nwos_used_private_blocks);
	nwos_4_uint8_to_uint32(nwos_disk_header.used_chunks, &nwos_used_private_chunks);

	nwos_total_private_chunks = nwos_total_private_blocks / BLOCKS_IN_CHUNK;

	/* Read the chunk info file if not a compressed file */

	if (!compressed_file)
	{
	    nwos_initialize_chunk_info(private_file_desc);
	}

	assert(nwos_total_private_blocks > 0);

	snprintf(log_msg, sizeof(log_msg), "   private blocks: %9u  used: %9u", nwos_total_private_blocks, nwos_used_private_blocks);
	nwos_log(log_msg);
    }
#endif

#ifndef NO_BACKUP
    /* Do a backup file? */

    backup_dir = nwos_get_backup_directory_path();

    if (backup_dir != NULL && *backup_dir != '\0' && !read_only)
    {
	if (!nwos_initialize_backup(backup_dir, &nwos_disk_header))    /* creating backup failed, shut 'er back down */
	{
	    nwos_terminate_disk_io();
	    exit(1);
	}
    }
#endif
}


/****************/
/* Open status. */
/****************/

bool nwos_disk_io_is_read_only()
{
    return read_only;
}

bool nwos_disk_io_is_read_write()
{
    return !read_only;
}


/*********************************************/
/* Function to terminate the disk I/O module */
/*********************************************/

void nwos_terminate_disk_io()
{
    char log_msg[128];

#ifndef PUBLIC_MODE
    nwos_terminate_bit_map();

    nwos_terminate_chunk_info(private_file_desc);
#endif

    if (modified)
    {
#ifdef PUBLIC_MODE
	snprintf(log_msg, sizeof(log_msg), "  used public blocks: %9u",
		 nwos_ref_to_word(&nwos_next_public_ref));
	nwos_log(log_msg);

	copy_reference((ObjRef*)&nwos_disk_header.used_blocks, &nwos_next_public_ref);

	nwos_get_time_stamp(nwos_disk_header.last_change);
#else
	nwos_uint32_to_4_uint8(&nwos_used_private_blocks, nwos_disk_header.used_blocks);
	nwos_uint32_to_4_uint8(&nwos_used_private_chunks, nwos_disk_header.used_chunks);

	if (nwos_archive_is_file())   /* totals could have changed */
	{
	    nwos_uint32_to_4_uint8(&nwos_total_private_blocks, nwos_disk_header.total_blocks);
	}

	snprintf(log_msg, sizeof(log_msg), "  private blocks: %9u  used: %9u", nwos_total_private_blocks, nwos_used_private_blocks);
	nwos_log(log_msg);

	nwos_get_time_stamp(nwos_disk_header.last_change);
#endif

#ifdef PUBLIC_MODE
	if (lseek(public_file_desc, (off_t)0, SEEK_SET) < 0)
#else
	if (lseek(private_file_desc, (off_t)0, SEEK_SET) < 0)
#endif
	{
	    perror(nwos_private_path);
	    exit(1);
	}

#ifdef PUBLIC_MODE
	if (write(public_file_desc, &nwos_disk_header, sizeof(nwos_disk_header)) != sizeof(nwos_disk_header))
#else
	if (write(private_file_desc, &nwos_disk_header, sizeof(nwos_disk_header)) != sizeof(nwos_disk_header))
#endif
	{
	    perror(nwos_private_path);
	    exit(1);
	}
    }
    else
    {
	nwos_log("  No changes made");
    }

    if (private_file_desc > 0 && close(private_file_desc) < 0)
    {
	perror(nwos_private_path);
    }

    private_file_desc = 0;
    nwos_private_path = NULL;

#ifndef NO_BACKUP
    nwos_terminate_backup(modified, &nwos_disk_header);
#endif

    if (public_file_desc > 0 && close(public_file_desc) < 0)
    {
	perror(nwos_public_path);
    }

    public_file_desc = 0;

    nwos_terminate_gen_id();
}


/**********************************************************************************************/
/* This function returns amount of private space used on disk 0.1 = 10%, 0.5 = 50%, 0.9 = 90% */
/**********************************************************************************************/

void nwos_write_empty_chunk(uint32 chunk_index)
{
    uint32 offset;
    static uint8* chunk;

    if (nwos_archive_is_file())
    {
	assert(chunk_index == nwos_total_private_chunks);
    }
    else
    {
	assert(chunk_index < nwos_total_private_chunks);
    }

    if (chunk == NULL)   /* first time this has been called, allocate memory */
    {
	chunk = malloc(CHUNK_SIZE);

	if (chunk == NULL)
	{
	    perror("allocate memory for creating new chunk");
	    exit(1);
	}

	memset(chunk, 0, CHUNK_SIZE);

	chunk[0] = 0xff;
	chunk[1] = 0xff;
	chunk[2] = 0xff;
	chunk[3] = 0xff;
    }

    offset = nwos_block_offset_to_chunks + (chunk_index * BLOCKS_IN_CHUNK);

    if (lseek(private_file_desc, (off_t)offset << 8, SEEK_SET) < 0)
    {
	perror(nwos_private_path);
	exit(1);
    }

    if (write(private_file_desc, chunk, CHUNK_SIZE) != CHUNK_SIZE)
    {
	perror(nwos_private_path);
	exit(1);
    }

    /* if the archve is a file, this expanded it, increase the counts */

    if (nwos_archive_is_file())
    {
	nwos_total_private_chunks++;

	nwos_total_private_blocks = nwos_total_private_chunks * BLOCKS_IN_CHUNK;
    }
}


/**********************************************************************************************/
/* This function returns amount of private space used on disk 0.1 = 10%, 0.5 = 50%, 0.9 = 90% */
/**********************************************************************************************/

float nwos_private_space_used()
{
    float result;

    result = (float)nwos_used_private_blocks / (float)nwos_total_private_blocks;

    assert(0.0f <= result && result <= 1.0f);

    return result;
}


void nwos_write_bit_map(int chunk_index, void* map)
{
    uint32 chunk;
    char log_msg[80];

    chunk = nwos_block_offset_to_chunks + chunk_index * BLOCKS_IN_CHUNK;

    assert(chunk < nwos_block_offset_to_chunks + nwos_total_private_blocks);

    if (lseek(private_file_desc, (off_t)chunk * FILE_BLOCK_SIZE, SEEK_SET) < 0)
    {
	snprintf(log_msg, sizeof(log_msg), "write_bit_map lseek chunk:%08x", chunk);
	perror(log_msg);
	exit(1);
    }

    if (write(private_file_desc, map, BIT_MAP_BYTES) != BIT_MAP_BYTES)
    {
	snprintf(log_msg, sizeof(log_msg), "write_bit_map write chunk:%08x", chunk);
	perror(log_msg);
	exit(1);
    }

    modified = true;
}


void nwos_read_bit_map(int chunk_index, void* map)
{
    uint32 chunk;
    char log_msg[80];

    chunk = nwos_block_offset_to_chunks + chunk_index * BLOCKS_IN_CHUNK;

    assert(chunk < nwos_block_offset_to_chunks + nwos_total_private_blocks);

    if (lseek(private_file_desc, (off_t)chunk * FILE_BLOCK_SIZE, SEEK_SET) < 0)
    {
	snprintf(log_msg, sizeof(log_msg), "read_bit_map lseek chunk:%08x", chunk);
	perror(log_msg);
	exit(1);
    }

    if (read(private_file_desc, map, BIT_MAP_BYTES) != BIT_MAP_BYTES)
    {
	snprintf(log_msg, sizeof(log_msg), "read_bit_map write chunk:%08x", chunk);
	perror(log_msg);
	exit(1);
    }
}


#ifndef PUBLIC_MODE

//static float chunk_usage(Bit_Map_Cache_Entry* entry)
//{
//    float result;

//    result = (float)chunk_info[entry->index].used / (float)USABLE_BLOCKS_PER_CHUNK;

//    assert(0.0f <= result && result <= 1.0f);

//    return result;
//}
#endif



/*****************************************/
/* Function to read a block from storage */
/*****************************************/

bool nwos_read_block(ObjRef* ref, uint8 block[FILE_BLOCK_SIZE])
{
    char ref_str[64];
    off_t offset = -1;
    int i = 0;
    int lower;
    int upper;
    int file_desc;
    uint32 ref_hash;
    uint32 mid_hash;
    uint32 hash;
    bool result = false;

    assert(!is_void_reference(ref));

    /*    printf("Read block: %02x%02x%02x%02x\n", ref->id[0], ref->id[1], ref->id[2], ref->id[3]); */


#if NO_DISK_ACCESS

    for (i = 0; i < 4096 && cache[i] != NULL; i++)
    {
	if (is_same_object(ref, (ObjRef*)&cache[i][4]))
	{
	    break;
	}
    }

    assert(i < 4096);

    if (cache[i] != NULL)
    {
	memcpy(block, cache[i], FILE_BLOCK_SIZE);
    }

    return cache[i] != NULL;

#else
    if (nwos_ref_to_word(ref) <= MAXIMUM_PUBLIC_REFERENCE)
    {
	file_desc = public_file_desc;
	offset = (off_t) nwos_ref_to_word(ref) * FILE_BLOCK_SIZE;
    }
#ifdef PUBLIC_MODE
    else
    {
	bool private_reference_in_public_mode = false;
	assert(private_reference_in_public_mode);
    }
#else
    else if (storage_index == NULL)     /* normal storage */
    {
	file_desc = private_file_desc;
	hash = nwos_hash_ref(ref);

	if (hash != 0 && nwos_test_bit_in_map(hash))
	{
	    offset = (off_t) hash << 8;
	}
    }
    else
    {
	file_desc = private_file_desc;

	lower = 1;
	upper = storage_count;

	ref_hash = nwos_ref_to_word(ref);

	while (lower <= upper)
	{
	    i = (lower + upper) / 2;

	    mid_hash = nwos_ref_to_word(&storage_index[i-1]);

	    if (mid_hash > ref_hash)
	    {
		upper = i - 1;
	    }
	    else if (mid_hash < ref_hash)
	    {
		lower = i + 1;
	    }
	    else
	    {
		i = i - 1;
		break;
	    }
	}

	if (lower > upper)
	{
	    int i;

	    fprintf(stderr, "Can't find %08x in index table:\n", ref_hash);
	    fflush(stderr);

	    if (ref_hash < nwos_ref_to_word(&storage_index[0]))
	    {
		i = 0;
	    }
	    else
	    {
		for (i = 1; i < storage_count; i++)
		{
		    if (nwos_ref_to_word(&storage_index[i-1]) < ref_hash && ref_hash < nwos_ref_to_word(&storage_index[i])) break;
		}
	    }

	    if (i > 0)
	    {
		fprintf(stderr, "  %d: %08x\n", i-1, nwos_ref_to_word(&storage_index[i-1]));
	    }

	    if (i < storage_count)
	    {
		fprintf(stderr, "  %d: %08x\n", i, nwos_ref_to_word(&storage_index[i]));
	    }
	}
	else
	{
	    assert(is_same_object(&storage_index[i], ref));

	    offset = (off_t)i * FILE_BLOCK_SIZE;
	}
    }
#endif

    if (offset < 0) return false;

    if (lseek(file_desc, offset, SEEK_SET) < 0)
    {
	uint32 offset_upr = (uint32)(offset >> 32);
	uint32 offset_lwr = (uint32)offset;
	snprintf(ref_str, sizeof(ref_str), "nwos_read_block lseek ref:%02x%02x%02x%02x offset:%x%08x",
		ref->id[0], ref->id[1], ref->id[2], ref->id[3], offset_upr, offset_lwr);

	perror(ref_str);
	exit(1);
    }

    result = read(file_desc, block, FILE_BLOCK_SIZE) == FILE_BLOCK_SIZE;

    /* if read was successful make sure this block contains the actual block requested */
    if (result)
    {
	assert((block[0] == 0) &
	       (block[1] == 0) &
	       (block[2] == 0) &
	       (block[3] == 0) &
	       (block[4] == ref->id[0]) &
	       (block[5] == ref->id[1]) &
	       (block[6] == ref->id[2]) &
	       (block[7] == ref->id[3]));
    }

    return result;
#endif
}


/****************************************/
/* Function to write a block to storage */
/****************************************/

void nwos_write_block(ObjRef* ref, uint8 block[FILE_BLOCK_SIZE])
{
    char log_msg[128];
    off_t offset;

#ifdef PUBLIC_MODE
    assert(nwos_reference_type(ref) == Public_Reference);
#else
    assert(nwos_reference_type(ref) != Public_Reference);
#endif

    assert(!is_void_reference(ref));
    assert((block[0] & 0x7f) == 0 && block[1] == 0 && block[2] == 0 && block[3] == 0);
    assert(is_same_object((ObjRef*)block + 1, ref));

#ifdef LOG_WRITE_BLOCK
    snprintf(log_msg, sizeof(log_msg), "write_block - ref:%02x%02x%02x%02x block:%08x",
	    ref->id[0], ref->id[1], ref->id[2], ref->id[3], nwos_hash_ref(ref));
    nwos_log(log_msg);
#endif

#if NO_DISK_ACCESS
{
    int i;
    for (i = 0; i < 4096 && cache[i] != NULL; i++)
    {
	if (is_same_object(ref, (ObjRef*)&cache[i][4]))
	{
	    printf("WARNING - rewritting the same block: %02x%02x%02x%02x\n", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	    break;
	}
    }

    assert(i < 4096);

    if (cache[i] == NULL)
    {
	cache[i] = malloc(FILE_BLOCK_SIZE);
	assert(cache[i] != NULL);
    }

    memcpy(cache[i], block, FILE_BLOCK_SIZE);
}
#else
    assert(storage_index == NULL);   /* at this time if there is a storage index it is read only */

    offset = nwos_ref_to_offset(ref);

    assert(offset >= 0);

#ifdef PUBLIC_MODE
    if (lseek(public_file_desc, offset, SEEK_SET) < 0)
#else
    if (lseek(private_file_desc, offset, SEEK_SET) < 0)
#endif
    {
	uint32 offset_upr = (uint32)(offset >> 32);
	uint32 offset_lwr = (uint32)offset;
	snprintf(log_msg, sizeof(log_msg), "write_block lseek ref:%02x%02x%02x%02x offset:%x%08x",
		ref->id[0], ref->id[1], ref->id[2], ref->id[3], offset_upr, offset_lwr);

	perror(log_msg);
	exit(1);
    }

#ifdef PUBLIC_MODE
    if (write(public_file_desc, block, FILE_BLOCK_SIZE) != FILE_BLOCK_SIZE)
#else
    if (write(private_file_desc, block, FILE_BLOCK_SIZE) != FILE_BLOCK_SIZE)
#endif
    {
	snprintf(log_msg, sizeof(log_msg), "write_block write ref:%02x%02x%02x%02x", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);

	perror(log_msg);
	exit(1);
    }

#ifndef NO_BACKUP
    nwos_backup_write_block(block);
#endif

    modified = true;
#endif
}


/************************************************/
/* Remove an object (by writing zero's over it) */
/************************************************/

#ifndef PUBLIC_MODE

void nwos_erase_block(ObjRef* ref)
{
    char log_msg[128];
    uint8 block[FILE_BLOCK_SIZE];
    off_t offset;
    uint32 hash;

    assert(nwos_read_block(ref, block));    /* verify that there is already a block there */

    snprintf(log_msg, sizeof(log_msg), "nwos_erase_block - %02x%02x%02x%02x", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
    nwos_log(log_msg);

    hash = nwos_hash_ref(ref);

    assert(nwos_test_bit_in_map(hash));
    assert(nwos_read_block(ref, block));
    assert(block[4] == ref->id[0] && block[5] == ref->id[1] && block[6] == ref->id[2] && block[7] == ref->id[3]);

    memset(block, 0, sizeof(block));

    assert(sizeof(block) == FILE_BLOCK_SIZE);

    offset = nwos_ref_to_offset(ref);

    assert(offset >= 0);

    if (lseek(private_file_desc, offset, SEEK_SET) < 0)
    {
	uint32 offset_upr = (uint32)(offset >> 32);
	uint32 offset_lwr = (uint32)offset;
	snprintf(log_msg, sizeof(log_msg), "nwos_erase_block lseek ref:%02x%02x%02x%02x offset:%x%08x",
		ref->id[0], ref->id[1], ref->id[2], ref->id[3], offset_upr, offset_lwr);

	perror(log_msg);
	exit(1);
    }

    if (write(private_file_desc, block, FILE_BLOCK_SIZE) != FILE_BLOCK_SIZE)
    {
	snprintf(log_msg, sizeof(log_msg), "nwos_erase_block write ref:%02x%02x%02x%02x", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);

	perror(log_msg);
	exit(1);
    }

    nwos_clear_bit_in_map(hash);

    modified = true;

    assert(!nwos_read_block(ref, block));   /* verifty that it was actually erased */
}

#endif

