/*
--             This file is part of the New World OS project
--                 Copyright (C) 2007-2009  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- This is a one time use program to add a bunch of features to the PERSON class
--
-- $Log: add_class.c,v $
-- Revision 1.24  2009/03/08 19:38:10  jsedwards
-- Fixed year in Copyright.
--
-- Revision 1.23  2009/03/08 16:59:09  jsedwards
-- Changed include objectify_private.h to objectify.h and added include
-- class_definition.h and disk_io.h.
--
-- Revision 1.22  2008/08/31 18:25:40  jsedwards
-- Change for new parameters to nwos_initialize_objectify without key stuff
-- and eliminate calle to nwos_set_root_object because initialize does that now.
--
-- Revision 1.21  2008/03/28 11:33:50  jsedwards
-- Add test to verify that class doesn't already exist!!
--
-- Revision 1.20  2008/03/14 12:50:58  jsedwards
-- Changed to add SHA512 class.
--
-- Revision 1.19  2007/08/17 16:44:30  jsedwards
-- Changed to add SHA256 class.
--
-- Revision 1.18  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.17  2007/06/09 12:05:59  jsedwards
-- Change to create 'Family Name' class.
--
-- Revision 1.16  2007/06/09 11:39:45  jsedwards
-- Changed to add "Given Name" class.
--
-- Revision 1.15  2007/05/28 15:15:44  jsedwards
-- Changed to create 'Unit' class again, but this time with more generic
-- 'Acquisition' and 'Disposal' features instead of 'Purchase' and 'Sale'
-- because somtimes stuff is given, thrown away, or stolen for example.
--
-- Revision 1.14  2007/05/28 10:36:54  jsedwards
-- Change to create 'Unit' class.
--
-- Revision 1.13  2007/05/20 17:49:20  jsedwards
-- Change to add SERIAL NUMBER class.
--
-- Revision 1.12  2007/05/20 17:36:15  jsedwards
-- Change to add "MODEL NUMBER" class.
--
-- Revision 1.11  2007/05/20 17:21:44  jsedwards
-- Change to add ACRONYM class.
--
-- Revision 1.10  2007/05/20 15:32:53  jsedwards
-- Changed to create Manufacturer class.
--
-- Revision 1.9  2007/05/06 02:56:55  jsedwards
-- Change to create "Checking" class.
--
-- Revision 1.8  2007/05/05 17:01:25  jsedwards
-- Change to add "account number" class.
--
-- Revision 1.7  2007/05/05 16:25:06  jsedwards
-- Changed to add "account" class.
--
-- Revision 1.6  2007/04/21 18:06:52  jsedwards
-- Change to create 'device' class.
--
-- Revision 1.5  2007/04/21 17:58:35  jsedwards
-- Changed to add 'URL' (Unified Resource Locator) class.
--
-- Revision 1.4  2007/04/19 12:48:51  jsedwards
-- Change abbreviation to acronym and add url list in credit union.
--
-- Revision 1.3  2007/04/14 02:09:46  jsedwards
-- Change to add the "credit union" class.
--
-- Revision 1.2  2007/04/13 12:20:53  jsedwards
-- Moved code to find class references to separate functions.
--
-- Revision 1.1  2007/04/11 14:26:53  jsedwards
-- Initial version, specifically adds "dollars and cents".
--
*/


#include <assert.h>
#include <ctype.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "../class_definition.h"
#include "../crc32.h"
#include "../disk_io.h"    /* define nwos_next_public_ref */
#include "../objectify.h"


#if 0
static size_t get_spelling_object_size(void* spelling_obj)
{
    assert(((C_struct_Spelling*)spelling_obj)->count > 0);
    return sizeof(C_struct_Spelling) + ((C_struct_Spelling*)spelling_obj)->count;
}

static size_t get_name_object_size(void* name_obj)
{
    assert(((C_struct_Name*)name_obj)->count > 0);
    return sizeof(C_struct_Name) + (((C_struct_Name*)name_obj)->count * sizeof(ObjRef));
}
#endif

static size_t get_class_object_size(void* class_obj)
{
    return sizeof(C_struct_Class_Definition) + (((C_struct_Class_Definition*)class_obj)->count * sizeof(ObjRef));
}


ObjRef* byte_class_ref()
{
    static ObjRef result;

    if (is_void_reference(&result))
    {
	if (!nwos_find_public_class_definition("BYTE", &result))
	{
	    fprintf(stderr, "Could not locate BYTE class definition\n");
	    nwos_terminate_objectify();
	    exit(1);
	}

	printf("byte_class_ref: %02x%02x%02x%02x\n",
	       result.id[0],
	       result.id[1],
	       result.id[2],
	       result.id[3]);
    }

    return &result;
}


ObjRef* character_class_ref()
{
    static ObjRef result;

    if (is_void_reference(&result))
    {
	if (!nwos_find_public_class_definition("CHARACTER", &result))
	{
	    fprintf(stderr, "Could not locate CHARACTER class definition\n");
	    nwos_terminate_objectify();
	    exit(1);
	}

	printf("byte_class_ref: %02x%02x%02x%02x\n",
	       result.id[0],
	       result.id[1],
	       result.id[2],
	       result.id[3]);
    }

    return &result;
}


ObjRef* objref_class_ref()
{
    static ObjRef result;

    if (is_void_reference(&result))
    {
	if (!nwos_find_public_class_definition("OBJECT REFERENCE", &result))
	{
	    fprintf(stderr, "Could not locate OBJECT REFERENCE class definition\n");
	    nwos_terminate_objectify();
	    exit(1);
	}

	printf("object_reference_class_ref: %02x%02x%02x%02x\n",
	       result.id[0],
	       result.id[1],
	       result.id[2],
	       result.id[3]);
    }

    return &result;
}


void add_feature(C_struct_Class_Definition* class_obj_ptr, ObjRef* type, char* name, int num_elements)
{
    int count;
    ObjRef feature_ref;

    assert(get_class_object_size(class_obj_ptr) < FILE_BLOCK_SIZE);

    count = class_obj_ptr->count;

    nwos_find_or_create_public_feature_definition(type, name, num_elements, &feature_ref);

    copy_reference(&class_obj_ptr->feature[count], &feature_ref);

    nwos_add_to_references(&class_obj_ptr->header.common.id, &feature_ref);

    class_obj_ptr->count = count + 1;

    printf("feature: %d - %s  new size: %d\n", count, name, (int)get_class_object_size(class_obj_ptr));
}


int main(int argc, char* argv[])
{
    ObjRef class_ref;
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Class_Definition* class_obj_ptr = (C_struct_Class_Definition*)kludge;


    nwos_log_arguments(argc, argv);

    nwos_initialize_objectify(PUBLIC, NULL);


#define NEW_CLASS_NAME "SHA512SUM"

    if (nwos_find_public_class_definition(NEW_CLASS_NAME, &class_ref))
    {
	fprintf(stderr, "\nERROR: class %s already exists: %08x\n\n", NEW_CLASS_NAME, nwos_ref_to_word(&class_ref));
	exit(1);
    }

    printf("Adding URL class on: %s\n", nwos_get_public_objects_path());

    printf("next_reference: %02x%02x%02x%02x\n", 
	   nwos_next_public_ref.id[0],
	   nwos_next_public_ref.id[1],
	   nwos_next_public_ref.id[2],
	   nwos_next_public_ref.id[3]);

    memset(kludge, 0, sizeof(kludge));

    nwos_generate_new_id(&class_ref);

    nwos_fill_in_common_header(&class_obj_ptr->header.common, &class_ref, &nwos_public_class_definition_class_ref);

    nwos_create_name(NEW_CLASS_NAME, &class_obj_ptr->name);

    nwos_create_reference_list(&class_ref, &class_obj_ptr->header.object.references);

    printf("class_ref: %02x%02x%02x%02x\n",
	   class_ref.id[0],
	   class_ref.id[1],
	   class_ref.id[2],
	   class_ref.id[3]);

    add_feature(class_obj_ptr, byte_class_ref(),   "sha512sum", 64);

    nwos_crc32_calculate((uint8*) &class_obj_ptr->header.object,
			 sizeof(ObjectHeader),
			 class_obj_ptr->header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &class_obj_ptr->name,
			 get_class_object_size(class_obj_ptr) - sizeof(EveryObject),
			 class_obj_ptr->header.common.data_chksum);

    nwos_write_object_to_disk(&class_ref, class_obj_ptr, get_class_object_size(class_obj_ptr));

    nwos_add_to_references(&class_ref, &nwos_public_class_definition_class_ref);
    nwos_add_to_references(&class_ref, &class_obj_ptr->name);

    printf("  number of features: %d\n", class_obj_ptr->count);

    nwos_terminate_objectify();

    return 0;
}

