/*
--             This file is part of the New World OS project
--                 Copyright (C) 2006-2009  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: update_files_0028.c,v $
-- Revision 1.9  2009/03/10 10:30:27  jsedwards
-- Eliminated the use of FILE_BLOCK_SIZE for kludge buffers.
--
-- Revision 1.8  2009/03/08 00:37:42  jsedwards
-- Changed include objectify_private.h to disk_io.h.
--
-- Revision 1.7  2008/09/01 03:13:45  jsedwards
-- Change for new nwos_initialize_objectify calling convention (doesn't pass
-- back root_object_reference anymore) and removed call to nwos_set_root_object
-- because initialize calls it now.
--
-- Revision 1.6  2008/08/31 21:53:54  jsedwards
-- Added an assert around calls to nwos_read_variable_sized_object_from_disk
-- and nwos_read_object_from_disk because now they return false when they fail
-- instead of asserting themselves.
--
-- Revision 1.5  2008/08/30 12:46:20  jsedwards
-- Removed code and variables to read pass phrase and pass it to initialize,
-- and change parameters passed to initialize.
--
-- Revision 1.4  2008/07/19 14:11:30  jsedwards
-- Added call to set security level in case updated FILE class is created and
-- call find_or_create FILE class instead of just find FILE class so that the
-- updated FILE class will be cloned if it hasn't been already.
--
-- Revision 1.3  2008/05/24 01:59:18  jsedwards
-- Changed to log arguments instead of disabling logging.
--
-- Revision 1.2  2008/05/21 11:43:55  jsedwards
-- Added code to update the objects and write them back to disk.
--
-- Revision 1.1  2008/05/20 13:43:12  jsedwards
-- Initial version created from attic/fix_img_files.c, which was created from
-- list_files.c.  This version only scans the files does NOT update them.
--
*/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "crc32.h"
#include "disk_io.h"       /* define nwos_get_public_objects_path */
#include "time_stamp.h"


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}


void print_time(TimeStamp ts)
{
    if (nwos_time_stamp_is_zero(ts))
    {
	printf("  unknown");
    }
    else
    {
	printf("  %u-%02u-%02u %02u:%02u:%02u",
	       nwos_extract_year_from_time_stamp(ts),
	       nwos_extract_month_from_time_stamp(ts),
	       nwos_extract_day_of_month_from_time_stamp(ts),
	       nwos_extract_hour_from_time_stamp(ts),
	       nwos_extract_minute_from_time_stamp(ts),
	       nwos_extract_second_from_time_stamp(ts));
    }
}


struct
{
  ObjRef prev_assoc;
  ObjRef prev_file;
  ObjRef next_assoc;
  ObjRef next_file;
} revision_list[4096];


int main(int argc, char* argv[])
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_Path_And_File_Association prev_assoc_obj;
    C_struct_Path_And_File_Association next_assoc_obj;
    uint8 kludge[MAX_PATH_OBJ_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    C_struct_File file_obj;
    C_struct_File prev_file_obj;
    C_struct_File next_file_obj;
    ObjRef object_class;
    ObjRef file_class_ref;
    ObjRef assoc_class_ref;
    ReferenceList* ref_list;
    int num_refs;
    int num_files = 0;
    int i;
    int j;
    char name[256];
    ObjRef file_001_class_ref;
    ObjRef file_002_class_ref;
    ObjRef prev_class;
    ObjRef next_class;
    int num_revised = 0;

    char* path = DEFAULT_FILE;

    if (argc != 1)
    {
	fprintf(stderr, "usage: %s\n", argv[0]);
	fprintf(stderr, "\n");
	fprintf(stderr, "  This program moves the file revision links from the FILE objects to the PATH\n");
	fprintf(stderr,	"  AND FILE ASSOCIATION objects.\n");
	exit(1);
    }

    nwos_log_arguments(argc, argv);

    printf("\n");

    nwos_initialize_objectify(READ_WRITE, path);


    nwos_set_security_level(Security_Very_Low);   /* this only needs to be set if there is no latest file class */

    if (!nwos_find_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref))
    {
	printf("No files are stored in the system!\n");
	exit(1);
    }

    /* make sure the latest file class exists */
    if (nwos_find_or_create_private_class_definition("FILE", &file_class_ref) == CREATED_NEW)
    {
	fprintf(stderr, "WARNING: created new file class: %08x\n", nwos_ref_to_word(&file_class_ref));
    }

    nwos_read_object_headers_from_disk(&file_class_ref, &class_def_obj.header);

    assert(!is_void_reference(&class_def_obj.header.object.prev_version));

    /* This could be bad: newly created objects have old public file in previous version field */

    if (nwos_reference_type(&class_def_obj.header.object.prev_version) == Private_Reference)
    {
	copy_reference(&file_002_class_ref, &class_def_obj.header.object.prev_version);

	nwos_read_object_headers_from_disk(&file_002_class_ref, &class_def_obj.header);

	assert(is_same_object(&class_def_obj.header.object.next_version, &file_class_ref));
	assert(!is_void_reference(&class_def_obj.header.object.prev_version));

	if (nwos_reference_type(&class_def_obj.header.object.prev_version) == Private_Reference)
	{
	    copy_reference(&file_001_class_ref, &class_def_obj.header.object.prev_version);

	    nwos_read_object_headers_from_disk(&file_001_class_ref, &class_def_obj.header);

	    assert(is_same_object(&class_def_obj.header.object.next_version, &file_002_class_ref));
	    assert(is_void_reference(&class_def_obj.header.object.prev_version));
	}
    }

    printf("PATH AND FILE ASSOCIATION ref: %08x\n", nwos_ref_to_word(&assoc_class_ref));
    printf("FILE 001 ref: %08x\n", nwos_ref_to_word(&file_001_class_ref));
    printf("FILE 002 ref: %08x\n", nwos_ref_to_word(&file_002_class_ref));
    printf("FILE 003 ref: %08x\n", nwos_ref_to_word(&file_class_ref));

    nwos_read_class_definition(&assoc_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    printf("num_refs: %d\n", num_refs);

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &assoc_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &assoc_obj, sizeof(assoc_obj)));

	    assert(nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj)));

	    if (!is_void_reference(&file_obj.header.object.prev_version) | !is_void_reference(&file_obj.header.object.next_version))

	    {
		assert(nwos_read_variable_sized_object_from_disk(&assoc_obj.path, kludge, sizeof(kludge), &get_path_object_size));

		/* remember ptr_path_obj points to the kludge buffer */

		for (j = 0; j < ptr_path_obj->count; j++) name[j] = ptr_path_obj->storage[j];

		name[j] = '\0';

		printf("%s", name);

		if (!is_void_reference(&file_obj.header.object.prev_version))
		{
		    nwos_get_object_class_without_update(&file_obj.header.object.prev_version, &prev_class);

		    assert(is_same_object(&prev_class, &file_001_class_ref) ||
			   is_same_object(&prev_class, &file_002_class_ref) ||
			   is_same_object(&prev_class, &file_class_ref));

		    for (j = 0; j < num_revised; j++)
		    {
			if (is_same_object(&revision_list[j].next_file, &assoc_obj.file)) break;
		    }

		    assert(j < num_revised);

		    assert(is_same_object(&file_obj.header.object.prev_version, &revision_list[j].prev_file));

		    copy_reference(&revision_list[j].next_assoc, &ref_list->references[i]);
		}
		else
		{
		    void_reference(&prev_class);
		}

		if (!is_void_reference(&file_obj.header.object.next_version))
		{
		    nwos_get_object_class_without_update(&file_obj.header.object.next_version, &next_class);

		    assert(is_same_object(&next_class, &file_001_class_ref) ||
			   is_same_object(&next_class, &file_002_class_ref) ||
			   is_same_object(&next_class, &file_class_ref));

		    copy_reference(&revision_list[num_revised].prev_assoc, &ref_list->references[i]);
		    copy_reference(&revision_list[num_revised].prev_file, &assoc_obj.file);
		    copy_reference(&revision_list[num_revised].next_file, &file_obj.header.object.next_version);

		    num_revised++;
		}
		else
		{
		    void_reference(&next_class);
		}

		printf("  assoc ref: %08x\n", nwos_ref_to_word(&ref_list->references[i]));
		printf("   path ref: %08x\n", nwos_ref_to_word(&assoc_obj.path));
		printf("   file ref: %08x\n", nwos_ref_to_word(&assoc_obj.file));
		printf("  file prev: %08x   class: %08x\n", nwos_ref_to_word(&file_obj.header.object.prev_version),
		       nwos_ref_to_word(&prev_class));
		printf("  file next: %08x   class: %08x\n", nwos_ref_to_word(&file_obj.header.object.next_version),
		       nwos_ref_to_word(&next_class));

		printf("\n");

		num_files++;
	    }

	    //    memcpy(file_ref, &ptr_path_obj->file, sizeof(*file_ref));
	}
    }

    printf("Number of files: %d\n", num_files);
    printf("Number of revisions: %d\n", num_revised);

    for (i = 0; i < num_revised; i++)
    {
	printf("prev_assoc: %08x  prev_file: %08x  next_assoc: %08x  next_file: %08x\n",
	       nwos_ref_to_word(&revision_list[i].prev_assoc),
	       nwos_ref_to_word(&revision_list[i].prev_file),
	       nwos_ref_to_word(&revision_list[i].next_assoc),
	       nwos_ref_to_word(&revision_list[i].next_file));

	assert(nwos_read_object_from_disk(&revision_list[i].prev_assoc, &prev_assoc_obj, sizeof(prev_assoc_obj)));
	assert(nwos_read_object_from_disk(&revision_list[i].prev_file,  &prev_file_obj,  sizeof(prev_file_obj)));
	assert(nwos_read_object_from_disk(&revision_list[i].next_assoc, &next_assoc_obj, sizeof(next_assoc_obj)));
	assert(nwos_read_object_from_disk(&revision_list[i].next_file,  &next_file_obj,  sizeof(next_file_obj)));

	assert(is_void_reference(&prev_assoc_obj.header.object.next_version));
	assert(is_void_reference(&next_assoc_obj.header.object.prev_version));

	assert(is_same_object(&prev_file_obj.header.object.next_version, &revision_list[i].next_file));
	assert(is_same_object(&next_file_obj.header.object.prev_version, &revision_list[i].prev_file));

	/* void the next_version in the previous file object */
	void_reference(&prev_file_obj.header.object.next_version);
	nwos_crc32_calculate((uint8*) &prev_file_obj.header.object, sizeof(ObjectHeader), prev_file_obj.header.common.header_chksum);

	nwos_overwrite_object_to_disk(&revision_list[i].prev_file, &prev_file_obj, sizeof(prev_file_obj));

	assert(nwos_read_object_from_disk(&revision_list[i].prev_file, &file_obj, sizeof(file_obj)));
	assert(memcmp(&prev_file_obj, &file_obj, sizeof(file_obj)) == 0);

	/* void the prev_version in the next file object */
	void_reference(&next_file_obj.header.object.prev_version);
	nwos_crc32_calculate((uint8*) &next_file_obj.header.object, sizeof(ObjectHeader), next_file_obj.header.common.header_chksum);
	nwos_overwrite_object_to_disk(&revision_list[i].next_file, &next_file_obj, sizeof(next_file_obj));

	assert(nwos_read_object_from_disk(&revision_list[i].next_file, &file_obj, sizeof(file_obj)));
	assert(memcmp(&next_file_obj, &file_obj, sizeof(file_obj)) == 0);

	/* point the previous assoc object to the next assoc object */
	copy_reference(&prev_assoc_obj.header.object.next_version, &revision_list[i].next_assoc);
	nwos_crc32_calculate((uint8*) &prev_assoc_obj.header.object, sizeof(ObjectHeader), prev_assoc_obj.header.common.header_chksum);
	nwos_overwrite_object_to_disk(&revision_list[i].prev_assoc, &prev_assoc_obj, sizeof(prev_assoc_obj));

	assert(nwos_read_object_from_disk(&revision_list[i].prev_assoc, &assoc_obj, sizeof(assoc_obj)));
	assert(memcmp(&prev_assoc_obj, &assoc_obj, sizeof(assoc_obj)) == 0);

	/* point the next assoc object to the previous assoc object */
	copy_reference(&next_assoc_obj.header.object.prev_version, &revision_list[i].prev_assoc);
	nwos_crc32_calculate((uint8*) &next_assoc_obj.header.object, sizeof(ObjectHeader), next_assoc_obj.header.common.header_chksum);
	nwos_overwrite_object_to_disk(&revision_list[i].next_assoc, &next_assoc_obj, sizeof(next_assoc_obj));

	assert(nwos_read_object_from_disk(&revision_list[i].next_assoc, &assoc_obj, sizeof(assoc_obj)));
	assert(memcmp(&next_assoc_obj, &assoc_obj, sizeof(assoc_obj)) == 0);
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    nwos_terminate_objectify();

    return 0;
}


