/*
--             This file is part of the New World OS project
--                 Copyright (C) 2006-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: disc_list.c,v $
-- Revision 1.28  2008/08/03 17:31:34  jsedwards
-- Add --ignore-empty-directories and --ignore-empty-files options.
--
-- Revision 1.27  2008/08/03 15:44:36  jsedwards
-- Changed to handle new error values returned by nwos_read_files_disc_list.
--
-- Revision 1.26  2008/07/26 15:09:02  jsedwards
-- Changed to use returned value from nwos_read_files_disc_list for exit code
-- if it fails.
--
-- Revision 1.25  2008/07/24 16:40:00  jsedwards
-- Moved read_file_names function to file.c and rename to
-- nwos_read_files_disc_list.
--
-- Revision 1.24  2008/07/24 14:20:59  jsedwards
-- Change to read subdirectories recursively.
--
-- Revision 1.23  2008/07/24 11:25:18  jsedwards
-- Move code that scans for all file names out of "main" and into new function
-- "read_file_names".  Also moved "num_files", "file_names" and "files"
-- variables out of "main" and made global to this file.
--
-- Revision 1.22  2008/05/22 10:04:55  jsedwards
-- Fix Bug #1954154 - Changed so that revised file links are in the PATH AND
-- FILE ASSOCIATION objects instead of the FILE objects.
--
-- Revision 1.21  2008/03/30 13:38:45  jsedwards
-- Moved code to get disc ID before checking public files as well.
--
-- Revision 1.20  2008/03/30 13:07:41  jsedwards
-- Move code to check public files out of the initial directory scan loop and
-- into it's own loop.  This way we can get the pass phrase first instead of
-- having a delay while it reads public files and then asking for the pass
-- phrase.
--
-- Revision 1.19  2008/03/21 14:48:22  jsedwards
-- Added code to check for public files.
--
-- Revision 1.18  2008/02/03 01:06:30  jsedwards
-- Changed DEFAULT_TYPE_RW to READ_WRITE.
--
-- Revision 1.17  2007/12/24 02:20:53  jsedwards
-- Added --add-revision option to connect any new files with any existing files.
--
-- Revision 1.16  2007/12/23 18:23:25  jsedwards
-- Added terminate_objectify call before exiting in disc already exists case.
--
-- Revision 1.15  2007/10/07 03:43:50  jsedwards
-- Renamed 'nwos_set_block_estimate' to 'nwos_check_blocks_available' and
-- changed to exit gracefully if not enough space available.
--
-- Revision 1.14  2007/09/02 19:47:25  jsedwards
-- Added call to set the block estimate.
--
-- Revision 1.13  2007/08/12 20:45:00  jsedwards
-- Change all of the "Encryption Level" stuff to "Security Level" because it
-- doesn't really change the encryption at all, all it does is change the
-- randomization of where objects are stored.
--
-- Revision 1.12  2007/07/01 19:44:11  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.11  2007/06/13 21:15:03  jsedwards
-- Comment out debugging print statement accidentally left in.
--
-- Revision 1.10  2007/06/13 20:51:39  jsedwards
-- Added code to free memory used for file names.
--
-- Revision 1.9  2007/06/13 20:13:54  jsedwards
-- Change to take a directory as a parameter instead of a list of files.
--
-- Revision 1.8  2007/02/11 16:58:26  jsedwards
-- Changed so DEFAULT_TYPE has to specify RO (Read-Only) or RW (Read-Write).
--
-- Revision 1.7  2007/01/20 13:07:26  jsedwards
-- Print blank line after entering pass phrase and move log_arguments call.
--
-- Revision 1.6  2007/01/07 20:31:55  jsedwards
-- Added call to log arguments.
--
-- Revision 1.5  2007/01/07 03:26:19  jsedwards
-- Changed default encryption level to 'Very_Low' instead of 'Low'.
--
-- Revision 1.4  2007/01/04 17:39:14  jsedwards
-- Add code to set encryption level (always "Low" for now).
--
-- Revision 1.3  2006/12/07 14:10:25  jsedwards
-- Removed call to file_setup, no longer needed.
--
-- Revision 1.2  2006/12/02 17:26:09  jsedwards
-- Changed to deal with file paths that can point to different files.
--
-- Revision 1.1  2006/11/19 13:53:12  jsedwards
-- Created "disc_list.c" from "log_disc.c".  This program just creates a disc
-- list and does not create a disc copy object.
--
-- Revision 1.6  2006/11/18 14:34:58  jsedwards
-- Changed so that discs can be stored in one of three different binders.
--
-- Revision 1.5  2006/11/11 12:01:04  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.4  2006/11/07 14:03:35  jsedwards
-- Add assert to verify file_path didn't come back void.
--
-- Revision 1.3  2006/11/06 13:54:07  jsedwards
-- Changed wording in print statement because disc_list now stores the
-- references to the file_path object instead of the file object.
--
-- Revision 1.2  2006/11/05 21:32:18  jsedwards
-- Add code to create storage_location and disc_copy objects.
--
-- Revision 1.1  2006/11/04 18:56:22  jsedwards
-- Program to scan all the files on a CD or DVD and make a disk_list object
-- of it.
--
*/

#include <assert.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>
#include "objectify_private.h"     /* this is kinda bad, but need overwrite to link old and new revisions */
#include "crc32.h"


static char* file_names[MAX_FILES_PER_DISC_LIST];
static ObjRef files[MAX_FILES_PER_DISC_LIST];


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}


void check_for_previous_version_and_link(ObjRef* new_assoc_ref)
{
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_Path_And_File_Association old_assoc_obj;
    C_struct_Path_And_File_Association new_assoc_obj;
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    ObjRef object_class;

    nwos_read_object_from_disk(new_assoc_ref, &new_assoc_obj, sizeof(new_assoc_obj));

    assert(is_void_reference(&new_assoc_obj.header.object.prev_version));

    nwos_read_variable_sized_object_from_disk(&new_assoc_obj.path, kludge, sizeof(kludge), &get_path_object_size);

    ref_list = nwos_malloc_reference_list(&ptr_path_obj->header.object.references);

    num_refs = ref_list->common_header.num_refs;

    assert(is_same_object(&ref_list->references[num_refs-1], new_assoc_ref));

    if (num_refs > 1)
    {
	for (i = 0; i < num_refs - 1; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);
	    if (is_same_object(&object_class, &new_assoc_obj.header.common.class_definition))
	    {
		nwos_read_object_from_disk(&ref_list->references[i], &old_assoc_obj, sizeof(old_assoc_obj));

		printf("Old association: %08x  file: %08x  next_version: %08x\n",
		       nwos_ref_to_word(&ref_list->references[i]),
		       nwos_ref_to_word(&old_assoc_obj.file),
		       nwos_ref_to_word(&old_assoc_obj.header.object.next_version));

		if (is_void_reference(&old_assoc_obj.header.object.next_version)) break;
	    }
	}

	if (i < num_refs - 1)
	{
	    assert(is_same_object(&new_assoc_obj.header.common.id, new_assoc_ref));
	    assert(is_void_reference(&old_assoc_obj.header.object.next_version));
	    assert(is_void_reference(&new_assoc_obj.header.object.prev_version));
	    assert(is_same_object(&old_assoc_obj.path, &new_assoc_obj.path));

	    copy_reference(&old_assoc_obj.header.object.next_version, &new_assoc_obj.header.common.id);
	    copy_reference(&new_assoc_obj.header.object.prev_version, &old_assoc_obj.header.common.id);

	    nwos_crc32_calculate((uint8*) &old_assoc_obj.header.object, sizeof(ObjectHeader), old_assoc_obj.header.common.header_chksum);
	    nwos_crc32_calculate((uint8*) &new_assoc_obj.header.object, sizeof(ObjectHeader), new_assoc_obj.header.common.header_chksum);

	    nwos_overwrite_object_to_disk(&old_assoc_obj.header.common.id, &old_assoc_obj, sizeof(old_assoc_obj));
	    nwos_overwrite_object_to_disk(&new_assoc_obj.header.common.id, &new_assoc_obj, sizeof(new_assoc_obj));

	    printf("linked - old: %08x and new: %08x\n",
		   nwos_ref_to_word(&old_assoc_obj.header.common.id),
		   nwos_ref_to_word(&new_assoc_obj.header.common.id));

	    nwos_read_object_from_disk(&old_assoc_obj.header.common.id, &assoc_obj, sizeof(assoc_obj));
	    assert(memcmp(&assoc_obj, &old_assoc_obj, sizeof(assoc_obj)) == 0);

	    nwos_read_object_from_disk(&new_assoc_obj.header.common.id, &assoc_obj, sizeof(assoc_obj));
	    assert(memcmp(&assoc_obj, &new_assoc_obj, sizeof(assoc_obj)) == 0);
	}

    }
}


int main(int argc, char* argv[])
{
    ObjRef root_object_ref;
    ObjRef public_ref;
    uint8 big_key[16 + 8 + 4];
    uint8 bf_key[16];
    uint32 linear;
    uint32 serial;
    ObjRef ref;
    ObjCreateResult result;
    int i;
    time_t start_time;
    char id[14];
    char *p;
    bool ok = false;
    uint32 num_files;
    ObjRef disc_list_ref;
    bool add_revision = false;
    bool ignore_public = false;
    bool public_file_mismatch = false;
    uint32 error_mask = DISC_LIST_ERROR_MASK;
    int argi = 1;

    while (argi < argc - 1)
    {
	if (strcmp(argv[argi], "--add-revision") == 0)
	{
	    add_revision = true;
	    argi++;
	}
	else if (strcmp(argv[argi], "--ignore-public") == 0)
	{
	    ignore_public = true;
	    argi++;
	}
	else if (strcmp(argv[argi], "--ignore-empty-directories") == 0)
	{
	    error_mask &= ~DISC_LIST_EMPTY_DIR_FLAG;
	    argi++;
	}
	else if (strcmp(argv[argi], "--ignore-empty-files") == 0)
	{
	    error_mask &= ~DISC_LIST_EMPTY_FILE_FLAG;
	    argi++;
	}
    }

    if (argi != argc - 1)
    {
	fprintf(stderr, "usage: %s [options] directory\n", argv[0]);
	fprintf(stderr, "  options:\n");
	fprintf(stderr, "    --add-revision              if file already exists with same name, make it a new revision\n");
	fprintf(stderr, "    --ignore-public             don't check public files for matches\n");
	fprintf(stderr, "    --ignore-empty-directories  ignore any empty directories\n");
	fprintf(stderr, "    --ignore-empty-files        ignore any empty files\n");
	exit(1);
    }

    printf("\n");

    /***********************************************************/
    /* First open the directory and read all of the file names */
    /***********************************************************/

    num_files = nwos_read_files_disc_list(argv[argi], NULL, file_names, 0);    /* NULL for subdirectory indicates this is the root */

    if ((num_files & DISC_LIST_NUM_FILES_MASK) == 0)
    {
	fprintf(stderr, "Error: directory '%s' is empty!\n\n", argv[argi]);

	exit(16);  /* 16 means directory is empty */
    }

    if ((num_files & error_mask) != 0)
    {
	fprintf(stderr, "\n");

	exit(num_files >> DISC_LIST_ERROR_SHIFT);  /* exit code is error code returned by nwos_read_files_disc_list 1-15 */
    }

    num_files &= DISC_LIST_NUM_FILES_MASK;

    /*********************************/
    /* Get the pass phrase from user */
    /*********************************/

    printf("\n");

    nwos_get_key_from_password(big_key, sizeof(big_key));

    printf("\n");


    /*******************/
    /* Get the disc id */
    /*******************/

    while (!ok)
    {
	printf("Disk id: ");
	fflush(stdout);

	fgets(id, sizeof(id), stdin);

	p = strchr(id, '\n');   /* find the newline char */

	if (p == NULL)    /* line was tool long */
	{
	    while (p == NULL) 
	    {
		fgets(id, sizeof(id), stdin);
		p = strchr(id, '\n');   /* find the newline char */
	    }
	    printf("id was too long, must be less than %zd characters\n", sizeof(id) - 2);
	}
	else     /* line was ok */
	{
	    *p = '\0';   /* eliminate the newline character */

	    if (strlen(id) < 12)
	    {
		printf("id was too short, must be %zd characters\n", sizeof(id) - 2);
	    }
	    else
	    {
		ok = true;   /* we should be good to go */
	    }
	}
    }


    /*******************************************************************************************/
    /* Before logging any files, open Objectify in public, read only mode and scan all of the  */
    /* files to see if any of their names match any public files.  If any of them do, checksum */
    /* them and if the checksum of any of them doesn't match report an error and exit.         */
    /*******************************************************************************************/

    if (!ignore_public)
    {
	nwos_initialize_objectify(NULL, 0, 0, PUBLIC, NULL);

	root_object_ref.id[0] = 0;
	root_object_ref.id[1] = 0;
	root_object_ref.id[2] = 0;
	root_object_ref.id[3] = 1;

	nwos_set_root_object(&root_object_ref);

	for (i = 0; i < num_files; i++)
	{
	    p = strrchr(file_names[i], '/');

	    if (p == NULL)    /* no path in file name */
	    {
		p = file_names[i];   /* use it as is */
	    }
	    else
	    {
		p++;   /* point to first character following the slash */
	    }

	    if (nwos_find_file_path(p, &public_ref))
	    {
		printf("checking public file %s: ", p);
		fflush(stdout);

		/* NOTE: when the time comes that we are doing recursive directory searches this will have to be
		         fixed so that the path includes the directory and the part of the path in the file name */
		if (nwos_find_matching_path_and_file_association(argv[argi], file_names[i], &public_ref, IgnoreTime))
		{
		    printf(" OK\n");
		}
		else
		{
		    printf(" does not match public file information!\n");
		    public_file_mismatch = true;
		}
	    }
	}

	nwos_terminate_objectify();
    }

    if (public_file_mismatch)
    {
	fprintf(stderr, "\n");
	fprintf(stderr, "ERROR: disc_list not created!  One or more files with the same name as a public\n");
	fprintf(stderr, "       file were found but the checksums don't match.  You may have a bad\n");
        fprintf(stderr, "       download.  To ignore any public files use the --ignore-public option.\n");
	fprintf(stderr, "\n");

	exit(1);
    }

    printf("\n");


    /***********************************************************************/
    /* Now open Objectify in private, read-write mode to do the real thing */
    /***********************************************************************/

    memcpy(bf_key, big_key, 16);
    linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
    memcpy(root_object_ref.id, big_key+20, 4);
    serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

    nwos_log_arguments(argc, argv);

    nwos_initialize_objectify(bf_key, linear, serial, READ_WRITE, DEFAULT_FILE);

    nwos_set_root_object(&root_object_ref);

    nwos_set_security_level(Security_Very_Low);

    if (!nwos_check_blocks_available(2 + num_files * 12)) /* 1 file = 14, 2 files = 26, 3 files = 38... */
    {
	fprintf(stderr, "Cannot add disc_list!\n");
	nwos_terminate_objectify();
	exit(1);
    }

    if (nwos_find_disc_list(id, &ref))
    {
	printf("That disc already exists!\n");
	nwos_terminate_objectify();
	exit(1);
    }

    memset(files, 0, sizeof(files));

    for (i = 0; i < num_files; i++)
    {
	start_time = time(NULL);

	result = nwos_create_file_without_storing_data(argv[argi], file_names[i], &files[i]);

	assert(!is_void_reference(&files[i]));

	if (result == CREATED_NEW)
	{
	    printf("created new file path: %02x%02x%02x%02x   time: %d seconds\n", 
		   files[i].id[0], files[i].id[1], files[i].id[2], files[i].id[3], 
		   (int) (time(NULL) - start_time));

	    if (add_revision)   /* check to see if there was a previous revision of this file */
	    {
		check_for_previous_version_and_link(&files[i]);
	    }
	}
	else
	{
	    printf("result: %d\n", result);
	}

	nwos_flush_bit_maps();
    }

    assert(nwos_create_disc_list(id, files, &disc_list_ref) == CREATED_NEW);

    printf("disc list: %02x%02x%02x%02x\n",
	   disc_list_ref.id[0], disc_list_ref.id[1], disc_list_ref.id[2], disc_list_ref.id[3]);

    for (i = 0; i < num_files; i++)
    {
	free(file_names[i]);
    }

    nwos_terminate_objectify();

    return 0;
}


