/*
--             This file is part of the New World OS project
--                   Copyright (C) 2007  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: device.c,v $
-- Revision 1.4  2007/07/01 19:44:11  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.3  2007/05/26 19:56:28  jsedwards
-- Fixed to handle lists of url for manufacturer.
--
-- Revision 1.2  2007/05/25 13:38:26  jsedwards
-- Fixed find_public_manufacturer function and created create_public_manufacturer
-- from code extracted from 'add_manufacturer.c'.
--
-- Revision 1.1  2007/05/24 13:24:49  jsedwards
-- Initial version, only function working is list_public_manufacturers.
--
*/


#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>   /* define memset */

#include "crc32.h"
#include "objectify_private.h"

#define MAX_NAME_SIZE 128
#define MAX_NUMBER_SIZE 18
#define MAX_ID_SIZE 4



bool nwos_find_public_manufacturer(char* name, ObjRef* ref)
{
    C_struct_Manufacturer mfr_obj;
    EveryObject name_header;
    ObjRef mfr_class_ref;
    ObjRef nora_ref;       /* name or acronym reference */
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    bool result = false;


    assert(nwos_find_public_class_definition("MANUFACTURER", &mfr_class_ref));

    if (nwos_find_public_name(name, &nora_ref) || nwos_find_public_acronym(name, &nora_ref))
    {
	nwos_read_object_headers_from_disk(&nora_ref, &name_header);

	ref_list = nwos_malloc_reference_list(&name_header.object.references);

	num_refs = ref_list->common_header.num_refs;

	/* printf("number of public manufacturers: %d\n", num_refs); */

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	    if (is_same_object(&object_class, &mfr_class_ref))   /* it is a word object */
	    {
		nwos_read_object_from_disk(&ref_list->references[i], &mfr_obj, sizeof(mfr_obj));

		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		result = true;
		break;
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }

    return result;
}


#ifdef PUBLIC_MODE
/* url_list needs to have a null at the end */
void nwos_create_public_manufacturer(char* brand_name, char* acronym, char* url_list[], ObjRef* ref)
{
    C_struct_Manufacturer mfr_obj;
    ObjRef mfr_class_ref;
    ObjRef url_ref[64];
    int i;

    assert(brand_name != NULL || acronym != NULL);   /* at least one has to be non-null */

    if (brand_name != NULL)
    {
	assert(!nwos_find_public_manufacturer(brand_name, ref));
    }

    if (acronym != NULL)
    {
	assert(!nwos_find_public_manufacturer(acronym, ref));
    }

    assert(nwos_find_public_class_definition("MANUFACTURER", &mfr_class_ref));

    memset(&mfr_obj, 0, sizeof(mfr_obj));

    nwos_generate_new_id(ref);

    nwos_fill_in_common_header(&mfr_obj.header.common, ref, &mfr_class_ref);

    if (brand_name != NULL)
    {
	nwos_create_name(brand_name, &mfr_obj.brand_name);
    }

    if (acronym != NULL)
    {
	nwos_create_public_acronym(acronym, &mfr_obj.acronym);
    }

    if (url_list != NULL)
    {
	nwos_create_reference_list(ref, &mfr_obj.url_list);
	for (i = 0; url_list[i] != NULL; i++)
	{
	    nwos_create_url(url_list[i], &url_ref[i]);
	    nwos_add_to_reference_list(&url_ref[i], &mfr_obj.url_list);
	}
    }

    nwos_create_reference_list(ref, &mfr_obj.header.object.references);

    nwos_crc32_calculate((uint8*) &mfr_obj.header.object, sizeof(ObjectHeader), mfr_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &mfr_obj.corporation, sizeof(mfr_obj) - sizeof(EveryObject), mfr_obj.header.common.data_chksum);

    nwos_write_object_to_disk(ref, &mfr_obj, sizeof(mfr_obj));

    nwos_add_to_references(ref, &mfr_class_ref);

    if (!is_void_reference(&mfr_obj.brand_name))
    {
	nwos_add_to_references(ref, &mfr_obj.brand_name);
    }

    if (!is_void_reference(&mfr_obj.acronym))
    {
	nwos_add_to_references(ref, &mfr_obj.acronym);
    }

    if (!is_void_reference(&mfr_obj.url_list))
    {
	for (i = 0; url_list[i] != NULL; i++)
	{
	    nwos_add_to_references(ref, &url_ref[i]);
	}
    }

    printf("Object created: %02x%02x%02x%02x\n", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
}
#endif


void nwos_list_public_manufacturers()
{
    C_struct_Manufacturer mfr_obj;
    ObjRef object_class;
    EveryObject class_ref_header;
    ObjRef mfr_class_ref;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    int j;
    char name[32];
    ReferenceList* url_list;
    int num_urls;

    assert(nwos_find_public_class_definition("MANUFACTURER", &mfr_class_ref));

    nwos_read_object_headers_from_disk(&mfr_class_ref, &class_ref_header);

    ref_list = nwos_malloc_reference_list(&class_ref_header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("mfr class num refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	if (is_same_object(&object_class, &mfr_class_ref))
	{
	    nwos_read_object_from_disk(&ref_list->references[i], &mfr_obj, sizeof(mfr_obj));

	    if (!is_void_reference(&mfr_obj.brand_name))
	    {
		nwos_name_to_string(&mfr_obj.brand_name, name, sizeof(name));
		printf("%s", name);
	    }

	    if (!is_void_reference(&mfr_obj.acronym))
	    {
		assert(nwos_acronym_to_string(&mfr_obj.acronym, name, sizeof(name)));
		if (!is_void_reference(&mfr_obj.brand_name)) printf(" - ");
		printf("%s", name);
	    }

	    if (!is_void_reference(&mfr_obj.url_list))
	    {
		url_list = nwos_malloc_reference_list(&mfr_obj.url_list);

		num_urls = url_list->common_header.num_refs;

		printf(" - ");

		for (j = 0; j < num_urls; j++)
		{
		    assert(nwos_url_to_string(&url_list->references[j], name, sizeof(name)));
		    if (j > 0) printf(", ");
		    printf("%s", name);
		}

		nwos_free_reference_list(url_list);
		url_list = NULL;
	    }

	    printf("\n");
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;
}

#if 0
bool nwos_find_private_credit_union(char* name, ObjRef* ref)
{
    ObjRef private_class_ref;
    ObjRef public_ref;
    ObjRef object_class;
    C_struct_Class_Definition class_def_obj;
    C_struct_Credit_Union cu_obj;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    bool result = false;

    assert(nwos_find_public_credit_union(name, &public_ref));

    if (nwos_find_private_class_definition("CREDIT UNION", &private_class_ref))
    {
	nwos_read_class_definition(&private_class_ref, &class_def_obj);

	ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

	num_refs = ref_list->common_header.num_refs;

	/* printf("num_refs: %d\n", num_refs); */

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);

	    if (is_same_object(&object_class, &private_class_ref))
	    {
		nwos_read_object_from_disk(&ref_list->references[i], &cu_obj, sizeof(cu_obj));

		if (is_same_object(&cu_obj.header.object.clone_of, &public_ref))   /* found a match */
		{
		    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		    result = true;
		    break;
		}
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }

    return result;
}


/**********************************************************************************************/
/* This routine finds a private credit union object or creates one by cloning the public one. */
/**********************************************************************************************/

ObjCreateResult nwos_find_or_create_private_credit_union(char* name, ObjRef* ref)
{
    ObjRef public_cu_ref;
    ObjRef private_class_ref;
    C_struct_Credit_Union cu_obj;
    char msg[128];
    ObjCreateResult result = FOUND_EXISTING;

    assert(nwos_find_public_credit_union(name, &public_cu_ref));

    if (!nwos_find_private_credit_union(name, ref))
    {
	nwos_find_or_create_private_class_definition("CREDIT UNION", &private_class_ref);

	nwos_read_object_from_disk(&public_cu_ref, &cu_obj, sizeof(cu_obj));

	nwos_generate_new_completely_random_id(ref);

	snprintf(msg, sizeof(msg), "clone_credit_union(%02x%02x%02x%02x) -> %02x%02x%02x%02x\n",
		 public_cu_ref.id[0], public_cu_ref.id[1], public_cu_ref.id[2], public_cu_ref.id[3],
		 ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	nwos_log(msg);

	copy_reference(&cu_obj.header.common.id, ref);

	nwos_get_time_stamp(cu_obj.header.common.creation_time);

	copy_reference(&cu_obj.header.common.class_definition, &private_class_ref);

	copy_reference(&cu_obj.header.object.clone_of, &public_cu_ref);

	nwos_create_reference_list(ref, &cu_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &cu_obj.header.object, sizeof(ObjectHeader), cu_obj.header.common.header_chksum);

	nwos_write_object_to_disk(ref, &cu_obj, sizeof(cu_obj));

	nwos_add_to_references(ref, &private_class_ref);

	result = CREATED_NEW;
    }

    return result;
}



static void get_input(char* descr, char* buffer, size_t size)
{
    char *ptr;

    while (1)
    {
	printf("%s: ", descr);
	fflush(stdout);
	fgets(buffer, size, stdin);
	ptr = strchr(buffer, '\n');
	if (ptr != NULL)
	{
	    *ptr = '\0';
	    break;
	}
	do { fgets(buffer, size, stdin); } while (strchr(buffer, '\n') == NULL);
	printf("input too long - try again!\n");
    }
}


void nwos_add_account()
{
    char name[MAX_NAME_SIZE+2];
    char number[MAX_NUMBER_SIZE+2];
    char id[MAX_ID_SIZE+2];
    ObjRef cu_ref;
    ObjRef number_ref;
    ObjRef account_ref;
    ObjRef checking_ref;
    bool ok;
    bool cu_existed;
    bool num_existed;

#if 0
    nwos_find_or_create_private_class_definition("PERSON", &person_class_ref);
#endif

    ok = false;
    while (!ok)
    {
	get_input("name of credit union", name, sizeof(name));
	if (*name == '\0') return;
	ok = nwos_find_public_credit_union(name, &cu_ref);
	if (!ok) printf("name not found in public credit union list.\n");
    }

    printf("public credit union: %02x%02x%02x%02x\n",
	   cu_ref.id[0], cu_ref.id[1], cu_ref.id[2], cu_ref.id[3]);


    ok = false;
    while (!ok)
    {
	get_input("account number", number, sizeof(number));
	if (*number == '\0') return;
	ok = valid_number(number);
	if (!ok) printf("number can only contain digits 0 to 9 and dash '-'.\n");
    }

    ok = false;
    while (!ok)
    {
	get_input("checking id", id, sizeof(id));
	if (*id == '\0') return;
	ok = valid_id(id);
	if (!ok) printf("id can only contain digits 0 to 9.\n");
    }

    printf("\n");
    printf("credit union: %s\n", name);
    printf("account number: %s\n", number);
    printf("checking id: %s\n", id);
    printf("\n");

    if (nwos_find_account_number(number, &number_ref))
    {
	printf("WARNING: account number: %s already exists\n\n", number);
    }

    if (nwos_ask_yes_or_no(NULL, "Create account"))
    {
	cu_existed = (nwos_find_or_create_private_credit_union(name, &cu_ref) == FOUND_EXISTING);

	num_existed = (nwos_find_or_create_account_number(number, &number_ref) == FOUND_EXISTING);

	if (cu_existed && num_existed && nwos_find_account(&cu_ref, &number_ref, &account_ref))
	{
	    printf("ERROR: that account already exists\n");
	    return;
	}

	nwos_create_account(&cu_ref, &number_ref, &account_ref);

	nwos_create_checking(&account_ref, id, &checking_ref);

	
	printf("Object created: %02x%02x%02x%02x\n",
	       checking_ref.id[0], checking_ref.id[1], checking_ref.id[2], checking_ref.id[3]);
    }
}


void nwos_list_accounts()
{
    ObjRef checking_class_ref;
    EveryObject class_ref_header;
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Checking* ptr_checking_obj = (C_struct_Checking*)kludge;
    C_struct_Account account_obj;
    C_struct_Credit_Union cu_obj;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    char name[MAX_NAME_SIZE+1];
    char number[MAX_NUMBER_SIZE+1];
    int i;

    if (!nwos_find_private_class_definition("CHECKING", &checking_class_ref))
    {
	printf("No checking accounts found\n");
    }
    else
    {
	nwos_read_object_headers_from_disk(&checking_class_ref, &class_ref_header);

	ref_list = nwos_malloc_reference_list(&class_ref_header.object.references);

	num_refs = ref_list->common_header.num_refs;

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	    if (is_same_object(&object_class, &checking_class_ref))
	    {
		nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &get_checking_object_size);

		nwos_read_object_from_disk(&ptr_checking_obj->account, &account_obj, sizeof(account_obj));

		nwos_read_object_from_disk(&account_obj.institution, &cu_obj, sizeof(cu_obj));

		nwos_account_number_to_string(&account_obj.number, number, sizeof(number));

		if (!is_void_reference(&cu_obj.short_name))
		{
		    nwos_name_to_string(&cu_obj.short_name, name, sizeof(name));
		}
		else
		{
		    nwos_name_to_string(&cu_obj.full_name, name, sizeof(name));
		}

		printf("%s - %s\n", name, number);
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }
}
#endif



#if 0
void nwos_list_credit_unions()
{
    C_struct_Person person_obj;
    ObjRef object_class;
    EveryObject class_ref_header;
    ObjRef person_class_ref;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    char name[32];

    assert(nwos_find_private_class_definition("PERSON", &person_class_ref));

    nwos_read_object_headers_from_disk(&person_class_ref, &class_ref_header);

    ref_list = nwos_malloc_reference_list(&class_ref_header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("person class num refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	if (is_same_object(&object_class, &person_class_ref))
	{
	    nwos_read_object_from_disk(&ref_list->references[i], &person_obj, sizeof(person_obj));

	    nwos_name_to_string(&person_obj.last_name, name, sizeof(name));

	    printf("%s, ", name);

	    nwos_name_to_string(&person_obj.first_name, name, sizeof(name));

	    printf("%s ", name);

	    if (!is_void_reference(&person_obj.middle_name))
	    {
		nwos_name_to_string(&person_obj.middle_name, name, sizeof(name));

		printf("%s", name);
	    }

	    printf("\n");
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;
}


void nwos_describe_person(char* name)
{
    C_struct_Person person_obj;
    ObjRef person_ref;
    char temp[64];
    
    if (!nwos_find_person(name, &person_ref))
    {
	printf("\nI'm sorry I don't know anyone named %s.\n\n", name);
    }
    else
    {
	nwos_read_object_from_disk(&person_ref, &person_obj, sizeof(person_obj));

	printf("full name: ");

	nwos_name_to_string(&person_obj.first_name, temp, sizeof(temp));
	printf("%s ", temp);

	if (!is_void_reference(&person_obj.middle_name))
	{
	    nwos_name_to_string(&person_obj.middle_name, temp, sizeof(temp));
	    printf("%s ", temp);
	}

	nwos_name_to_string(&person_obj.last_name, temp, sizeof(temp));
	printf("%s\n", temp);


	if (!is_void_reference(&person_obj.goes_by))
	{
	    printf("goes by: ");
	    nwos_name_to_string(&person_obj.goes_by, temp, sizeof(temp));
	    printf("%s\n", temp);
	}

	if (!is_void_reference(&person_obj.nickname))
	{
	    printf("nickname: ");
	    nwos_name_to_string(&person_obj.nickname, temp, sizeof(temp));
	    printf("%s\n", temp);
	}

	if (!is_void_reference(&person_obj.maiden_name))
	{
	    printf("maiden_name: ");
	    nwos_name_to_string(&person_obj.maiden_name, temp, sizeof(temp));
	    printf("%s\n", temp);
	}

	if (!is_void_reference(&person_obj.birth_date))
	{
	    printf("born: ");
	    nwos_date_to_string(&person_obj.birth_date, temp, sizeof(temp));
	    printf("%s\n", temp);
	}
    }

    printf("\n");
}
#endif

