/*
--             This file is part of the New World OS project
--                 Copyright (C) 2004-2007  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: reference_list_0014.c,v $
-- Revision 1.3  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.2  2006/12/18 18:34:54  jsedwards
-- Changes to be used to read old compressed version 0014 files.
--
-- Revision 1.1  2006/12/17 14:00:30  jsedwards
-- Version copied from root directory of Version 0014.
--
-- Revision 1.2  2006/12/01 14:24:31  jsedwards
-- Added malloc and free reference list functions.
--
-- Revision 1.1  2006/11/11 15:21:40  jsedwards
-- Moved all reference list code out of objectify.c and into this file.
--
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "../crc32.h"
#include "../objectify_private.h"
#include "objectify_0014.h"



#define REF_LIST_CACHE_SIZE 64

typedef struct {
  ObjRef ref;
  int num_refs;
  Ref_List_First_Block* first_block;
  int age;
  uint8 ivec[IVEC_SIZE];
} Block_Cache;


static int ref_list_tick = 0;

static Block_Cache ref_list_cache[REF_LIST_CACHE_SIZE];


static void check_ref_list_data(ObjRef* ref, uint8* data, size_t data_size, uint8 stored[4])
{
    uint8 computed[4];

#if 0
    printf("Total size: %d\n", total_size);
    printf("ObjectHeader: %d\n", sizeof(EveryObject));
    printf("object pointer: %p\n", object);
    {
      int i;
      uint8* ptr = (uint8*)object + sizeof(EveryObject);
      printf("data pointer: %p\n", ptr);
      printf("data: ");
      for (i = 0; i < data_size; i++) printf("%02x%c", *ptr++, i%16==15?'\n':' ');
      printf("\n");
    }
#endif

    nwos_crc32_calculate((uint8*)data, data_size, computed);

    if (stored[0] != computed[0] || stored[1] != computed[1] || stored[2] != computed[2] || stored[3] != computed[3])
    {
	printf("reference list: %02x%02x%02x%02x - bad data checksum, ", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	printf("size %zd - ", data_size);
	printf("computed %02x%02x%02x%02x - ", computed[0], computed[1], computed[2], computed[3]);
	printf("stored %02x%02x%02x%02x\n", stored[0], stored[1], stored[2], stored[3]);

	printf("\n");
	{
	  int i;
	  for (i = 0; i < data_size; i++)
	  {
	      printf("%02x%c", data[i], (i % 16) == 15 ? '\n' : ' ');
	  }
	  printf("\n");
	}
	exit(1);
    }
}


static void read_ref_list_into_cache(Block_Cache* cache)
{
    uint8 ivec[IVEC_SIZE];
    Ref_List_Extra_Block* prev_ptr;            /* points to the previous block */
    Ref_List_Extra_Block* next_ptr;            /* points to the next block */
    int i;
    int seq;

    assert(sizeof(Ref_List_First_Block) == FILE_BLOCK_SIZE + sizeof(void*));
    assert(sizeof(Ref_List_Extra_Block) == FILE_BLOCK_SIZE + sizeof(void*));

    memset(ivec, 0, sizeof(ivec));

    cache->first_block = nwos_malloc(sizeof(Ref_List_First_Block));
    assert(cache->first_block != NULL);

    memset(ivec, 0, sizeof(ivec));
    memcpy(cache->ivec, ivec, sizeof(cache->ivec));

    cache->first_block->next_block_ptr = NULL;

    nwos_read_object_from_disk_and_decrypt_0014(&cache->ref, &cache->first_block->list, FILE_BLOCK_SIZE, ivec, nwos_random_sequence[0]);

    assert(is_same_object(&cache->first_block->list.common_header.class_definition, &nwos_reference_list_class_ref_0014));

    cache->num_refs = 0;

    while (cache->num_refs < MAX_REFS_IN_REF_LIST)
    {
	if (is_void_reference(&cache->first_block->refs[cache->num_refs])) break;
	cache->num_refs++;
    }

    /* use the flags as a pointer to the next block */
    prev_ptr = (Ref_List_Extra_Block*) cache->first_block;

    check_ref_list_data(&cache->ref,
			(uint8*) cache->first_block->refs,
			(cache->num_refs + 1) * sizeof(ObjRef),                 /* include void or next_block_ref */
			cache->first_block->list.common_header.data_chksum);

    if (cache->num_refs < MAX_REFS_IN_REF_LIST)
    {
	prev_ptr->next_block_ref.word = 0;
    }

    assert(prev_ptr->next_block_ptr == NULL);

    seq = 1;    /* use different sequence tables for each block in turn */

    while (!is_void_reference(&prev_ptr->next_block_ref))    /* more blocks in this list */
    {
	next_ptr = nwos_malloc(sizeof(Ref_List_Extra_Block));

	assert(next_ptr != NULL);

	next_ptr->next_block_ptr = NULL;

	memcpy(cache->ivec, ivec, sizeof(cache->ivec));  /* save this ivec in case this block has to be written back to disk */

	nwos_read_object_from_disk_and_decrypt_0014(&prev_ptr->next_block_ref, &next_ptr->dirty, FILE_BLOCK_SIZE, ivec, nwos_random_sequence[seq]);

	for (i = 0; i < MAX_REFS_IN_SIDECAR; i++)
	{
	    if (is_void_reference(&next_ptr->refs[i])) break;
	    cache->num_refs++;
	}

	check_ref_list_data(&cache->ref, (uint8*) next_ptr->refs, (i + 1) * sizeof(ObjRef), next_ptr->checksum);

	if (i < MAX_REFS_IN_SIDECAR)    /* the link will be garbage, make it void */
	{
	    next_ptr->next_block_ref.word = 0;
	}
	else
	{
	    assert(!is_same_object(&prev_ptr->next_block_ref, &next_ptr->next_block_ref));
	}

	prev_ptr->next_block_ptr = next_ptr;

	prev_ptr = next_ptr;

	seq = (seq + 1) % NUM_STORED_SEQ;    /* wrap around if we've used all of them */
    }
}



static void write_reference_list(Block_Cache* cache)
{
    uint8 ivec[IVEC_SIZE];
    Ref_List_Extra_Block *block = NULL;
    int running_total;
    int refs_remaining;
    Ref_List_Extra_Block* prev_ptr = NULL;
    Ref_List_Extra_Block* next_ptr = NULL;
    ObjRef ref;
    int i;
    int seq;

    memset(ivec, 0, sizeof(ivec));

    if (cache->num_refs <= MAX_REFS_IN_REF_LIST)    /* there is just one block */
    {
	assert(cache->first_block->next_block_ptr == NULL);  /* the next block pointer should be null */
	assert(is_void_reference(&cache->first_block->refs[cache->num_refs]));

	if (cache->first_block->list.common_header.flags == 0xffffffff)    /* block is dirty, needs to be written to disk */
	{
	    for (i = cache->num_refs + 1; i < MAX_REFS_IN_REF_LIST; i++)
	    {
		cache->first_block->refs[i].word = (random() << 1) ^ (random() >> 1);
	    }

	    cache->first_block->next_block_ref.word = (random() << 1) ^ (random() >> 1);

	    nwos_crc32_calculate((uint8*) &cache->first_block->refs, 
				 (cache->num_refs + 1) * sizeof(ObjRef),               /* include void ptr or next_block_ref */
				 cache->first_block->list.common_header.data_chksum);

	    block = (Ref_List_Extra_Block*)cache->first_block;     /* flag that it needs to be written */

	    seq = 0;
	}
	else                    /* we don't need to write this block so we can get rid of it now */
	{
	    nwos_free(cache->first_block);
	}
    }
    else
    {
	assert(cache->first_block->next_block_ptr != NULL);  /* the next block pointer should not be null */

	running_total = MAX_REFS_IN_REF_LIST;
	prev_ptr = (Ref_List_Extra_Block*)cache->first_block;

	seq = 0;

	while (running_total < cache->num_refs)    /* read more blocks */
	{
	    next_ptr = prev_ptr->next_block_ptr;

	    assert(is_same_object(&prev_ptr->next_block_ref, &next_ptr->id));  /* next pointer ref matches */

	    nwos_free(prev_ptr);   /* won't be needing this anymore */

	    refs_remaining = cache->num_refs - running_total;

	    if (refs_remaining <= MAX_REFS_IN_SIDECAR)
	    {
		assert(next_ptr->next_block_ptr == NULL);  /* the next block pointer should be null */
		assert(is_void_reference(&next_ptr->refs[refs_remaining]));

		if (next_ptr->dirty == -1)    /* block is dirty, needs to be written to disk */
		{
		    for (i = refs_remaining + 1; i < MAX_REFS_IN_SIDECAR; i++)
		    {
			next_ptr->refs[i].word = (random() << 1) ^ (random() >> 1);
		    }

		    if (refs_remaining < MAX_REFS_IN_SIDECAR)
		    {
			next_ptr->next_block_ref.word = (random() << 1) ^ (random() >> 1);    /* fill next with random junk */
		    }
		    else
		    {
			next_ptr->next_block_ref.word = 0;    /* mark the end */
		    }

		    nwos_crc32_calculate((uint8*) &next_ptr->refs, 
					 (refs_remaining + 1) * sizeof(ObjRef),        /* include void ptr or next_block_ref */
					 next_ptr->checksum);

		    block = next_ptr;     /* flag that it needs to be written */
		}

		running_total += refs_remaining;
	    }
	    else
	    {
		assert(next_ptr->next_block_ptr != NULL); /* the next block pointer should not be null */
		running_total += MAX_REFS_IN_SIDECAR;
	    }
	    
	    prev_ptr = next_ptr;

	    seq = (seq + 1) % NUM_STORED_SEQ;    /* wrap around if we've used all of them */
	}


	if (block == NULL)      /* we don't need to write this block so we can get rid of it now */
	{
	    assert(next_ptr != NULL);

	    nwos_free(next_ptr);
	}

	assert(running_total == cache->num_refs);
    }

    if (block != NULL)    /* it needs to be written */
    {
	assert(block->dirty == 0xffffffff);

	/* write object expects these to be null right now */
	block->dirty = 0;

	copy_reference(&ref, &block->id);

	/* printf("writing block: %02x%02x%02x%02x\n", ref.id[0], ref.id[1], ref.id[2], ref.id[3]); */

	assert(false);
#if 0
	nwos_write_object_to_disk_and_encrypt(&ref, &block->dirty, FILE_BLOCK_SIZE, cache->ivec, nwos_random_sequence[seq]);
#endif
	nwos_free(block);
    }

    /* just in case, nuke these */
    cache->first_block = NULL;
    memset(&cache->ref, 0, sizeof(cache->ref));
}


static Block_Cache* find_ref_list_in_cache(ObjRef* ref)
{
    int i;
    Block_Cache* result;

    assert(!is_void_reference(ref));

    for (i = 0; i < REF_LIST_CACHE_SIZE; i++)
    {
	if (is_same_object(&ref_list_cache[i].ref, ref)) break;
    }

    if (i < REF_LIST_CACHE_SIZE)   /* found it */
    {
	result = &ref_list_cache[i];
    }
    else                           /* didn't find it */
    {
	/* find an empty one or find the oldest */

	result = ref_list_cache;

	for (i = 0; i < REF_LIST_CACHE_SIZE; i++)
	{
	    if (is_void_reference(&ref_list_cache[i].ref))
	    {
		result = &ref_list_cache[i];
		break;
	    }

	    if (ref_list_cache[i].age < result->age)
	    {
		result = &ref_list_cache[i];
	    }
	}

	if (i == REF_LIST_CACHE_SIZE)    /* didn't find an empty one, write the oldest one out */
	{
	    write_reference_list(result);
	}

	memcpy(&result->ref, ref, sizeof(result->ref));

	read_ref_list_into_cache(result);
    }

    ref_list_tick++;
    result->age = ref_list_tick;

    return result;
}


/* WARNING: this only works on reference lists */

size_t nwos_reference_list_size_0014(ObjRef* ref)
{
    Block_Cache* cache;

    assert(!is_void_reference(ref));

    cache = find_ref_list_in_cache(ref);

    return sizeof(ReferenceList) + (cache->num_refs * sizeof(ObjRef));
}


void nwos_read_reference_list_from_disk_0014(ObjRef* ref, ReferenceList* object, size_t size)
{
    Block_Cache* cache;
    int running_total;
    int refs_remaining;
    Ref_List_Extra_Block* next_ptr;

    assert(!is_void_reference(ref));

    cache = find_ref_list_in_cache(ref);

    assert(size >= sizeof(ReferenceList) + cache->num_refs * sizeof(ObjRef));

    /* copy the first block (which is a reference list object */

    if (cache->num_refs <= MAX_REFS_IN_REF_LIST)    /* there is just one block */
    {
	assert(cache->first_block->next_block_ptr == NULL);         /* the next block pointer should be null */
	assert(is_void_reference(&cache->first_block->refs[cache->num_refs])); /* this should always be null */

	memcpy(object, &cache->first_block->list, sizeof(ReferenceList) + cache->num_refs * sizeof(ObjRef));
    }
    else
    {
	assert(cache->first_block->next_block_ptr != NULL);         /* the next block pointer should not be null */
	assert(sizeof(ReferenceList) + MAX_REFS_IN_REF_LIST * sizeof(ObjRef) == FILE_BLOCK_SIZE - 4);

	memcpy(object, &cache->first_block->list, sizeof(ReferenceList) + MAX_REFS_IN_REF_LIST * sizeof(ObjRef));

	running_total = MAX_REFS_IN_REF_LIST;
	next_ptr = *(Ref_List_Extra_Block**)cache->first_block;        /* get pointer to first sidecar */

	assert(is_same_object(&cache->first_block->next_block_ref, &next_ptr->id));  /* next pointer ref matches */

	while (running_total < cache->num_refs)    /* read more blocks */
	{
	    refs_remaining = cache->num_refs - running_total;

	    if (refs_remaining <= MAX_REFS_IN_SIDECAR)
	    {
		assert(next_ptr->next_block_ptr == NULL);  /* the next block pointer should be null */
		assert(is_void_reference(&next_ptr->refs[refs_remaining]));
		memcpy(&object->references[running_total], next_ptr->refs, refs_remaining * sizeof(ObjRef));
		running_total += refs_remaining;
	    }
	    else
	    {
		assert(next_ptr->next_block_ptr != NULL);  /* the next block pointer should not be null */
		memcpy(&object->references[running_total], next_ptr->refs, MAX_REFS_IN_SIDECAR * sizeof(ObjRef));
		running_total += MAX_REFS_IN_SIDECAR;

		/* next pointer ref matches */
		assert(is_same_object(&next_ptr->next_block_ref, &next_ptr->next_block_ptr->id));
	    }

	    next_ptr = next_ptr->next_block_ptr;
	}
	assert(running_total == cache->num_refs);
    }
}


ReferenceList* nwos_malloc_reference_list_0014(ObjRef* ref)
{
    size_t ref_list_size;
    ReferenceList* ref_list;

    ref_list_size = nwos_reference_list_size_0014(ref);

    ref_list = nwos_malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading reference list");
	exit(1);
    }

    nwos_read_reference_list_from_disk_0014(ref, ref_list, ref_list_size);

    ref_list->common_header.num_refs = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

    return ref_list;
}


void nwos_free_reference_list_0014(ReferenceList* ref_list)
{
    nwos_free(ref_list);
}

