/*
--             This file is part of the New World OS project
--                    Copyright (C) 2007 QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   This program takes a compressed file as input, scans it for chunks
--   that only have one block in them and attempts to move that block
--   into an adjacent block to free up that chunk.
--
-- $Log: eliminate_one_block_chunks.c,v $
-- Revision 1.5  2007/12/02 15:50:06  jsedwards
-- Added counting the number of files checked and verify the block count is
-- correct for each file.
--
-- Revision 1.4  2007/11/30 14:15:22  jsedwards
-- Added code to log and print out the number of single blocks that are used
-- int the reference list first block and extra blocks.
--
-- Revision 1.3  2007/11/29 16:06:40  jsedwards
-- Added code to scan the remainder of each reference list.
--
-- Revision 1.2  2007/11/29 13:51:50  jsedwards
-- Added code to search the first block in a reference list for single blocks.
--
-- Revision 1.1  2007/11/29 12:19:40  jsedwards
-- Move to attic.
--
-- Revision 1.4  2007/11/29 05:36:36  jsedwards
-- Added subroutine to scan the files for the blocks found in block table.
--
-- Revision 1.3  2007/11/29 05:20:57  jsedwards
-- Took code that scanned compressed file for single block chunks and made a
-- new function find_single_block_chunks.
--
-- Revision 1.2  2007/11/28 15:22:23  jsedwards
-- Changed blocks table to store previous and subsequent refs so that single
-- block chunks in a row can be detected.
--
-- Revision 1.1  2007/11/28 14:57:15  jsedwards
-- Initial version - only scans for one block chunks.
--
*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>    /* define sleep() */

#include "../crc32.h"
#include "../objectify_private.h"


#define DATA_STORAGE_SIZE (FILE_BLOCK_SIZE - 12)   /* 4 bytes each for flags, id, checksum */


int single_block_chunks = 0;
int total_chunks = 0;
int blocks_found = 0;
int first_block_blocks = 0;
int extra_block_blocks = 0;
int files_checked = 0;


struct
{
    uint32 previous;
    uint32 single;
    uint32 subsequent;
    uint32 new;
    ObjRef file;
} blocks[4096];


void find_single_block_chunks(char* compressed_file_name)
{
    FILE* fp1;
    uint8 buf1[FILE_BLOCK_SIZE];
    Disk_Header header1;
    size_t read1;
    uint32 ref;
    uint32 prev_ref1;
    uint32 prev_ref2;
    uint32 chunk;
    uint32 prev_chunk1;
    uint32 prev_chunk2;
    int block_count;
    int block1;


    /* Open the old file and check it */

    fp1 = fopen(compressed_file_name, "r");
    if (fp1 == NULL)
    {
	perror(compressed_file_name);
	exit(1);
    }

    read1 = fread(buf1, 1, sizeof(buf1), fp1);

    if (read1 != FILE_BLOCK_SIZE)
    {
	if (ferror(fp1))
	{
	    perror(compressed_file_name);
	}
	else
	{
	    fprintf(stderr, "Unexpected end of file: %s\n", compressed_file_name);
	}
	fclose(fp1);
	exit(1);
    }

    memcpy(&header1, buf1, sizeof(header1));

    if (memcmp(header1.magic_number, "NWOS", 4) != 0)
    {
	fprintf(stderr, "Not an Objectify file: %s\n", compressed_file_name);
	fclose(fp1);
	exit(1);
    }


    read1 = fread(buf1, 1, sizeof(buf1), fp1);

    block_count = 0;
    ref = 0;
    prev_ref1 = 0;
    prev_ref2 = 0;
    chunk = 0xffffffff;
    prev_chunk1 = 0xffffffff;
    prev_chunk2 = 0xffffffff;

    while (!feof(fp1) && read1 == FILE_BLOCK_SIZE)
    {
	if (buf1[0] != 0 || buf1[1] != 0 || buf1[2] != 0 || buf1[3] != 0)
	{
	    printf("\n%s block %d - first four bytes not zero: %02x%02x%02x%02x\n",
		   compressed_file_name, block1, buf1[0], buf1[1], buf1[2], buf1[3]);
	    break;
	}

	nwos_4_uint8_to_uint32(&buf1[4], &ref);

	chunk = (ref - RESERVED_PUBLIC_BLOCKS) / USABLE_BLOCKS_PER_CHUNK;

	if (prev_chunk2 != 0xffffffff && prev_chunk2 != prev_chunk1 && prev_chunk1 != chunk)
	{
	    printf("%c  before: %08x (%05u.%05u)  single: %08x (%05u.%05u)  after: %08x (%05u.%05u)\n",
		   single_block_chunks > 0 && blocks[single_block_chunks-1].single == prev_ref2 ? '+' : '-',
		   prev_ref2, prev_chunk2, (prev_ref2 - RESERVED_PUBLIC_BLOCKS) % USABLE_BLOCKS_PER_CHUNK,
		   prev_ref1, prev_chunk1, (prev_ref1 - RESERVED_PUBLIC_BLOCKS) % USABLE_BLOCKS_PER_CHUNK,
		   ref,       chunk,       (ref - RESERVED_PUBLIC_BLOCKS) % USABLE_BLOCKS_PER_CHUNK);
	    fflush(stdout);

	    blocks[single_block_chunks].previous = prev_ref2;
	    blocks[single_block_chunks].single = prev_ref1;
	    blocks[single_block_chunks].subsequent = ref;
	    single_block_chunks++;
	}

	if (prev_chunk1 != chunk)
	{
	    total_chunks++;
	}

	prev_ref2 = prev_ref1;
	prev_ref1 = ref;

	prev_chunk2 = prev_chunk1;
	prev_chunk1 = chunk;

	read1 = fread(buf1, 1, sizeof(buf1), fp1);
    }

    if (!feof(fp1) && read1 != FILE_BLOCK_SIZE)
    {
	perror(compressed_file_name);
    }

    fclose(fp1);
}


static inline uint64 file_size_to_uint64(uint8 size[5])
{
    return ((uint64)size[0] << 32) |
	   ((uint64)size[1] << 24) |
	   ((uint64)size[2] << 16) |
	   ((uint64)size[3] << 8)  |
	    (uint64)size[4];
}


static int find_single_block(uint32 ref)
{
    uint32 mid;
    int i;
    int lower = 1;
    int upper = single_block_chunks;

    while (lower <= upper)
    {
	i = (lower + upper) / 2;

	mid = blocks[i-1].single;

	if (mid > ref)
	{
	    upper = i - 1;
	}
	else if (mid < ref)
	{
	    lower = i + 1;
	}
	else
	{
	    return i - 1;
	}
    }

    return -1;
}


static void check_ref_list_data(ObjRef* ref, uint8* data, size_t data_size, uint8 stored[4])
{
    uint8 computed[4];

#if 0
    printf("Total size: %d\n", total_size);
    printf("ObjectHeader: %d\n", sizeof(EveryObject));
    printf("object pointer: %p\n", object);
    {
      int i;
      uint8* ptr = (uint8*)object + sizeof(EveryObject);
      printf("data pointer: %p\n", ptr);
      printf("data: ");
      for (i = 0; i < data_size; i++) printf("%02x%c", *ptr++, i%16==15?'\n':' ');
      printf("\n");
    }
#endif

    nwos_crc32_calculate((uint8*)data, data_size, computed);

    if (stored[0] != computed[0] || stored[1] != computed[1] || stored[2] != computed[2] || stored[3] != computed[3])
    {
	printf("reference list: %02x%02x%02x%02x - bad data checksum, ", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	printf("size %zd - ", data_size);
	printf("computed %02x%02x%02x%02x - ", computed[0], computed[1], computed[2], computed[3]);
	printf("stored %02x%02x%02x%02x\n", stored[0], stored[1], stored[2], stored[3]);

	printf("\n");
	{
	  int i;
	  for (i = 0; i < data_size; i++)
	  {
	      printf("%02x%c", data[i], (i % 16) == 15 ? '\n' : ' ');
	  }
	  printf("\n");
	}
	exit(1);
    }
}


static bool check_file(ObjRef* file_ref)
{
    bool result = true;
    int i;
    uint64 file_length;
    uint32 num_blocks;
    uint32 block_count = 0;
    C_struct_File file_obj;
    C_struct_Path_And_File_Association assoc_obj;
//    DataObject data_obj;
    struct timeval tv[2];      /* [0] is access time and [1] is modification time, see man utimes */
//    MD5_CTX md5_context;           /* MD5 checksum context */
//    struct sha1_ctx sha1_context;
    uint8 md5_digest[16];
    uint8 sha1_digest[20];
    C_struct_MD5sum md5_object;
    C_struct_SHA1sum sha1_object;
    uint8 ivec[IVEC_SIZE];
    uint8 chksum[4];
    Ref_List_First_Block first_block;
    Ref_List_Extra_Block extra_block;
    int block_index;
    int seq;
    ObjRef next_ref;


    memset(ivec, 0, sizeof(ivec));

    nwos_read_object_from_disk(file_ref, &file_obj, sizeof(file_obj));  /* read the file object */

    if (!is_void_reference(&file_obj.block_list))
    {
	files_checked++;

	block_index = find_single_block(nwos_ref_to_word(&file_obj.block_list));

	if (block_index >= 0)
	{
	    printf("block_list: %08x  file: %08x\n", blocks[block_index].single, nwos_ref_to_word(file_ref));
	    first_block_blocks++;
	}

	file_length = file_size_to_uint64(file_obj.size);

	num_blocks = (file_length + DATA_STORAGE_SIZE - 1) / DATA_STORAGE_SIZE;

	nwos_read_object_from_disk_and_decrypt(&file_obj.block_list, &first_block.list, FILE_BLOCK_SIZE, ivec, nwos_random_sequence[0]);

	for (i = 0; i < MAX_REFS_IN_REF_LIST; i++)
	{
	    if (is_void_reference(&first_block.list.references[i])) break;

	    block_index = find_single_block(nwos_ref_to_word(&first_block.list.references[i]));

	    if (block_index >= 0)
	    {
		assert(is_void_reference(&blocks[block_index].file));

		copy_reference(&blocks[block_index].file, file_ref);

		printf("block: %08x  file: %08x\n", blocks[block_index].single, nwos_ref_to_word(file_ref));

		blocks_found++;
	    }

	    block_count++;
	}

	check_ref_list_data(&file_obj.block_list,
			    (uint8*) first_block.refs,
			    (i + 1) * sizeof(ObjRef),                 /* include void or next_block_ref */
			    first_block.list.common_header.data_chksum);

	if (i == MAX_REFS_IN_REF_LIST && !is_void_reference(&first_block.next_block_ref))   /* more than one block */
	{
	    copy_reference(&extra_block.next_block_ref, &first_block.next_block_ref);

	    seq = 1;    /* use different sequence tables for each block in turn */

	    while (!is_void_reference(&extra_block.next_block_ref))    /* more blocks in this list */
	    {
		copy_reference(&next_ref, &extra_block.next_block_ref);

		block_index = find_single_block(nwos_ref_to_word(&next_ref));

		if (block_index >= 0)
		{
		    printf("ref_list: %08x  file: %08x\n", blocks[block_index].single, nwos_ref_to_word(file_ref));
		    extra_block_blocks++;
		}

		nwos_read_object_from_disk_and_decrypt(&next_ref,
						       &extra_block.dirty,
						       FILE_BLOCK_SIZE, 
						       ivec,
						       nwos_random_sequence[seq]);

		for (i = 0; i < MAX_REFS_IN_SIDECAR; i++)
		{
		    if (is_void_reference(&extra_block.refs[i])) break;

		    block_index = find_single_block(nwos_ref_to_word(&extra_block.refs[i]));

		    if (block_index >= 0)
		    {
			assert(is_void_reference(&blocks[block_index].file));

			copy_reference(&blocks[block_index].file, file_ref);

			printf("block: %08x  file: %08x\n", blocks[block_index].single, nwos_ref_to_word(file_ref));

			blocks_found++;
		    }

		    block_count++;
		}

		check_ref_list_data(&next_ref, (uint8*) extra_block.refs, (i + 1) * sizeof(ObjRef), extra_block.checksum);

		if (i < MAX_REFS_IN_SIDECAR)
		{
		    void_reference(&extra_block.next_block_ref);
		}

		seq++;
	    }
	}

	if (block_count != num_blocks)
	{
	    printf("Block count mismatch - computed %d  calculated: %d\n", num_blocks, block_count);
	}
    }


#if 0
    ref_list_ptr = nwos_malloc_reference_list(&file_obj.block_list);

    assert(num_blocks == ref_list_ptr->common_header.num_refs);

//    printf("num_blocks: %u  block_list: %02x%02x%02x%02x\n", num_blocks,
//	   file_obj.block_list.id[0],
//	   file_obj.block_list.id[1],
//	   file_obj.block_list.id[2],
//	   file_obj.block_list.id[3]);

    {
    while (file_length > DATA_STORAGE_SIZE)   /* do all except the last block */
    {
	nwos_read_object_from_disk_and_decrypt(&ref_list_ptr->references[i], &data_obj, sizeof(data_obj), ivec, nwos_random_sequence[i%NUM_STORED_SEQ]);

	    nwos_crc32_calculate((uint8*) &data_obj.storage, sizeof(data_obj.storage), chksum);

	    if (memcmp(chksum, data_obj.checksum, sizeof(chksum)) != 0)
	    {
		printf("checksum error in file block - sequence: %d\n", i);
		exit(1);
	    }

	    MD5Update(&md5_context, data_obj.storage, DATA_STORAGE_SIZE);    /* include this data in the md5 checksum */
	    sha1_process_bytes(data_obj.storage, DATA_STORAGE_SIZE, &sha1_context);    /* and in the sha1 checksum */

	    if (fwrite(data_obj.storage, sizeof(uint8), DATA_STORAGE_SIZE, fp) != DATA_STORAGE_SIZE)
	    {
		perror(path);
		exit(1);
	    }

	    file_length -= DATA_STORAGE_SIZE;

	    i++;
	}

	nwos_read_object_from_disk_and_decrypt(&ref_list_ptr->references[i], &data_obj, sizeof(data_obj), ivec, nwos_random_sequence[i%NUM_STORED_SEQ]);

	nwos_crc32_calculate((uint8*) &data_obj.storage, file_length, chksum);

	if (memcmp(chksum, data_obj.checksum, sizeof(chksum)) != 0)
	{
	    printf("checksum error in file block - sequence: %d\n", i);
	    exit(1);
	}

	MD5Update(&md5_context, data_obj.storage, file_length);    /* include this data in the md5 checksum */
	sha1_process_bytes(data_obj.storage, file_length, &sha1_context);    /* and in the sha1 checksum */

	if (fwrite(data_obj.storage, sizeof(uint8), file_length, fp) != file_length)
	{
	    perror(path);
	    exit(1);
	}

	if (fclose(fp) != 0)
	{
	    perror(path);
	    exit(1);
	}

	if (result)   /* the file was ok, restore it's modification time */
	{
	    nwos_convert_time_stamp_to_timeval(assoc_obj.modification_time, &tv[1]);
	    tv[0] = tv[1];  /* copy modification time into access time since we don't save the access time */
	    utimes(path, tv);
	}

	MD5Final(md5_digest, &md5_context);   /* finish computing the md5 sum */
	sha1_finish_ctx(&sha1_context, sha1_digest);  /* and the sha1 sum */

	if (result)
	{
	    nwos_read_object_from_disk(&file_obj.md5sum, &md5_object, sizeof(md5_object));

	    if (memcmp(md5_digest, md5_object.md5sum, sizeof(md5_digest)) == 0)
	    {
		int j;
		printf("MD5 sum OK: ");
		for (j = 0; j < 16; j++) printf("%02x", md5_digest[j]);
		printf("\n");
	    }
	    else
	    {
		int j;
		printf("MD5 checksum error, expected: ");
		for (j = 0; j < 16; j++) printf("%02x", md5_object.md5sum[j]);
		printf("\n                    received: ");
		for (j = 0; j < 16; j++) printf("%02x", md5_digest[j]);
		printf("\n");

		result = false;
	    }

	    nwos_read_object_from_disk(&file_obj.sha1sum, &sha1_object, sizeof(sha1_object));

	    if (memcmp(sha1_digest, sha1_object.sha1sum, sizeof(sha1_digest)) == 0)
	    {
		int j;
		printf("SHA1 sum OK: ");
		for (j = 0; j < 20; j++) printf("%02x", sha1_digest[j]);
		printf("\n");
	    }
	    else
	    {
		int j;
		printf("SHA1 checksum error, expected: ");
		for (j = 0; j < 20; j++) printf("%02x", sha1_object.sha1sum[j]);
		printf("\n                    received: ");
		for (j = 0; j < 20; j++) printf("%02x", sha1_digest[j]);
		printf("\n");

		result = false;
	    }
	}

	nwos_free_reference_list(ref_list_ptr);
	ref_list_ptr = NULL;
    }
#endif

    return result;
}


bool scan_for_matching_blocks(char* compressed_file_name)
{
    uint8 big_key[16 + 8 + 4];
    uint8 bf_key[16];
    uint32 linear;
    uint32 serial;
    ObjRef root_object_ref;
    ObjRef object_class;
    C_struct_Class_Definition class_def_obj;
    ReferenceList* ref_list;
    int num_refs;
    ObjRef file_ref;
    ObjRef file_class_ref;
    int i;


    nwos_get_key_from_password(big_key, sizeof(big_key));

    memcpy(bf_key, big_key, 16);
    linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
    memcpy(root_object_ref.id, big_key+20, 4);
    serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

    nwos_initialize_objectify(bf_key, linear, serial,   Compressed_File_RO, compressed_file_name);

    nwos_set_root_object(&root_object_ref);


    assert(nwos_find_private_class_definition("FILE", &file_class_ref));

    nwos_read_class_definition(&file_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    printf("num_refs: %d\n", num_refs);

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &file_class_ref))
	{
	    check_file(&ref_list->references[i]);
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    if (!is_void_reference(&class_def_obj.header.object.prev_version))
    {
	nwos_read_class_definition(&class_def_obj.header.object.prev_version, &class_def_obj);

	ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

	num_refs = ref_list->common_header.num_refs;

	printf("num_refs: %d\n", num_refs);

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);

	    if (is_same_object(&object_class, &file_class_ref))
	    {
		check_file(&ref_list->references[i]);
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }

    nwos_terminate_objectify();

    return false;  /* for now only do once */
}


int main(int argc, char* argv[])
{
    FILE* fp2;
    FILE* fp3 = NULL;
    uint8 buf2[FILE_BLOCK_SIZE];
    int block2;
    size_t read2;
    size_t write3;
    Disk_Header header2;
    bool wrapped_file = false;     /* true if older file that wasn't sequencial */
    int header1_version;
    int header2_version;
    uint32 ref_offset;
    uint32 new_ref;
    uint32 new_chunk;

    if (argc != 3)
    {
	fprintf(stderr, "usage: %s old_file new_file\n", argv[0]);
	exit(1);
    }

    nwos_log_arguments(argc, argv);

    find_single_block_chunks(argv[1]);

    scan_for_matching_blocks(argv[1]);

#if 0
    fp3 = fopen(argv[3], "w");

    if (fp3 == NULL)
    {
	perror(argv[3]);
	fclose(fp1);
	fclose(fp2);
	exit(1);
    }

    write3 = fwrite(buf2, 1, sizeof(buf2), fp3);

    if (write3 != FILE_BLOCK_SIZE)
    {
	perror(argv[3]);
	fclose(fp1);
	fclose(fp2);
	fclose(fp3);
	exit(1);
    }

	    ref_offset = (ref - RESERVED_PUBLIC_BLOCKS) % USABLE_BLOCKS_PER_CHUNK;
	    new_ref = RESERVED_PUBLIC_BLOCKS + USABLE_BLOCKS_PER_CHUNK * chunk + random() % ref_offset;

	    new_chunk = (new_ref - RESERVED_PUBLIC_BLOCKS) / USABLE_BLOCKS_PER_CHUNK;

	    assert(prev_ref1 < new_ref && new_ref < ref);
	    assert(new_chunk == chunk);
	    assert((new_ref - RESERVED_PUBLIC_BLOCKS) % USABLE_BLOCKS_PER_CHUNK < ref_offset);


    while(something)
    {
	if (buf1[4] == buf2[4] && buf1[5] == buf2[5] && buf1[6] == buf2[6] && buf1[7] == buf2[7])
	{
	    /* blocks have the same id number, must have changed */

	    if (memcmp(buf1, buf2, FILE_BLOCK_SIZE) != 0)
	    {
		printf("%02x%02x%02x%02x: changed\n",
		       buf1[4], buf1[5], buf1[6], buf1[7]);

		write3 = fwrite(buf2, 1, sizeof(buf2), fp3);
	    }
	    else   /* block stayed the same write the old block */
	    {
		write3 = fwrite(buf1, 1, sizeof(buf1), fp3);
	    }

	    if (write3 != FILE_BLOCK_SIZE)
	    {
		perror(argv[2]);
		fclose(fp1);
		fclose(fp2);
		fclose(fp3);
		exit(1);
	    }

	    read1 = fread(buf1, 1, sizeof(buf1), fp1);
	    block1++;
	    read2 = fread(buf2, 1, sizeof(buf2), fp2);
	    block2++;
	}
	else   /* id is different, assume this an add, since right now can't delete */
	{
	    if (wrapped_file)
	    {
		/* if we've found the proper place to insert it */
		if (buf1[4] == buf2[0] && buf1[5] == buf2[1] && buf1[6] == buf2[2] && buf1[7] == buf2[3])
		{
		    printf("%02x%02x%02x%02x: added\n",
			   buf2[4], buf2[5], buf2[6], buf2[7]);

		    buf2[0] = 0;
		    buf2[1] = 0;
		    buf2[2] = 0;
		    buf2[3] = 0;

		    write3 = fwrite(buf2, 1, sizeof(buf2), fp3);

		    read2 = fread(buf2, 1, sizeof(buf2), fp2);
		    block2++;
		}
		else  /* not time to insert yet, just write the old block */
		{
		    write3 = fwrite(buf1, 1, sizeof(buf1), fp3);

		    read1 = fread(buf1, 1, sizeof(buf1), fp1);
		    block1++;
		}
	    }
	    else   /* new non-wrapped file that is sequential */
	    {
		/* if we've found the proper place to insert it */
		if (buf1[4] > buf2[4] || 
		    (buf1[4] == buf2[4] && buf1[5] > buf2[5]) ||
		    (buf1[4] == buf2[4] && buf1[5] == buf2[5] && buf1[6] > buf2[6]) ||
		    (buf1[4] == buf2[4] && buf1[5] == buf2[5] && buf1[6] == buf2[6] && buf1[7] > buf2[7])
		    )
		{
		    printf("%02x%02x%02x%02x: added\n",
			   buf2[4], buf2[5], buf2[6], buf2[7]);

		    write3 = fwrite(buf2, 1, sizeof(buf2), fp3);

		    read2 = fread(buf2, 1, sizeof(buf2), fp2);
		    block2++;
		}
		else  /* not time to insert yet, just write the old block */
		{
		    write3 = fwrite(buf1, 1, sizeof(buf1), fp3);

		    read1 = fread(buf1, 1, sizeof(buf1), fp1);
		    block1++;
		}
	    }

	    if (write3 != FILE_BLOCK_SIZE)
	    {
		perror(argv[3]);
		fclose(fp1);
		fclose(fp2);
		fclose(fp3);
		exit(1);
	    }
	}
    }

    /* finish writing output */

    while (!feof(fp1) && read1 == FILE_BLOCK_SIZE)
    {
	write3 = fwrite(buf1, 1, sizeof(buf1), fp3);

	if (write3 != FILE_BLOCK_SIZE)
	{
	    perror(argv[3]);
	    fclose(fp1);
	    fclose(fp2);
	    fclose(fp3);
	    exit(1);
	}

	read1 = fread(buf1, 1, sizeof(buf1), fp1);
	block1++;
    }

    /* more in the patch file */

    while (!feof(fp2) && read2 == FILE_BLOCK_SIZE)
    {
	printf("%02x%02x%02x%02x: added\n", buf2[4], buf2[5], buf2[6], buf2[7]);

	write3 = fwrite(buf2, 1, sizeof(buf2), fp3);

	if (write3 != FILE_BLOCK_SIZE)
	{
	    perror(argv[3]);
	    fclose(fp1);
	    fclose(fp2);
	    fclose(fp3);
	    exit(1);
	}

	read2 = fread(buf2, 1, sizeof(buf2), fp2);
	block1++;
    }
#endif
    printf("\n");

#if 0
    if (memcmp(header1.version_string, header2.version_string, 4) != 0)
    {
	fprintf(stderr, "WARNING, versions don't match - %s: %c%c%c%c  %s: %c%c%c%c\n",
		argv[1], 
		header1.version_string[0], header1.version_string[1],
		header1.version_string[2], header1.version_string[3],
		argv[2], 
		header2.version_string[0], header2.version_string[1],
		header2.version_string[2], header2.version_string[3]);
    }

    if (!feof(fp2) && read2 != FILE_BLOCK_SIZE)
    {
	perror(argv[2]);
    }

    fclose(fp2);

    if (fclose(fp3) != 0)
    {
	perror(argv[3]);
    }
#endif

    printf("Total chunks: %d\n", total_chunks);
    printf("Single block chunks: %d\n", single_block_chunks);
    printf("Files checked: %d\n", files_checked);
    printf("Blocks found: %d\n", blocks_found);
    printf("First block found: %d\n", first_block_blocks);
    printf("Extra block found: %d\n", extra_block_blocks);

    return 0;
}
