/*             This file is part of the New World OS project
--                   Copyright (C) 2006  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: old_multifile_objectify.c,v $
-- Revision 1.1  2006/12/05 12:42:02  jsedwards
-- Moved old_multifile_objectify files to attic.
--
-- Revision 1.5  2006/11/30 01:48:01  jsedwards
-- Changed strlcpy back to strcpy because Linux doesn't have strlcpy and the
-- function I defined in objectify.c isn't visible in this file.
--
-- Revision 1.4  2006/11/30 01:01:12  jsedwards
-- Changed calls to strcpy to strlcpy and sprintf to snprintf so the linker
-- won't complain so much.
--
-- Revision 1.3  2006/11/11 12:01:05  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.2  2006/10/26 01:51:28  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.2  2006/10/19 01:50:04  jsedwards
-- Added 'z' to format specifier for size_t variables.
--
-- Revision 1.1.2.1  2006/09/10 20:46:34  jsedwards
-- Old objectify.c (and other) routines that dealt with each object stored in
-- a separate file.
--
*/

#include <assert.h>
#include <ctype.h>
#include <openssl/blowfish.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "crc32.h"
#include "old_multifile_objectify.h"


#define MAGIC_NUMBER "NWOS"
#define HEADER_VERSION "0005"

#define COMPATIBLE_HEADER_VERSIONS { "0004", "0005" }

#define OBJECT_DIRECTORY "/obj/"

#define OBJ_PATH_SIZE (strlen(OBJECT_DIRECTORY) + (sizeof(ObjRef) * 2) + 1)    /* size of entire path name /obj/filename'\0' */


static int random_sequence[FILE_BLOCK_SIZE];
static BF_KEY blowfish_key;

#define MAX_NUMBER_SIZE 512    /* number can be up to 4096 bits */

#define MULTIPLIER 3141592621U
#define ADDEND     101U

static uint32 linear;
static uint32 serial;

ObjRef old_class_definition_class_ref;
ObjRef old_reference_list_class_ref;


void old_seed_sequence(uint32 linear_seed, uint32 serial_seed)
{
    linear = linear_seed;
    serial = serial_seed;
}


uint32 old_next_sequence()
{
    uint32 feedback;

    linear = (linear * MULTIPLIER) + ADDEND;

    /* feedback_mask: BIT 32 is 0x80004000 */
    /* exclusive or of bits 31 and 14 */
    feedback = ((serial >> 31) ^ (serial >> 14)) & 1;

    serial = (serial << 1) | feedback;

    return linear ^ serial;
}



static void convert_pass_phrase_to_reference(char* pass_phrase, uint8 key[], size_t key_size)
{
    int i;
    int j;
    uint8 number[MAX_NUMBER_SIZE];
    uint8 mult;
    uint32 acc;
    char *p;

    mult = '~' - ' ' + 2;

#if 0
    printf("mult: %02x\n", mult);
#endif

    memset(&number, 0, MAX_NUMBER_SIZE);

    for (p = pass_phrase; *p != '\0'; p++)
    {
	acc = *p;

	if (' ' <= acc && acc <= '~')
	{
	    acc = acc - ' ' + 1;
	}
	else
	{
	    acc = 0;
	}

	/* multiply and add big endian ([0] is most significant byte) */
	for (i = MAX_NUMBER_SIZE; i > 0; i--)
	{
	    acc = (number[i-1] * mult) + acc;
	    number[i-1] = acc;
	    acc = acc >> 8;
	}

	for (i = 0; i < MAX_NUMBER_SIZE; i++) if (number[i] != 0) break;
#if 0
	while (i < MAX_NUMBER_SIZE)
	  {
	    printf("%02x", number[i++]);
	  }
	printf("\n");
#endif
    }

    /* exclusive or each reference size chunk into the result */

    memset(key, 0, key_size);

    j = key_size - 1;
    for (i = MAX_NUMBER_SIZE; i > 0; i--)
    {
	key[j] = key[j] ^ number[i-1];
	j--;
	if (j < 0) j = key_size - 1;
    }
}


void old_get_key_from_password(uint8 key[], size_t key_size)
{
    bool ok = false;
    char buffer[MAX_NUMBER_SIZE];
    char *p;
    int chars_needed;

    chars_needed = (key_size * 80) / 65; /* we get 6.5 bits for each character entered, we need 8 for each output byte */

    while (!ok)
    {
	fprintf(stderr, "Pass phrase: ");
	fflush(stderr);

	fgets(buffer, MAX_NUMBER_SIZE, stdin);

	p = strchr(buffer, '\n');   /* find the newline char */

	if (p == NULL)    /* line was tool long */
	{
	    while (p == NULL) 
	    {
		fgets(buffer, MAX_NUMBER_SIZE, stdin);
		p = strchr(buffer, '\n');   /* find the newline char */
	    }
	    printf("password was too long, must be less than %d characters\n", MAX_NUMBER_SIZE);
	}
	else     /* line was ok */
	{
	    *p = '\0';   /* eliminate the newline character */

	    if (strlen(buffer) < chars_needed)
	    {
		printf("password was too short, must be at least %d characters\n", chars_needed);
	    }
	    else
	    {
		ok = true;   /* we should be good to go */
	    }
	}
    }

    convert_pass_phrase_to_reference(buffer, key, key_size);
}


static void ref_to_path(ObjRef* ref, char path[OBJ_PATH_SIZE])
{
    strcpy(path, OBJECT_DIRECTORY);
    old_ref_to_name(ref, &path[strlen(path)]);
}


static size_t object_size(char* path)
{
    struct stat st;

    if (stat(path, &st) != 0)
    {
	perror(path);
	exit(1);
    }

    return st.st_size;
}


bool old_object_exists(ObjRef* ref)
{
    char path[OBJ_PATH_SIZE];
    struct stat st;

    ref_to_path(ref, path);

    return (stat(path, &st) == 0);
}


void old_initialize_objectify(uint8 bf_key[16], uint32 linear, uint32 serial)
{
    uint8 filled[FILE_BLOCK_SIZE];    /* remember which slots are already filled */
    int i, j;

    /*    initialize_random_number_generator(); */
    nwos_crc32_initialize();

    BF_set_key(&blowfish_key, 16, bf_key);

    /* Create the sequence table from the linear and serial parameters */

    old_seed_sequence(linear, serial);

    /* mark all slots as empty */
    for (i = 0; i < FILE_BLOCK_SIZE; i++) filled[i] = 0;

    /* then mark the first 8 as used */
    for (i = 0; i < 8; i++) filled[i] = 1;

    /* start at 0 because it is used already, so while loop will fail the first time. */
    j = 0;
    assert(filled[j] != 0);

    for (i = 8; i < FILE_BLOCK_SIZE; i++)
    {
	/* find where the next byte is stored */
	while(filled[j] != 0) j = old_next_sequence() % FILE_BLOCK_SIZE;
	random_sequence[i] = j;   /* store it in the table */
	filled[j] = 1;            /* mark it as used */
    }

#if 0
    {
    int i;
    for (i = 0; i < 16; i++) printf("%02x", bf_key[i]);
    printf("\n");
    }
#endif

    /*    nwos_log("nwos_initialize_objectify()"); */
}



bool old_find_class_definition(char* name, ObjRef* class_ref);  /* forward reference */

void old_set_root_object(ObjRef* ref)
{
    C_struct_Root root_obj;

    assert(old_object_exists(ref));

    old_read_object_from_disk(ref, &root_obj, sizeof(root_obj));   /* root object structure changed */
    memcpy(&old_class_definition_class_ref, &root_obj.class_definition_class, sizeof(old_class_definition_class_ref));

    assert(old_find_class_definition("REFERENCE LIST", &old_reference_list_class_ref));
}


static bool read_object_from_disk(char* path, void* object, size_t size)
{
    char msg[OBJ_PATH_SIZE + 16];
    FILE* fp;
    size_t bytes_read;
    bool result;

    fp = fopen(path, "r");

    if (fp == NULL) {
	assert(snprintf(msg, sizeof(msg), "opening %s", path) < sizeof(msg));
	perror(path);
	exit(1);
    }

    bytes_read = fread(object, 1, size, fp);
    if (bytes_read != size)
    {
	if (ferror(fp))
	{
	    assert(snprintf(msg, sizeof(msg), "reading %s", path) < sizeof(msg));
	    perror(path);
	}
	else
	{
	    fprintf(stderr, "object %s too small: %zd - expected: %zd\n", path, bytes_read, size); 
	}
	exit(1);
    }

    result = (fgetc(fp) == EOF);

    if (fclose(fp) != 0) {
	assert(snprintf(msg, sizeof(msg), "closing %s", path) < sizeof(msg));
	perror(path);
	exit(1);
    }

    return result;
}


static void read_object_from_disk_and_decrypt(char* path, void* object, size_t size)
{
    uint8 buffer1[FILE_BLOCK_SIZE];
    uint8 buffer2[FILE_BLOCK_SIZE];
    uint8* ptrobj = (uint8*)object;
    int i;
    int j;

    if (!read_object_from_disk(path, buffer2, FILE_BLOCK_SIZE))
    {
	fprintf(stderr, "object %s too large: %zd - expected: %zd\n", path, object_size(path), size); 
	exit(1);
    }


#ifdef DISABLE_SECURITY_FEATURES  /* just copy the object into the buffer */
    memcpy(object, buffer2, size);
#else
    /* copy the first 8 bytes directly, not encrypted */
    memcpy(buffer1, buffer2, 8);

    /* decrypt the remaining 504 bytes, 8 bytes at a time */
    for (i = 8; i < FILE_BLOCK_SIZE; i += 8)
    {
	BF_ecb_encrypt(buffer2+i, buffer1+i, &blowfish_key, BF_DECRYPT);
    }

    assert(size <= FILE_BLOCK_SIZE);

    /* copy the first 8 bytes directly accross */
    for (i = 0; i < 8; i++) ptrobj[i] = buffer1[i];

    for (i = 8; i < size; i++)
    {
	j = random_sequence[i];    /* find where the next byte is stored */
	ptrobj[i] = buffer1[j];    /* get this byte from there */
	buffer2[j] = 1;            /* mark it as used */
    }
#endif

#if 0
    printf("read obj path: %s\n", path);
    printf("read obj pointer: %p\n", object);
    {
      int i;
      for (i = 0; i < size; i++) printf("%02x%c", *((uint8*)object+i), i%16==15?'\n':' ');
      printf("\n");
    }
#endif
}


static bool is_compatible_version(char* header_version)
{
    static char* compatible[] = COMPATIBLE_HEADER_VERSIONS;
    const int num_compatible = sizeof(compatible) / sizeof(char*);
    int i;

    for (i = 0; i < num_compatible; i++) 
    {
	if (memcmp(header_version, compatible[i], strlen(HEADER_VERSION)) == 0) break;
    }

    return (i < num_compatible);    /* return true if found a match */
}


static void check_common_header(char* path, CommonHeader* common)
{
#if 0
    printf("common header pointer: %p\n", common);
#endif

    if (memcmp(common->magic_number, MAGIC_NUMBER, sizeof(common->magic_number)))
    {
	printf("object: %s - missing magic number\n", path);
	exit(1);
    }

    if (!is_compatible_version((char*)common->version))
    {
	printf("object: %s - incompatible version: ", path);
	if (isprint(common->version[0]) && isprint(common->version[1]) &&
	    isprint(common->version[2]) && isprint(common->version[3]))
	{
	    printf("%c%c%c%c\n", common->version[0], common->version[1], common->version[2], common->version[3]);
	}
	else
	{
	    printf("%02x %02x %02x %02x\n", common->version[0], common->version[1], common->version[2], common->version[3]);
	}
	exit(1);
    }
}


/* path is only needed for error messages */

static void check_object_header(char* path, ObjectHeader* obj_header, uint8 stored[4])
{
    uint8 computed[4];

    nwos_crc32_calculate((uint8*)obj_header, sizeof(*obj_header), computed);

    if (stored[0] != computed[0] || stored[1] != computed[1] || stored[2] != computed[2] || stored[3] != computed[3])
    {
	printf("object: %s - bad header checksum, ", path);
	printf("size %zd - ", sizeof(*obj_header));
	printf("computed %02x%02x%02x%02x - ", computed[0], computed[1], computed[2], computed[3]);
	printf("stored %02x%02x%02x%02x\n", stored[0], stored[1], stored[2], stored[3]);
	exit(1);
    }
}


/* path is only needed for error messages */

static void check_object_data(char* path, uint8* object, size_t total_size, uint8 stored[4])
{
    uint8 computed[4];
    size_t data_size;

    data_size = total_size - sizeof(EveryObject);

#if 0
    printf("Total size: %d\n", total_size);
    printf("ObjectHeader: %d\n", sizeof(EveryObject));
    printf("object pointer: %p\n", object);
    {
      int i;
      uint8* ptr = (uint8*)object + sizeof(EveryObject);
      printf("data pointer: %p\n", ptr);
      printf("data: ");
      for (i = 0; i < data_size; i++) printf("%02x%c", *ptr++, i%16==15?'\n':' ');
      printf("\n");
    }
#endif

    nwos_crc32_calculate((uint8*)object + sizeof(EveryObject), data_size, computed);

    if (stored[0] != computed[0] || stored[1] != computed[1] || stored[2] != computed[2] || stored[3] != computed[3])
    {
	printf("object: %s - bad data checksum, ", path);
	printf("size %zd - ", data_size);
	printf("computed %02x%02x%02x%02x - ", computed[0], computed[1], computed[2], computed[3]);
	printf("stored %02x%02x%02x%02x\n", stored[0], stored[1], stored[2], stored[3]);

	printf("\n");
	{
	  int i;
	  for (i = 0; i < sizeof(EveryObject) + data_size; i++)
	  {
	      printf("%02x%c", object[i], (i % 16) == 15 ? '\n' : ' ');
	  }
	  printf("\n");
	}
	exit(1);
    }
}


void old_ref_to_name(ObjRef* ref, char name[])
{
    static char xlat[17] = "0123456789abcdef";

    name[0] = xlat[ref->id[0] >> 4];
    name[1] = xlat[ref->id[0] & 0xF];
    name[2] = xlat[ref->id[1] >> 4];
    name[3] = xlat[ref->id[1] & 0xF];
    name[4] = xlat[ref->id[2] >> 4];
    name[5] = xlat[ref->id[2] & 0xF];
    name[6] = xlat[ref->id[3] >> 4];
    name[7] = xlat[ref->id[3] & 0xF];
    name[8] = xlat[16];
}


size_t old_reference_list_size(ObjRef* ref)
{
    char path[OBJ_PATH_SIZE];

    ref_to_path(ref, path);

    return object_size(path);
}


void old_get_object_class(ObjRef* obj, ObjRef* object_class)
{
    char path[OBJ_PATH_SIZE];
    EveryObject header;

    ref_to_path(obj, path);

    read_object_from_disk_and_decrypt(path, &header, sizeof(header));

    check_common_header(path, &header.common);

    check_object_header(path, &header.object, header.common.header_chksum);

    memcpy(object_class, &header.common.class_definition, sizeof(ObjRef));
}



void old_read_object_from_disk(ObjRef* ref, void* object, size_t size)
{
    char path[OBJ_PATH_SIZE];
    EveryObject* header;

    ref_to_path(ref, path);

#if 0
    char log_msg[128];

    sprintf(log_msg, "old_read_object_from_disk - %s size: %d", path, size);
    nwos_log(log_msg);
#endif

    assert(size > sizeof(CommonHeader));  /* make sure this wasn't called to just get the header */

    read_object_from_disk_and_decrypt(path, object, size);

#if 0
    printf("read obj path: %s\n", path);
    printf("read obj pointer: %p\n", object);
    {
      int i;
      for (i = 0; i < size; i++) printf("%02x%c", *((uint8*)object+i), i%16==15?'\n':' ');
      printf("\n");
    }
#endif

    header = (EveryObject*) object;

    check_common_header(path, &header->common);

    check_object_header(path, &header->object, header->common.header_chksum);

    check_object_data(path, (uint8*)object, size, header->common.data_chksum);
}


void old_read_variable_sized_object_from_disk(ObjRef* ref, void* object, size_t (*size_function)(void*))
{
    char path[OBJ_PATH_SIZE];
    EveryObject* header;

    ref_to_path(ref, path);

#if 0
    char log_msg[128];

    sprintf(log_msg, "old_read_variable_size_object_from_disk - %s size: %d", path, size);
    nwos_log(log_msg);
#endif

    read_object_from_disk_and_decrypt(path, object, FILE_BLOCK_SIZE);

#if 0
    printf("read obj path: %s\n", path);
    printf("read obj pointer: %p\n", object);
    {
      int i;
      for (i = 0; i < size; i++) printf("%02x%c", *((uint8*)object+i), i%16==15?'\n':' ');
      printf("\n");
    }
#endif

    header = (EveryObject*) object;

    check_common_header(path, &header->common);

    check_object_header(path, &header->object, header->common.header_chksum);

    check_object_data(path, (uint8*)object, (*size_function)(object), header->common.data_chksum);
}


void old_read_reference_list_from_disk(ObjRef* ref, ReferenceList* object, size_t size)
{
    char path[OBJ_PATH_SIZE];

    assert(size >= sizeof(CommonHeader));  /* make sure this wasn't called to just get the header */

    ref_to_path(ref, path);

    read_object_from_disk(path, object, size);
}



void old_remove_object(ObjRef* ref)
{
    char path[OBJ_PATH_SIZE];
    /* char log_msg[128]; */

    assert(old_object_exists(ref));

    ref_to_path(ref, path);
#if 0
    sprintf(log_msg, "old_remove_object - %s", path);
    nwos_log(log_msg);
#endif
    assert(unlink(path) == 0);
}



bool old_find_class_definition(char* name, ObjRef* class_ref)
{
    C_struct_class_definition class_def_obj;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_classes;
    ObjRef object_class;
    char buffer[128];
    int i;
    bool result = false;

    memset(class_ref, 0, sizeof(class_ref));

    old_read_object_from_disk(&old_class_definition_class_ref, &class_def_obj, sizeof(class_def_obj));

    ref_list_size = old_reference_list_size(&class_def_obj.header.object.references);

    ref_list = malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("find class definition, reading class definition class reference list");
	exit(1);
    }

    old_read_reference_list_from_disk(&class_def_obj.header.object.references, ref_list, ref_list_size);

    num_classes = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

    /* printf("num_classes (in find_class_definition): %d\n", num_classes); */

    for (i = 0; i < num_classes; i++)
    {
	old_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	if (is_same_object(&object_class, &old_class_definition_class_ref))   /* it is a class definition object */
	{
	    old_read_object_from_disk(&ref_list->references[i], &class_def_obj, sizeof(class_def_obj));

	    old_name_to_string(&class_def_obj.name, buffer, sizeof(buffer));   /* get it's name */

	    if (strcasecmp(name, buffer) == 0)
	    {
		memcpy(class_ref, &ref_list->references[i], sizeof(ObjRef));
		result = true;
		break;
	    }
	}
    }

    free(ref_list);
    ref_list = NULL;

    return result;
}


static size_t get_name_object_size(void* name_obj)
{
    assert(((C_struct_Name*)name_obj)->count > 0);

    return sizeof(C_struct_Name) + (((C_struct_Name*)name_obj)->count * sizeof(ObjRef));
}

static size_t get_spelling_object_size(void* spelling_obj)
{
    assert(((C_struct_Spelling*)spelling_obj)->count > 0);

    return sizeof(C_struct_Spelling) + ((C_struct_Spelling*)spelling_obj)->count;
}


bool old_name_to_string(ObjRef* ref, char* string, size_t size)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Name* ptr_name_obj;
    C_struct_Spelling* ptr_spelling_obj;
    int word;
    int i;
    int j;
    size_t name_obj_size;

    /* read the name object into the kludge buffer and then after we know what size it is malloc space and copy it there */
    old_read_variable_sized_object_from_disk(ref, kludge, &get_name_object_size);
    ptr_name_obj = (C_struct_Name*)kludge;
    name_obj_size = sizeof(C_struct_Name) + (ptr_name_obj->count * sizeof(ObjRef));
    /* printf("name_obj_size: %d\n", name_obj_size); */
    assert(name_obj_size > sizeof(C_struct_Name));
    ptr_name_obj = malloc(name_obj_size);
    memcpy(ptr_name_obj, kludge, name_obj_size);

    ptr_spelling_obj = (C_struct_Spelling*)kludge;

    i = 0;
    for (word = 0; word < ptr_name_obj->count; word++)
    {
	assert(!is_void_reference(&ptr_name_obj->spelling[word]));
	old_read_variable_sized_object_from_disk(&ptr_name_obj->spelling[word], kludge, &get_spelling_object_size);

	string[i++] = toupper(ptr_spelling_obj->storage[0]);    /* first letter is upper case */

	for (j = 1; j < ptr_spelling_obj->count; j++) 
	{
	    assert(i < size);
	    string[i++] = ptr_spelling_obj->storage[j];
	}

	assert(i < size);

	if (word + 1 == ptr_name_obj->count)   /* it's the last word */
	{
	    string[i++] = '\0';
	}
	else
	{
	    string[i++] = ' ';
	}
    }

    free(ptr_name_obj);

    return true;
}


