/*             This file is part of the New World OS project
--                Copyright (C) 2006, 2007  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: import_file.c,v $
-- Revision 1.9  2007/02/11 16:58:26  jsedwards
-- Changed so DEFAULT_TYPE has to specify RO (Read-Only) or RW (Read-Write).
--
-- Revision 1.8  2007/01/27 14:59:03  jsedwards
-- Changed to handle new possible return values (duplicate path or file).
--
-- Revision 1.7  2007/01/17 13:17:43  jsedwards
-- Added security level options.
--
-- Revision 1.6  2007/01/07 20:31:55  jsedwards
-- Added call to log arguments.
--
-- Revision 1.5  2007/01/07 03:27:06  jsedwards
-- Added call to set encryption level (Low by default).
--
-- Revision 1.4  2006/12/07 14:10:25  jsedwards
-- Removed call to file_setup, no longer needed.
--
-- Revision 1.3  2006/11/11 12:01:03  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.2  2006/10/26 01:51:27  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.4  2006/10/13 12:58:52  jsedwards
-- Added printing of time to import each file.
--
-- Revision 1.1.2.3  2006/10/13 11:48:27  jsedwards
-- Add call to flush block maps after each file is read.
--
-- Revision 1.1.2.2  2006/10/03 12:53:07  jsedwards
-- Changed so that instead of calling a separate routine after initialize to
-- change the already opened storage, you call it now with a type of storage
-- parameter and a path to the storage.  The problem with the other way was
-- if you tried reading a compressed file on another machine it tried to open
-- the default file which didn't exist.
--
-- Revision 1.1.2.1  2006/09/18 01:45:37  jsedwards
-- Program to store a file into objectify.
--
*/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "objectify.h"


struct options {
  Encryption_Level level;
  char* option;
  char* description;
} encryption_options[] = 
  {
    { Encryption_Minimal,  "--minimal-security",  "minimal"  },
    { Encryption_Very_Low, "--very-low-security", "very low" },
    { Encryption_Low,      "--low-security",      "low"      },
    { Encryption_Medium,   "--medium-security",   "medium"   },
    { Encryption_Extreme,  "--extreme-security",  "extreme"  },
  };

#define NUM_OPTIONS (sizeof(encryption_options)/sizeof(struct options))


int main(int argc, char* argv[])
{
    ObjRef root_object_ref;
    uint8 big_key[16 + 8 + 4];
    uint8 bf_key[16];
    uint32 linear;
    uint32 serial;
    ObjRef ref;
    ObjCreateResult result;
    int i;
    int j;
    time_t start_time;
    int result_counts[NUM_OBJ_RESULTS];

    for (i = 0; i < NUM_OBJ_RESULTS; i++) result_counts[i] = 0;

    printf("\n");

    for (i = 1; i < argc; i++)
    {
	if (argv[i][0] == '-')
	{
	    for (j = 0; j < NUM_OPTIONS; j++)
	    {
		if (strcmp(argv[i], encryption_options[j].option) == 0) break;
	    }

	    if (j == NUM_OPTIONS)
	    {
		fprintf(stderr, "Unknown option: %s\n", argv[i]);
		exit(1);
	    }
	}
    }

    nwos_log_arguments(argc, argv);

    nwos_get_key_from_password(big_key, sizeof(big_key));

    memcpy(bf_key, big_key, 16);
    linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
    memcpy(root_object_ref.id, big_key+20, 4);
    serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

    nwos_initialize_objectify(bf_key, linear, serial, DEFAULT_TYPE_RW, DEFAULT_FILE);

    nwos_set_root_object(&root_object_ref);

    nwos_set_encryption_level(Encryption_Low);


    for (i = 1; i < argc; i++)
    {
	if (argv[i][0] == '-')
	{
	    for (j = 0; j < NUM_OPTIONS; j++)
	    {
		if (strcmp(argv[i], encryption_options[j].option) == 0) break;
	    }

	    assert(j < NUM_OPTIONS);

	    printf("setting encryption level to: %s\n", encryption_options[j].description);

	    nwos_set_encryption_level(encryption_options[j].level);
	}
	else
	{
	    start_time = time(NULL);

	    result = nwos_create_file(argv[i], &ref);

	    result_counts[result]++;

	    switch(result)
	    {
	      case CREATED_NEW:
		printf("file created: %02x%02x%02x%02x   time: %d seconds\n", 
		       ref.id[0], ref.id[1], ref.id[2], ref.id[3], 
		       (int) (time(NULL) - start_time));
		break;

	      case FOUND_EXISTING:
		printf("existing file found: %02x%02x%02x%02x\n",
		       ref.id[0], ref.id[1], ref.id[2], ref.id[3]);
		break;

	      case DUPLICATE_PATH:
		printf("WARNING: a file by the name of: %s already exists.\n", argv[i]);
		printf("         In this version it will be skipped!!\n");
		break;

	      case DUPLICATE_FILE:
		printf("WARNING: a file with the same size, md5, and sha1 sums already exists.\n");
		printf("         In this version it will be skipped!!\n");
		break;

	      case ERROR_OCCURRED:
	      case NUM_OBJ_RESULTS:   /* keep the compiler form complaining */
		break;
	    }

	    nwos_flush_bit_maps();
	}
    }

    printf("files created:   %d\n", result_counts[CREATED_NEW]);
    printf("existing files:  %d\n", result_counts[FOUND_EXISTING]);
    printf("duplicate paths: %d\n", result_counts[DUPLICATE_PATH]);
    printf("duplicate files: %d\n", result_counts[DUPLICATE_FILE]);
    printf("files w/errors:  %d\n", result_counts[ERROR_OCCURRED]);

    nwos_terminate_objectify();

    return 0;
}


