/*             This file is part of the New World OS project
--                   Copyright (C) 2007  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- This is a one time use program to add a feature to the "Credit Union" class
--
-- $Log: add_to_class.c,v $
-- Revision 1.13  2007/06/03 12:19:19  jsedwards
-- Changed to add 'date discontinued' to device.
--
-- Revision 1.12  2007/05/28 19:57:50  jsedwards
-- Change yet again to add 'specifications url' feature to 'Device' class.
--
-- Revision 1.11  2007/05/28 15:53:21  jsedwards
-- Change to add generic name and introduction date to 'Device', and rearrange
-- order.
--
-- Revision 1.10  2007/05/28 10:29:05  jsedwards
-- Add even more features to Device.
--
-- Revision 1.9  2007/05/28 10:08:47  jsedwards
-- Added remove_feature() function and modified Device class by removing
-- features that are specific to a particular unit and added more features
-- that are not specific to a particular unit.
--
-- Revision 1.8  2007/05/25 02:48:32  jsedwards
-- Change to add 'location' to 'device' class.
--
-- Revision 1.7  2007/05/22 02:15:15  jsedwards
-- Change to add 'purchase' and 'sale' features to 'Device' class.
--
-- Revision 1.6  2007/05/06 00:02:45  jsedwards
-- Changed to add 'pin' to 'account' class.
--
-- Revision 1.5  2007/05/05 23:43:25  jsedwards
-- Added functions to find built in types from add_class.c.
--
-- Revision 1.4  2007/04/22 13:49:45  jsedwards
-- Added another 2 features to 'credit union' class.
--
-- Revision 1.3  2007/04/21 15:35:50  jsedwards
-- Made yet another addition to the 'credit union' class.
--
-- Revision 1.2  2007/04/19 14:40:29  jsedwards
-- Changed to add two more features to credit union.
--
-- Revision 1.1  2007/04/19 13:19:13  jsedwards
-- Created from add_to_check.c program, changed to add to credit union class.
--
*/


#include <assert.h>
#include <ctype.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "../crc32.h"
#include "../objectify_private.h"


#if 0
static size_t get_spelling_object_size(void* spelling_obj)
{
    assert(((C_struct_Spelling*)spelling_obj)->count > 0);
    return sizeof(C_struct_Spelling) + ((C_struct_Spelling*)spelling_obj)->count;
}

static size_t get_name_object_size(void* name_obj)
{
    assert(((C_struct_Name*)name_obj)->count > 0);
    return sizeof(C_struct_Name) + (((C_struct_Name*)name_obj)->count * sizeof(ObjRef));
}
#endif

static size_t get_class_object_size(void* class_obj)
{
    return sizeof(C_struct_Class_Definition) + (((C_struct_Class_Definition*)class_obj)->count * sizeof(ObjRef));
}


ObjRef* byte_class_ref()
{
    static ObjRef result;

    if (is_void_reference(&result))
    {
	if (!nwos_find_public_class_definition("BYTE", &result))
	{
	    fprintf(stderr, "Could not locate BYTE class definition\n");
	    nwos_terminate_objectify();
	    exit(1);
	}

	printf("byte_class_ref: %02x%02x%02x%02x\n",
	       result.id[0],
	       result.id[1],
	       result.id[2],
	       result.id[3]);
    }

    return &result;
}


ObjRef* character_class_ref()
{
    static ObjRef result;

    if (is_void_reference(&result))
    {
	if (!nwos_find_public_class_definition("CHARACTER", &result))
	{
	    fprintf(stderr, "Could not locate CHARACTER class definition\n");
	    nwos_terminate_objectify();
	    exit(1);
	}

	printf("byte_class_ref: %02x%02x%02x%02x\n",
	       result.id[0],
	       result.id[1],
	       result.id[2],
	       result.id[3]);
    }

    return &result;
}


ObjRef* objref_class_ref()
{
    static ObjRef result;

    if (is_void_reference(&result))
    {
	if (!nwos_find_public_class_definition("OBJECT REFERENCE", &result))
	{
	    fprintf(stderr, "Could not locate OBJECT REFERENCE class definition\n");
	    nwos_terminate_objectify();
	    exit(1);
	}

	printf("object_reference_class_ref: %02x%02x%02x%02x\n",
	       result.id[0],
	       result.id[1],
	       result.id[2],
	       result.id[3]);
    }

    return &result;
}


void add_feature(C_struct_Class_Definition* class_obj_ptr, ObjRef* type, char* name, int num)
{
    int count;
    ObjRef feature_ref;

    assert(get_class_object_size(class_obj_ptr) < FILE_BLOCK_SIZE);

    count = class_obj_ptr->count;

    nwos_find_or_create_public_feature_definition(type, name, num, &feature_ref);

    copy_reference(&class_obj_ptr->feature[count], &feature_ref);

    nwos_add_to_references(&class_obj_ptr->header.common.id, &feature_ref);

    class_obj_ptr->count = count + 1;

    printf("feature: %d - %s  new size: %d\n", count, name, (int)get_class_object_size(class_obj_ptr));
}



void remove_feature(C_struct_Class_Definition* class_obj_ptr, ObjRef* type, char* name, int num)
{
    int count;
    ObjRef feature_ref;
    int i;
    int j;

    assert(get_class_object_size(class_obj_ptr) < FILE_BLOCK_SIZE);

    count = class_obj_ptr->count;

    assert(nwos_find_public_feature_definition(type, name, num, &feature_ref));

    for (i = 0; i < count; i++)
    {
	if (is_same_object(&feature_ref, &class_obj_ptr->feature[i])) break;
    }

    count--;

    if (i != count)    /* not removing the last one in the list */
    {
	for (j = i; j < count; j++)
	{
	    copy_reference(&class_obj_ptr->feature[j], &class_obj_ptr->feature[j+1]);
	}
    }

    void_reference(&class_obj_ptr->feature[count]);

    nwos_remove_from_references(&class_obj_ptr->header.common.id, &feature_ref);

    class_obj_ptr->count = count;

    printf("feature removed: %d - %s  new count: %d  new size: %d\n",
	   i, name, count, (int)get_class_object_size(class_obj_ptr));
}



int main(int argc, char* argv[])
{
    ObjRef root_ref;
    ObjRef class_ref;
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Class_Definition* class_obj_ptr = (C_struct_Class_Definition*)kludge;

    char* class_name = "DEVICE";     /* <---- CHANGE THIS --------- */
    int old_num_features = 13;        /* <---- AND THIS --------- */

    nwos_log_arguments(argc, argv);

    nwos_initialize_objectify(NULL, 0, 0, DEFAULT_TYPE_RW, DEFAULT_FILE);

    nwos_set_encryption_level(Encryption_None);

    printf("Updating '%s' class on: %s\n", class_name, DEFAULT_FILE);

    printf("next_reference: %02x%02x%02x%02x\n", 
	   nwos_next_public_ref.id[0],
	   nwos_next_public_ref.id[1],
	   nwos_next_public_ref.id[2],
	   nwos_next_public_ref.id[3]);

    root_ref.id[0] = 0;
    root_ref.id[1] = 0;
    root_ref.id[2] = 0;
    root_ref.id[3] = 1;

    nwos_set_root_object(&root_ref);

    if (!nwos_find_public_class_definition(class_name, &class_ref))
    {
	fprintf(stderr, "Could not locate %s class definition\n", class_name);
	nwos_terminate_objectify();
	exit(1);
    }

    printf("class_ref: %02x%02x%02x%02x\n",
	   class_ref.id[0],
	   class_ref.id[1],
	   class_ref.id[2],
	   class_ref.id[3]);

    nwos_read_variable_sized_object_from_disk(&class_ref, kludge, sizeof(kludge), &get_class_object_size);

    printf("  number of features: %d\n", class_obj_ptr->count);

    if (class_obj_ptr->count != old_num_features)
    {
	fprintf(stderr, "%s class has already been changed, has more than %d features!\n",
		class_name, old_num_features);
	nwos_terminate_objectify();
    }
    else
      {
	/* going to do something tricky here to have it put the new features in certain place */

	/* features:            index old  new
	   ObjRef manufacturer;        0 ->  0
           ObjRef name                 1 ->  1
	   ObjRef model_name;          2 ->  2
	   ObjRef model_number;        3 ->  3
           ObjRef introduction date    4 ->  4
                  discontinued date          5
	   ObjRef specifications;      5 ->  6
           ObjRef specifications_url   6 ->  7
	   ObjRef manual_url;          7 ->  8
	   ObjRef component_list;      8 ->  9
	   ObjRef software_list;       9 -> 10
	   ObjRef accessories;        10 -> 11
	   ObjRef photo_url_list;     11 -> 12 
	   ObjRef online_photo_list;  12 -> 13
	*/

	copy_reference(&class_obj_ptr->feature[13], &class_obj_ptr->feature[12]);
	copy_reference(&class_obj_ptr->feature[12], &class_obj_ptr->feature[11]);
	copy_reference(&class_obj_ptr->feature[11], &class_obj_ptr->feature[10]);
	copy_reference(&class_obj_ptr->feature[10], &class_obj_ptr->feature[9]);
	copy_reference(&class_obj_ptr->feature[9],  &class_obj_ptr->feature[8]);
	copy_reference(&class_obj_ptr->feature[8],  &class_obj_ptr->feature[7]);
	copy_reference(&class_obj_ptr->feature[7],  &class_obj_ptr->feature[6]);
	copy_reference(&class_obj_ptr->feature[6],  &class_obj_ptr->feature[5]);
	/* leave hole [5] for discontinued date */

	class_obj_ptr->count = 5;
	add_feature(class_obj_ptr, objref_class_ref(), "date discontinued", 1);

	class_obj_ptr->count = 14;

	nwos_crc32_calculate((uint8*) &class_obj_ptr->header.object,
			     sizeof(ObjectHeader),
			     class_obj_ptr->header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &class_obj_ptr->name,
			     get_class_object_size(class_obj_ptr) - sizeof(EveryObject),
			     class_obj_ptr->header.common.data_chksum);

	nwos_overwrite_object_to_disk(&class_ref, class_obj_ptr, get_class_object_size(class_obj_ptr));
    }

    nwos_terminate_objectify();

    return 0;
}

