/*             This file is part of the New World OS project
--                   Copyright (C) 2007  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: add_manufacturers.c,v $
-- Revision 1.2  2007/05/26 19:58:14  jsedwards
-- Fixed to handle manufacturer name being null (only has an acronym) and
-- allow multiple URLs.
--
-- Revision 1.1  2007/05/26 13:26:35  jsedwards
-- Initial version - incomplete.
--
*/

#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "../objectify_private.h"
#include "../crc32.h"


struct manufacturer
{
    char* brand_name;
    char* acronym;
    char* urls[3];    /* WARNING: create expects this list to be NULL terminated */
};

struct manufacturer manufacturers[] =
  {
    { "Acoustic Research",                "AR",   { "www.audiovox.com" } },
    { "Adaptec",                          NULL,   { "adaptec.com" } },
    { "Advanced Micro Devices",           "AMD",  { "www.amd.com" } },
    { "American Power Conversion",        "APC",  { "www.apc.com" } },
    { "Apple",                            NULL,   { "www.apple.com" } },
    { "Audiovox",                         NULL,   { "www.audiovox.com" } },
    { "Averatec",                         NULL,   { "www.averatec.com" } },
    { "Boston Acoustics",                 NULL,   { "bostonacoustics.com" } },
    { "Canon",                            NULL,   { "www.canon.com" } },
/* D-Link - must fix name to handle hyphen */
    { "Daewoo",                           NULL,   { "www.daewoo.com" } },
    { "Dell",                             NULL,   { "www.dell.com" } },
    { "Denon",                            NULL,   { "denon.com" } },
    { "Dual",                             NULL,   { "www.dualav.com", "www.dual-reference.com" } },
    { "Eclipse",                          NULL,   { "www.eclipse-web.com" } },
    { "Emerson",                          NULL,   { "www.gotoemerson.com" } },
    { "Epson",                            NULL,   { "epson.com" } },
    { "Fujitsu",                          NULL,   { "www.fujitsu.com" } },
    { "Gateway",                          NULL,   { "www.gateway.com" } },
    { "General Electric",                 "GE",   { "www.ge.com" } },
    { "Harman Kardon",                    NULL,   { "www.harmankardon.com" } },
    { "Hewlett Packard",                  "HP",   { "www.hp.com" } },
    { "Hitachi",                          NULL,   { "hitachi.com" } },
    { "Intel",                            NULL,   { "www.intel.com" } },
    { "International Business Machines",  "IBM",  { "www.ibm.com" } },
    { NULL,                               "JBL",  { "jbl.com", "www.jblpro.com" } },
    { "Japan Victor Company",             "JVC",  { "jvc.com" } },
    { "Jensen",                           NULL,   { "www.audiovox.com" } },
    { "Kawasaki",                         NULL,   { "kawasaki.com" } },
    { "Kenwood",                          NULL,   { "kenwood.com" } },
    { "Klipsch",                          NULL,   { "klipsch.com" } },
    { "Kodak",                            NULL,   { "www.kodak.com" } },
    { "Konica",                           NULL,   { "konicaminolta.com" } },
    { "Lenovo",                           NULL,   { "www.lenovo.com" } },
    { "Lexmark",                          NULL,   { "www.lexmark.com" } },
    { "Linksys",                          NULL,   { "www.linksys.com" } },
/* MUST fix name to handle hyphen
    { "M-Audio",                          NULL,   { "www.m-audio.com" } },
*/
    { "Magnavox",                         NULL,   { "magnavox.com" } },
    { "Marantz",                          NULL,   { "marantz.com" } },
    { "Maxtor",                           NULL,   { "www.seagate.com/maxtor" } },
    { "Memorex",                          NULL,   { "memorex.com" } },
    { "Micro Star International",         "MSI",  { "www.msicomputer.com" } },
    { "Minolta",                          NULL,   { "konicaminolta.com" } },
    { "Mitsubishi",                       NULL,   { "www.mitsubishi.com" } },
    { "Motorola",                         NULL,   { "www.motorola.com" } },
    { "Nakamichi",                        NULL,   { "www.nakamichi.co.jp", "www.nakamichi.co.jp" } },
    { "Netgear",                          NULL,   { "netgear.com" } },
    { "Nikon",                            NULL,   { "nikon.com" } },
    { "Olympus",                          NULL,   { "www.olympus-global.com", "olympusamerica.com" } },
    { "Onkyo",                            NULL,   { "onkyo.com" } },
    { "Panasonic",                        NULL,   { "www.panasonic.com" } },
    { "Pentax",                           NULL,   { "www.pentax.com" } },
    { "Philips",                          NULL,   { "www.philips.com" } },
    { "Pioneer",                          NULL,   { "www.pioneerelectronics.com" } },
    { "Polaroid",                         NULL,   { "www.polaroid.com" } },
    { "Radio Corporation of America",     "RCA",  { "home.rca.com" } },
    { NULL,                               "SMC",  { "www.smc.com" } },
    { "Samsung",                          NULL,   { "samsung.com" } },
    { "Sanyo",                            NULL,   { "sanyo.com" } },
    { "Seagate",                          NULL,   { "www.seagate.com" } },
    { "Sony",                             NULL,   { "www.sony.com" } },
    { "Sharp",                            NULL,   { "www.sharpusa.com" } },
    { "Sherwood",                         NULL,   { "www.sherwoodusa.com", "www.sherwood.de" } },
    { "Sylvania",                         NULL,   { "www.sylvania.com" } },
    { NULL,                               "TDK",  { "tdk.com" } },
    { NULL,                               "TEAC", { "www.teac.com" } },
    { "Technics",                         NULL,   { "www.panasonic-europe.com" } },
    { "TiVo",                             NULL,   { "www.tivo.com" } },
    { "Toshiba",                          NULL,   { "toshiba.com" } },
    { "Western Digital",                  NULL,   { "westerndigital.com" } },
    { "Xerox",                            NULL,   { "www.xerox.com" } },
    { "Yamaha",                           NULL,   { "www.yamaha.com" } },
    { "Zenith",                           NULL,   { "www.zenith.com" } },
    { "Zonet",                            NULL,   { "www.zonetusa.com" } },
};

const int num_mfrs = (sizeof(manufacturers) / sizeof(struct manufacturer));



int main(int argc, char* argv[])
{
    ObjRef root_ref;
    ObjRef reference;
    C_struct_Manufacturer test_mfr_obj;
    char test_string[128];
    ReferenceList* ref_list;
    int num_refs;
    int i;


    nwos_log_arguments(argc, argv);

    nwos_initialize_objectify(NULL, 0, 0, DEFAULT_TYPE_RW, DEFAULT_FILE);

    nwos_set_encryption_level(Encryption_None);

    printf("Adding manufacturer on: %s\n", DEFAULT_FILE);


    root_ref.id[0] = 0;
    root_ref.id[1] = 0;
    root_ref.id[2] = 0;
    root_ref.id[3] = 1;

    nwos_set_root_object(&root_ref);

    for (i = 0; i < num_mfrs; i++)
    {
	if (manufacturers[i].brand_name != NULL &&
	    nwos_find_public_manufacturer(manufacturers[i].brand_name, &reference))
	{
	    printf("WARNING: manufacturer %s already in database\n", manufacturers[i].brand_name);
	    continue;
	}

	if (manufacturers[i].acronym != NULL &&
	    nwos_find_public_manufacturer(manufacturers[i].acronym, &reference))
	{
	    printf("WARNING: manufacturer acronym %s already in database\n", manufacturers[i].acronym);
	    continue;
	}

	nwos_create_public_manufacturer(manufacturers[i].brand_name,
					manufacturers[i].acronym,
					manufacturers[i].urls,
					&reference);

	printf("manufacturer: %02x%02x%02x%02x: ", 
	       reference.id[0],
	       reference.id[1],
	       reference.id[2],
	       reference.id[3]);

	nwos_read_object_from_disk(&reference, &test_mfr_obj, sizeof(test_mfr_obj));

	if (!is_void_reference(&test_mfr_obj.brand_name))
	{
	    assert(nwos_name_to_string(&test_mfr_obj.brand_name, test_string, sizeof(test_string)));
	    printf("%s, ", test_string);
	}

	if (!is_void_reference(&test_mfr_obj.acronym))
	{
	    assert(nwos_acronym_to_string(&test_mfr_obj.acronym, test_string, sizeof(test_string)));
	    printf("%s", test_string);
	}

	if (!is_void_reference(&test_mfr_obj.url_list))
	{
	    ref_list = nwos_malloc_reference_list(&test_mfr_obj.url_list);

	    num_refs = ref_list->common_header.num_refs;

	    if (num_refs == 1)
	    {
		assert(nwos_url_to_string(&ref_list->references[0], test_string, sizeof(test_string)));
		printf(", %s", test_string);
	    }
	    else if (num_refs == 2)
	    {
		assert(nwos_url_to_string(&ref_list->references[0], test_string, sizeof(test_string)));
		printf(", %s", test_string);
		assert(nwos_url_to_string(&ref_list->references[1], test_string, sizeof(test_string)));
		printf(", %s", test_string);
	    }
	    else
	    {
		printf("Error: number of URLs is not 1 or 2: %d\n", num_refs);
		break;
	    }

	    nwos_free_reference_list(ref_list);
	    ref_list = NULL;
	}

	printf("\n");
    }

    nwos_terminate_objectify();

    return 0;
}


