/*             This file is part of the New World OS project
--                 Copyright (C) 2004-2007  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: objectify.c,v $
-- Revision 1.114  2007/06/23 14:28:40  jsedwards
-- Added code to close the public file in nwos_terminate.
--
-- Revision 1.113  2007/06/22 22:06:31  jsedwards
-- Fix so it will compile correctly in PUBLIC_MODE.
--
-- Revision 1.112  2007/06/22 15:04:15  jsedwards
-- Added testing public object file header for correct version, etc.
--
-- Revision 1.111  2007/06/22 14:16:04  jsedwards
-- Changed location of public object file to a define in config.h.
--
-- Revision 1.110  2007/06/21 22:37:28  jsedwards
-- Added temporary code to verify new get_object_size function returns the
-- same value as the class specific function passed into read_variable_sized.
--
-- Revision 1.109  2007/06/20 03:46:43  jsedwards
-- Added special function to set the total_private_blocks for expand_sparse.
--
-- Revision 1.108  2007/06/19 18:48:32  jsedwards
-- Changed to use separate public objects file instead of having it part of
-- private storage.
--
-- Revision 1.107  2007/05/10 14:42:48  jsedwards
-- Add conditionals for compiling export_c_structs (NO_INDEX_FILE).  Fix
-- to free malloc'd path name.
--
-- Revision 1.106  2007/05/10 13:48:23  jsedwards
-- Fix problems with storing index file.
--
-- Revision 1.105  2007/05/10 12:36:15  jsedwards
-- Add code to save index into file and use it if possible.
--
-- Revision 1.104  2007/04/19 12:01:54  jsedwards
-- Move test for bad open right after open.
--
-- Revision 1.103  2007/04/19 11:55:16  jsedwards
-- Split two lines that somehow got concatenated into one line. NO code changes!
--
-- Revision 1.102  2007/04/15 16:18:54  jsedwards
-- Change generate random id to try longer to find a free block.
--
-- Revision 1.101  2007/04/14 12:30:32  jsedwards
-- Remove hack that forced it to not use IDs in the first pass of the storage
-- device, which was intended to balance out old archives that only allocated
-- IDs in the disk range.
--
-- Revision 1.100  2007/04/14 12:28:10  jsedwards
-- Change to allow sparse (regular) file.
--
-- Revision 1.99  2007/04/05 01:45:18  jsedwards
-- Changed 'hash_ref' to 'nwos_hash_ref' so it could be external.
--
-- Revision 1.98  2007/03/27 11:41:11  jsedwards
-- Made total and used, public and private block counts global so that
-- expand_sparse can access them.
--
-- Revision 1.97  2007/03/26 12:32:39  jsedwards
-- Separate last change time into public and private.
--
-- Revision 1.96  2007/03/24 02:22:51  jsedwards
-- Increment used_public_blocks when we allocate a new one.
--
-- Revision 1.95  2007/03/09 13:47:07  jsedwards
-- Fixed to use RESERVED_PUBLIC_BLOCKS to determine if an reference id is
-- public or private (this was broken unless total_public_blocks was exactly
-- equal to RESERVED_PUBLIC_BLOCKS.
--
-- Revision 1.94  2007/03/08 13:51:22  jsedwards
-- Change so output about indexing compressed file goes to stderr instead of
-- stdout.  This way if the standard output doesn't have junk in it.
--
-- Revision 1.93  2007/03/08 13:42:19  jsedwards
-- Changes to allow reading a public compressed file.
--
-- Revision 1.92  2007/03/08 13:04:58  jsedwards
-- Rearranged and added ifndefs to make it compile in public mode without
-- warnings.
--
-- Revision 1.91  2007/03/08 01:14:27  jsedwards
-- Fix hash_uint32 in public_mode to return correct value.
--
-- Revision 1.90  2007/03/02 14:27:31  jsedwards
-- Fixed generate_random_id to generate all 32 bits instead of just 31.
--
-- Revision 1.89  2007/02/27 12:24:30  jsedwards
-- Added test_bit_in_map() and made hash_uint32_ref(), write_block(),
-- set_bit_in_map(), clear_bit_in_map() external.
--
-- Revision 1.88  2007/02/27 03:29:08  jsedwards
-- Change the ID hash to disc block to skip over bit maps instead of leaving
-- holes in the available IDs.
--
-- Revision 1.87  2007/02/25 20:57:01  jsedwards
-- Changed bit map tests in generate new id functions to use the hashed value
-- for testing the bit map instead of the id itself (because the two are no
-- longer the same thing).
--
-- Revision 1.86  2007/02/25 20:53:09  jsedwards
-- Moved test for inside bit map from ref_to_offset into hash_uint32_ref, so
-- it is consistant for all three routines.
--
-- Revision 1.85  2007/02/25 20:48:45  jsedwards
-- Fixed so it doesn't generate a completely random id everytime id is out
-- of total blocks on disk range.
--
-- Revision 1.84  2007/02/24 15:56:03  jsedwards
-- Added clear_bit_in_map function and use it when removing object.
--
-- Revision 1.83  2007/02/24 15:53:37  jsedwards
-- Change reference id hash back to allowing all 32 bits instead of just ids
-- that map directly to a disk block.  This way when the size of the disk or
-- partition changes, it is much more graceful.
--
-- Revision 1.82  2007/02/23 13:27:04  jsedwards
-- Move the strlcpy and strlcat functions from objectify.c to log.c so that
-- log.o can be used without having to drag in the whole objecitfy library.
--
-- Revision 1.81  2007/02/11 16:17:56  jsedwards
-- Added "strlcat" function only for Linux, because it seems to be missing.
-- Also fixed bug in "strlcpy".
--
-- Revision 1.80  2007/02/11 14:28:47  jsedwards
-- Changed all the 'sprintf' calls to 'snprintf' so OpenBSD won't whine so much.
--
-- Revision 1.79  2007/02/11 14:23:01  jsedwards
-- Change all 'off64_t' and 'lseek64' references to 'off_t' and 'lseek',
-- because BSD doesn't dig the whole brain damaged 64 bit thing.
--
-- Revision 1.78  2007/01/27 01:52:16  jsedwards
-- Added global 'verbose' boolean.
--
-- Revision 1.77  2007/01/17 13:15:30  jsedwards
-- Change Encryption_Fast level name to Encryption_Minimal.
--
-- Revision 1.76  2007/01/14 03:58:30  jsedwards
-- Fix create_root in public mode.
--
-- Revision 1.75  2007/01/13 20:00:52  jsedwards
-- Changed so that the bit maps are used to determine if a block is used
-- instead of whether the block is all zeros.  Also set the bit when the
-- reference id is generated so that we don't have the problem when a block
-- isn't written right away.
--
-- Revision 1.74  2007/01/07 03:23:54  jsedwards
-- Added push and pop encryption level routines.  If'd out generate_in_range
-- routine and added generate_new_id_one_in_N_routine.
--
-- Revision 1.73  2007/01/04 17:35:04  jsedwards
-- Fix log message in terminate.  Add routines to compute the percentage of
-- private space used and percentage of chunk used.  Add routine for generating
-- block ids for low level encryption.
--
-- Revision 1.72  2007/01/02 11:18:52  jsedwards
-- Changed so that root object class is always the public class, I can't think
-- of any reason to have a private root object class.  Also changed for the
-- time being that you can't pass the public root in private mode.
--
-- Revision 1.71  2006/12/30 03:45:41  jsedwards
-- Moved NUM_PUBLIC_IDS and NUM_PRIVATE_IDS defines from objectify_private.h
-- to here, because they are only used for older versions (0014 for example).
--
-- Revision 1.70  2006/12/28 23:15:13  jsedwards
-- Added printing of public or private size to terminate when objects were
-- added.
--
-- Revision 1.69  2006/12/27 14:15:24  jsedwards
-- Move log functions out of objectify.c and into a new file log.c.
--
-- Revision 1.68  2006/12/27 12:25:06  jsedwards
-- Add function to put program arguments into log.
--
-- Revision 1.67  2006/12/25 12:04:52  jsedwards
-- Fixes to run in "public_mode".
--
-- Revision 1.66  2006/12/21 13:13:38  jsedwards
-- Change to call find_public_class_definition for getting reference list.
--
-- Revision 1.65  2006/12/20 12:35:13  jsedwards
-- Changed to have two class definition refs, public and private.
--
-- Revision 1.64  2006/12/19 13:38:33  jsedwards
-- Fixed to increment the used_private_blocks.
--
-- Revision 1.63  2006/12/15 14:59:11  jsedwards
-- Changed back to only having two disk areas (Public and Private).
--
-- Revision 1.62  2006/12/14 14:24:53  jsedwards
-- Removed creation of reference list for root object.
--
-- Revision 1.61  2006/12/14 12:19:21  jsedwards
-- Fixed bug so private id gets incremented everytime.
--
-- Revision 1.60  2006/12/13 14:23:38  jsedwards
-- Added encryption level stuff.
--
-- Revision 1.59  2006/12/12 03:11:13  jsedwards
-- Fixed generate_new_private_id function.
--
-- Revision 1.58  2006/12/11 14:40:58  jsedwards
-- Tried to repair some things but now needs more work.
--
-- Revision 1.57  2006/12/11 14:22:09  jsedwards
-- Fixed generate_private_id to use block maps. Incomplete, won't compile!
--
-- Revision 1.56  2006/12/09 01:42:40  jsedwards
-- Added some more "public_mode" ifdef statements.
--
-- Revision 1.55  2006/12/08 17:17:24  jsedwards
-- Added ifdef statements to handle "public_mode" that changes it to deal with
-- public objects.  Fixed bug in generate_new_public_id and added code to
-- assert that it incremented the id by one.  Moved read_class_definition to
-- class_definition.c.
--
-- Revision 1.54  2006/12/08 11:31:21  jsedwards
-- Change to use new disk_header structure.
--
-- Revision 1.53  2006/12/08 10:42:05  jsedwards
-- Change to allow security parameters to be null.
--
-- Revision 1.52  2006/12/05 13:55:33  jsedwards
-- Fix bug in write_public_object where it wasn't passing the clean block to
-- write_block, it was passing the object that wasn't the size of a block.
--
-- Revision 1.51  2006/12/05 04:20:00  jsedwards
-- More changes for new disk header layout.  Removed UNENCRYPTED_FLAG as now
-- unencrypted/encrypted is determined from the reference itself.
--
-- Revision 1.50  2006/12/04 14:58:17  jsedwards
-- First changes for new header, etc.  This version doesn't work!
--
-- Revision 1.49  2006/12/01 05:16:46  jsedwards
-- Added a test in get_object_class to make sure it is not a reference list
-- before trying to compute the header checksum.
--
-- Revision 1.48  2006/12/01 05:11:48  jsedwards
-- Add a strlcpy function only if compiling on Linux because it doesn't have
-- it in the library.
--
-- Revision 1.47  2006/11/27 14:28:26  jsedwards
-- Added test for unencrypted objects in read_and_decrypt and write_and_encrypt
-- functions and skip over encryption stuff if set.
--
-- Revision 1.46  2006/11/27 13:46:46  jsedwards
-- Added functions to encode and decode variable sized counts.
--
-- Revision 1.45  2006/11/19 13:44:30  jsedwards
-- Add time stamp to disk header so we can tell when it was last changed.
--
-- Revision 1.44  2006/11/18 15:09:10  jsedwards
-- Added "max_size" parameter to read_variable_sized_object_from_disk because
-- objects are no longer limited to one file block.
--
-- Revision 1.43  2006/11/18 14:26:47  jsedwards
-- Changed to read and write objects larger than FILE_BLOCK_SIZE (256 bytes).
--
-- Revision 1.42  2006/11/17 12:18:11  jsedwards
-- Added "read_only" flag to keep it from trying to write the header back when
-- read only.
--
-- Revision 1.41  2006/11/11 15:20:37  jsedwards
-- Moved all reference list code into a separate file: reference_list.c.
--
-- Revision 1.40  2006/11/11 14:24:37  jsedwards
-- Made nwos_malloc and nwos_free public (not static) and moved
-- find_class_definiton function to class_definition.c.
--
-- Revision 1.39  2006/11/11 13:58:00  jsedwards
-- Add new generate_new_public_id function.
--
-- Revision 1.38  2006/11/11 13:34:32  jsedwards
-- Made disk header a static global that is read in initialize(), updated with
-- next public reference and written back out in terminate().  Also added a
-- new global variable with the next public reference.
--
-- Revision 1.37  2006/11/11 12:01:05  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.36  2006/11/10 13:39:16  jsedwards
-- Uncommented write_block in write_public_object_to_disk and removed print
-- statements.
--
-- Revision 1.35  2006/11/10 13:32:05  jsedwards
-- Moved Ref_List_First_Block and Ref_List_Extra_Block definitions from
-- objectify.c to objectify_private.h so that import_public_csv can use
-- them also.
--
-- Revision 1.34  2006/11/10 04:11:02  jsedwards
-- Add "write_public_object_to_disk" function and fix hash to deal with
-- public references.
--
-- Revision 1.33  2006/11/04 18:51:07  jsedwards
-- Made a #define in the config.h file for the log file.
--
-- Revision 1.32  2006/11/02 11:49:28  jsedwards
-- Fixed all cases where 'z' was used as a format for 'off64_t' values because
-- the older compiler complains.
--
-- Revision 1.31  2006/10/26 01:22:51  jsedwards
-- Replace the main truck version (1.30) with the latest version from the
-- alpha_05_branch.
--
-- Revision 1.24.2.57  2006/10/25 12:22:29  jsedwards
-- Changed C_struct_class_definition to C_struct_Class_Definition so the case
-- is consistent with all the other C_struct objects.
--
-- Revision 1.24.2.56  2006/10/22 13:05:16  jsedwards
-- Changed so that blocks on disk and reserved public blocks is variable
-- based on values store in disk header.
--
-- Revision 1.24.2.55  2006/10/22 12:40:47  jsedwards
-- Corrected version string error message.
--
-- Revision 1.24.2.54  2006/10/22 00:16:43  jsedwards
-- More fixes for the changing to the separate next_block_ptr to the add to
-- reference list and write reference list.
--
-- Revision 1.24.2.53  2006/10/20 13:25:58  jsedwards
-- Change to have a separate next_block_ptr (for linked list) instead of
-- using the first 4 (unused) bytes of each reference list block as the
-- pointer to the next block.  This wasn't happy when pointers aren't 4 bytes.
--
-- Revision 1.24.2.52  2006/10/17 12:38:37  jsedwards
-- Changed formats in printf statementss to be more compatible with different
-- types (size_t, off64_t) being different sizes.  Also changed format for
-- uint32 to be unsigned instead of long unsigned (unsigned is 32 bits on
-- both 32 and 64 bit machines, whereas long unsigned is 64 on amd64).
--
-- Revision 1.24.2.51  2006/10/16 13:01:05  jsedwards
-- Add code to rewind disk after indexing a compressed file so reading the
-- header works correctly.
--
-- Revision 1.24.2.50  2006/10/14 13:13:45  jsedwards
-- Added code to check the disk header and make sure it is a nwos disk, the
-- version string matches, and the disk size matches what was compiled in.
--
-- Revision 1.24.2.49  2006/10/13 11:47:44  jsedwards
-- Comment out print statement of setting bits in block maps.
--
-- Revision 1.24.2.48  2006/10/12 12:12:22  jsedwards
-- Fixed bug in last ref for generating references in sequence.  Also added
-- function to flush dirty block maps back to disk.
--
-- Revision 1.24.2.47  2006/10/11 13:24:50  jsedwards
-- Add code to set the bits in the bit maps when a block is written.  Note:
-- this version does not use the bit maps for anything.
--
-- Revision 1.24.2.46  2006/10/10 07:30:35  jsedwards
-- Change to 8192 bytes per bit map instead of 256 bytes.
--
-- Revision 1.24.2.45  2006/10/07 22:27:02  jsedwards
-- Changed the generate id function so that it checks to see if we are
-- operating in the sequential mode and it calls the correct function.
--
-- Revision 1.24.2.44  2006/10/06 04:39:24  jsedwards
-- Changed the names of some of the public and private, blocks and id
-- definitions and added new function to generate ids that scan across
-- the disk drive, instead of totally random.
--
-- Revision 1.24.2.43  2006/10/03 12:50:16  jsedwards
-- Changed so that instead of calling a separate routine after initialize to
-- change the already opened storage, you call it now with a type of storage
-- parameter and a path to the storage.  The problem with the other way was
-- if you tried reading a compressed file on another machine it tried to open
-- the default file which didn't exist.
--
-- Revision 1.24.2.42  2006/10/02 14:02:45  jsedwards
-- Changed to just read the file while indexing, instead of doing the seeks.
--
-- Revision 1.24.2.41  2006/10/02 13:29:54  jsedwards
-- Changed to use a binary search for the indexes instead of the linear.
--
-- Revision 1.24.2.40  2006/10/02 12:33:44  jsedwards
-- Split out reference hash into a separate function and initialized next_ptr
-- so that gcc doesn't worry about it possibly being uninitialized.
--
-- Revision 1.24.2.39  2006/10/02 02:15:33  jsedwards
-- Added the capability of reading blocks from a compressed archive file
-- instead of the normal disk drive or partition.
--
-- Revision 1.24.2.38  2006/10/01 11:49:47  jsedwards
-- Increased the number of tries in finding a random id.  Made the code to
-- not access the hard drive for testing purposes ifdef'd so it can be enabled
-- or disabled with a -D option.
--
-- Revision 1.24.2.37  2006/09/29 04:12:39  jsedwards
-- Fixed "object_exists" and "block_used" functions again.  Added if'd out
-- code to write to memory instead of the disk (for testing).
--
-- Revision 1.24.2.36  2006/09/28 13:38:46  jsedwards
-- Re-arranged testing of reference in block to do object exists better.
-- Added printing of offset in lseek error statements.
--
-- Revision 1.24.2.35  2006/09/26 13:22:38  jsedwards
-- Changed to use a disk drive or partition for storage instead of a file.
-- This required hashing the id to fit in the disk or partition size.
--
-- Revision 1.24.2.34  2006/09/24 13:46:14  jsedwards
-- Previous change was a mistake, added a new function that checks to see if
-- the block is used and changed "object_exists" function back to the way it
-- was.
--
-- Revision 1.24.2.33  2006/09/24 13:38:06  jsedwards
-- Add test to "object_exists" function to verify it is not just a zero block.
--
-- Revision 1.24.2.32  2006/09/19 14:10:45  jsedwards
-- Fix bug where next block wasn't zeroed when reference list sidecar filled
-- up exactly.  Added printing of num_blocks in log.
--
-- Revision 1.24.2.31  2006/09/18 02:07:04  jsedwards
-- Add asserts to make sure the reference passed into routines is not void.
--
-- Revision 1.24.2.30  2006/09/18 01:32:52  jsedwards
-- Renamed read_file_and_decrypt and write_file with "nwos" prefix and removed
-- the static so that they can be accessed by other modules (like file.c).
--
-- Revision 1.24.2.29  2006/09/09 13:03:33  jsedwards
-- Moved "create_root_object" routine from security.c to objectify.c so that
-- the security module didn't have references to so many other parts of the
-- system.
--
-- Revision 1.24.2.28  2006/09/08 13:28:41  jsedwards
-- Changed to use a different sequence block for objects beyond the first for
-- objects larger than one block.
--
-- Revision 1.24.2.27  2006/09/08 11:07:34  jsedwards
-- Changed so that random_sequence table is a two dimentional array so we can
-- store different sequence tables for multiple block objects.
--
-- Revision 1.24.2.26  2006/09/08 11:01:23  jsedwards
-- Changed so that the sequence table is passed into "write_object_to_disk"
-- and "read_object_from_disk_and_decrypt" routines.
--
-- Revision 1.24.2.25  2006/09/07 12:59:46  jsedwards
-- Separated the sequence table generation code into a separate function.
--
-- Revision 1.24.2.24  2006/09/06 13:15:25  jsedwards
-- Removed nwos_seed_sequence function and instead pass pointers to the values
-- in the call to next_sequence function.
--
-- Revision 1.24.2.23  2006/09/05 12:59:34  jsedwards
-- Added checksums (crc32) to reference lists.
--
-- Revision 1.24.2.22  2006/09/05 01:26:41  jsedwards
-- Converted add_to_reference_list to use the new block structures.
--
-- Revision 1.24.2.21  2006/09/05 00:29:16  jsedwards
-- Converted the read_reference_list_from_disk routine to use the new block
-- structures.
--
-- Revision 1.24.2.20  2006/09/05 00:07:48  jsedwards
-- Converted the write_reference_list routine to use the new block structures.
--
-- Revision 1.24.2.19  2006/09/04 15:51:14  jsedwards
-- Change read_reference_list_into_cache routine to use new block structures.
--
-- Revision 1.24.2.18  2006/09/04 13:26:06  jsedwards
-- Added reference list block structures (for the blocks written to disk).
-- Changed cache to point to a reference list first block structure instead
-- of a reference list itself.  The code really doesn't use them yet.
--
-- Revision 1.24.2.17  2006/09/04 01:01:24  jsedwards
-- Split up the create_reference_list routine so it is possible to create a
-- reference list with a given id instead of generating a new one.  This is
-- helpful when doing the conversion from the old multiple file style (we
-- can just keep the old reference ids and not have to switch them around).
--
-- Revision 1.24.2.16  2006/09/03 13:56:35  jsedwards
-- Moved defintion of DEFAULT_FILE to objectify_private.h so the big_bang can
-- use it to create the file.
--
-- Revision 1.24.2.15  2006/09/03 13:14:55  jsedwards
-- Rearranged the order of functions in the file and added some comments to
-- mark the different sections.  NO code changes were made.
--
-- Revision 1.24.2.14  2006/09/03 12:20:40  jsedwards
-- Removed a couple more debugging print statements.
--
-- Revision 1.24.2.13  2006/09/03 12:13:57  jsedwards
-- Add code to check that all malloc'd blocks are freed during testing.
--
-- Revision 1.24.2.12  2006/09/03 11:09:14  jsedwards
-- Removed many of the debugging print statements, added a routine to allow
-- an object to be overwritted (instead of having to remove it and then write
-- it again.  Also rearranged the checksum checking code slightly.
--
-- Revision 1.24.2.11  2006/09/03 10:22:14  jsedwards
-- Initialize ivec in nwos_get_object_class function.
--
-- Revision 1.24.2.10  2006/09/02 15:08:16  jsedwards
-- Add reference list class reference to root object because now it has to
-- read the reference list to get it's size and we can't easily read it
-- without the read routine verifying it's class.
--
-- Fixed add reference routine to clean dirty flag before writing block.
--
-- Fixed remove object routine to do the seek before nuking the object.
--
-- Revision 1.24.2.9  2006/09/01 13:20:00  jsedwards
-- Fixed some bugs in new storing objects in one file.  Still NOT working!
--
-- Revision 1.24.2.8  2006/09/01 11:49:59  jsedwards
-- Merged version with file additions and version that writes objects into one
-- large file.  This vesion DOES NOT work correctly.
--
-- Revision 1.24.2.7  2006/08/25 13:06:54  jsedwards
-- Fixed daylight savings to work correctly when computing the end of time.
--
-- Revision 1.24.2.6  2006/08/25 12:43:37  jsedwards
-- Change to compute the end of time value in local time and save it.  This
-- way the time stamps on the file actually read 9/9/99...
--
-- Revision 1.24.2.5  2006/08/25 12:24:00  jsedwards
-- Added code to set the time stamps on all of the objects to 9/9/99 9:09:09.
--
-- Revision 1.24.2.4  2006/08/16 12:47:56  jsedwards
-- Changed the name of the static function "log" to "nwos_log" and added a
-- cast so that new versions of GCC don't complain.
-- doesn't complain about the type of a built in function being redefined
--
-- Revision 1.24.2.3  2006/08/13 16:36:43  jsedwards
-- Made a define for the block size of the file called FILE_BLOCK_SIZE, that
-- is the minimum size of the file for each object.  The amount of disk space
-- used for a one byte file.
--
-- Revision 1.24.2.2  2006/08/13 16:27:14  jsedwards
-- Change to compute the random sequence table only one time and save it.
--
-- Revision 1.24.2.1  2006/08/13 12:53:06  jsedwards
-- Added comments to encrypt and decrypt routines (NO code changes).
--
-- Revision 1.24  2006/01/12 02:57:28  jsedwards
-- Fixed "ask yes or no" routine to actually work.
--
-- Revision 1.23  2006/01/12 02:03:36  jsedwards
-- Added code to dump the object when the data checksum fails.
--
-- Revision 1.22  2006/01/08 15:27:28  jsedwards
-- Removed unused variable.
--
-- Revision 1.21  2006/01/03 03:21:38  jsedwards
-- Added "is compatible version" routine to determine if a particular version
-- of an object is still compatible with the current version.
--
-- Revision 1.20  2006/01/01 19:47:33  jsedwards
-- Added "is quit command" and "ask yes or no" routines.
--
-- Revision 1.19  2005/12/30 14:09:50  jsedwards
-- Commented out printing of blowfish key, was for debugging.
--
-- Revision 1.18  2005/12/29 19:08:00  jsedwards
-- Fixed some of the ifdef DISABLE_SECURITY_FEATURES that were accidentally
-- entered as #if DISABLE_SECURITY_FEATURES instead.  Also added a disable
-- security features to the unique id routine to make sure it finds a unique
-- one by incrementing (since we can punch the random number generator all
-- we want and it isn't going to fix that problem).
--
-- Revision 1.17  2005/12/29 18:30:42  jsedwards
-- Added ifdefs to disable/enable the security features.
--
-- Revision 1.16  2005/12/29 18:20:24  jsedwards
-- Split up routines, added read variable length objects and other changes to
-- make encryption work, HOWEVER, ENCRYPTION IS DISABLED IN THIS VERSION.
--
-- Revision 1.15  2005/12/28 12:53:06  jsedwards
-- First attempt at adding security.  Objects are encrypted before being
-- written to disk and decrypted when read from disk.  It is incomplete at
-- this point and doesn't work completely, for example, the read object
-- headers function won't work.  This version should only be used for testing.
--
-- Revision 1.14  2005/12/27 19:44:40  jsedwards
-- Added routine to set the root object.
--
-- Revision 1.13  2005/12/27 18:14:08  jsedwards
-- Changed ObjRefs to be 4 bytes instead of 8 bytes and made them random
-- instead of somehow based upon the contents.
--
-- Revision 1.12  2005/12/24 16:18:26  jsedwards
-- Removed "host" id from object references (ObjRef).  Host redirection will
-- be done using a "redirection" object in the future.
--
-- Revision 1.11  2005/12/21 23:25:57  jsedwards
-- Added better error messages when the size of an object doesn't match the
-- number of bytes read.
--
-- Revision 1.10  2005/12/11 16:52:56  jsedwards
-- Added a log message when an object is added to a reference list and fixed
-- the error messages printed when adding to a reference list fails.
--
-- Revision 1.9  2005/12/10 15:03:36  jsedwards
-- Fixed header to say the GPL is in the LICENSE file instead of COPYING.
--
-- Revision 1.8  2005/12/05 19:03:06  jsedwards
-- Moved local read_object_from_disk (not nwos_read_object_from_disk) function
-- earlier in the file so that get_object_class function can call it instead
-- of nwos_read_object_from_disk which wasn't happy about it just reading the
-- common header.
--
-- Revision 1.7  2005/12/05 05:18:53  jsedwards
-- Ifdefd out logging each time a file is read, too much spam into log file.
--
-- Revision 1.6  2005/12/05 05:16:22  jsedwards
-- Made a separate routine 'read_reference_list_from_disk' to read reference
-- lists, that doesn't do checksums.  Changed 'read_object_from_disk' to
-- check both the header and data checksums.  Also added a log routine to
-- log each time a file is written to and read from disk.
--
-- Revision 1.5  2005/12/03 22:15:38  jsedwards
-- Added get_object_class and add_to_references functions.
--
-- Revision 1.4  2005/12/02 20:30:11  jsedwards
-- Changed so that reference list id is generated from the object that the
-- reference list is for's id.
--
-- Revision 1.3  2005/11/30 13:53:31  jsedwards
-- Added routines to get object size and read class definition.  Also added
-- 'nwos' prefix to generate_new_id routine and made it global.
--
-- Revision 1.2  2005/11/26 15:36:59  jsedwards
-- Added 'nwos' prefix to functions and added 'static' to local functions.
--
-- Revision 1.1.1.1  2005/11/25 12:44:27  jsedwards
-- Copied from 'lab'.
--
-- Revision 1.5  2005/11/24 15:47:47  jsedwards
-- Changed to match new object header layouts.
--
-- Revision 1.4  2005/11/22 13:57:04  jsedwards
-- Made a common subroutine to fill in the common header.
--
-- Revision 1.3  2005/11/22 13:09:58  jsedwards
-- Added read object from disk and create simple object routines.
--
-- Revision 1.2  2005/11/12 23:02:58  jsedwards
-- Rearranged headers.
--
-- Revision 1.1  2005/11/12 14:55:12  jsedwards
-- Routines for creating objects.
--
*/

/*---------------------------------------------------------------------------*/
/* The new plan for storing objects:                                         */
/*                                                                           */
/* Instead of storing objects in separate files (as was done previously),    */
/* the plan is to store them all in one sparse file.                         */
/*                                                                           */
/* One of the major problems with the previous plan was how to cope with     */
/* block sizes of all the different file systems.  I was using Resier4       */
/* because it's block size was 512 bytes and so if you created a file that   */
/* was only say 92 bytes you only wasted 430 bytes.  However when it was     */
/* used on an ext3 file system with 4096 blocks you wasted 4004 bytes.       */
/* I also suspsect there may have been more problems when there got to be    */
/* a large number of files with exceeding the number that were possible      */
/* in a directory and running out of inodes.                                 */
/*                                                                           */
/* So the new plan is to store all of the objects in a sparse file, indexed  */
/* by their ID number.  My current theory then is to use 256 byte blocks     */
/* for each object.  The first 8 bytes of each block is it's ID number.      */
/* If the first 8 bytes of the block are zeros then that block is unused.    */
/*                                                                           */
/* Variable sized objects that are larger than 256 bytes use multiple        */
/* blocks.  The last 4 bytes of the block point to the next block.           */
/*---------------------------------------------------------------------------*/


#define _LARGEFILE64_SOURCE

#include <assert.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <openssl/blowfish.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "crc32.h"
#include "objectify.h"
#include "objectify_private.h"
#include "time_stamp.h"


/* Global variables... BAD!... need to fix */

bool verbose = false;

ObjRef nwos_public_class_definition_class_ref;
ObjRef nwos_private_class_definition_class_ref;
ObjRef nwos_reference_list_class_ref;
ObjRef nwos_next_public_ref;

static Encryption_Level encryption_level = Encryption_High;

#ifndef PUBLIC_MODE
static uint32 linear_after_first;
static uint32 serial_after_first;
#endif

uint8  nwos_random_sequence[NUM_STORED_SEQ][FILE_BLOCK_SIZE];     /* this will only work for blocks <= 256 bytes */

static BF_KEY blowfish_key;


static Disk_Header disk_header;

static int    public_file_desc;
static int    obj_file_desc;
uint32 nwos_block_offset_to_chunks;
uint32 nwos_total_private_blocks;
uint32 nwos_used_public_blocks;
uint32 nwos_used_private_blocks;

static bool    read_only;
static bool    modified;
static char*   public_path = DEFAULT_PUBLIC_PATH;
static char*   storage_path;
static ObjRef* storage_index;
static int     storage_count;


static struct timeval end_of_time;   /* 9/9/99 9:09:09.999999 */


static bool read_block(ObjRef* ref, uint8 block[FILE_BLOCK_SIZE]);

#if 0
static uint32 block_spacing;     /* maximum spacing between positions on the disk, to fit the file 0 = disabled */
static int block_direction;
#endif

/* static uint8* cache[MAX_BLOCKS_CACHED]; until we implement this don't use space */


#define BIT_MAP_CACHE_SIZE 256

typedef struct {
  uint32 chunk;
  int age;
  bool dirty;
  uint8* map;
} Bit_Map_Cache_Entry;

static Bit_Map_Cache_Entry bit_map_cache[BIT_MAP_CACHE_SIZE];

static int bit_map_tick = 0;

static void write_bit_map(Bit_Map_Cache_Entry* entry);


/**************************************************************************/
/* Special malloc and free routines that can help check for memory leaks. */
/**************************************************************************/

#ifdef CHECK_MEMORY_LEAKS
#define MAX_MEM_ADDR_SAVED 16384
void* save_malloc_addr[MAX_MEM_ADDR_SAVED];
int cur_malloc_index;
#endif

void* nwos_malloc(size_t size)
{
#ifdef CHECK_MEMORY_LEAKS
    int i;
#endif

    void* result = malloc(size);

    if (result == NULL)
    {
	char msg[64];
	snprintf(msg, sizeof(msg), "%zd bytes", size);
	perror(msg);
	assert(result);
	exit(1);
    }

#ifdef CHECK_MEMORY_LEAKS
    for (i = 0; i < MAX_MEM_ADDR_SAVED; i++)
    {
	if (save_malloc_addr[cur_malloc_index] == NULL) break;
	cur_malloc_index++;
	if (cur_malloc_index >= MAX_MEM_ADDR_SAVED) cur_malloc_index = 0;  /* wrap if necessary */
    }

    assert(i < MAX_MEM_ADDR_SAVED);

    save_malloc_addr[cur_malloc_index] = result;
    cur_malloc_index++;
    if (cur_malloc_index >= MAX_MEM_ADDR_SAVED) cur_malloc_index = 0;  /* wrap if necessary */
#endif

    return result;
}

void nwos_free(void* ptr)
{
#ifdef CHECK_MEMORY_LEAKS
    int i;
#endif

    free(ptr);

#ifdef CHECK_MEMORY_LEAKS
    for (i = 0; i < MAX_MEM_ADDR_SAVED; i++)
    {
	cur_malloc_index--;
	if (cur_malloc_index < 0) cur_malloc_index = MAX_MEM_ADDR_SAVED - 1;  /* wrap if necessary */
	if (save_malloc_addr[cur_malloc_index] == ptr) break;
    }

    assert(i < MAX_MEM_ADDR_SAVED);

    save_malloc_addr[cur_malloc_index] = NULL;    /* erase it */
#endif
}


/***********************************/
/* Variable Length Count functions */
/***********************************/

uint32 nwos_decode_variable_sized_count(uint8 count[4])
{
    uint32 result;

    result = count[0];   /* get the first byte */

    if (result > 127)   /* two bytes */
    {
	assert((result & 0xc0) == 0x80);  /* for now upper two bits must be 10 */

	result = (((result & 0x3f) << 8) | count[1]) + 128;
    }
    else
    {
	/* this byte should always be zero if first bytes is < 128.  the original */
	/* plan was to not store the extra bytes, but the compiler did it anyway */
	assert(count[1] == 0);
    }

    return result;
}

void nwos_encode_variable_sized_count(uint32 count, uint8 result[4])
{
    int adjusted;

    assert(count < (16384 + 128));   /* for now we can only handle two bytes */

    if (count > 127)   /* two bytes */
    {
	adjusted  = count - 128;
	result[0] = (0x80 | (adjusted >> 8));
	result[1] = adjusted & 0xff;
    }
    else
    {
	result[0] = count;
	result[1] = 0;
    }

    result[2] = 0;
    result[3] = 0;

    assert(nwos_decode_variable_sized_count(result) == count);
}



#ifdef DISABLE_SECURITY_FEATURES
uint32 randomxxx;
#endif


/**********************/
/* Sequence generator */
/**********************/

#ifndef PUBLIC_MODE
static void generate_sequence_table(uint32* linear, uint32* serial, uint8 table[FILE_BLOCK_SIZE])
{
    int i;
    uint8 j;
    uint8 filled[FILE_BLOCK_SIZE];    /* remember which slots are already filled */

    /* mark all slots as empty */
    for (i = 0; i < FILE_BLOCK_SIZE; i++) filled[i] = 0;

    /* then mark the first 8 as used */
    for (i = 0; i < 8; i++) filled[i] = 1;

    /* start at 0 because it is used already, so while loop will fail the first time. */
    j = 0;
    assert(filled[j] != 0);

    for (i = 8; i < FILE_BLOCK_SIZE; i++)
    {
	/* find where the next byte is stored */
	while(filled[j] != 0) j = nwos_next_sequence(linear, serial) % FILE_BLOCK_SIZE;
	table[i] = j;   /* store it in the table */
	filled[j] = 1;            /* mark it as used */
    }
}
#endif


/**********************************/
/* Initialization and Termination */
/**********************************/

#define INDEX_ALLOC_SIZE 262144 

void index_compressed_file()
{
    int i;
    int size_of_index = INDEX_ALLOC_SIZE;
    size_t bytes_read;
    uint8 buffer[FILE_BLOCK_SIZE];
    uint8 header[FILE_BLOCK_SIZE];
    char* index_path = NULL;
    FILE* index_fp;

#ifdef NO_INDEX_FILE
    index_fp = NULL;
#else
    assert(obj_file_desc > 0);

    /* first see if there is an existing index file */

    i = strlen(storage_path);
    index_path = nwos_malloc(i+5);  /* 4 for .ndx and 1 for null */
    strcpy(index_path, storage_path);

    if (index_path[i-4] == '.' && index_path[i-3] == 'o' && index_path[i-2] == 'b' && index_path[i-1] == 'j')
    {
	i = i - 4;   /* write over the old .obj */
    }

    index_path[i++] = '.';
    index_path[i++] = 'n';
    index_path[i++] = 'd';
    index_path[i++] = 'x';
    index_path[i++] = '\0';

    fprintf(stderr, "index_path: %s\n", index_path);

    index_fp = fopen(index_path, "r");
#endif

    storage_index = malloc(size_of_index * sizeof(ObjRef));   /* allocate space for the first batch */

    assert(storage_index != NULL);

    /* read the first block, it contains the header */
    bytes_read = read(obj_file_desc, header, sizeof(header));

    assert(bytes_read == sizeof(header));

    /* if there is an index file see if the header matches */
    if (index_fp != NULL)
    {
	bytes_read = fread(buffer, 1, FILE_BLOCK_SIZE, index_fp);
	assert(bytes_read == sizeof(buffer));

	fprintf(stderr, "Index file found: ");
	fflush(stderr);

	if (memcmp(header, buffer, FILE_BLOCK_SIZE) == 0)   /* we have a winner */
	{
	    fprintf(stderr, "match\n");
	}
	else
	{
	    fprintf(stderr, "no match\n");
	    fclose(index_fp);
	    index_fp = NULL;
	}
    }
    else /* no index file */
    {
	errno = 0;   /* clear errno for later */
    }

    fprintf(stderr, "Indexing");
    fflush(stderr);

    i = 1;
    while (1)
    {
	if (i % 262144 == 1)
	{
	    fprintf(stderr, ".");
	    fflush(stderr);
	}

	if (i == size_of_index)   /* need to allocate more space */
	{
	    size_of_index += INDEX_ALLOC_SIZE;
	    storage_index = realloc(storage_index, size_of_index * sizeof(ObjRef));
	    assert(storage_index != NULL);
	}

	assert(i < size_of_index);

	if (index_fp != NULL)   /* read index from file */
	{
	    bytes_read = fread(&storage_index[i], sizeof(ObjRef), 1, index_fp);

	    if (bytes_read != 1) break;
	}
	else
	{
	    bytes_read = read(obj_file_desc, buffer, sizeof(buffer));

	    if (bytes_read != sizeof(buffer)) break;

	    memcpy(&storage_index[i], &buffer[4], sizeof(ObjRef));
	}

	i++;
    }

    storage_count = i;

    fprintf(stderr, "\nblocks: %d\n", i);

    if (index_fp != NULL)
    {
	if (ferror(index_fp))
	{
	    perror(index_path);
	    exit(1);
	}
	fclose(index_fp);
    }
    else
    {
	if (errno != 0)
	{
	    perror(storage_path);
	    exit(1);
	}
#ifndef NO_INDEX_FILE
	fprintf(stderr, "Writing index file: %s\n", index_path);

	index_fp = fopen(index_path, "w");

	if (fwrite(header, 1, sizeof(header), index_fp) != sizeof(header))
	{
	    perror(index_path);
	    exit(1);
	}

	if (fwrite(&storage_index[1], sizeof(ObjRef), i, index_fp) != i)
	{
	    perror(index_path);
	    exit(1);
	}

	if (fclose(index_fp))
	{
	    perror(index_path);
	    exit(1);
	}
#endif
    }

    assert(bytes_read == 0);

    if (index_path != NULL)
    {
	nwos_free(index_path);
    }
}


void initialize_random_number_generator()
{
    struct timespec ts;

    clock_gettime(CLOCK_REALTIME, &ts);
    srandom((uint32)ts.tv_sec ^ (uint32)ts.tv_nsec);

#ifdef DISABLE_SECURITY_FEATURES
    randomxxx = 0x00000100;     /* start objects here */
#endif
}


void nwos_initialize_objectify(uint8 bf_key[16], uint32 linear_seed, uint32 serial_seed, StorageType type, char* path)
{
    struct tm bdt;
#ifndef PUBLIC_MODE
    int i;
    int j;
    int k;
    int tries;
#endif
    char log_msg[128];

    /* make sure the storage is something we can deal with */
    assert(type == Drive_Or_Partition_RO || type == Drive_Or_Partition_RW || 
	   type == Sparse_File_RO || type == Sparse_File_RW || type == Compressed_File_RO);

    assert(storage_path == NULL);   /* make sure this is only called once */
    assert(obj_file_desc == 0);
    assert(public_file_desc == 0);

    nwos_log("nwos_initialize_objectify():");

    storage_path = path;

    initialize_random_number_generator();
    nwos_crc32_initialize();

    /* are we starting up with security or not? */

    if (bf_key == NULL)    /* this is kind of cheesy since bf_key is declared as an array */
    {
	assert(linear_seed == 0 && serial_seed == 0);
    }
    else
    {
#if 0
	int i;
	printf("blowfish key: ");
	for (i = 0; i < 16; i++) printf("%02x", bf_key[i]);
	printf("\n");
#endif

#ifdef PUBLIC_MODE
	bool Initialize_Called_With_Non_Null_Key = false;
	assert(Initialize_Called_With_Non_Null_Key);
#else
	BF_set_key(&blowfish_key, 16, bf_key);

	linear_after_first = linear_seed;
	serial_after_first = serial_seed;

	/* Create all of the sequence tables from the linear and serial parameters */
	for (i = 0; i < NUM_STORED_SEQ; i++)
	{
	    tries = 0;

	    do
	    {
		tries++;

		assert(tries < 4);

		generate_sequence_table(&linear_after_first, &serial_after_first, nwos_random_sequence[i]);

		for (j = 0; j < i; j++)
		{
		    for (k = 8; k < FILE_BLOCK_SIZE; k++) if (nwos_random_sequence[i][k] != nwos_random_sequence[j][k]) break;

		    if (k == FILE_BLOCK_SIZE) break;
		}
	    }
	    while (j < i);    /* if j is less than i it means one of the tables matched and needs to be regenerated */
	}
#endif
    }

    bdt.tm_sec = 9;
    bdt.tm_min = 9;
    bdt.tm_hour = 9;
    bdt.tm_mday = 9;
    bdt.tm_mon = 9 - 1;
    bdt.tm_year = 1999 - 1900;
    bdt.tm_isdst = -1;            /* hopefully mktime can figure this out! */

    end_of_time.tv_sec = mktime(&bdt);
    end_of_time.tv_usec = 999999;

    /********************************/
    /* open the public objects file */
    /********************************/

    public_file_desc = open(public_path, O_RDONLY | O_LARGEFILE);

    if (public_file_desc < 0)
    {
	perror(public_path);
	exit(1);
    }

    if (read(public_file_desc, &disk_header, sizeof(disk_header)) != sizeof(disk_header))
    {
	snprintf(log_msg, sizeof(log_msg), "reading disk header from: %s", public_path);
	perror(log_msg);
	exit(1);
    }

    if (memcmp(disk_header.magic_number, MAGIC_NUMBER, 4) != 0)
    {
	fprintf(stderr, "Missing magic number in disk header\n");
	exit(1);
    }

    if (memcmp(disk_header.version_string, VERSION_STRING, 4) != 0)
    {
	fprintf(stderr, "Incorrect version string in disk header\n");
	exit(1);
    }

    nwos_4_uint8_to_uint32(disk_header.used_blocks, &nwos_used_private_blocks);

    snprintf(log_msg, sizeof(log_msg), " used public blocks: %9u", nwos_used_public_blocks);
    nwos_log(log_msg);


    /***************************************************/
    /* if a private objects file was specified open it */
    /***************************************************/

    if (storage_path != NULL)
    {
	read_only = ((type & 1) == 0);

	if (read_only)
	{
	    obj_file_desc = open(storage_path, O_RDONLY | O_LARGEFILE);
	}
	else
	{
	  obj_file_desc = open(storage_path, O_RDWR | O_LARGEFILE);
	}

	if (obj_file_desc < 0)
	{
	    perror(storage_path);
	    exit(1);
	}

	if (type == Compressed_File_RO)
	{
	    index_compressed_file();   /* build an index into the compressed file */

	    if (lseek(obj_file_desc, 0LL, SEEK_SET) < 0)
	    {
		perror("rewinding after indexing");
		exit(1);
	    }
	}

	if (read(obj_file_desc, &disk_header, sizeof(disk_header)) != sizeof(disk_header))
	{
	    snprintf(log_msg, sizeof(log_msg), "reading disk header from: %s", storage_path);
	    perror(log_msg);
	    exit(1);
	}

	if (memcmp(disk_header.magic_number, MAGIC_NUMBER, 4) != 0)
	{
	    fprintf(stderr, "Missing magic number in disk header\n");
	    exit(1);
	}

	if (memcmp(disk_header.version_string, VERSION_STRING, 4) != 0)
	{
	    fprintf(stderr, "Incorrect version string in disk header\n");
	    exit(1);
	}

	nwos_4_uint8_to_uint32(disk_header.block_offset_to_chunks, &nwos_block_offset_to_chunks);
	nwos_4_uint8_to_uint32(disk_header.total_blocks, &nwos_total_private_blocks);
	nwos_4_uint8_to_uint32(disk_header.used_blocks, &nwos_used_private_blocks);

	if (type == Drive_Or_Partition_RO || type == Drive_Or_Partition_RW)
	{
	    assert(nwos_total_private_blocks > 0);
	}

	snprintf(log_msg, sizeof(log_msg), "   private blocks: %9u  used: %9u", nwos_total_private_blocks, nwos_used_private_blocks);
	nwos_log(log_msg);
    }
}


/* returns true if the object was created, false if it already existed */

bool nwos_create_root(ObjRef* ref)
{
    C_struct_Root root_obj;
    ObjRef root_class_ref;
    bool result = false;

#ifndef PUBLIC_MODE
    nwos_set_bit_in_map(nwos_hash_ref(ref));
#endif

    /* first find out if we already have this root */


    if (!nwos_object_exists(ref))   /* it already exists */
    {
	/* currently I can't think of any reason to have a private root class, since there is no list of root objects */
	assert(nwos_find_public_class_definition("ROOT", &root_class_ref));

	memset(&root_obj, 0, sizeof(root_obj));  /* zero it out */

	nwos_fill_in_common_header(&root_obj.header.common, ref, &root_class_ref);
#ifdef PUBLIC_MODE
	memcpy(&root_obj.class_definition_class, &nwos_public_class_definition_class_ref, sizeof(root_obj.class_definition_class));
#else
	memcpy(&root_obj.class_definition_class, &nwos_private_class_definition_class_ref, sizeof(root_obj.class_definition_class));
#endif
	memcpy(&root_obj.reference_list_class, &nwos_reference_list_class_ref, sizeof(root_obj.reference_list_class));

	nwos_crc32_calculate((uint8*) &root_obj.header.object, sizeof(ObjectHeader), root_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &root_obj.class_definition_class, sizeof(root_obj) - sizeof(EveryObject), root_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &root_obj, sizeof(root_obj));

#ifdef PUBLIC_MODE
	nwos_add_to_references(ref, &root_class_ref);
#endif

	result = true;
    }

    return result;
}


void nwos_set_root_object(ObjRef* ref)
{
    C_struct_Root root_obj;
    ObjRef verify_ref;
    C_struct_Class_Definition class_def;

    nwos_read_object_from_disk(ref, &root_obj, sizeof(root_obj));

    if (nwos_reference_type(ref) == Public_Reference)
    {
	assert(storage_path == NULL);   /* only allow if no private objects are in play */

	copy_reference(&nwos_public_class_definition_class_ref, &root_obj.class_definition_class);
    }
    else
    {
	assert(storage_path != NULL);   /* only allow if private objects are in play */

	copy_reference(&nwos_private_class_definition_class_ref, &root_obj.class_definition_class);

	nwos_read_class_definition(&nwos_private_class_definition_class_ref, &class_def);

	copy_reference(&nwos_public_class_definition_class_ref, &class_def.header.object.clone_of);
    }

    copy_reference(&nwos_reference_list_class_ref, &root_obj.reference_list_class);

    assert(nwos_find_public_class_definition("REFERENCE LIST", &verify_ref));
    assert(is_same_object(&verify_ref, &nwos_reference_list_class_ref));
}

void special_update_total_blocks_for_expand_0021(uint32 new_reserved_blocks)
{
    nwos_total_private_blocks = new_reserved_blocks;
    nwos_uint32_to_4_uint8(&nwos_total_private_blocks, disk_header.total_blocks);
}


void nwos_terminate_objectify()
{
    int i;
    char log_msg[128];

    nwos_log("nwos_terminate_objectify():");

    nwos_flush_dirty_ref_lists();  /* flush any reference lists that are dirty out to disk */

    for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
    {
	if (bit_map_cache[i].chunk != 0 && bit_map_cache[i].dirty)
	{
	    write_bit_map(&bit_map_cache[i]);

	    bit_map_cache[i].chunk = 0;

	    free(bit_map_cache[i].map);
	    bit_map_cache[i].map = NULL;
	}
    }

    if (modified)
    {
#ifdef PUBLIC_MODE
	snprintf(log_msg, sizeof(log_msg), "  used public blocks: %9u",
		 nwos_ref_to_word(&nwos_next_public_ref));
	nwos_log(log_msg);

	copy_reference((ObjRef*)&disk_header.used_blocks, &nwos_next_public_ref);

	nwos_get_time_stamp(disk_header.last_change);
#else
	nwos_uint32_to_4_uint8(&nwos_used_private_blocks, disk_header.used_blocks);

	snprintf(log_msg, sizeof(log_msg), "  private blocks: %9u  used: %9u", nwos_total_private_blocks, nwos_used_private_blocks);
	nwos_log(log_msg);

	nwos_get_time_stamp(disk_header.last_change);
#endif

	if (lseek(obj_file_desc, (off_t)0, SEEK_SET) < 0)
	{
	    perror(storage_path);
	    exit(1);
	}

	if (write(obj_file_desc, &disk_header, sizeof(disk_header)) != sizeof(disk_header))
	{
	    perror(storage_path);
	    exit(1);
	}
    }
    else
    {
	nwos_log("  No changes made");
    }

    if (close(obj_file_desc) < 0)
    {
	perror(storage_path);
	exit(1);
    }

    obj_file_desc = 0;
    storage_path = NULL;

    if (close(public_file_desc) < 0)
    {
	perror(public_path);
	exit(1);
    }

    public_file_desc = 0;
    public_path = NULL;

#ifdef CHECK_MEMORY_LEAKS
    for (i = 0; i < MAX_MEM_ADDR_SAVED; i++)
    {
	if (save_malloc_addr[i] != NULL) printf("WARNING - memory block not freed: %p\n", save_malloc_addr[i]);
    }
#endif
}

void nwos_flush_bit_maps()
{
    int i;

    for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
    {
	if (bit_map_cache[i].chunk != 0 && bit_map_cache[i].dirty)
	{
	    write_bit_map(&bit_map_cache[i]);
	}
    }
}


/*******************************************************************/
/* Reference utility functions: conversion, testing and generation */
/*******************************************************************/

#if 0

/* these were taken from objectify_private.h */
#define NUM_PUBLIC_IDS 268435456

#define NUM_PRIVATE_IDS ((uint32) (4294967296LL - NUM_PUBLIC_IDS))

static inline uint32 hash_ref_0014(ObjRef* ref)
{
    uint32 result;

    result = ((uint32)(ref->id[0]) << 24) | ((uint32)(ref->id[1]) << 16) | 
             ((uint32)(ref->id[2]) << 8) | ((uint32)(ref->id[3]));

    if ((ref->id[0] & 0xF0) != 0)  /* it's a private reference */
    {
	result = (result - NUM_PUBLIC_IDS) % private_blocks_on_disk + reserved_public_blocks;
    }

    return result;
}
#endif



/* returns amount of private space used on disk 0.1 = 10%, 0.5 = 50%, 0.9 = 90% */

float nwos_private_space_used()
{
    float result;

    result = (float)nwos_used_private_blocks / (float)nwos_total_private_blocks;

    assert(0.0f <= result && result <= 1.0f);

    return result;
}

/*********************************************************************************************/
/* this function maps a uint32 reference to a block number                                   */
/*                                                                                           */
/* The new formula is to skip over the bit maps instead of leaving holes for them, so every  */
/* 32 bit number is available for use.                                                       */
/*********************************************************************************************/

#define USABLE_BLOCKS_PER_CHUNK (BLOCKS_IN_CHUNK - BIT_MAP_BLOCKS)

uint32 nwos_hash_uint32_ref(uint32 ref)
{
#ifdef PUBLIC_MODE
    assert(ref < nwos_used_public_blocks);
    return ref;
#else
    uint32 result;
    uint32 chunk;
    uint32 offset;
    static uint32 total_private_chunks = 0;

    if (ref < RESERVED_PUBLIC_BLOCKS)
    {
	result = ref;
    }
    else
    {
	if (total_private_chunks == 0)
	{
	    total_private_chunks = nwos_total_private_blocks / BLOCKS_IN_CHUNK;
	}

	chunk = ((ref - RESERVED_PUBLIC_BLOCKS) / USABLE_BLOCKS_PER_CHUNK) % total_private_chunks;
	offset = (ref - RESERVED_PUBLIC_BLOCKS) % USABLE_BLOCKS_PER_CHUNK;

	result = nwos_block_offset_to_chunks + (chunk * BLOCKS_IN_CHUNK) + BIT_MAP_BLOCKS + offset;

//	printf("total blocks: %u chunks: %u  chunk: %u  offset: %u  result: %u (%08x)\n",
//	       nwos_total_private_blocks, total_private_chunks, chunk, offset, result, result);
    }

    return result;
#endif
}


uint32 nwos_hash_ref(ObjRef* ref)
{
    uint32 uint32_ref;

    uint32_ref = ((uint32)(ref->id[0]) << 24) | ((uint32)(ref->id[1]) << 16) | 
                                  ((uint32)(ref->id[2]) << 8) | ((uint32)(ref->id[3]));

    return nwos_hash_uint32_ref(uint32_ref);
}


static inline off_t ref_to_offset(ObjRef* ref) 
{
    uint32 hash;

    assert(FILE_BLOCK_SIZE == 256);

    hash = nwos_hash_ref(ref);

    return (off_t) hash << 8;
}


/* Doesn't return values where the uppermost nibble is zero */

static void generate_random_id(ObjRef* ref)
{
    uint32 word = 0;

    while (word < RESERVED_PUBLIC_BLOCKS)
    {
	word = random() ^ (random() << 1);
    }

    nwos_word_to_ref(word, ref);

//    printf("generate_random_id: %02x%02x%02x%02x\n", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
#ifdef DISABLE_SECURITY_FEATURES  /* use an incremental value instead of random */
    randomxxx++;
    ref->id[0] = randomxxx >> 24;
    ref->id[1] = randomxxx >> 16;
    ref->id[2] = randomxxx >> 8;
    ref->id[3] = randomxxx;
#endif
}


bool block_contains_object(uint8 block[FILE_BLOCK_SIZE], ObjRef* ref)
{
    uint8 marker[8];

    marker[0] = 0;
    marker[1] = 0;
    marker[2] = 0;
    marker[3] = 0;
    marker[4] = ref->id[0];
    marker[5] = ref->id[1];
    marker[6] = ref->id[2];
    marker[7] = ref->id[3];

    return (memcmp(block, marker, sizeof(marker)) == 0);   /* return true if the object marker is there */
}


bool nwos_object_exists(ObjRef* ref)
{
    uint8 block[FILE_BLOCK_SIZE];

    return read_block(ref, block) && block_contains_object(block, ref);
}


void write_bit_map(Bit_Map_Cache_Entry* entry)
{
    char log_msg[80];

    snprintf(log_msg, sizeof(log_msg), "write bit_map: %08x", entry->chunk);

    nwos_log(log_msg);

    assert(nwos_block_offset_to_chunks <= entry->chunk && entry->chunk < nwos_block_offset_to_chunks + nwos_total_private_blocks);

    if (lseek(obj_file_desc, (off_t)entry->chunk << 8, SEEK_SET) < 0)
    {
	snprintf(log_msg, sizeof(log_msg), "write_bit_map lseek chunk:%08x", entry->chunk);
	perror(log_msg);
	exit(1);
    }

    if (write(obj_file_desc, entry->map, BIT_MAP_BYTES) != BIT_MAP_BYTES)
    {
	snprintf(log_msg, sizeof(log_msg), "write_bit_map write chunk:%08x", entry->chunk);
	perror(log_msg);
	exit(1);
    }

    entry->dirty = false;

    modified = true;
}


void read_bit_map_into_cache(Bit_Map_Cache_Entry* entry)
{
    char log_msg[80];
#if 0
    snprintf(log_msg, sizeof(log_msg), "write bit_map: %08x\n", entry->chunk);

    nwos_log(log_msg);
#endif
    assert(nwos_block_offset_to_chunks <= entry->chunk && entry->chunk < nwos_block_offset_to_chunks + nwos_total_private_blocks);

    if (lseek(obj_file_desc, (off_t)entry->chunk << 8, SEEK_SET) < 0)
    {
	snprintf(log_msg, sizeof(log_msg), "read_bit_map lseek chunk:%08x", entry->chunk);
	perror(log_msg);
	exit(1);
    }

    if (entry->map == NULL)
    {
	entry->map = malloc(BIT_MAP_BYTES);
	assert(entry->map != NULL);
    }

    if (read(obj_file_desc, entry->map, BIT_MAP_BYTES) != BIT_MAP_BYTES)
    {
	snprintf(log_msg, sizeof(log_msg), "read_bit_map write chunk:%08x", entry->chunk);
	perror(log_msg);
	exit(1);
    }

    entry->dirty = false;
}


static Bit_Map_Cache_Entry* find_bit_map_in_cache(uint32 block)
{
    int i;
    Bit_Map_Cache_Entry* result = NULL;

    assert(nwos_block_offset_to_chunks <= block && block < nwos_block_offset_to_chunks + nwos_total_private_blocks);

    /* assert((block % CHUNK_SIZE) != 0);  what was this for? */

    /* see if it's already in the cache */
    for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
    {
	if (bit_map_cache[i].chunk == (block & BIT_MAP_BLOCK_MASK)) break;
    }

    if (i < BIT_MAP_CACHE_SIZE)   /* found it */
    {
	result = &bit_map_cache[i];
    }
    else                           /* didn't find it */
    {
	/* find an empty one or find the oldest */

	result = bit_map_cache;   /* so we have an age to compare to */

	for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
	{
	    if (bit_map_cache[i].chunk == 0)
	    {
		result = &bit_map_cache[i];
		break;
	    }

	    if (bit_map_cache[i].age < result->age)
	    {
		result = &bit_map_cache[i];
	    }
	}

	if (i == BIT_MAP_CACHE_SIZE && result->dirty)    /* didn't find an empty one, write the oldest one out */
	{
	    write_bit_map(result);
	}

	result->chunk = block & BIT_MAP_BLOCK_MASK;

	read_bit_map_into_cache(result);
    }

    bit_map_tick++;
    result->age = bit_map_tick;

    assert(result != NULL);

    return result;
}


void nwos_set_bit_in_map(uint32 block)
{
    int byte_num;
    int bit_num;
    Bit_Map_Cache_Entry* entry;

    entry = find_bit_map_in_cache(block);

    byte_num = (block % BLOCKS_IN_CHUNK) / 8;
    assert(byte_num < BIT_MAP_BYTES);
    bit_num = block % 8;

    if ((entry->map[byte_num] & (0x80 >> bit_num)) == 0)    /* don't count a block that was already used */
    {
	entry->map[byte_num] |= (0x80 >> bit_num);

/* printf("set_bit_in_map block: %08x  chunk: %08x  byte: %d  bit: %d\n", block, entry->chunk, byte_num, bit_num); */

	entry->dirty = true;

	nwos_used_private_blocks++;
    }
}


void nwos_clear_bit_in_map(uint32 block)
{
    int byte_num;
    int bit_num;
    Bit_Map_Cache_Entry* entry;

    entry = find_bit_map_in_cache(block);

    byte_num = (block % BLOCKS_IN_CHUNK) / 8;
    assert(byte_num < BIT_MAP_BYTES);
    bit_num = block % 8;

    if ((entry->map[byte_num] & (0x80 >> bit_num)) != 0)    /* don't count a block that was already clear */
    {
	entry->map[byte_num] &= ~(0x80 >> bit_num);

/* printf("clear_bit_in_map block: %08x  chunk: %08x  byte: %d  bit: %d\n", block, entry->chunk, byte_num, bit_num); */

	entry->dirty = true;

	nwos_used_private_blocks--;
    }
}


bool nwos_test_bit_in_map(uint32 block)
{
    int byte_num;
    int bit_num;
    Bit_Map_Cache_Entry* entry;

    entry = find_bit_map_in_cache(block);

    byte_num = (block % BLOCKS_IN_CHUNK) / 8;
    assert(byte_num < BIT_MAP_BYTES);
    bit_num = block % 8;

    return (entry->map[byte_num] & (0x80 >> bit_num)) != 0;
}


#ifndef PUBLIC_MODE

static int8 bits_in_byte[256] = 
{
/*        0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f */
/* 0 */   0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4,
/* 1 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 2 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 3 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 4 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 5 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 6 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 7 */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* 8 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 9 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* a */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* b */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* c */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* d */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* e */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* f */   4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
};

static float chunk_usage(Bit_Map_Cache_Entry* entry)
{
    float result;
    int i;
    int blocks_used = 0;

    for (i = 0; i < BIT_MAP_BYTES; i++)
    {
	blocks_used += bits_in_byte[entry->map[i]];
    }

    result = (float)blocks_used / (float)BLOCKS_IN_CHUNK;

    assert(0.0f <= result && result <= 1.0f);

    return result;
}
#endif


Reference_Type nwos_reference_type(ObjRef* ref)
{
    if (nwos_ref_to_word(ref) < RESERVED_PUBLIC_BLOCKS)
    {
	return Public_Reference;
    }

    return Private_Reference;
}


bool nwos_block_used(ObjRef* ref)
{
#ifdef PUBLIC_MODE
    uint8 block[FILE_BLOCK_SIZE];

    return  read_block(ref, block) && !is_void_reference((ObjRef*)&block[4]);
#else
    uint32 block;
    int byte_num;
    int bit_num;
    Bit_Map_Cache_Entry* entry;

    block = nwos_hash_ref(ref);

    entry = find_bit_map_in_cache(block);

    byte_num = (block % BLOCKS_IN_CHUNK) / 8;
    assert(byte_num < BIT_MAP_BYTES);
    bit_num = block % 8;

    return ((entry->map[byte_num] & (0x80 >> bit_num)) != 0);
#endif
}


static Encryption_Level previous_encryption_level;

void nwos_set_encryption_level(Encryption_Level new_level)
{
    assert(Encryption_None <= new_level && new_level <= Encryption_Extreme);

    encryption_level = new_level;

    previous_encryption_level = Encryption_None;
}


void nwos_push_encryption_level(Encryption_Level new_level)
{
    assert(Encryption_None <= new_level && new_level <= Encryption_Extreme);
    assert(previous_encryption_level == Encryption_None);

    previous_encryption_level = encryption_level;

    encryption_level = new_level;
}


void nwos_pop_encryption_level()
{
    assert(Encryption_None < previous_encryption_level && previous_encryption_level <= Encryption_Extreme);

    encryption_level = previous_encryption_level;

    previous_encryption_level = Encryption_None;
}


#if 0
void nwos_set_sequential_blocks(uint32 approx_blocks)
{
  assert(false);   /* FIX THIS FUNCTION! */
    assert(approx_blocks == 0 || approx_blocks > 3);

    if (approx_blocks == 0)
    {
	block_spacing = 0;
    }
    else
    {
	block_spacing = blocks_on_disk / approx_blocks;
    }

    block_direction = 0;

    assert(block_spacing == 0 || block_spacing > 5);   /* let's say file shouldn't use 20% of disk */
}
#endif


void generate_new_completely_random_id(ObjRef* ref)
{
    unsigned long dummy;
    int state = 1;
    int n;
    bool unable_to_find_unique_id = false;
    static int high_water_mark = 16;

    while (state > 0)
    {
	generate_random_id(ref);

	if (!nwos_block_used(ref)) 
	{
	    if (state > high_water_mark)
	    {
		fprintf(stderr, "Warning: search %d for random id\n", state);
		high_water_mark = state;
	    }
	    state = 0;   /* exit loop */
	}
	else      /* found an unused id */
	{
#ifdef DISABLE_SECURITY_FEATURES
	  /* ignore the duplicate and just keep counting up until we find one */
#else
	    switch(state)
	    {
	      default: /* the default is to just keep trying */
		state++;
		break;

	      case 17:    /* spin the dammed thing a few times */
		n = random() % 13;
		while ((random() % 17) != n) { }
		state++;
		break;

	      case 101:    /* reinitialize in case we've hit upon a sequence we've done before */	
		initialize_random_number_generator();
		state++;
		break;

	      case 313:    /* this is bad - wait a random amount of time and then try reinitializing again */
		dummy = random();
		usleep(dummy % 7919 + 2082);
		initialize_random_number_generator();
		state++;
		break;

	      case 991:    /* stop and figure out what is wrong */
		assert(unable_to_find_unique_id);
		break;
	    }
#endif
	}
    }
}


void nwos_generate_new_completely_random_id(ObjRef* ref)
{
    generate_new_completely_random_id(ref);
    nwos_set_bit_in_map(nwos_hash_ref(ref));
}


#ifndef PUBLIC_MODE
/* density is how many blocks to use 1-7 out of 8 blocks. */

static void nwos_generate_new_closely_spaced_id(ObjRef* last_ref, int density, ObjRef* new_ref)
{
    Bit_Map_Cache_Entry* entry = NULL;
    uint32 block;
    uint32 hash;
    float  private_space_used;
    float  chunk_used;
    int    count;
    int    byte_num;
    int    bit_num;
    uint8  byte;

    assert(1 <= density && density <= 7);

    private_space_used = nwos_private_space_used();

    /* for now don't do anything if disc is more than 90% full */
    assert(private_space_used < 0.9f);

    if (is_void_reference(last_ref))
    {
	for (count = 0; count < 10000; count++)
	{
	    generate_new_completely_random_id(new_ref);

	    block = nwos_ref_to_word(new_ref);

	    chunk_used = chunk_usage(find_bit_map_in_cache(block));

	    if (chunk_used < private_space_used + 0.01f)   /* if less than 1% over average */
	    {
		break;
	    }
	}

	assert(count < 10000);

	block--;    /* adjust for increment below */
    }
    else
    {
	block = nwos_ref_to_word(last_ref);
    }

    while (true)
    {
	block++;

	if ((random() % 8) < density)
	{
	    if (entry == NULL || entry->chunk != (block & BIT_MAP_BLOCK_MASK))
	    {
		entry = find_bit_map_in_cache(nwos_hash_uint32_ref(block));
	    }

	    hash = nwos_hash_uint32_ref(block);

	    assert(entry->chunk == (hash & BIT_MAP_BLOCK_MASK));

	    byte_num = (hash % BLOCKS_IN_CHUNK) / 8;
	    assert(byte_num < BIT_MAP_BYTES);
	    bit_num = hash % 8;

	    byte = entry->map[byte_num];

	    if (bits_in_byte[byte] < density && (byte & (0x80 >> bit_num)) == 0) break;
	}
    }

    nwos_set_bit_in_map(nwos_hash_uint32_ref(block));

    nwos_word_to_ref(block, new_ref);
}


/* generate approximately 1 block out of N */

static void nwos_generate_new_1_in_N_id(ObjRef* last_ref, uint32 n, ObjRef* new_ref)
{
    Bit_Map_Cache_Entry* entry = NULL;
    uint32 mask;
    uint32 block;
    uint32 hash;
    int    byte_num;
    int    bit_num;
    uint8  byte;
    int    i;


    /* make sure it's a binary multiple between 8 and 32768 */

    for (i = 3; i < 16; i++) if ((1 << i) == n) break;

    assert(i < 16);


    mask = n - 1;

    if (is_void_reference(last_ref))
    {
	generate_new_completely_random_id(new_ref);

	block = nwos_ref_to_word(new_ref);
    }
    else
    {
	/* pick random bit in next byte */
	block = ((nwos_ref_to_word(last_ref) & ~mask) | (random() & mask)) + n;
    }

    while (true)
    {
	if ((~block & ~mask) == 0)  /* continue this somewhere else */
	{
	    generate_new_completely_random_id(new_ref);

	    block = nwos_ref_to_word(new_ref);
	}	    

	if (entry == NULL || entry->chunk != (block & BIT_MAP_BLOCK_MASK))
	{
	    entry = find_bit_map_in_cache(nwos_hash_uint32_ref(block));
	}

	hash = nwos_hash_uint32_ref(block);

	assert(entry->chunk == (hash & BIT_MAP_BLOCK_MASK));

	byte_num = (hash % BLOCKS_IN_CHUNK) / 8;
	assert(byte_num < BIT_MAP_BYTES);
	bit_num = hash % 8;

	byte = entry->map[byte_num];
	
	if ((byte & (0x80 >> bit_num)) == 0) break;

	block++;
    }

    nwos_set_bit_in_map(nwos_hash_uint32_ref(block));

    nwos_word_to_ref(block, new_ref);
}
#endif


#if 0
#define MAX_POSSIBLE_REFS ((int) (NUM_PRIVATE_IDS / blocks_on_disk + 1))  /* could be 1 too many if even */

static int nwos_generate_new_id_in_range(ObjRef* last_ref, uint32 spacing, int direction, ObjRef* new_ref)
{
    int new_direction = direction;
  assert(false);  /* FIX THIS FUNCTION */
    uint32 old_block;
    uint32 new_block;
    int offset;
    int i;
    ObjRef possible_ref[MAX_POSSIBLE_REFS];

    assert((spacing & 0x8000000) == 0);     /* make sure it will convert to a positive integer */

    assert(spacing < (blocks_on_disk / 2));   /* and it isn't going to cause any strange behaviour by being too large */

    /* if this is the first time, choose a direction */

    old_block = nwos_hash_ref(last_ref);

    if (new_direction == 0)
      {
	if (old_block < reserved_public_blocks + spacing)
	{
	    new_direction = 1;
	}
	else if (old_block > blocks_on_disk - spacing)
	{
	    new_direction = -1;
	}
	else if ((random() & 1) == 0)
	{
	    new_direction = 1;
	}
	else
	{
	    new_direction = -1;
	}
    }


    /* Are we getting close to one end and need to reverse direction? */

    if (new_direction < 0)
    {
	if (old_block < (reserved_public_blocks + spacing)) new_direction = 1;    /* go forward now */
    }
    else 
    {
	if (old_block > (blocks_on_disk - spacing)) new_direction = -1;    /* go backwards now */
    }


    /* compute a new block number */

    do
    {
	offset = random() % (int)spacing;

	if (new_direction < 0)
	{
	    new_block = old_block - offset;    /* go backwards */
	}
	else 
	{
	    new_block = old_block + offset;    /* go forward */
	}

	assert(new_block > reserved_public_blocks);
	assert(new_block < blocks_on_disk);

	nwos_word_to_ref(new_block, &possible_ref[0]);
    }
    while (nwos_block_used(&possible_ref[0]));


    /* now convert the block into a random reference that will equate to that block */

    for (i = 1; i < MAX_POSSIBLE_REFS; i++)
    {
	if (((new_block - reserved_public_blocks) * i) > NUM_PRIVATE_IDS)
	{
	    assert(i + 1 == MAX_POSSIBLE_REFS);
	    break;
	}
	else
	{
	    nwos_word_to_ref((new_block - reserved_public_blocks) + private_blocks_on_disk * i + NUM_PUBLIC_IDS, &possible_ref[i]);
	}

	assert(nwos_hash_ref(&possible_ref[i]) == new_block);
    }

    i = random() % i;   /* pick on of the possible ids */

    assert(0 <= i && i < MAX_POSSIBLE_REFS);

    copy_reference(new_ref, &possible_ref[i]);
    return new_direction;
}
#endif


void nwos_generate_new_id(ObjRef* ref)
{
#ifdef PUBLIC_MODE
    nwos_generate_new_public_id(ref);

    assert(nwos_reference_type(ref) == Public_Reference);
#else
    bool Cant_Use_Encryption_None = false;
    bool Encryption_Not_Set = false;
    static ObjRef last_ref;

    switch (encryption_level)
    {
      case Encryption_None:
	assert(Cant_Use_Encryption_None);
	break;

      case Encryption_Minimal:
	nwos_generate_new_closely_spaced_id(&last_ref, 7, ref);
	break;

      case Encryption_Very_Low:
	nwos_generate_new_1_in_N_id(&last_ref, 8, ref);
	break;

      case Encryption_Low:
	nwos_generate_new_1_in_N_id(&last_ref, 64, ref);
	break;

      case Encryption_Medium:
	nwos_generate_new_1_in_N_id(&last_ref, 512, ref);
	break;

      default:
	assert(Encryption_Not_Set);
	break;

      case Encryption_Extreme:
	nwos_generate_new_completely_random_id(ref);
	break;
    }

    copy_reference(&last_ref, ref);

    assert(nwos_reference_type(ref) == Private_Reference);
#endif
}


#ifdef PUBLIC_MODE
void nwos_generate_new_public_id(ObjRef* ref)
{
    int i;
    int acc;
    uint8 diff[4];

    copy_reference(ref, &nwos_next_public_ref);

    for (i = sizeof(nwos_next_public_ref)-1; i > -1; i--)
    {
	nwos_next_public_ref.id[i]++;
	if (nwos_next_public_ref.id[i] != 0) break;
    }

    acc = (int)nwos_next_public_ref.id[3] - (int)ref->id[3];
    diff[3] = acc;
    acc = (int)nwos_next_public_ref.id[2] - (int)ref->id[2] + (acc >> 8);
    diff[2] = acc;
    acc = (int)nwos_next_public_ref.id[1] - (int)ref->id[1] + (acc >> 8);
    diff[1] = acc;
    acc = (int)nwos_next_public_ref.id[0] - (int)ref->id[0] + (acc >> 8);
    diff[0] = acc;

    assert(diff[0] == 0 && diff[1] == 0 && diff[2] == 0 && diff[3] == 1);

    nwos_used_public_blocks++;
}
#endif

#if 0
// Save this in case we change to allocate public ids this way
void nwos_generate_new_public_id(ObjRef* ref)
{
    static int next_free_index = 0;
    static int bit = 0;
    static uint32 orig_chunk;
    static Bit_Map_Cache_Entry* current_cache_entry;
    uint32 block;

    if (next_free_index == 0)
    {
	orig_chunk = next_private_chunk;

	current_cache_entry = find_bit_map_in_cache(orig_chunk);

	while (next_free_index < BIT_MAP_BYTES && current_cache_entry->map[next_free_index] == 0xff)
	{
	    next_free_index++;
	}

	while (bit < 8 && (current_cache_entry->map[next_free_index] & (0x80 >> bit)) != 0)
	{
	    bit++;
	}
    }

    assert(next_free_index < BIT_MAP_BYTES);

    assert(bit < 8);

    assert((current_cache_entry->map[next_free_index] & (0x80 >> bit)) == 0);

    block = next_private_chunk + (next_free_index * 8) + bit;

    /* don't mark as used here, let the write_block routine mark it when it's written */
    /* we just always move to next empty one so the ones behind might not be zero yet */
    /*    current_private_block_map[next_free_index] |= (0x80 >> bit); */   /* mark as used */

    do
    {
	bit++;

	if (bit > 7)
	{
	    while (next_free_index < BIT_MAP_BYTES && current_cache_entry->map[next_free_index] == 0xff)
	    {
		next_free_index++;

		if (next_free_index == BIT_MAP_BYTES)   /* this chunk is full goto next */
		{
		    next_private_chunk += BLOCKS_IN_CHUNK;

		    assert(next_private_chunk != orig_chunk);   /* we've wrapped clear around, disk full?  FIX THIS! */

		    if (next_private_chunk >= nwos_block_offset_to_chunks + nwos_total_private_blocks)
		    {
			next_private_chunk = nwos_block_offset_to_chunks;   /* go back to the beginning */
		    }

		    current_cache_entry = find_bit_map_in_cache(orig_chunk);

		    next_free_index = 0;
		}
	    }

	    bit = 0;
	}
    }
    while ((current_cache_entry->map[next_free_index] & (0x80 >> bit)) != 0);

    assert(next_free_index < BIT_MAP_BYTES);

    assert(bit < 8);

    assert((current_cache_entry->map[next_free_index] & (0x80 >> bit)) == 0);

    nwos_uint32_to_4_uint8(&block, ref->id);
}
#endif


/*************************/
/* Verification routines */
/*************************/

#if 0
static bool is_compatible_version(char* header_version)
{
    static char* compatible[] = COMPATIBLE_HEADER_VERSIONS;
    const int num_compatible = sizeof(compatible) / sizeof(char*);
    int i;

    for (i = 0; i < num_compatible; i++) 
    {
	if (memcmp(header_version, compatible[i], strlen(HEADER_VERSION)) == 0) break;
    }

    return (i < num_compatible);    /* return true if found a match */

    return true;
}
#endif


static void check_common_header(ObjRef* ref, CommonHeader* common)
{
#if 0
#if 0
    printf("common header pointer: %p\n", common);
#endif

    if (memcmp(common->magic_number, MAGIC_NUMBER, sizeof(common->magic_number)))
    {
	printf("object: %s - missing magic number\n", path);
	exit(1);
    }

    if (!is_compatible_version((char*)common->version))
    {
	printf("object: %s - incompatible version: ", path);
	if (isprint(common->version[0]) && isprint(common->version[1]) &&
	    isprint(common->version[2]) && isprint(common->version[3]))
	{
	    printf("%c%c%c%c\n", common->version[0], common->version[1], common->version[2], common->version[3]);
	}
	else
	{
	    printf("%02x %02x %02x %02x\n", common->version[0], common->version[1], common->version[2], common->version[3]);
	}
	exit(1);
    }
#endif
}


/* ref is only needed for error messages */

static void check_object_header(ObjRef* ref, ObjectHeader* obj_header, uint8 stored[4])
{
    uint8 computed[4];
    bool checksum_ok;

    nwos_crc32_calculate((uint8*)obj_header, sizeof(*obj_header), computed);

    checksum_ok = (stored[0] == computed[0] || stored[1] == computed[1] || stored[2] == computed[2] || stored[3] == computed[3]);

    if (!checksum_ok)
    {
	printf("object: %02x%02x%02x%02x - bad header checksum, ", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	printf("size %zd - ", sizeof(*obj_header));
	printf("computed %02x%02x%02x%02x - ", computed[0], computed[1], computed[2], computed[3]);
	printf("stored %02x%02x%02x%02x\n", stored[0], stored[1], stored[2], stored[3]);
	assert(checksum_ok);
    }
}


static void check_object_data(ObjRef* ref, uint8* object, size_t total_size, uint8 stored[4])
{
    uint8 computed[4];
    size_t data_size;

    data_size = total_size - sizeof(EveryObject);

#if 0
    printf("Total size: %d\n", total_size);
    printf("ObjectHeader: %d\n", sizeof(EveryObject));
    printf("object pointer: %p\n", object);
    {
      int i;
      uint8* ptr = (uint8*)object + sizeof(EveryObject);
      printf("data pointer: %p\n", ptr);
      printf("data: ");
      for (i = 0; i < data_size; i++) printf("%02x%c", *ptr++, i%16==15?'\n':' ');
      printf("\n");
    }
#endif

    nwos_crc32_calculate((uint8*)object + sizeof(EveryObject), data_size, computed);

    if (stored[0] != computed[0] || stored[1] != computed[1] || stored[2] != computed[2] || stored[3] != computed[3])
    {
	printf("object: %02x%02x%02x%02x - bad data checksum, ", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	printf("size %zd - ", data_size);
	printf("computed %02x%02x%02x%02x - ", computed[0], computed[1], computed[2], computed[3]);
	printf("stored %02x%02x%02x%02x\n", stored[0], stored[1], stored[2], stored[3]);

	printf("\n");
	{
	  int i;
	  for (i = 0; i < sizeof(EveryObject) + data_size; i++)
	  {
	      printf("%02x%c", object[i], (i % 16) == 15 ? '\n' : ' ');
	  }
	  printf("\n");
	}
	exit(1);
    }
}



/*****************************************/
/* Routines to read blocks, objects, etc */
/*****************************************/

#ifdef DEBUG_ENCRYPTION
uint8 save_unencrypted[FILE_BLOCK_SIZE];
uint8 save_add_random[FILE_BLOCK_SIZE];
uint8 save_mixed_up[FILE_BLOCK_SIZE];
uint8 save_encrypted[FILE_BLOCK_SIZE];
#endif

static bool read_block(ObjRef* ref, uint8 block[FILE_BLOCK_SIZE])
{
    char ref_str[64];
    off_t offset;
    int i;
    int lower;
    int upper;
    int file_desc;
    uint32 ref_hash;
    uint32 mid_hash;

    assert(!is_void_reference(ref));

    /*    printf("Read block: %02x%02x%02x%02x\n", ref->id[0], ref->id[1], ref->id[2], ref->id[3]); */


#if NO_DISK_ACCESS

    for (i = 0; i < 4096 && cache[i] != NULL; i++)
    {
	if (is_same_object(ref, (ObjRef*)&cache[i][4]))
	{
	    break;
	}
    }

    assert(i < 4096);

    if (cache[i] != NULL)
    {
	memcpy(block, cache[i], FILE_BLOCK_SIZE);
    }

    return cache[i] != NULL;

#else
    if (nwos_ref_to_word(ref) < RESERVED_PUBLIC_BLOCKS)
    {
	file_desc = public_file_desc;
	offset = (off_t) nwos_ref_to_word(ref) * FILE_BLOCK_SIZE;
    }
    else if (storage_index == NULL)     /* normal storage */
    {
	file_desc = obj_file_desc;
	offset = ref_to_offset(ref);
    }
    else
    {
	file_desc = obj_file_desc;

	lower = 1;
	upper = storage_count;

	ref_hash = nwos_hash_ref(ref);

	while (lower <= upper)
	{
	    i = (lower + upper) / 2;

	    mid_hash = nwos_hash_ref(&storage_index[i-1]);

	    if (mid_hash > ref_hash)
	    {
		upper = i - 1;
	    }
	    else if (mid_hash < ref_hash)
	    {
		lower = i + 1;
	    }
	    else
	    {
		i = i - 1;
		break;
	    }
	}

	assert(lower <= upper);

	assert(is_same_object(&storage_index[i], ref));

	offset = (off_t)i * FILE_BLOCK_SIZE;
    }

    if (lseek(file_desc, offset, SEEK_SET) < 0)
    {
	uint32 offset_upr = (uint32)(offset >> 32);
	uint32 offset_lwr = (uint32)offset;
	snprintf(ref_str, sizeof(ref_str), "read_block lseek ref:%02x%02x%02x%02x offset:%x%08x",
		ref->id[0], ref->id[1], ref->id[2], ref->id[3], offset_upr, offset_lwr);

	perror(ref_str);
	exit(1);
    }

    return read(file_desc, block, FILE_BLOCK_SIZE) == FILE_BLOCK_SIZE;

#endif
}


#ifdef DEBUG_ENCRYPTION
void compare_bufs(uint8 buf1[FILE_BLOCK_SIZE], uint8 buf2[FILE_BLOCK_SIZE], char* descr)
{
  int i;
  for (i = 8; i < FILE_BLOCK_SIZE; i++) if (buf1[i] != buf2[i]) break;
  if (i == FILE_BLOCK_SIZE)
    {
      printf("%s - OK\n", descr);
    }
  else
    {
      printf("%s - bad\n", descr);
      for (i = 0; i < FILE_BLOCK_SIZE; i++)
	{
	  printf("%02x-%02x%c", buf1[i], buf2[i], i%16==15 ? '\n' : ' ');
	}
    }
}
#endif


void nwos_read_object_from_disk_and_decrypt(ObjRef* ref, void* object, size_t size, uint8 ivec[IVEC_SIZE], uint8 seq_table[FILE_BLOCK_SIZE])
{
    uint8 buffer1[FILE_BLOCK_SIZE];
    uint8 buffer2[FILE_BLOCK_SIZE];
    uint8* ptrobj = (uint8*)object;
    int i;
    int j;

    assert(!is_void_reference(ref));

    read_block(ref, buffer2);

    assert(block_contains_object(buffer2, ref));

#ifdef DEBUG_ENCRYPTION
    compare_bufs(buffer2, save_encrypted, "Encrypted");
#endif

#ifdef DISABLE_SECURITY_FEATURES
    if (true)
#else
    if (nwos_reference_type(ref) == Public_Reference)
#endif
    {
	memcpy(object, buffer2, size);    /* just copy the buffer into the object */
    }
    else
    {
	/* copy the first 8 bytes directly, not encrypted */
	memcpy(buffer1, buffer2, 8);

	/* decrypt the remaining bytes */
	BF_cbc_encrypt((buffer2 + 8), (buffer1 + 8), (FILE_BLOCK_SIZE - 8), &blowfish_key, ivec, BF_DECRYPT);

#if 0
	printf("ivec: ");
	for (i = 0; i < 8; i++) printf("%02x ", ivec[i]);
	printf("\n");
#endif

#ifdef DEBUG_ENCRYPTION
	compare_bufs(buffer1, save_mixed_up, "Mixed up");
#endif

	assert(size <= FILE_BLOCK_SIZE);

	/* copy the first 8 bytes directly accross */
	for (i = 0; i < 8; i++) ptrobj[i] = buffer1[i];

	for (i = 8; i < size; i++)
	{
	    j = seq_table[i];    /* find where the next byte is stored */
	    ptrobj[i] = buffer1[j];    /* get this byte from there */
	}

    }

#ifdef DEBUG_ENCRYPTION
    compare_bufs(ptrobj, save_unencrypted, "Unencrypted");
#endif
}


#define SIZE_EXTRA_BLOCK_DATA (FILE_BLOCK_SIZE - sizeof(uint32) - sizeof(ObjRef) - sizeof(ObjRef))

typedef struct
{
    uint8  flags[4];
    ObjRef id;
    uint8  data[SIZE_EXTRA_BLOCK_DATA];
    ObjRef next_block;
} Extra_Data_Block;


void nwos_read_object_from_disk(ObjRef* ref, void* object, size_t size)
{
    EveryObject* header;
    uint8 ivec[IVEC_SIZE];
    uint8* ptr_obj;
    Extra_Data_Block extra;
    ObjRef next_ref;
    size_t bytes_remaining;
    int i;
    int seq = 0;

    assert(!is_void_reference(ref));

#if 0
    char log_msg[128];

    snprintf(log_msg, sizeof(log_msg), "nwos_read_object_from_disk - %s size: %d", path, size);
    nwos_log(log_msg);
#endif

    assert(size > sizeof(CommonHeader));  /* make sure this wasn't called to just get the header */

    memset(ivec, 0, sizeof(ivec));

    bytes_remaining = size;

    ptr_obj = object;

    nwos_read_object_from_disk_and_decrypt(ref, &extra, sizeof(extra), ivec, nwos_random_sequence[seq]);

    assert(sizeof(ObjRef) == 4);

    *ptr_obj++ = extra.flags[0];
    *ptr_obj++ = extra.flags[1];
    *ptr_obj++ = extra.flags[2];
    *ptr_obj++ = extra.flags[3];
    *ptr_obj++ = extra.id.id[0];
    *ptr_obj++ = extra.id.id[1];
    *ptr_obj++ = extra.id.id[2];
    *ptr_obj++ = extra.id.id[3];

    while (bytes_remaining > FILE_BLOCK_SIZE)
    {
	for (i = 0; i < SIZE_EXTRA_BLOCK_DATA; i++)
	{
	    *ptr_obj++ = extra.data[i];
	    bytes_remaining--;
	}

	/* save the reference because the extra block is going to get written over, we don't want it to change */
	copy_reference(&next_ref, &extra.next_block);

	seq++;

	nwos_read_object_from_disk_and_decrypt(&next_ref, &extra, sizeof(extra), ivec, nwos_random_sequence[seq]);
    }

    assert(bytes_remaining <= FILE_BLOCK_SIZE);

    for (i = 0; i < bytes_remaining - 8; i++) *ptr_obj++ = extra.data[i];

#if 0
    printf("read obj path: %s\n", path);
    printf("read obj pointer: %p\n", object);
    {
      int i;
      for (i = 0; i < size; i++) printf("%02x%c", *((uint8*)object+i), i%16==15?'\n':' ');
      printf("\n");
    }
#endif

    header = (EveryObject*) object;

    check_common_header(ref, &header->common);

    check_object_header(ref, &header->object, header->common.header_chksum);

    check_object_data(ref, (uint8*)object, size, header->common.data_chksum);
}


void nwos_read_variable_sized_object_from_disk(ObjRef* ref, void* object, size_t max_size, size_t (*size_function)(void*))
{
    EveryObject* header;
    uint8 ivec[IVEC_SIZE];
    uint8* ptr_obj;
    Extra_Data_Block extra;
    ObjRef next_ref;
    size_t bytes_remaining;
    int i;
    int seq = 0;
    size_t xyzzy;
    char msg[128];

    assert(!is_void_reference(ref));

#if 0
    char log_msg[128];

    snprintf(log_msg, sizeof(log_msg), "nwos_read_variable_size_object_from_disk - %s size: %d", path, size);
    nwos_log(log_msg);
#endif

    memset(ivec, 0, sizeof(ivec));

    ptr_obj = object;

    nwos_read_object_from_disk_and_decrypt(ref, &extra, sizeof(extra), ivec, nwos_random_sequence[seq]);

    assert(sizeof(ObjRef) == 4);

    *ptr_obj++ = extra.flags[0];
    *ptr_obj++ = extra.flags[1];
    *ptr_obj++ = extra.flags[2];
    *ptr_obj++ = extra.flags[3];
    *ptr_obj++ = extra.id.id[0];
    *ptr_obj++ = extra.id.id[1];
    *ptr_obj++ = extra.id.id[2];
    *ptr_obj++ = extra.id.id[3];

    bytes_remaining = (*size_function)(&extra);
    xyzzy = nwos_get_object_size(&extra);

    //    assert(xyzzy == 0 || bytes_remaining == xyzzy);
    if (xyzzy != 0 && bytes_remaining != xyzzy)
    {
	snprintf(msg, sizeof(msg), "Warning: size_function: %zd  get_object_size: %zd", bytes_remaining, xyzzy);
	nwos_log(msg);
	printf("%s\n", msg);
    }

    assert(bytes_remaining <= max_size);

    while (bytes_remaining > FILE_BLOCK_SIZE)
    {
	for (i = 0; i < SIZE_EXTRA_BLOCK_DATA; i++)
	{
	    *ptr_obj++ = extra.data[i];
	    bytes_remaining--;
	}

	/* save the reference because the extra block is going to get written over, we don't want it to change */
	copy_reference(&next_ref, &extra.next_block);

	seq++;

	nwos_read_object_from_disk_and_decrypt(&next_ref, &extra, sizeof(extra), ivec, nwos_random_sequence[seq]);
    }

    assert(bytes_remaining <= FILE_BLOCK_SIZE);

    for (i = 0; i < bytes_remaining - 8; i++) *ptr_obj++ = extra.data[i];

#if 0
    printf("read obj path: %s\n", path);
    printf("read obj pointer: %p\n", object);
    {
      int i;
      for (i = 0; i < size; i++) printf("%02x%c", *((uint8*)object+i), i%16==15?'\n':' ');
      printf("\n");
    }
#endif

    header = (EveryObject*) object;

    check_common_header(ref, &header->common);

    check_object_header(ref, &header->object, header->common.header_chksum);

    check_object_data(ref, (uint8*)object, (*size_function)(object), header->common.data_chksum);
}


void nwos_read_object_headers_from_disk(ObjRef* ref, EveryObject* header)
{
    uint8 ivec[IVEC_SIZE];

    assert(!is_void_reference(ref));

    memset(ivec, 0, sizeof(ivec));
    nwos_read_object_from_disk_and_decrypt(ref, header, sizeof(*header), ivec, nwos_random_sequence[0]);

#if 0
    printf("read header path: %s\n", path);
    printf("read header pointer: %p\n", header);
    {
      int i;
      for (i = 0; i < sizeof(EveryObject); i++) printf("%02x%c", *((uint8*)header+i), i%16==15?'\n':' ');
      printf("\n");
    }
#endif

    check_common_header(ref, &header->common);

    check_object_header(ref, &header->object, header->common.header_chksum);
}


/******************************************/
/* Routines to write blocks, objects, etc */
/******************************************/

void nwos_write_block(ObjRef* ref, uint8 block[FILE_BLOCK_SIZE])
{
    char log_msg[128];
    off_t offset;

#ifdef PUBLIC_MODE
    assert(nwos_reference_type(ref) == Public_Reference);
#else
    assert(nwos_reference_type(ref) != Public_Reference);
#endif

    assert(!is_void_reference(ref));
    assert((block[0] & 0x7f) == 0 && block[1] == 0 && block[2] == 0 && block[3] == 0);
    assert(is_same_object((ObjRef*)block + 1, ref));

    snprintf(log_msg, sizeof(log_msg), "write_block - ref:%02x%02x%02x%02x block:%08x",
	    ref->id[0], ref->id[1], ref->id[2], ref->id[3], nwos_hash_ref(ref));
    nwos_log(log_msg);

#if NO_DISK_ACCESS
{
    int i;
    for (i = 0; i < 4096 && cache[i] != NULL; i++)
    {
	if (is_same_object(ref, (ObjRef*)&cache[i][4]))
	{
	    printf("WARNING - rewritting the same block: %02x%02x%02x%02x\n", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	    break;
	}
    }

    assert(i < 4096);

    if (cache[i] == NULL)
    {
	cache[i] = malloc(FILE_BLOCK_SIZE);
	assert(cache[i] != NULL);
    }

    memcpy(cache[i], block, FILE_BLOCK_SIZE);
}
#else
    assert(storage_index == NULL);   /* at this time if there is a storage index it is read only */

    offset = ref_to_offset(ref);
    if (lseek(obj_file_desc, offset, SEEK_SET) < 0)
    {
	uint32 offset_upr = (uint32)(offset >> 32);
	uint32 offset_lwr = (uint32)offset;
	snprintf(log_msg, sizeof(log_msg), "write_block lseek ref:%02x%02x%02x%02x offset:%x%08x",
		ref->id[0], ref->id[1], ref->id[2], ref->id[3], offset_upr, offset_lwr);

	perror(log_msg);
	exit(1);
    }

    if (write(obj_file_desc, block, FILE_BLOCK_SIZE) != FILE_BLOCK_SIZE)
    {
	snprintf(log_msg, sizeof(log_msg), "write_block write ref:%02x%02x%02x%02x", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);

	perror(log_msg);
	exit(1);
    }

    modified = true;
#endif
}


void nwos_write_object_to_disk_and_encrypt(ObjRef* ref, void* object, size_t size, uint8 ivec[IVEC_SIZE], uint8 seq_table[FILE_BLOCK_SIZE])
{
    uint8 buffer1[FILE_BLOCK_SIZE];
    uint8 buffer2[FILE_BLOCK_SIZE];
    uint8* ptrobj = (uint8*)object;
    int i;
    int j;

    assert(!is_void_reference(ref));
    assert(size <= FILE_BLOCK_SIZE);

#ifdef DEBUG_ENCRYPTION
    memcpy(save_unencrypted, object, FILE_BLOCK_SIZE);
#endif

#ifdef DISABLE_SECURITY_FEATURES    /* always do the unencryped code */
    if (true)
#else
    if (nwos_reference_type(ref) == Public_Reference)
#endif
    {
	memset(buffer2, 0, FILE_BLOCK_SIZE);    /* just copy the object into the buffer */
	memcpy(buffer2, object, size);
    }
    else
    {
	/* copy the first 8 bytes directly accross */
	for (i = 0; i < 8; i++) buffer1[i] = ptrobj[i];

	/* fill any empty slots with random junk */
	if (size < FILE_BLOCK_SIZE)
	{
	    for (i = 8; i < FILE_BLOCK_SIZE; i++) buffer1[i] = random();
	}

	for (i = 8; i < size; i++)
	{
	    /* find where the next byte is stored */
	    j = seq_table[i];    /* find where the next byte is stored */
	    buffer1[j] = ptrobj[i];    /* save this byte there */
	}

#ifdef DEBUG_ENCRYPTION
	memcpy(save_mixed_up, buffer1, FILE_BLOCK_SIZE);
#endif

	/* copy the first 8 bytes straight across */
	memcpy(buffer2, buffer1, 8);

	/* and encrypt the remainder 8 bytes at a time */
	BF_cbc_encrypt((buffer1 + 8), (buffer2 + 8), (FILE_BLOCK_SIZE - 8), &blowfish_key, ivec, BF_ENCRYPT);
    }

#ifdef DEBUG_ENCRYPTION
    memcpy(save_encrypted, buffer2, FILE_BLOCK_SIZE);
#endif

    nwos_write_block(ref, buffer2);
}



void nwos_write_object_to_disk(ObjRef* ref, void* object, size_t size)
{
    uint8 ivec[IVEC_SIZE];
    uint8* ptr_obj;
    Extra_Data_Block extra;
    size_t bytes_remaining;
    int i;
    int seq = 0;
    uint8 block[FILE_BLOCK_SIZE];

    memset(ivec, 0, sizeof(ivec));


    assert(sizeof(extra) == FILE_BLOCK_SIZE);

#ifndef PUBLIC_MODE
    assert(nwos_block_used(ref));      /* assert bit in map was set by generate id */
#endif

    assert(!read_block(ref, block) || is_void_reference((ObjRef*)&block[1]));   /* make sure block is zero */

    ptr_obj = (uint8*) object;

    assert(ptr_obj[0] == 0 && ptr_obj[1] == 0 && ptr_obj[2] == 0 && ptr_obj[3] == 0);

    ptr_obj += 4;

    assert(is_same_object(ref, (ObjRef*)ptr_obj));

    ptr_obj += 4;

    bytes_remaining = size;

    extra.flags[0] = 0;
    extra.flags[1] = 0;
    extra.flags[2] = 0;
    extra.flags[3] = 0;

    copy_reference(&extra.id, ref);

    while (bytes_remaining > FILE_BLOCK_SIZE)
    {
	for (i = 0; i < SIZE_EXTRA_BLOCK_DATA; i++)
	{
	    extra.data[i] = *ptr_obj++;
	    bytes_remaining--;
	}

	nwos_generate_new_id(&extra.next_block);

	nwos_write_object_to_disk_and_encrypt(&extra.id, &extra, sizeof(extra), ivec, nwos_random_sequence[seq]);

	copy_reference(&extra.id, &extra.next_block);

	seq++;
    }

    assert(bytes_remaining <= FILE_BLOCK_SIZE);

    for (i = 0; i < bytes_remaining - 8; i++) extra.data[i] = *ptr_obj++;

    nwos_write_object_to_disk_and_encrypt(&extra.id, &extra, bytes_remaining, ivec, nwos_random_sequence[seq]);
}


void nwos_overwrite_object_to_disk(ObjRef* ref, void* object, size_t size)
{
    uint8 ivec[IVEC_SIZE];

    assert(nwos_object_exists(ref));

    memset(ivec, 0, sizeof(ivec));

    nwos_write_object_to_disk_and_encrypt(ref, object, size, ivec, nwos_random_sequence[0]);
}


void nwos_write_public_object_to_disk(ObjRef* ref, void* object, size_t size)
{
    uint8 block[FILE_BLOCK_SIZE];
//    int i;

    assert(size <= FILE_BLOCK_SIZE);
    assert((ref->id[0] & 0xF0) == 0);

    memset(block, 0, sizeof(block));

    memcpy(block, object, size);

    nwos_write_block(ref, block);

//    printf("write_public_object: %02x%02x%02x%02x", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
}


/***************/
/* Class stuff */
/***************/

void nwos_get_object_class(ObjRef* obj, ObjRef* object_class)
{
    EveryObject header;
    uint8 ivec[IVEC_SIZE];

    assert(!is_void_reference(obj));

    memset(ivec, 0, sizeof(ivec));

    nwos_read_object_from_disk_and_decrypt(obj, &header, sizeof(header), ivec, nwos_random_sequence[0]);

    check_common_header(obj, &header.common);

    /* if it is not a reference list check the header checksum */

    if (!is_same_object(&header.common.class_definition, &nwos_reference_list_class_ref))
    {
	check_object_header(obj, &header.object, header.common.header_chksum);
    }

    memcpy(object_class, &header.common.class_definition, sizeof(object_class));
}


void nwos_fill_in_common_header(CommonHeader* common, ObjRef* ref, ObjRef* class_definition_ref)
{
    memset(common, 0, sizeof(CommonHeader));
    memcpy(&common->id, ref, sizeof(ObjRef));
    nwos_get_time_stamp(common->creation_time);
    memcpy(&common->class_definition, class_definition_ref, sizeof(common->class_definition));
}



/*************************************************/
/* Remove and object (by writing zero's over it) */
/*************************************************/

void nwos_remove_object(ObjRef* ref)
{
    char log_msg[128];
    uint8 block[FILE_BLOCK_SIZE];
    off_t offset;

    assert(nwos_object_exists(ref));

    snprintf(log_msg, sizeof(log_msg), "nwos_remove_object - %02x%02x%02x%02x", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
    nwos_log(log_msg);

    memset(block, 0, sizeof(block));

    assert(sizeof(block) == FILE_BLOCK_SIZE);

    offset = ref_to_offset(ref);
    if (lseek(obj_file_desc, offset, SEEK_SET) < 0)
    {
	uint32 offset_upr = (uint32)(offset >> 32);
	uint32 offset_lwr = (uint32)offset;
	snprintf(log_msg, sizeof(log_msg), "remove_object lseek ref:%02x%02x%02x%02x offset:%x%08x",
		ref->id[0], ref->id[1], ref->id[2], ref->id[3], offset_upr, offset_lwr);

	perror(log_msg);
	exit(1);
    }

    if (write(obj_file_desc, block, FILE_BLOCK_SIZE) != FILE_BLOCK_SIZE)
    {
	snprintf(log_msg, sizeof(log_msg), "nwos_remove_object write ref:%02x%02x%02x%02x", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);

	perror(log_msg);
	exit(1);
    }

    nwos_clear_bit_in_map(nwos_hash_ref(ref));

    modified = true;

    assert(!nwos_object_exists(ref));
}

#if 0
void create_object(uint64 class_def, EveryObject* header)
{
    nwos_fill_in_common_header(&header->common, 'l', class_def);

    memset(&header->object, 0, sizeof(ObjectHeader));  /* zero it out */

    nwos_create_reference_list(&header->object.references);

    nwos_crc32_calculate((uint8*) &header->object, sizeof(ObjectHeader), header->common.header_chksum);
}
#endif



bool nwos_is_quit_command(char* command)
{
    if (strcasecmp(command, "quit") == 0) return true;
    if (strcasecmp(command, "quit now") == 0) return true;
    if (strcasecmp(command, "exit") == 0) return true;
    if (strcasecmp(command, "stop") == 0) return true;
    if (strcasecmp(command, "stop now") == 0) return true;
    if (strcasecmp(command, "done") == 0) return true;
    if (strcasecmp(command, "done now") == 0) return true;
    if (strcasecmp(command, "finish") == 0) return true;
    if (strcasecmp(command, "finish now") == 0) return true;
    if (strcasecmp(command, "finished") == 0) return true;
    if (strcasecmp(command, "terminate") == 0) return true;

    return false;
}


bool nwos_ask_yes_or_no(char* statement, char* question)
{
    char buffer[16];

    if (statement != NULL)
    {
	printf("%s.  %s? (y/n): ", statement, question);
    }
    else
    {
	printf("%s? (y/n): ", question);
    }

    while (1)
    {
	fflush(stdout);
	fgets(buffer, sizeof(buffer), stdin);
	printf("\n");

	if (strchr(buffer, '\n') == NULL)  /* too much input */
	{
	    while(strchr(buffer, '\n') == NULL) fgets(buffer, sizeof(buffer), stdin);
	    buffer[0] = '\0';
	}

	if (strcasecmp(buffer, "y\n") == 0 || strcasecmp(buffer, "yes\n") == 0) break;

	if (strcasecmp(buffer, "n\n") == 0 || strcasecmp(buffer, "no\n") == 0) break;

	printf("Please answer the question: %s (yes or no)\n", question);
    }

    return (tolower(buffer[0]) == 'y');
}


