--             This file is part of the New World OS project
--                   Copyright (C) 2006  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file COPYING.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: nwos_spelling_index.e,v $
-- Revision 1.1  2007/04/14 02:24:20  jsedwards
-- Move Fine (Eiffel) files to next_gen directory.
--
-- Revision 1.18  2006/11/11 12:29:19  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.17  2006/02/19 15:38:09  jsedwards
-- Added inheritance from NWOS_LETTER_OR_SPELLING_INDEX to allow the spelling
-- indexes to point back to either the first letter object or the previous
-- spelling index.  Changed the read and write routines to always save the
-- previous index as it should never be void.  Also changed the read and write
-- routines to skip the spelling object without a void (it is either there or
-- not).
--
-- Revision 1.16  2006/02/19 14:57:02  jsedwards
-- Changed attributes to routines that make sure the object is loaded before
-- returing object.
--
-- Revision 1.15  2006/02/19 02:51:23  jsedwards
-- Removed calls to load alphabet and spelling objects, these are done
-- automatically now.
--
-- Revision 1.14  2006/02/13 14:15:31  jsedwards
-- Changed to deal with hyphens and apostrophes in words.
--
-- Revision 1.13  2006/02/12 01:49:54  jsedwards
-- Wrapped print statements with 'debug'.
--
-- Revision 1.12  2006/02/11 22:17:24  jsedwards
-- Added more debugging print statements.
--
-- Revision 1.11  2006/02/11 21:11:26  jsedwards
-- Added void references when optional fields are missing.  Implemented
-- read_body_from_stream routine.  Commented out class invariant because
-- it fails when object is created with make_unresolved.
--
-- Revision 1.10  2006/02/10 13:55:34  jsedwards
-- Changed "load" in creation clause to "make_unresolved".
--
-- Revision 1.9  2006/02/09 13:06:49  jsedwards
-- Added reference parameter to "initialize header" call and removed "next
-- identifier" feature that is now in nwos_header.  Changed to the new
-- method of doing the class defintion stuff, with the "class reference"
-- and "class revision".
--
-- Revision 1.8  2006/02/08 14:24:04  jsedwards
-- Added "load" to creation clause, a dummy read_body_from_stream routine,
-- and class reference to class definition "make" call.
--
-- Revision 1.7  2006/02/06 13:43:43  jsedwards
-- Changed domain from English to Latin, because spellings in all latin based
-- languages should be allocated from a common place.
--
-- Revision 1.6  2006/02/05 17:01:29  jsedwards
-- Added "item", "put", and "put_spelling" features.
--
-- Revision 1.5  2006/02/05 14:41:38  jsedwards
-- Fix Id so it gets filled in correctly.
--
-- Revision 1.4  2006/02/05 14:18:51  jsedwards
-- Updated to new data and deferred class definition formats.
--
-- Revision 1.3  2006/02/02 02:05:44  jsedwards
-- Removed passing class reference to "initialize header".
--
-- Revision 1.2  2006/01/31 19:31:08  jsedwards
-- Added new kludgy class definition stuff.
--
-- Revision 1.1  2006/01/30 05:04:40  jsedwards
-- Initial version.
--


class NWOS_SPELLING_INDEX

inherit NWOS_LETTER_OR_SPELLING_INDEX


creation make, make_unresolved


feature

   alphabet: NWOS_ALPHABET is
      do
         load_if_not_already
         Result := alphabet_obj
      ensure
         Result /= Void
      end

   spelling: NWOS_SPELLING is   -- spelling at this point in the index tree
      do
         load_if_not_already
         Result := spelling_obj

         -- spelling can be void, so no ensure that it isn't
      end

   previous_index: NWOS_LETTER_OR_SPELLING_INDEX is
      do
         load_if_not_already
         Result := prev_index_obj
      ensure
         Result /= Void
      end

   next_index: ARRAY[NWOS_SPELLING_INDEX] is
      do
         load_if_not_already
         Result := next_index_obj
      ensure
         Result /= Void
      end


feature

   item(symbol: NWOS_SYMBOL): NWOS_SPELLING_INDEX is
         -- next index for given letter
      local
         letter: NWOS_LETTER
      do
         if alphabet.has(symbol) then
            letter ?= symbol  -- since the alphabet reports having this as a letter, by all rights it should be a letter
            check
               letter /= Void  -- verify it
            end
            Result := next_index.item(letter.index)
         elseif symbol.identifier.is_equal(Apostrophe_ref) then
            Result := next_index.item(alphabet.upper + 1)
         elseif symbol.identifier.is_equal(Hyphen_ref) then
            Result := next_index.item(alphabet.upper + 2)
         else  -- unknown symbol, we can't handle
            std_error.put_string("NWOS_SPELLING_INDEX.item unkown symbol: ")
            std_error.put_string(symbol.identifier.to_string)
            std_error.put_new_line
         end
      end

   put(new_index: NWOS_SPELLING_INDEX; symbol: NWOS_SYMBOL) is
         -- add a new index
      require
         item(symbol) = Void  -- don't allow writing over an existing one
      local
         letter: NWOS_LETTER
      do
         if alphabet.has(symbol) then
            letter ?= symbol  -- since the alphabet reports having this as a letter, by all rights it should be a letter
            check
               letter /= Void  -- verify it
            end
            next_index.put(new_index, letter.index)
         elseif symbol.identifier.is_equal(Apostrophe_ref) then
            next_index.put(new_index, alphabet.upper + 1)
         elseif symbol.identifier.is_equal(Hyphen_ref) then
            next_index.put(new_index, alphabet.upper + 2)
         else  -- unknown symbol, we can't handle
            std_error.put_string("NWOS_SPELLING_INDEX.put unkown symbol: ")
            std_error.put_string(symbol.identifier.to_string)
            std_error.put_new_line
         end
      end

   put_spelling(spell: NWOS_SPELLING) is
      require
         spell /= Void
      do
         check
            spelling_obj = Void  -- don't allow to be overwritten
         end

         spelling_obj := spell
      ensure
         spelling_obj = spell
      end

feature {NONE}

   write_body_to_stream(stream: BINARY_OUTPUT_STREAM) is
      local
         i: INTEGER
      do
         alphabet_obj.class_definition.identifier.write_to_stream(stream)
         alphabet_obj.identifier.write_to_stream(stream)

         if spelling_obj /= Void then   -- there is a spelling at this juncture
            spelling_obj.class_definition.identifier.write_to_stream(stream)
            spelling_obj.identifier.write_to_stream(stream)
         end

         -- this can be either the first letter or the previous spelling index
         prev_index_obj.class_definition.identifier.write_to_stream(stream)
         prev_index_obj.identifier.write_to_stream(stream)

         from
            i := next_index.lower
         variant
            next_index.upper - i
         until
            i > next_index.upper
         loop
            if next_index.item(i) = Void then
               Void_reference.write_to_stream(stream)
            else
               next_index.item(i).class_definition.identifier.write_to_stream(stream)
               next_index.item(i).identifier.write_to_stream(stream)
            end
            i := i + 1
         end
      end

   read_body_from_stream(stream: BINARY_INPUT_STREAM) is
      local
         index: NWOS_SPELLING_INDEX
         class_ref: NWOS_REFERENCE
         object_ref: NWOS_REFERENCE
         i: INTEGER
      do
         !!class_ref.make_from_stream(stream)
         !!object_ref.make_from_stream(stream)

         alphabet_obj ?= object_ref.get_object(class_ref)

         if alphabet_obj = Void then
            std_error.put_string("NWOS_SPELLING_INDEX.read_body_from_stream: alphabet failed to load%N")
            die_with_code(exit_failure_code)
         end

         !!class_ref.make_from_stream(stream)
debug
std_output.put_string("NWOS_SPELLING_INDEX.read_body_from_stream%N")
std_output.put_string("  spelling class: ")
std_output.put_string(class_ref.to_string)
std_output.put_new_line
end
         if class_ref.is_equal(Class_spelling_ref) then
            !!object_ref.make_from_stream(stream)
debug
std_output.put_string("  spelling object: ")
std_output.put_string(object_ref.to_string)
std_output.put_new_line
end
            spelling_obj ?= object_ref.get_object(class_ref)

            if spelling_obj = Void then
               std_error.put_string("NWOS_SPELLING_INDEX.read_body_from_stream: spelling failed to load%N")
               die_with_code(exit_failure_code)
            end

            !!class_ref.make_from_stream(stream)  -- read the next one because we consumed this one
         end

         !!object_ref.make_from_stream(stream)

         prev_index_obj ?= object_ref.get_object(class_ref)

         if prev_index_obj = Void then
            std_error.put_string("NWOS_SPELLING_INDEX.read_body_from_stream: previous_index failed to load%N")
            die_with_code(exit_failure_code)
         end

         !!next_index_obj.make(alphabet_obj.lower, alphabet_obj.upper)
debug
std_output.put_string("NWOS_SPELLING_INDEX.read_body_from_stream%N")
std_output.put_string(identifier.to_string)
std_output.put_string("  next_index upper ")
std_output.put_integer(next_index_obj.upper)
std_output.put_new_line
end
         from
            i := next_index_obj.lower
         variant
            next_index_obj.upper - i
         until
            i > next_index_obj.upper
         loop
            !!class_ref.make_from_stream(stream)
debug
std_output.put_string("NWOS_SPELLING_INDEX.read_body_from_stream%N")
std_output.put_string(identifier.to_string)
std_output.put_string("  next_index class ")
std_output.put_integer(i)
std_output.put_string(": ")
std_output.put_string(class_ref.to_string)
std_output.put_new_line
end
            if not class_ref.is_void then
debug
std_output.put_string("  next_index object: ")
std_output.put_string(object_ref.to_string)
std_output.put_new_line
end
               !!object_ref.make_from_stream(stream)

               index ?= object_ref.get_object(class_ref)

               if index = Void then
                  std_error.put_string("NWOS_SPELLING_INDEX.read_body_from_stream: next_index(")
                  std_error.put_integer(i)
                  std_error.put_string(") failed to load%N")
                  die_with_code(exit_failure_code)
               end

               next_index_obj.put(index, i)
            end
            i := i + 1
         end
      end

   make(letters: like alphabet; previous: like previous_index) is
      require
         letters /= Void
      do
         initialize_header(next_latin_identifier)

         alphabet_obj := letters

         spelling_obj := Void

         prev_index_obj := previous

         !!next_index_obj.make(alphabet_obj.lower, alphabet_obj.upper + 2)  -- allow room for apostrophes and hyphens
      end


feature {NONE}  -- actual storage

   alphabet_obj: like alphabet

   spelling_obj: like spelling

   prev_index_obj: like previous_index

   next_index_obj: like next_index


feature {NONE}  -- once again the kludgy class definition stuff

   class_revision: STRING is
      once
         !!Result.copy("$Revision: 1.1 $")
         Result.remove_prefix("$Revision: ")
         Result.remove_suffix(" $")
      end

   class_reference: NWOS_REFERENCE is
      do
         Result := Class_spelling_index_ref
      end


--invariant

--   alphabet /= Void

end

