/*             This file is part of the New World OS project
--                   Copyright (C) 2006  QRW Software
--             J. Scott Edwards - j.scott.edwards@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: prep_disk.c,v $
-- Revision 1.2  2006/10/26 01:51:28  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.5  2006/10/22 12:32:30  jsedwards
-- Added code to handle variable number of reserved public blocks.
--
-- Revision 1.1.2.4  2006/10/09 13:10:35  jsedwards
-- Change bit map size from 256 to 8192 (to improve speed, disk_usage program
-- took over an hour to run with 256 byte bit maps).
--
-- Revision 1.1.2.3  2006/10/07 22:28:40  jsedwards
-- Changed to write block bit map into the first block of every 2048 block
-- chunk.
--
-- Revision 1.1.2.2  2006/09/26 20:15:51  jsedwards
-- Added warning and verification before nuking the disk or partition.
--
-- Revision 1.1.2.1  2006/09/26 14:03:50  jsedwards
-- Kludged program to prepare a disk or partition for being used with
-- objectify.
--
*/

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "objectify_private.h"

/* Now there is a bit map of used blocks as the first block in every chunk */
/* 8192 * 8 = 65536 blocks * 256 bytes per block = 16777216 bytes in chunk */

#define BUFFER_SIZE CHUNK_SIZE


void print_usage(char* name)
{
    fprintf(stderr, "\nusage: %s [number of reserved public chunks]\n\n", name);
    fprintf(stderr, "  a chunk is %u blocks (%d bytes),", BLOCKS_IN_CHUNK, CHUNK_SIZE);
    fprintf(stderr, " the default is %u chunks\n\n", (NUM_PUBLIC_IDS / BLOCKS_IN_CHUNK) - 1);
}


int main(int argc, char* argv[])
{
    int obj_fd;
    int blocks;
    int bytes_written;
    int i;
    uint8 *buf;
    char answer[8];
    char *p;
    uint32 reserved_public_chunks = 0;
    uint32 reserved_public_blocks = NUM_PUBLIC_IDS; /* public blocks not specified default to number of public ids */

    assert(NUM_PUBLIC_IDS % BLOCKS_IN_CHUNK == 0);

    if (argc > 2)
    {
	print_usage(argv[0]);
	exit(1);
    }
    else if (argc == 2)
    {
	for (p = argv[1]; *p != '\0'; p++)
	{
	    if (!isdigit(*p))
	    {
		fprintf(stderr, "\nInvalid number of chunks: %s\n", argv[1]);
		print_usage(argv[0]);
		exit(1);
	    }

	    reserved_public_chunks = reserved_public_chunks * 10 + (*p - '0');

	    if (reserved_public_chunks > NUM_PUBLIC_IDS - 1)
	    {
		fprintf(stderr, "\nThat is too many chunks, maximum useful is: %u\n\n",
			NUM_PUBLIC_IDS / BLOCKS_IN_CHUNK);
		exit(1);
	    }

	    reserved_public_blocks = (reserved_public_chunks + 1) * BLOCKS_IN_CHUNK;
	}
    }
   
    printf("\nreserved_public_blocks: %u\n\n", reserved_public_blocks);

    printf("\nWARNING: all of the contents of %s will be ERASED!!!\n\n", DEFAULT_FILE);

    printf("Do you want to continue? (enter `yes' to erase %s) ", DEFAULT_FILE);

    fflush(stdout);

    fgets(answer, sizeof(answer), stdin);

    printf("\n");

    if (strcmp(answer, "yes\n") != 0)
    {
	exit(1);
    }

    buf = malloc(BUFFER_SIZE);

    if (buf == NULL)
    {
	perror("allocating buffer");
	exit(1);
    }

    obj_fd = open(DEFAULT_FILE, O_WRONLY);    /* don't need large file if not seeking? */

    if (obj_fd < 0)
    {
	perror(DEFAULT_FILE);
	exit(1);
    }

    for (i = 1; i < BUFFER_SIZE; i++) buf[i] = 0;

    /* mark off 32 bits for the bit map itself */
    buf[0] = 0xff;
    buf[1] = 0xff;
    buf[2] = 0xff;
    buf[3] = 0xff;

    blocks = 0;
    bytes_written = BUFFER_SIZE;

    while (bytes_written == BUFFER_SIZE)
    {
	bytes_written = write(obj_fd, buf, BUFFER_SIZE);

	if (bytes_written % 512 != 0)
	{
	    fprintf(stderr, "bytes written not a multiple of 512: %d\n", bytes_written);
	    close(obj_fd);
	    exit(1);
	}

	blocks += bytes_written / 256;

	if (bytes_written < BUFFER_SIZE || blocks % 4194304 == 0)
	{
	    printf("%d - %d\n", (blocks / 4194304), blocks);
	}
    }

    if (errno != 0)
    {
	perror("after last write");
    }

    if (lseek(obj_fd, 0, SEEK_SET) < 0)
    {
	perror("lseek 0");
	exit(1);
    }
    
    memcpy(&buf[0], MAGIC_NUMBER, 4);
    memcpy(&buf[4], VERSION_STRING, 4);

    buf[8]  = reserved_public_blocks >> 24;
    buf[9]  = reserved_public_blocks >> 16;
    buf[10] = reserved_public_blocks >> 8;
    buf[11] = reserved_public_blocks;

    buf[12] = blocks >> 24;
    buf[13] = blocks >> 16;
    buf[14] = blocks >> 8;
    buf[15] = blocks;

    for (i = 0; i < 16; i++) printf("%02x", buf[i]);
    printf("\n");

    if (write(obj_fd, buf, 16) != 16)
    {
	perror(DEFAULT_FILE);
	exit(1);
    }

    if (close(obj_fd) != 0)
    {
	perror(DEFAULT_FILE);
	exit(1);
    }

    return 0;
}

